package clinical.utils;

import gnu.trove.TIntHashSet;
import gnu.trove.TIntIterator;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import clinical.web.vo.HeaderFieldType;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class AFNIBrikReader {
	private int width;
	private int height;
	private int noSlices;
	private String brikFile;
	private boolean swapBytes = false;
	private int brikDataType = FLOAT;
	public final static int BYTE = 0;
	public final static int SHORT = 1;
	public final static int FLOAT = 3;

	public AFNIBrikReader(String brikFile) throws IOException {
		this.brikFile = brikFile;
		String headFile = brikFile.replaceFirst("\\.BRIK", ".HEAD");

		AFNIHeaderReader headerReader = new AFNIHeaderReader(headFile);
		List<HeaderFieldType> hftList = new ArrayList<HeaderFieldType>(5);
		hftList.add(new HeaderFieldType("BRICK_TYPES", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("DATASET_DIMENSIONS",
				HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("BYTEORDER_STRING",
				HeaderFieldType.STRING));

		headerReader.readHeader();
		headerReader.extractFields(hftList);

		Field<Integer> dimField = headerReader
				.getIntField("DATASET_DIMENSIONS");
		List<Integer> values = dimField.getValues();
		Assertion.assertTrue(values.size() >= 3);
		this.width = values.get(0);
		this.height = values.get(1);
		this.noSlices = values.get(2);

		Field<String> byteOrderField = headerReader
				.getStringField("BYTEORDER_STRING");
		if (byteOrderField != null
				&& byteOrderField.getValue().equals("LSB_FIRST")) {
			swapBytes = true;
		}

		Field<Integer> brikTypeField = headerReader.getIntField("BRICK_TYPES");
		this.brikDataType = brikTypeField.getValue().intValue();
	}

	public float[] load() throws IOException {
		if (brikDataType != FLOAT && brikDataType != SHORT
				&& brikDataType != BYTE) {
			throw new RuntimeException(
					"Unsupported data type for AFNI Brik (code:" + brikDataType
							+ ")!");
		}
		int noVoxels = width * height * noSlices;
		float[] data = new float[noVoxels];
		DataInputStream in = null;
		try {
			in = new DataInputStream(new BufferedInputStream(
					new FileInputStream(brikFile), 4096));
			for (int i = 0; i < noVoxels; i++) {
				if (brikDataType == FLOAT) {
					int intValue = in.readInt();
					if (swapBytes) {
						intValue = swapInt(intValue);
					}
					float value = Float.intBitsToFloat(intValue);
					data[i] = value;
				} else if (brikDataType == SHORT) {
					short shortValue = in.readShort();
					if (swapBytes) {
						shortValue = swapShort(shortValue);
					}
					data[i] = shortValue;
				} else if (brikDataType == BYTE) {
					int byteValue = in.readUnsignedByte();
					data[i] = byteValue;
				}
			}
		} finally {
			FileUtils.close(in);
		}
		return data;
	}

	public static void showStats(float[] data) {
		float min = Float.POSITIVE_INFINITY;
		float max = Float.NEGATIVE_INFINITY;
		for (float val : data) {
			if (max < val) {
				max = val;
			}
			if (min > val) {
				min = val;
			}
		}
		System.out.println("noVoxels:" + data.length);
		System.out.println("min:" + min + " max:" + max);
	}

	public static TIntHashSet getUniqueMaskValues(float[] data) {
		TIntHashSet maskValSet = new TIntHashSet();
		for (float val : data) {
			int intVal = (int) val;
			if (intVal == 0) {
				continue;
			}
			if (!maskValSet.contains(intVal)) {
				maskValSet.add(intVal);
			}
		}
		return maskValSet;
	}

	public static TIntHashSet union(TIntHashSet set1, TIntHashSet set2) {
		TIntHashSet unionSet = new TIntHashSet(Math.max(set1.size(),
				set2.size()));
		for (TIntIterator it = set1.iterator(); it.hasNext();) {
			unionSet.add(it.next());
		}
		for (TIntIterator it = set2.iterator(); it.hasNext();) {
			int val = it.next();
			if (!unionSet.contains(val)) {
				unionSet.add(val);
			}
		}
		return unionSet;
	}

	public static int[] getSortedROINoList(TIntHashSet unionSet) {
		int[] roiNoArr = new int[unionSet.size()];
		int i = 0;
		for (TIntIterator it = unionSet.iterator(); it.hasNext();) {
			roiNoArr[i++] = it.next();
		}
		Arrays.sort(roiNoArr);
		return roiNoArr;
	}

	public static int swapInt(int value) {
		int a = (value >> 0) & 0xff;
		int b = (value >> 8) & 0xff;
		int c = (value >> 16) & 0xff;
		int d = (value >> 24) & 0xff;
		return a << 24 | b << 16 | c << 8 | d << 0;
	}

	public static short swapShort(short value) {
		int a = value & 0xff;
		int b = (value >> 8) & 0xff;

		return (short) (a << 8 | b << 0);
	}

	public static void main(String[] args) throws Exception {
		int val = 2000;
		System.out.println("int:" + val + " byteSwapped:" + swapInt(val));

		String dir = "/home/bozyurt/download_cache/upload_4ga/caffeine_roi_masks_1342647562392_files/Masks/111120/100812/post/";

		AFNIBrikReader brikReader = new AFNIBrikReader(dir
				+ "20_aparc+aseg_+reg+rs+orig.BRIK");

		float[] data = brikReader.load();

		showStats(data);
		TIntHashSet maskValues = getUniqueMaskValues(data);
		System.out.println("Unique ROI Numbers (" + maskValues.size() + ")");
		for (TIntIterator it = maskValues.iterator(); it.hasNext();) {
			int maskValue = it.next();
			System.out.println(maskValue);
		}
	}
}
