package clinical.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import com.pixelmed.dicom.DicomFileUtilities;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: FileUtils.java 794 2013-04-09 23:34:08Z bozyurt $
 */
public class FileUtils {
	protected FileUtils() {
	}

	public static void copyFile(String sourceFile, String destFile)
			throws FileNotFoundException, IOException {

		BufferedInputStream bin = null;
		BufferedOutputStream bout = null;
		try {
			bin = new BufferedInputStream(new FileInputStream(sourceFile));
			bout = new BufferedOutputStream(new FileOutputStream(destFile));
			byte[] buffer = new byte[4096];
			int readBytes = 0;
			while ((readBytes = bin.read(buffer)) != -1) {
				bout.write(buffer, 0, readBytes);
			}

		} finally {
			if (bin != null)
				try {
					bin.close();
				} catch (Exception x) {
				}
			if (bout != null)
				try {
					bout.close();
				} catch (Exception x) {
				}
		}
	}

	public static void copyDirRecursively(File srcDir, File destDir)
			throws Exception {
		Assertion.assertTrue(srcDir.isDirectory());
		destDir.mkdirs();
		if (!destDir.isDirectory()) {
			throw new Exception("Cannot make destination dir:" + destDir);
		}

		List<File> srcFiles = getAllFilesUnderDir(srcDir.getAbsolutePath());
		for (File srcFile : srcFiles) {
			String relPath = srcFile.getAbsolutePath();
			relPath = relPath.substring(srcDir.getAbsolutePath().length());
			if (relPath.startsWith("/")) {
				relPath = relPath.substring(1);
			}
			File destFile = new File(destDir, relPath);
			destFile.getParentFile().mkdirs();
			copyFile(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
		}
	}

	public static void copyContent(String sourceFile, OutputStream destOut)
			throws FileNotFoundException, IOException {
		BufferedInputStream bin = null;
		try {
			bin = new BufferedInputStream(new FileInputStream(sourceFile));
			byte[] buffer = new byte[4096];
			int readBytes = 0;
			while ((readBytes = bin.read(buffer)) != -1) {
				destOut.write(buffer, 0, readBytes);
			}

		} finally {
			if (bin != null)
				try {
					bin.close();
				} catch (Exception x) {
				}
		}
	}

	public static String loadTextFile(String filename) throws IOException {
		BufferedReader in = null;
		StringBuilder buf = new StringBuilder(4096);
		try {
			in = new BufferedReader(new FileReader(filename));
			String line = null;

			while ((line = in.readLine()) != null) {
				buf.append(line);
				buf.append('\n');
			}
			return buf.toString();
		} finally {
			close(in);
		}
	}

	public static List<String> readLines(String filename) throws IOException {
		BufferedReader in = null;
		List<String> lines = new LinkedList<String>();
		try {
			in = new BufferedReader(new FileReader(filename));
			String line = null;

			while ((line = in.readLine()) != null) {
				lines.add(line);
			}
			return lines;
		} finally {
			close(in);
		}
	}

	public static void save2File(String content, String filename)
			throws IOException {
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(filename));
			out.write(content);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public static Element loadXML(String xmlFile) throws Exception {
		SAXBuilder builder = new SAXBuilder(false);
		Document doc = builder.build(xmlFile);
		return doc.getRootElement();
	}

	public static Element loadXML(Reader reader) throws Exception {
		SAXBuilder builder = new SAXBuilder(false);
		Document doc = builder.build(reader);
		return doc.getRootElement();
	}

	public static void saveXML(Element rootElem, String xmlFile)
			throws Exception {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(xmlFile));
			xout.output(rootElem, out);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public static void saveXML(Element rootElem, OutputStream outStream)
			throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(rootElem, outStream);
		} finally {
			close(outStream);
		}
	}

	public static void saveXML(Element rootElem, Writer out) throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(rootElem, out);
		} finally {
			close(out);
		}
	}

	public static void saveXML(Document doc, Writer out) throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(doc, out);
		} finally {
			close(out);
		}
	}

	public static void close(InputStream in) {
		if (in != null) {
			try {
				in.close();
			} catch (Exception x) {
			}
		}
	}

	public static void close(OutputStream out) {
		if (out != null) {
			try {
				out.close();
			} catch (Exception x) {
			}
		}
	}

	public static void close(Writer out) {
		if (out != null) {
			try {
				out.close();
			} catch (Exception x) {
			}
		}
	}

	public static void close(Reader in) {
		if (in != null) {
			try {
				in.close();
			} catch (Exception x) {
			}
		}
	}

	public static List<File> getAllFilesMatching(File dir, FilenameFilter filter) {
		List<File> filteredList = new ArrayList<File>();
		getAllFilesMatching(dir, filter, filteredList);
		return filteredList;
	}

	static void getAllFilesMatching(File dir, FilenameFilter filter,
			List<File> filteredList) {
		File[] files = dir.listFiles();
		for (File f : files) {
			if (f.isDirectory()) {
				getAllFilesMatching(f, filter, filteredList);
			} else {
				if (filter.accept(f.getParentFile(), f.getName())) {
					filteredList.add(f);
				}
			}
		}
	}

	public static void findAndDeleteLeafPaths(File parent, List<File> dirList) {
		if (parent.isFile()) {
			parent.delete();
		} else if (parent.exists()) {
			File[] files = parent.listFiles();
			if (files == null || files.length == 0) {
				dirList.add(parent);
			} else {
				for (int i = 0; i < files.length; i++) {
					findAndDeleteLeafPaths(files[i], dirList);
				}
			}
		}
	}

	public static void deleteRecursively(File dir) {
		if (dir.isFile()) {
			dir.delete();
		} else if (dir.exists()) {
			File[] files = dir.listFiles();
			if (files != null && files.length > 0) {
				for (int i = 0; i < files.length; i++) {
					if (files[i].isDirectory()) {
						deleteRecursively(files[i]);
					} else {
						files[i].delete();
					}
				}
			}
			dir.delete();
		}
	}

	public static void deleteSubdirs(String rootDir, String path) {
		File f = new File(path);
		List<File> dirList = new ArrayList<File>();
		if (f.isDirectory()) {
			findAndDeleteLeafPaths(f, dirList);
		}

		for (Iterator<File> it = dirList.iterator(); it.hasNext();) {
			File dir = it.next();
			do {
				if (dir.isDirectory()) {
					// don't care if deletion is not successful make an attempt
					dir.delete();
				}
				dir = dir.getParentFile();
			} while (!dir.getPath().equals(rootDir));
		}
	}

	public static String getRelativePath(String rootDir, String path) {
		int idx = path.indexOf(rootDir);
		assert (idx == 0);
		String relPath = path.substring(rootDir.length());
		relPath = relPath.replaceFirst("^\\/+", "");
		return relPath;
	}

	public static String getTopDir(String rootDir, File path) {
		String relPath = getRelativePath(rootDir, path.getAbsolutePath());
		int idx = relPath.indexOf("/");
		if (idx == -1) {
			return null;
		}
		return relPath.substring(0, idx);
	}

	public static boolean buildLocalPath(String rootDir, String relativePath) {
		File parent = new File(rootDir);
		Assertion.assertTrue(parent.isDirectory());
		File f = new File(parent, relativePath);
		f.getParentFile().mkdirs();
		return f.getParentFile().isDirectory();
	}

	/**
	 * build a full path by combining a root directory path and a relative path
	 * making sure the there is only one <code>/</code> between those.
	 * 
	 * @param rootDir
	 * @param relativePath
	 * @return
	 */
	public static String createFullPath(String rootDir, String relativePath) {
		StringBuffer buf = new StringBuffer();
		buf.append(rootDir);
		if (rootDir.endsWith("/")) {
			if (relativePath.startsWith("/")) {
				buf.append(relativePath.substring(1));
			} else {
				buf.append(relativePath);
			}
		} else {
			if (relativePath.startsWith("/")) {
				buf.append(relativePath);
			} else {
				buf.append('/').append(relativePath);
			}
		}
		return buf.toString();
	}

	public static List<File> getAllUnderDir(String rootPath) {
		File dir = new File(rootPath);
		if (!dir.isDirectory()) {
			return new ArrayList<File>(0);
		}
		List<File> paths = new ArrayList<File>();
		prepBranchAll(dir, paths);
		return paths;
	}

	protected static void prepBranchAll(File parentDir, List<File> paths) {
		if (parentDir == null)
			return;
		File[] files = parentDir.listFiles();
		if (files == null)
			return;
		for (int i = 0; i < files.length; i++) {
			if (files[i].isFile()) {
				paths.add(files[i]);
			} else {
				paths.add(files[i]);
				prepBranchAll(files[i], paths);
			}
		}
	}

	/**
	 * 
	 * @param rootPath
	 * @return
	 * @throws Exception
	 */
	public static List<File> getAllFilesUnderDir(String rootPath) {
		List<File> paths = new ArrayList<File>();
		File dir = new File(rootPath);
		prepBranch(dir, paths);
		return paths;
	}

	protected static void prepBranch(File parentDir, List<File> paths) {
		if (parentDir == null)
			return;
		File[] files = parentDir.listFiles();
		if (files == null)
			return;
		for (int i = 0; i < files.length; i++) {
			if (files[i].isFile()) {
				paths.add(files[i]);
			} else {
				prepBranch(files[i], paths);
			}
		}
	}

	/**
	 * Given a (composite) object, and a filename, serialize the state of the
	 * object to the given filename.
	 * 
	 * @param o
	 *            the object to serialize
	 * @param filename
	 *            the filename to which the serialized object is written
	 * @throws IOException
	 */
	public static void serialize(Object o, String filename) throws IOException {
		assert (o != null);
		ObjectOutputStream out = null;
		try {
			out = new ObjectOutputStream(new BufferedOutputStream(
					new FileOutputStream(filename), 4096));
			out.writeObject(o);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {
				}
			}
		}
	}

	/**
	 * Resurrects a serialized object.
	 * 
	 * @param filename
	 *            the filename from which the serialized object is read
	 * @return the deserialized object.
	 * @throws IOException
	 * @throws java.lang.ClassNotFoundException
	 */
	public static Object deserialize(String filename) throws IOException,
			ClassNotFoundException {
		assert (filename != null && new File(filename).isFile());
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(new BufferedInputStream(
					new FileInputStream(filename), 4096));
			Object o = in.readObject();
			return o;
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public static long getTotalSizeUnderDir(String relPath, String localRootDir) {
		File f = new File(localRootDir, relPath);
		return getTotalSizeUnderDir(f);
	}

	public static long getTotalSizeUnderDir(File dir) {
		List<File> allFilesUnderDir = FileUtils.getAllFilesUnderDir(dir
				.getAbsolutePath());
		long totSize = 0;
		for (File file : allFilesUnderDir) {
			if (file.isFile())
				totSize += file.length();
		}
		return totSize;
	}

	public static String getFileSuffix(String filename) {
		if (filename.indexOf('.') == -1) {
			return null;
		}
		int idx = filename.lastIndexOf('.');
		return filename.substring(idx);
	}

	public static String getFullFileSuffix(String filename) {
		if (filename.indexOf('.') == -1) {
			return null;
		}
		int idx = filename.indexOf('.');
		return filename.substring(idx);
	}

	public static String getBasename(String filename) {
		String basename = filename.replaceFirst("\\.([^\\.])+$", "");
		return basename;
	}

	public static String getBasenameNoSuffix(String filename) {
		String basename = filename.replaceFirst("\\.([^\\.])+$", "");
		while (basename.indexOf('.') != -1) {
			basename = basename.replaceFirst("\\.([^\\.])+$", "");
		}
		return basename;
	}

	public static String toExperimentName(String name) {
		return name.replaceAll("\\s+", "_");
	}

	public static String findCommonRoot(List<String> files) {
		List<String[]> pathPartsList = new ArrayList<String[]>(files.size());

		String[] shortestPathParts = null;
		int minPathLen = Integer.MAX_VALUE;

		for (String path : files) {
			String[] pathParts = path.split("\\/+");
			pathPartsList.add(pathParts);

			if (pathParts.length < minPathLen) {
				shortestPathParts = pathParts;
				minPathLen = pathParts.length;
			}
		}
		int idx = minPathLen - 1;
		String commonPart = shortestPathParts[idx];
		String refCommonPath = buildUnixPath(shortestPathParts, minPathLen);
		while (idx >= 0) {
			boolean ok = true;
			for (String[] pathParts : pathPartsList) {
				if (pathParts[idx].equals(commonPart)) {
					ok = false;
					break;
				} else {
					String prefix = buildUnixPath(pathParts, idx + 1);
					if (!refCommonPath.equals(prefix)) {
						ok = false;
						break;
					}
				}
			}
			if (ok) {
				break;
			}
			refCommonPath = buildUnixPath(shortestPathParts, idx);
			--idx;
		}
		if (idx < 0) {
			return null;
		}

		return refCommonPath;
	}

	static String buildUnixPath(String[] pathParts, int uptoIdx) {
		StringBuilder sb = new StringBuilder();
		int len = (uptoIdx > 0) ? uptoIdx : pathParts.length;
		for (int i = 0; i < len; i++) {
			sb.append(pathParts[i]);
			if ((i + 1) < len) {
				sb.append('/');
			}
		}
		return sb.toString();
	}

	public static String findCommonRootOld(List<String> files) {
		File[] fileArr = new File[files.size()];
		for (int i = 0; i < fileArr.length; i++) {
			fileArr[i] = new File(files.get(i));
		}

		File parent = fileArr[0].getParentFile();
		while (parent != null) {
			boolean ok = true;
			for (File f : fileArr) {
				if (!f.getParentFile().equals(parent)) {
					ok = false;
					break;
				}
			}
			if (ok) {
				break;
			} else {
				parent = parent.getParentFile();
				if (parent == null) {
					break;
				}
				for (int i = 0; i < fileArr.length; i++) {
					fileArr[i] = fileArr[i].getParentFile();
				}

			}
		}
		if (parent == null)
			return null;
		return parent.getAbsolutePath();
	}

	public static boolean moveTo(File src, File dest) {
		boolean ok = src.renameTo(dest);
		if (!ok) {
			try {
				copyFile(src.getAbsolutePath(), dest.getAbsolutePath());
				if (dest.exists()) {
					src.delete();
					ok = true;
				}
			} catch (IOException e) {
				e.printStackTrace();
				return false;
			}
		}
		return ok;
	}

	public static boolean isPFile(String filePath) {
		if (filePath.endsWith(".7")) {
			return true;
		}
		// TODO other checks
		return false;
	}

	public static boolean isDICOMFile(String filePath) {
		return DicomFileUtilities.isDicomOrAcrNemaFile(filePath);
	}

	public static boolean isAFNIHeader(String filePath) {
		if (filePath.endsWith(".HEAD")) {
			return true;
		}
		// TODO other checks
		return false;
	}

	public static boolean isAFNIBrick(String filePath) {
		if (filePath.endsWith(".BRIK")) {
			return true;
		}
		// TODO other checks
		return false;
	}
	
	public static boolean isAnalyzeHeader(File filePath) {
		if (filePath.getName().endsWith(".hdr") ) {
			if ( filePath.length() == 348) {
				return true;
			}
		}
		return false;
	}
	
	public static boolean isAnalyzeImage(File filePath) {
		if (filePath.getName().endsWith(".img") ) {
				return true;
		}
		// TODO other checks?
		return false;
	}
	

	public static boolean isNiftiFile(String filePath) {
		if (filePath.endsWith(".nii") || filePath.endsWith(".nii.gz")) {
			return true;
		}
		return false;
	}

}
