package clinical.web.common;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;

import clinical.web.Constants;
import clinical.web.common.query.MediatedQueryWorker;
import clinical.web.common.query.Operator;
import clinical.web.common.query.ParallelQueryResult;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.SubjectDerivedDataValueSummary;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: MediatedQueryHelper.java 366 2011-05-05 20:06:27Z bozyurt $
 */
public class MediatedQueryHelper {
	HttpSession session;
	private static Log log = LogFactory.getLog(MediatedQueryHelper.class);

	public MediatedQueryHelper(HttpSession session) {
		this.session = session;
	}

	protected static boolean isEnded(MediatedQueryWorker mqw) {
		return (mqw.getStatus() == MediatedQueryWorker.FINISHED || mqw
				.getStatus() == MediatedQueryWorker.ERROR);
	}

	protected static boolean endedWithError(MediatedQueryWorker mqw) {
		return (mqw.getStatus() == MediatedQueryWorker.ERROR);
	}

	public Map<?, ?> prepareAssesmentMapping() throws BaseException {
		Map<?, ?> siteAsMap = (Map<?, ?>) session
				.getAttribute(Constants.SITE_ASSESSMENT_MAP_KEY);
		if (siteAsMap != null) {
			return siteAsMap;
		}
		Map<?, ?> workerQueue = (Map<?, ?>) session
				.getAttribute(Constants.WORKER_THREAD_QUEUE_KEY);
		MediatedQueryWorker asNameWorker = (MediatedQueryWorker) workerQueue
				.get("assessment_names");
		MediatedQueryWorker scoreNameWorker = (MediatedQueryWorker) workerQueue
				.get("score_names");

		while (true) {
			if (isEnded(asNameWorker)) {
				break;
			}
			try {
				synchronized (asNameWorker.getLock()) {
					if (isEnded(asNameWorker))
						break;
					asNameWorker.getLock().wait();
				}
			} catch (InterruptedException ie) {
				if (isEnded(asNameWorker))
					break;
			}
		}

		// wait till the score name mediated query worker ends
		while (true) {
			if (isEnded(scoreNameWorker)) {
				break;
			}
			try {
				synchronized (scoreNameWorker.getLock()) {
					if (isEnded(scoreNameWorker))
						break;
					scoreNameWorker.getLock().wait();
				}
			} catch (InterruptedException ie) {
				if (isEnded(scoreNameWorker))
					break;
			}
		}

		workerQueue.remove("assessment_names");
		workerQueue.remove("score_names");

		if (endedWithError(asNameWorker) || endedWithError(scoreNameWorker)) {
			BaseException be = new BaseException();

			if (endedWithError(asNameWorker))
				be.addException(asNameWorker.getException());
			if (endedWithError(scoreNameWorker))
				be.addException(scoreNameWorker.getException());
			throw be;
		}

		return siteAsMap;
	}

	public static Map<Integer, List<AssessmentMapping>> prepareAssesmentMapping(
			String mappingFile) throws BaseException {
		Map<Integer, List<AssessmentMapping>> siteAsMap = new HashMap<Integer, List<AssessmentMapping>>(
				17);
		System.out.println("**** checking mapping file " + mappingFile);
		if (!new java.io.File(mappingFile).exists()) {
			return null;
		}
		SAXBuilder builder = new SAXBuilder(false);
		try {
			Document doc = builder.build(new java.io.File(mappingFile));
			System.out.println("**** loading mapping file " + mappingFile);
			loadMappings(doc.getRootElement(), siteAsMap);
		} catch (Exception x) {
			throw new BaseException(x);
		}
		return siteAsMap;
	}

	protected static void loadMappings(Element rootElement,
			Map<Integer, List<AssessmentMapping>> siteAsMap) {
		List<?> children = rootElement.getChildren("as-map");
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element asMapElem = (Element) iter.next();
			String name = asMapElem.getAttributeValue("name");
			int asID = Integer.parseInt(asMapElem.getAttributeValue("id"));
			AssessmentMapping am = new AssessmentMapping(name, asID);
			if (asMapElem.getAttribute("first-db") != null) {
				am.setFirstDB(asMapElem.getAttributeValue("first-db"));
			}
			if (asMapElem.getAttribute("second-db") != null) {
				am.setSecondDB(asMapElem.getAttributeValue("second-db"));
			}

			Integer key = new Integer(asID);
			List<AssessmentMapping> amList = siteAsMap.get(key);
			if (amList == null) {
				amList = new LinkedList<AssessmentMapping>();
				siteAsMap.put(key, amList);
			}
			if (!amList.contains(am))
				amList.add(am);

			Element scoreMapsElem = asMapElem.getChild("score-maps");
			List<?> scoreElems = scoreMapsElem.getChildren("score");
			for (Iterator<?> it2 = scoreElems.iterator(); it2.hasNext();) {
				Element scoreElem = (Element) it2.next();
				String scoreName = scoreElem.getAttributeValue("name");
				String mapName = scoreElem.getAttributeValue("map");
				int mapAsID = Integer.parseInt(scoreElem
						.getAttributeValue("map-as-id"));

				AssessmentMapping.ScoreMap scoreMap = am.addScoreMap(scoreName,
						mapName, mapAsID);
				Element tmElem = scoreElem.getChild("type-map");
				if (tmElem != null) {
					loadTypeMap(scoreMap, tmElem);
				}
			}

		}
	}

	protected static void loadTypeMap(AssessmentMapping.ScoreMap scoreMap,
			Element tmElem) {
		String secondDBType = tmElem.getAttributeValue("second-db-type");
		String firstDBType = tmElem.getAttributeValue("first-db-type");
		List<?> values = tmElem.getChildren("value");
		AssessmentMapping.TypeMap typeMap = new AssessmentMapping.TypeMap(
				firstDBType, secondDBType);
		scoreMap.setTypeMap(typeMap);

		for (Iterator<?> iter = values.iterator(); iter.hasNext();) {
			Element ve = (Element) iter.next();
			typeMap.addValueMap(convertType(ve.getAttributeValue("first-db"),
					firstDBType), convertType(
					ve.getAttributeValue("second-db"), secondDBType));
		}
	}

	protected static Object convertType(String value, String type) {
		if (type.equals("varchar"))
			return value;
		if (type.equals("integer"))
			return new Integer(value);
		if (type.equals("float"))
			return new Float(value);
		if (type.equals("boolean"))
			return new Boolean(value);
		if (type.equals("timestamp"))
			return Timestamp.valueOf(value);
		else
			return new RuntimeException("Unknown type: " + type);
	}

	public static ParallelQueryResult doForkAndJoin(Operator asRoot,
			List<AssessmentSelectionInfo> asList,
			Map<Integer, List<AssessmentMapping>> siteAsMap,
			List<SubCorticalVarInfo> derivedDataInfos,
			Operator derivedDataQueryOpRoot, String dbID) throws Exception {
		MediatedQueryWorker worker1 = new MediatedQueryWorker("as_query",
				MediatedQueryWorker.ASSESSMENT_QUERY, dbID);
		worker1.addArgument(asRoot);
		worker1.addArgument(asList);
		worker1.addArgument(siteAsMap);

		MediatedQueryWorker worker2 = new MediatedQueryWorker("subcort_query",
				MediatedQueryWorker.SUBCORTICAL_QUERY, dbID);
		worker2.addArgument(derivedDataInfos);
		worker2.addArgument(derivedDataQueryOpRoot);

		Thread thread1 = new Thread(worker1);
		Thread thread2 = new Thread(worker2);
		thread1.setDaemon(true);
		thread1.setPriority(Thread.NORM_PRIORITY - 1);
		thread2.setDaemon(true);
		thread2.setPriority(Thread.NORM_PRIORITY - 1);
		thread1.start();
		thread2.start();
		try {
			thread1.join();
			thread2.join();
		} catch (InterruptedException ie) {
			log.error(ie);
		}

		if (endedWithError(worker1) || endedWithError(worker2)) {
			BaseException be = new BaseException();

			if (endedWithError(worker1))
				be.addException(worker1.getException());
			if (endedWithError(worker2))
				be.addException(worker2.getException());
			throw be;
		}
		// filter the subcortical query results
		log.info("parallel queries are finished.");
		List<?> summaryList = (List<?>) worker1.getResult();
		List<?> derivedDataSummaryList = (List<?>) worker2.getResult();

		List<String> uniqueSubjectIds = getUniqueSubjectIds(summaryList);
		Map<String, String> sidMap = new HashMap<String, String>();
		for (Iterator<String> iter = uniqueSubjectIds.iterator(); iter
				.hasNext();) {
			String subjectID = iter.next();
			sidMap.put(subjectID, subjectID);
		}
		List<SubjectDerivedDataValueSummary> filteredDerivedDataSummaryList = new LinkedList<SubjectDerivedDataValueSummary>();
		for (Iterator<?> iter = derivedDataSummaryList.iterator(); iter
				.hasNext();) {
			SubjectDerivedDataValueSummary sddvs = (SubjectDerivedDataValueSummary) iter
					.next();
			if (sidMap.get(sddvs.getSubjectID()) != null) {
				filteredDerivedDataSummaryList.add(sddvs);
			}
		}
		derivedDataSummaryList = null;
		List<SubjectAsScoreValueSummary> sasvsList = new ArrayList<SubjectAsScoreValueSummary>(
				summaryList.size());
		for (Iterator<?> it = summaryList.iterator(); it.hasNext();) {
			SubjectAsScoreValueSummary sasvs = (SubjectAsScoreValueSummary) it
					.next();
			sasvsList.add(sasvs);
		}

		return new ParallelQueryResult(sasvsList,
				filteredDerivedDataSummaryList);

	}

	public static List<String> getUniqueSubjectIds(List<?> summaryList) {
		List<String> sids = null;
		Map<String, String> sidMap = new TreeMap<String, String>();
		for (Iterator<?> iter = summaryList.iterator(); iter.hasNext();) {
			SubjectAsScoreValueSummary sasv = (SubjectAsScoreValueSummary) iter
					.next();
			if (sidMap.get(sasv.getSubjectID()) == null) {
				sidMap.put(sasv.getSubjectID(), sasv.getSubjectID());
			}
		}
		sids = new ArrayList<String>(sidMap.size());
		for (Iterator<String> iter = sidMap.values().iterator(); iter.hasNext();) {
			String sid = iter.next();
			sids.add(sid);
		}
		return sids;
	}

}
