package clinical.web.common.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.struts.util.LabelValueBean;
import org.jdom.Element;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * abstract class representing the common part of the model for the part of the
 * query view of a query builder screen. A user query is composed of combination
 * of query parts representing a search predicate.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractQueryPartInfo.java,v 1.2 2006/05/02 01:00:39 bozyurt
 *          Exp $
 */
public abstract class AbstractQueryPartInfo implements java.io.Serializable {
   private static final long serialVersionUID = 1L;
   protected Collection<LabelValueBean> logicalOps;
   protected int operator;
   protected String rhs = "*";
   protected String lowBound;
   protected String uppBound;
   protected boolean useBetween = false;
   /** if true, user will select from a list of values */
   protected boolean hasEnums = false;
   /** list of possible values for RHS */
   protected Collection<LabelValueBean> rhsEnums;
   /** index of the selected value from the enumeration drop down */
   protected int rhsIdx;

   /** maps a score code label to its corresponding score code */
   protected Map<String, String> scoreCodeMap;
   /** 0 based index of the query part in the full filter */
   protected int partIdx = -1;

   public AbstractQueryPartInfo() {}

   public AbstractQueryPartInfo(AbstractQueryPartInfo other) {
      this.operator = other.operator;
      this.rhs = other.rhs;
      this.lowBound = other.lowBound;
      this.uppBound = other.uppBound;
      this.useBetween = other.useBetween;
      this.hasEnums = other.hasEnums;
      this.rhsEnums = other.rhsEnums;
      this.rhsIdx = other.rhsIdx;
      this.scoreCodeMap = other.scoreCodeMap;
   }

   protected void prepareNumericLogicalOps() {
      logicalOps = new ArrayList<LabelValueBean>(6);
      logicalOps.add(new LabelValueBean("=", String
            .valueOf(SearchPredicate.EQUAL)));
      logicalOps.add(new LabelValueBean(">", String
            .valueOf(SearchPredicate.GREATER)));
      logicalOps.add(new LabelValueBean("<", String
            .valueOf(SearchPredicate.LESS)));
      logicalOps.add(new LabelValueBean("<>", String
            .valueOf(SearchPredicate.NOT_EQUAL)));
      logicalOps.add(new LabelValueBean(">=", String
            .valueOf(SearchPredicate.GREATER_EQUAL)));
      logicalOps.add(new LabelValueBean("<=", String
            .valueOf(SearchPredicate.LESS_EQUAL)));
   }

   protected void prepareVarcharLogicalOps() {
      logicalOps = new ArrayList<LabelValueBean>(4);
      logicalOps.add(new LabelValueBean("=", String
            .valueOf(SearchPredicate.EQUAL)));
      logicalOps.add(new LabelValueBean("starts with", String
            .valueOf(SearchPredicate.STARTS_WITH)));
      logicalOps.add(new LabelValueBean("ends with", String
            .valueOf(SearchPredicate.ENDS_WITH)));
      logicalOps.add(new LabelValueBean("any of", String
            .valueOf(SearchPredicate.ANY)));
   }
   
   protected void prepareDateLogicalOps() {
	   logicalOps = new ArrayList<LabelValueBean>(2);
	   logicalOps.add(new LabelValueBean(">=", String
	            .valueOf(SearchPredicate.GREATER_EQUAL)));
	   logicalOps.add(new LabelValueBean("<=", String
	            .valueOf(SearchPredicate.LESS_EQUAL)));
   }

   public void prepareRHSEnumerations(List<String> possibleValues) {
      if (possibleValues == null || possibleValues.isEmpty()) {
         return;
      }

      setHasEnums(true);
      setUseBetween(false);
      setOperator(SearchPredicate.EQUAL);

      rhsEnums = new ArrayList<LabelValueBean>(possibleValues.size() + 1);
      rhsEnums.add(new LabelValueBean("*", "0"));
      int count = 1;
      for (String value : possibleValues) {
         rhsEnums.add(new LabelValueBean(value, String.valueOf(count++)));
      }
   }

   public void prepareRHSEnumerations(List<String> possibleValues,
         List<String> scoreCodes) {
      if (possibleValues == null || possibleValues.isEmpty()) {
         return;
      }

      setHasEnums(true);
      setUseBetween(false);
      setOperator(SearchPredicate.EQUAL);

      rhsEnums = new ArrayList<LabelValueBean>(possibleValues.size() + 1);
      rhsEnums.add(new LabelValueBean("*", "0"));
      int i = 1;
      scoreCodeMap = new HashMap<String, String>(11);
      Iterator<String> it = scoreCodes.iterator();

      for (String value : possibleValues) {
         String scoreCode = (String) it.next();
         scoreCodeMap.put(value, scoreCode);
         rhsEnums.add(new LabelValueBean(value, String.valueOf(i++)));
      }
   }

   public String findScoreCode(String scoreLabel) {
      return scoreCodeMap.get(scoreLabel);
   }

   public void setRhsIdx(String label) {
      int i = 0;
      for (Iterator<LabelValueBean> iter = rhsEnums.iterator(); iter.hasNext();) {
         LabelValueBean lvb = iter.next();
         if (lvb.getLabel().equals(label)) {
            this.rhsIdx = i;
            break;
         } else {
            String scoreCode = findScoreCode(lvb.getLabel());
            if (scoreCode != null && scoreCode.equals(label)) {
               this.rhsIdx = i;
               System.out.println("rhsIdx is set to " + rhsIdx);
               break;
            }
         }
         ++i;
      }
   }

   public void setRhsIdx(int newRhsIdx) {
      this.rhsIdx = newRhsIdx;
      LabelValueBean lvb = ((List<LabelValueBean>) rhsEnums).get(rhsIdx);
      String scoreCode = findScoreCode(lvb.getLabel());
      if (scoreCode == null) {
         this.rhs = lvb.getLabel();
      } else {
         this.rhs = scoreCode;
      }
   }

   public int getRhsIdx() {
      return this.rhsIdx;
   }

   public String getRhsEnums(int rhsIdx) {
      String valueStr = String.valueOf(rhsIdx);
      for (Iterator<LabelValueBean> iter = rhsEnums.iterator(); iter.hasNext();) {
         LabelValueBean lvb = iter.next();
         if (lvb.getValue().equals(valueStr))
            return lvb.getLabel();
      }
      return null;
   }

   public Collection<LabelValueBean> getRhsEnums() {
      return rhsEnums;
   }

   public String getLogicalOps(int operator) {
      String opStr = String.valueOf(operator);
      if (logicalOps == null)
         return null;
      for (Iterator<LabelValueBean> it = logicalOps.iterator(); it.hasNext();) {
         LabelValueBean lvb = it.next();
         if (lvb.getValue().equals(opStr))
            return lvb.getLabel();
      }
      return null;
   }

   public Collection<LabelValueBean> getLogicalOps() {
      return logicalOps;
   }

   // ---------------------- setters --------------
   public void setOperator(int newOperator) {
      this.operator = newOperator;
   }

   public void setRhs(String newRhs) {
      this.rhs = newRhs;
   }

   public void setLowBound(String newLowBound) {
      this.lowBound = newLowBound;
   }

   public void setUppBound(String newUppBound) {
      this.uppBound = newUppBound;
   }

   public void setUseBetween(boolean newUseBetween) {
      this.useBetween = newUseBetween;
   }

   public void setHasEnums(boolean newHasEnums) {
      this.hasEnums = newHasEnums;
   }

   // ---------------------- getters --------------
   public int getOperator() {
      return this.operator;
   }

   public String getRhs() {
      return this.rhs;
   }

   public String getLowBound() {
      return this.lowBound;
   }

   public String getUppBound() {
      return this.uppBound;
   }

   public boolean getUseBetween() {
      return this.useBetween;
   }

   public boolean getHasEnums() {
      return this.hasEnums;
   }

   public Element toXML() {
      Element elem = new Element("query-part-info");
      elem.setAttribute("useBetween", String.valueOf(useBetween));

      elem.setAttribute("hasEnums", String.valueOf(hasEnums));

      elem.setAttribute("operator", String.valueOf(operator));
      if (rhs != null)
         elem.setAttribute("rhs", rhs);
      if (lowBound != null && lowBound.trim().length() == 0)
         elem.setAttribute("lowBound", lowBound.trim());
      if (uppBound != null && uppBound.trim().length() == 0)
         elem.setAttribute("uppBound", uppBound.trim());

      if (rhsEnums != null && !rhsEnums.isEmpty()) {
         Element enumsElem = new Element("rhs-enums");
         elem.addContent(enumsElem);
         for (Iterator<LabelValueBean> iter = rhsEnums.iterator(); iter
               .hasNext();) {
            LabelValueBean lvb = iter.next();
            // don't persist the wildcard
            if (lvb.getLabel().equals("*"))
               continue;
            Element e = new Element("enum");
            e.setAttribute("value", lvb.getLabel());
            enumsElem.addContent(e);
         }
      }
      if (partIdx >= 0) {
    	  elem.setAttribute("partIdx", String.valueOf(partIdx));
      }
      return elem;
   }

   public JSONObject toJSON(JSONObject js) throws JSONException {
      js.put("useBetween", useBetween);
      js.put("hasEnums", hasEnums);
      js.put("operator", operator);
      if (rhs != null)
         js.put("rhs", rhs);
      if (lowBound != null && lowBound.trim().length() == 0)
         js.put("lowBound", lowBound.trim());
      if (uppBound != null && uppBound.trim().length() == 0)
         js.put("uppBound", uppBound.trim());
      if (rhsEnums != null && !rhsEnums.isEmpty()) {
         JSONArray jsArr = new JSONArray();
         for (Iterator<LabelValueBean> iter = rhsEnums.iterator(); iter
               .hasNext();) {
            LabelValueBean lvb = iter.next();
            // don't persist the wildcard
            if (lvb.getLabel().equals("*"))
               continue;
            jsArr.put(lvb.getLabel());
         }
         js.put("rhsEnums", jsArr);
      }
      if (partIdx >= 0) {
    	  js.put("partIdx", partIdx);
      }
      return js;
   }

public int getPartIdx() {
	return partIdx;
}

public void setPartIdx(int partIdx) {
	this.partIdx = partIdx;
}

}