package clinical.web.common.query;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.web.DBUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractQueryProcessor.java,v 1.4 2007/11/22 02:10:14 bozyurt
 *          Exp $
 */
public abstract class AbstractQueryProcessor implements IQueryProcessor {
	/**
	 * The class of the value object for which a query will be build and
	 * optionally executed.
	 */
	protected Class<?> voClass;

	/**
	 * holds the search predicate list used to build the where clause of the SQL
	 * query
	 */
	protected SearchPredicateList spList;

	/**
	 * provides explicit information about the methods, properties, events, etc,
	 * of the value object bean.
	 */
	protected BeanInfo bi;
	protected SimpleDateFormat df = new SimpleDateFormat("dd-MMM-yy");

	private static Log log = LogFactory.getLog(AbstractQueryProcessor.class);

	public AbstractQueryProcessor(Class<?> valueObjectClass,
			SearchPredicateList spList) throws Exception {
		this.voClass = valueObjectClass;
		this.spList = spList;
		bi = Introspector.getBeanInfo(voClass, java.lang.Object.class);
	}

	protected abstract String getTableName();

	public List<?> doQuery(Connection con, String query) throws SQLException,
			InstantiationException, IllegalAccessException,
			InvocationTargetException {
		List<Object> results = new LinkedList<Object>();
		java.sql.Statement st = null;

		Map<String, PropertyDescriptor> propMap = new HashMap<String, PropertyDescriptor>(
				19);
		PropertyDescriptor[] pds = bi.getPropertyDescriptors();
		for (int i = 0; i < pds.length; i++) {
			propMap.put(pds[i].getName().toLowerCase(), pds[i]);
		}

		Method[] setters = null;
		try {
			st = con.createStatement();
			ResultSet rs = st.executeQuery(query);
			ResultSetMetaData rsmd = rs.getMetaData();
			setters = new Method[rsmd.getColumnCount()];
			int[] colTypes = new int[rsmd.getColumnCount()];
			for (int i = 0; i < rsmd.getColumnCount(); i++) {
				String dbColName = rsmd.getColumnName(i + 1).toLowerCase();
				PropertyDescriptor pd = propMap.get(dbColName);
				if (pd == null) {
					dbColName = removeChar(dbColName, '_');
					pd = propMap.get(dbColName);
				}
				setters[i] = pd.getWriteMethod();
				colTypes[i] = rsmd.getColumnType(i + 1);
			}

			while (rs.next()) {
				Object vo = voClass.newInstance();
				for (int i = 0; i < rsmd.getColumnCount(); i++) {
					Object value = rs.getObject(i + 1);

					invokeSetter(setters[i], vo, value, colTypes[i]);
				}
				results.add(vo);
			}
			rs.close();
		} finally {
			DBUtils.close(st);
		}
		return results;
	}

	protected abstract void invokeSetter(Method m, Object vo, Object value,
			int colType) throws SQLException, InvocationTargetException,
			IllegalArgumentException, IllegalAccessException;

	public String buildQuery(String dbType) throws Exception {
		StringBuffer query = new StringBuffer(256);
		Map<String, PropertyDescriptor> propMap = new HashMap<String, PropertyDescriptor>(
				19);
		PropertyDescriptor[] pds = bi.getPropertyDescriptors();
		for (int i = 0; i < pds.length; i++) {
			if (log.isDebugEnabled()) {
				log.debug(pds[i].getName());
			}
			propMap.put(pds[i].getName().toLowerCase(), pds[i]);
		}

		String tableName = getTableName();

		query.append("select ");
		for (int i = 0; i < pds.length; i++) {
			if (pds[i].getReadMethod() == null) {
				continue;
			}
			String columnName = toDBColumnName(pds[i].getReadMethod().getName());
			if (log.isDebugEnabled()) {
				log.debug("columnName=" + columnName + " propertyName="
						+ pds[i].getName());
			}
			query.append(columnName.toUpperCase());
			if ((i + 1) < pds.length)
				query.append(',');
		}
		query.append(" from ").append(tableName).append(' ');

		boolean first = true;
		for (Iterator<SearchPredicateList.SearchPredicateInfo> iter = spList
				.iterator(); iter.hasNext();) {
			SearchPredicateList.SearchPredicateInfo spi = iter.next();
			String attr = (String) spi.getSearchPredicate().getAttribute();
			if (propMap.get(attr.toLowerCase()) != null) {
				if (first)
					query.append("WHERE ");
				else {
					if (spi.getLogicOp() == SearchPredicateList.AND)
						query.append(" AND ");
					else if (spi.getLogicOp() == SearchPredicateList.OR)
						query.append(" OR ");
				}
				// String colName = getDBColumnName(tableName,
				// attr.toLowerCase()); // orig
				String colName = toDBColumnName(attr);
				if (log.isDebugEnabled()) {
					log.debug(">>colName=" + colName + "attr=" + attr
							+ "tableName=" + tableName);
				}
				query.append(buildPredicate(spi.getSearchPredicate(), colName));

				first = false;
			}
		}
		return query.toString();
	}	

	public static String removeChar(String s, char ch) {
		StringBuffer buf = new StringBuffer(s);
		StringBuffer obuf = new StringBuffer();
		for (int i = 0; i < buf.length(); i++) {
			if (buf.charAt(i) != ch) {
				obuf.append(buf.charAt(i));
			}
		}
		return obuf.toString();
	}
	

	protected String toDBColumnName(String getMethodName) {
		String pn = getMethodName;
		pn = pn.replaceFirst("^(get|is)", "");
		StringBuffer buf = new StringBuffer();

		char[] carr = pn.toCharArray();
		for (int i = 0; i < carr.length; i++) {
			if (Character.isUpperCase(carr[i])) {
				if (i > 0) {
					buf.append('_');
				}
				buf.append(Character.toLowerCase(carr[i]));
			} else {
				buf.append(carr[i]);
			}
		}

		return buf.toString();
	}

	public static String getBaseClassName(String fullClassName) {
		int idx = fullClassName.lastIndexOf('.');
		if (idx != -1) {
			return fullClassName.substring(idx + 1);
		}
		return fullClassName;
	}

	protected String buildPredicate(SearchPredicate sp, String colName) {
		StringBuffer buf = new StringBuffer();
		buf.append("( ");
		buf.append(colName);
		int op = sp.getOperator();

		if (sp.getValue() == null
				|| sp.getValue().toString().trim().equals("*")) {
			buf.append(" is not null ) ");
			return buf.toString();
		}
		buf.append(QueryUtils.getRelationalOperator(op));

		if (op == SearchPredicate.BETWEEN) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			buf.append(range.getLowBound()).append(" AND ").append(
					range.getUppBound());
		} else if (op == SearchPredicate.STARTS_WITH) {
			buf.append("'").append(sp.getValue()).append("%' ");
		} else if (op == SearchPredicate.ENDS_WITH) {
			buf.append("'%").append(sp.getValue()).append("' ");
		} else {
			if (sp.getType() == SearchPredicate.STRING) {
				buf.append("'").append(sp.getValue()).append("' ");
			} else if (sp.getType() == SearchPredicate.DATE) {
				preparePredicateDateValue(sp, buf);
			} else {
				buf.append(sp.getValue()).append(' ');
			}
		}
		buf.append(") ");

		return buf.toString();
	}

	protected static String getFullyQualifedClassName(String voName) {
		if (isFullyQualifiedClass(voName)) {
			return voName;
		}
		StringBuffer buf = new StringBuffer();
		buf.append("clinical.server.vo.").append(voName);
		return buf.toString();
	}

	protected static boolean isFullyQualifiedClass(String className) {
		return className.indexOf('.') != -1;
	}

	protected abstract void preparePredicateDateValue(SearchPredicate sp,
			StringBuffer buf);

}
