package clinical.web.helpers;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;

import clinical.server.vo.ConfParams;
import clinical.server.vo.ConfParamsGroup;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.MinimalServiceFactory;
import clinical.web.services.AppConfigService;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AppConfigHelper.java 366 2011-05-05 20:06:27Z bozyurt $
 */
public class AppConfigHelper {

	private AppConfigHelper() {}

	public static String toJavaVarName(String confPropsName) {
		StringBuffer buf = new StringBuffer();
		char[] cpnArr = confPropsName.toCharArray();
		boolean needsCap = false;
		for (int i = 0; i < cpnArr.length; i++) {
			if (!needsCap && cpnArr[i] != '.') {
				buf.append(cpnArr[i]);
			} else if (cpnArr[i] == '.') {
				needsCap = true;
			} else if (needsCap) {
				if (cpnArr[i] != '.') {
					buf.append(Character.toUpperCase(cpnArr[i]));
					needsCap = false;
				}
			}
		}
		return buf.toString();
	}

	public static String toConfPropsName(String javaVarName) {
		StringBuffer buf = new StringBuffer();
		char[] jvnArr = javaVarName.toCharArray();
		for (int i = 0; i < jvnArr.length; i++) {
			if (Character.isUpperCase(jvnArr[i])) {
				buf.append('.').append(Character.toLowerCase(jvnArr[i]));
			} else {
				buf.append(jvnArr[i]);
			}
		}
		return buf.toString();
	}

	public static int getConfPropsIndexNumber(String confPropsName) {
		int idx = confPropsName.lastIndexOf('.');
		if (idx == -1)
			return -1;
		String suffix = confPropsName.substring(idx + 1);
		return GenUtils.toInt(suffix, -1);
	}

	public static Map<String, PropertyDescriptor> getPropertyDescriptorMap(
			Class<?> clazz) throws ClassNotFoundException, IntrospectionException {
		BeanInfo bi = Introspector.getBeanInfo(clazz);
		PropertyDescriptor[] descriptors = (PropertyDescriptor[]) bi
				.getPropertyDescriptors().clone();
		Map<String, PropertyDescriptor> descriptorsMap = new HashMap<String, PropertyDescriptor>(
				19);
		for (int i = 0; i < descriptors.length; i++) {
			descriptorsMap.put(descriptors[i].getName(), descriptors[i]);
		}
		return descriptorsMap;
	}

	public static void setProperty(Object object, String value,
			PropertyDescriptor pd) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException {

		Method m = pd.getWriteMethod();
		m.invoke(object, new Object[] { value });
	}

	public static String getProperty(Object object, PropertyDescriptor pd)
			throws IntrospectionException, IllegalAccessException,
			InvocationTargetException {
		Method m = pd.getReadMethod();
		return (String) m.invoke(object, new Object[0]);
	}

	/**
	 * populates the database with default configuration parameter groups and
	 * parameter values.
	 * 
	 * @param xmlConfFile
	 *           HID web app default Xml configuration file (eg
	 *           clinical-conf.xml)
	 * @throws Exception
	 */
	public static void confParamsFromXml2DB(String xmlConfFile) throws Exception {
		InputStream is = AppConfigHelper.class.getClassLoader()
				.getResourceAsStream(xmlConfFile);
		SAXBuilder builder = null;
		builder = new SAXBuilder(false);

		Document doc = builder.build(is);
		Element root = doc.getRootElement();
		Map<String, Group> groupMap = new HashMap<String, Group>(7);
		List<Param> paramList = new ArrayList<Param>(20);
		List<?> groupList = root.getChild("groups").getChildren("group");
		for (Iterator<?> it = groupList.iterator(); it.hasNext();) {
			Element gEl = (Element) it.next();
			Group group = new Group(gEl.getAttributeValue("id"), gEl
					.getAttributeValue("name"));
			groupMap.put(group.id, group);
			if (gEl.getChild("description") != null) {
				group.description = gEl.getChildTextTrim("description");
			}
		}

		List<?> peList = root.getChild("params").getChildren("param");
		for (Iterator<?> it = peList.iterator(); it.hasNext();) {
			Element pel = (Element) it.next();
			Param p = new Param(pel.getAttributeValue("name"), pel
					.getAttributeValue("value"), pel.getAttributeValue("type"));
			String gid = pel.getAttributeValue("gid");
			Group g = groupMap.get(gid);
			p.group = g;
			if (pel.getChild("description") != null) {
				p.description = pel.getChildTextTrim("description");
			}
			paramList.add(p);
		}

		MinimalServiceFactory.setMimimalOpMode(true);
		ConnectionSupportMixin csm = null;
		try {
			csm = new ConnectionSupportMixin("users.xml");

			csm.startup();
			AppConfigService acs = AppConfigService.getInstance(csm
					.getDbPoolService(), csm.getDbID());

			for (Group g : groupMap.values()) {
				ConfParamsGroup pg = acs.getParamGroup(g.name);
				if (pg == null) {
					acs.addParameterGroup(g.name, g.description);
				}
			}

			for (Param p : paramList) {
				ConfParams cp = acs.getParam(p.name);

				if (cp == null) {
					String value = p.value;
					if (value == null || value.trim().length() == 0) {
						value = " ";
					}
					acs.addParameter(p.name, value, p.type, p.description,
							p.group.name);
				} else if (cp.getValue() == null
						|| (p.value.trim().length() > 0 && !cp.getValue().equals(
								p.value))) {
					System.out.println("updating " + p);
					acs.updateParameter(p.name, p.value);
				}
			}
			System.out
					.println("finished populating HID with default application configuration data.");
		} finally {
			if (csm != null)
				csm.shutdown();
		}
	}

	public static class Group {
		String id, name;
		String description;

		public Group(String id, String name) {
			this.id = id;
			this.name = name;
		}
	}// ;

	public static class Param {
		String name, value;
		String type;
		Group group;
		String description;

		public Param(String name, String value, String type) {
			this.name = name;
			this.value = value;
			this.type = type;
		}

		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.append("Param::[");
			sb.append("name:").append(name);
			sb.append(",value:").append(value);
			sb.append(']');
			return sb.toString();
		}
	}// ;

	public static void main(String[] args) throws Exception {

		AppConfigHelper.confParamsFromXml2DB("clinical-conf.xml");
	}
}
