package clinical.web.helpers;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.json.JSONArray;
import org.json.JSONObject;

import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Assessmentscorecode;
import clinical.server.vo.Brainsegmentdata;
import clinical.server.vo.Experiment;
import clinical.server.vo.Storedquery;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.IAssessmentService;
import clinical.web.IDerivedDataService;
import clinical.web.ServiceFactory;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.IAuthorizationService.PrivilegeLabel;
import clinical.web.common.query.QueryPartInfo;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.exception.BaseException;
import clinical.web.forms.AsQueryBuilderForm;
import clinical.web.forms.StoredQuerySelector;
import clinical.web.vo.RangeInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: QueryWizardHelper.java,v 1.9.2.1 2008/07/10 23:56:00 bozyurt
 *          Exp $
 */
public class QueryWizardHelper {
	private static Log log = LogFactory.getLog(QueryWizardHelper.class);

	public QueryWizardHelper() {
	}

	public String previousPage(AsQueryBuilderForm queryForm, String currentState) {
		if (currentState.equals(Constants.SEL_SCORE)) {
			return Constants.SEL_ASSESSMENT;
		} else if (currentState.equals(Constants.SEL_DERIVED)) {
			return Constants.SEL_SCORE;
		} else if (currentState.equals(Constants.COLLECT_QUERY)) {
			if (hasSubcorticalVariables(queryForm)) {
				return Constants.SEL_DERIVED;
			} else {
				return Constants.SEL_SCORE;
			}
		}
		return null;
	}

	public void dumpScoresForAssessment(AsScoreInfo si,
			AsQueryBuilderForm queryForm) {
		System.out.println("dumpScoresForAssessment");
		for (AssessmentSelectionInfo asi : queryForm.getAssessments()) {
			if (asi.getAssessmentID().equals(si.getAssessmentID())) {
				for (AsScoreInfo score : asi.getScores()) {
					System.out.println(score.toString());
				}
			}
		}
	}

	public void populateScoresForSelectedAssessments(UserInfo ui,
			IAssessmentService asService,
			List<AssessmentSelectionInfo> asiList, AsQueryBuilderForm queryForm)
			throws Exception {
		for (AssessmentSelectionInfo asi : asiList) {
			if (!asi.isSelected()) {
				// unselect all the scores selected (if any)
				changeScoresSelection(asi, false);
				continue;
			}
			if (asi.getScores().isEmpty()) {
				Assessment a = new Assessment();
				a.setAssessmentid(asi.getAssessmentID());
				List<Assessmentscore> scoreList = asService
						.getScoresForAssessment(ui, a);

				// sort by score order
				Collections.sort(scoreList, new Comparator<Object>() {
					public int compare(final Object o1, final Object o2) {
						final Assessmentscore s1 = (Assessmentscore) o1;
						final Assessmentscore s2 = (Assessmentscore) o2;
						return s1.getScoresequence().intValue()
								- s2.getScoresequence().intValue();
					}
				});

				log.debug("got the list of scores size= " + scoreList.size());
				for (Assessmentscore score : scoreList) {
					AsScoreInfo si = new AsScoreInfo(asi.getAssessmentID());
					si.setName(score.getScorename());
					si.setType(score.getScoretype());
					si.setParentScore(score.getParentscore());
					si.setLevel(score.getScorelevel() == null ? -1 : score
							.getScorelevel().intValue());
					asi.addScore(si);
				}
				// create score hierarchy
				createTree(asi);
			}
		}
	}

	// Assumption: score names are unique within an assessment
	public static void createTree(AssessmentSelectionInfo asi) {
		List<AsScoreInfo> topLevelScores = new LinkedList<AsScoreInfo>();
		Map<String, AsScoreInfo> scoreMap = new HashMap<String, AsScoreInfo>();
		for (AsScoreInfo si : asi.getScores()) {
			scoreMap.put(si.getName(), si);
			if (si.getParentScore() == null) {
				topLevelScores.add(si);
			}
		}

		for (AsScoreInfo si : asi.getScores()) {
			if (si.getParentScore() == null)
				continue;
			String parentScore = si.getParentScore();
			AsScoreInfo parent = scoreMap.get(parentScore);
			parent.addChild(si);
		}
		asi.setTopLevelScores(topLevelScores);
	}

	protected void changeScoresSelection(AssessmentSelectionInfo asi,
			boolean selected) {
		for (AsScoreInfo si : asi.getScores()) {
			si.setSelected(selected);
		}
	}

	/**
	 * creates a list of QueryPartInfo objects and stores them in the
	 * AsQueryBuilderForm, also sets the combinators.
	 * 
	 * @param queryForm
	 * @param asiList
	 *            a list of {@link clinical.web.common.vo.AssessmentSelectionInfo}
	 *            objects.
	 */
	public void createQueryParts(AsQueryBuilderForm queryForm,
			List<AssessmentSelectionInfo> asiList) {
		Map<AsScoreInfo, AsScoreInfo> alreadySelectedScoresMap = new HashMap<AsScoreInfo, AsScoreInfo>(
				11);
		if (queryForm.getSavedQueryLoaded()) {
			for (AssessmentSelectionInfo asi : asiList) {
				if (asi.isSelected()) {
					for (AsScoreInfo si : asi.getScores()) {
						if (si.isSelected()) {
							alreadySelectedScoresMap.put(si, si);
						}
					}
				}
			}
		} else {
			queryForm.clearQueryParts();
		}

		for (AssessmentSelectionInfo asi : asiList) {
			if (!asi.isSelected()) {
				continue;
			}
			for (AsScoreInfo si : asi.getScores()) {
				if (!si.isSelected())
					continue;

				QueryPartInfo queryPart = new QueryPartInfo(asi, si);
				if (queryForm.getSavedQueryLoaded()) {
					if (!queryForm.getQueryParts().contains(queryPart)) {
						log.info("queryPart=" + queryPart);
						int idx = getScoreInfoQueryPartIndexForInsert(queryForm
								.getQueryParts());
						queryForm.getQueryParts().add(idx, queryPart);
					}
				} else {
					queryForm.addQueryPart(queryPart);
				}
			}
		}

		// now set the combinators
		setCombinators(queryForm);
	}

	protected int getScoreInfoQueryPartIndexForInsert(
			List<QueryPartInfo> queryParts) {
		int idx = 0;
		for (QueryPartInfo qp : queryParts) {
			if (qp.getScoreInfo() == null) {
				return idx;
			}
			++idx;
		}
		return idx;
	}

	/**
	 * Determines the selected scores for each assessment from the web form info
	 * and sets the selected flag for each <code>ScoreInfo</code> object.
	 * 
	 * @param request
	 * @param queryForm
	 * @return
	 */
	public List<AssessmentSelectionInfo> populateScores(
			HttpServletRequest request, AsQueryBuilderForm queryForm) {
		List<AssessmentSelectionInfo> asiList = queryForm.getAssessments();

		int aidx = 0;
		for (AssessmentSelectionInfo asi : asiList) {
			if (!asi.isSelected()) {
				++aidx;
				continue;
			}
			int sidx = 0;
			for (AsScoreInfo si : asi.getScores()) {
				StringBuffer buf = new StringBuffer();
				buf.append("score_").append(aidx).append('_').append(sidx);
				String selValue = request.getParameter(buf.toString());
				if (selValue != null) {
					if (log.isDebugEnabled()) {
						log.debug("selecting " + buf.toString());
					}
					log.info("selecting " + buf.toString());
					si.setSelected(true);
					log.info(si.toString() + " ");
					;
				}
				++sidx;
			}
			++aidx;
		}
		return asiList;
	}

	public int getNumberOfScores(AsQueryBuilderForm queryForm) {
		List<AssessmentSelectionInfo> asiList = queryForm.getAssessments();
		int numScores = 0;
		for (AssessmentSelectionInfo asi : asiList) {
			if (asi.isSelected()) {
				numScores += asi.getScores().size();
			}
		}
		return numScores;
	}

	/**
	 * 
	 * @param asiList
	 * @param assessmentsAlso
	 */
	public void resetSelections(List<AssessmentSelectionInfo> asiList,
			boolean assessmentsAlso) {
		for (AssessmentSelectionInfo asi : asiList) {
			if (assessmentsAlso) {
				asi.setSelected(false);
			}
			if (asi.getScores() != null) {
				for (AsScoreInfo si : asi.getScores()) {
					si.setSelected(false);
				}
			}
		}
	}

	/**
	 * retrieves subcortical variables from the database and adds them (@link
	 * clinical.web.helpers.SubCorticalVarInfo}) to the passed queryForm.
	 * 
	 * @param dbID
	 *            database ID
	 * @param ui
	 *            user information for database connection
	 * @param queryForm
	 *            Struts Form bean for the query wizard.
	 * @return true if the database has subcortical variables
	 * @throws java.lang.Exception
	 */
	public boolean prepareSubcorticalVariables(String dbID, UserInfo ui,
			AsQueryBuilderForm queryForm) throws Exception {
		// queryForm.getSubcorticalVars().clear();
		boolean hasSubcorticalData = true;
		if (queryForm.getSubcorticalVars().isEmpty()) {
			IDerivedDataService dService = ServiceFactory
					.getDerivedDataService(dbID);
			List<Brainsegmentdata> bsds = dService
					.getAllSubCorticalVariables(ui);
			log.info("got subcortical variables >>" + bsds.size());
			if (bsds.isEmpty()) {
				synchronized (this) {
					log.info("hasSubcorticalData >> false");
					hasSubcorticalData = false;
				}
			}
			Map<String, BrainSegmentVarInfo> map = new LinkedHashMap<String, BrainSegmentVarInfo>();
			for (Brainsegmentdata bsd : bsds) {
				BrainSegmentVarInfo bsvi = map.get(bsd.getBrainregionname());
				if (bsvi == null) {
					List<String> lateralities = new ArrayList<String>(2);
					bsvi = new BrainSegmentVarInfo(bsd.getBrainregionname(),
							lateralities, bsd.getMeasurementtype(), bsd
									.getUnit());
					map.put(bsd.getBrainregionname(), bsvi);
				}
				bsvi.lateralities.add(bsd.getLaterality());
			}
			for (Map.Entry<String, BrainSegmentVarInfo> entry : map.entrySet()) {
				String brainRegionName = entry.getKey();
				BrainSegmentVarInfo bsvi = entry.getValue();
				// String[] latStrs = null;
				String[] latStrs = bsvi.prepLateralityStrings();
				if (bsvi.lateralities.size() >= 1
						&& !bsvi.lateralities.get(0).equals("n/a")) {
					if (bsvi.lateralities.size() == 2) {
						latStrs = new String[3];
						latStrs[0] = "both";
						latStrs[1] = bsvi.lateralities.get(0);
						latStrs[2] = bsvi.lateralities.get(1);
					}
				}
				SubCorticalVarInfo scvi = new SubCorticalVarInfo(
						brainRegionName, latStrs, bsvi.measurementType,
						bsvi.unit);

				queryForm.addSubcorticalVar(scvi);
			}
		}
		return hasSubcorticalData;
	}

	public boolean hasSubcorticalVariables(AsQueryBuilderForm queryForm) {
		List<QueryPartInfo> queryParts = queryForm.getQueryParts();
		for (QueryPartInfo qpi : queryParts) {
			if (qpi.getSubCorticalVarInfo() != null)
				return true;
		}
		return false;
	}

	protected void resetQueryParts(AsQueryBuilderForm queryForm, boolean all) {
		List<QueryPartInfo> queryParts = queryForm.getQueryParts();
		for (Iterator<QueryPartInfo> iter = queryParts.iterator(); iter
				.hasNext();) {
			QueryPartInfo qpi = iter.next();
			if (all) {
				iter.remove();
			} else if (qpi.getSubCorticalVarInfo() != null) {
				iter.remove();
			}
		}
	}

	protected void resetSubcorticalVarSelections(List<SubCorticalVarInfo> scvis) {
		for (SubCorticalVarInfo scvi : scvis) {
			scvi.setSelected(false);
		}
	}

	/**
	 * Determines the selected subcortical variables and prepares the
	 * {@link clinical.web.common.query.QueryPartInfo} object for the selected
	 * subcortical variables and adds them to the queryForm.
	 * 
	 * @param request
	 * @param queryForm
	 */
	public void prepareSubCorticalQueryParts(HttpServletRequest request,
			AsQueryBuilderForm queryForm) {

		if (!queryForm.getSavedQueryLoaded()) {
			// delete only the previous subcortical query parts.
			resetQueryParts(queryForm, false);
		}

		List<SubCorticalVarInfo> scvis = queryForm.getSubcorticalVars();
		Map<SubCorticalVarInfo, SubCorticalVarInfo> alreadySelectedMap = new HashMap<SubCorticalVarInfo, SubCorticalVarInfo>(
				5);

		if (queryForm.getSavedQueryLoaded()) {
			for (SubCorticalVarInfo scvi : scvis) {
				if (scvi.isSelected()) {
					alreadySelectedMap.put(scvi, scvi);
				}
			}
		} else {
			log.info("resetting subcortical var selections");
			resetSubcorticalVarSelections(scvis);
		}

		int idx = 0;
		for (SubCorticalVarInfo scvi : scvis) {
			StringBuffer sb = new StringBuffer();
			sb.append("scvi_").append(idx);
			String selValue = request.getParameter(sb.toString());
			if (selValue != null) {
				scvi.setSelected(true);
				log.info("selecting " + scvi.getBrainRegionName());
			}
			++idx;
		}

		for (SubCorticalVarInfo scvi : scvis) {
			if (scvi.isSelected()) {
				// skip the ones loaded from the saved query
				if (alreadySelectedMap.get(scvi) != null)
					continue;

				if (scvi.getLaterality() == null) {
					SubCorticalVarInfo sv = new SubCorticalVarInfo(scvi
							.getBrainRegionName(), "", scvi
							.getMeasurementType(), scvi.getUnit());
					queryForm.addQueryPart(new QueryPartInfo(sv));

				} else if (scvi.getLaterality().equals("both")) {
					SubCorticalVarInfo sv = new SubCorticalVarInfo("Left "
							+ scvi.getBrainRegionName(), "left", scvi
							.getMeasurementType(), scvi.getUnit());
					// needs to be selected
					sv.setSelected(true);
					queryForm.addQueryPart(new QueryPartInfo(sv));
					sv = new SubCorticalVarInfo("Right "
							+ scvi.getBrainRegionName(), "right", scvi
							.getMeasurementType(), scvi.getUnit());
					// needs to be selected
					sv.setSelected(true);
					queryForm.addQueryPart(new QueryPartInfo(sv));
				} else {
					SubCorticalVarInfo sv = new SubCorticalVarInfo(GenUtils
							.toTitleCase(scvi.getLaterality())
							+ " " + scvi.getBrainRegionName(), scvi
							.getLaterality(), scvi.getMeasurementType(), scvi
							.getUnit());
					queryForm.addQueryPart(new QueryPartInfo(sv));
				}
			}
		}
		// now set the combinators
		setCombinators(queryForm);
	}

	/**
	 * Sets the ranges for the query parts.
	 * 
	 * @param ui
	 *            user information for database connection
	 * @param queryForm
	 *            Struts Form bean for the query wizard.
	 * @param asService
	 *            {@link clinical.web.IAssessmentService}
	 * @param skipScores
	 * @throws java.lang.Exception
	 */
	public void prepareRangeInfos(UserInfo ui, AsQueryBuilderForm queryForm,
			IAssessmentService asService, boolean skipScores) throws Exception {
		List<SubCorticalVarInfo> subcorticals = new LinkedList<SubCorticalVarInfo>();
		List<AsScoreInfo> scoreInfos = new LinkedList<AsScoreInfo>();
		Map<Serializable, QueryPartInfo> map = new HashMap<Serializable, QueryPartInfo>(
				11);
		for (QueryPartInfo qp : queryForm.getQueryParts()) {
			if (qp.getSubCorticalVarInfo() != null) {
				subcorticals.add(qp.getSubCorticalVarInfo());
				map.put(qp.getSubCorticalVarInfo(), qp);
			} else {
				if (!skipScores) {
					scoreInfos.add(qp.getScoreInfo());
					map.put(qp.getScoreInfo(), qp);
				}
			}
		}

		if (subcorticals.isEmpty()) {
			subcorticals = null;
		}
		List<RangeInfo> rangeInfos = null;
		if (skipScores) {
			rangeInfos = asService.getRangeInfos(ui, null, subcorticals);
		} else {
			rangeInfos = asService.getRangeInfos(ui, scoreInfos, subcorticals);
		}
		// set the range infos
		for (RangeInfo ri : rangeInfos) {
			QueryPartInfo qp = map.get(ri.getVariable());
			qp.setRangeInfo(ri);
		}
	}

	/**
	 * Looks for any enumerations in the query parts, if it finds any, prepares
	 * RHS enumerations.
	 * 
	 * @param ui
	 *            user information for database connection
	 * @param queryForm
	 *            Struts Form bean for the query wizard.
	 * @param dbCache
	 *            {@link clinical.web.common.IDBCache}
	 * @throws java.lang.Exception
	 */
	public void prepareEnumerations(UserInfo ui, AsQueryBuilderForm queryForm,
			IDBCache dbCache) throws Exception {
		Map<String, List<Assessmentscorecode>> scoreCodeMap = dbCache
				.getScoreCodeMap(ui, false);
		for (QueryPartInfo qp : queryForm.getQueryParts()) {
			if (qp.getScoreInfo() != null) {
				AsScoreInfo si = qp.getScoreInfo();
				String key = si.getAssessmentID().toString() + "_"
						+ si.getName();
				List<Assessmentscorecode> scList = scoreCodeMap.get(key);
				if (scList != null) {
					List<String> possibleValues = new ArrayList<String>(scList
							.size());
					List<String> scoreCodes = new LinkedList<String>();
					for (Assessmentscorecode asc : scList) {
						possibleValues.add(asc.getScorecodevalue());
						scoreCodes.add(asc.getScorecode());
					}
					qp.prepareRHSEnumerations(possibleValues, scoreCodes);
				}
			}
		}
	}

	protected void setCombinators(AsQueryBuilderForm queryForm) {
		if (queryForm.getQueryParts().size() > 1) {
			String[] combs = new String[queryForm.getQueryParts().size() - 1];
			for (int i = 0; i < combs.length; ++i)
				combs[i] = "AND";
			queryForm.setCombinators(combs);
		}
	}

	public static void prepareExperimentSelector(String dbID, UserInfo ui,
			AsQueryBuilderForm queryForm, IDBCache dbCache) throws Exception {
		int selectedExpID = ExperimentSelector.ALL_EXPERIMENTS;
		if (queryForm.getExpSelector() != null) {
			selectedExpID = queryForm.getExpSelector().getSelectedExpID();
		}
		List<Experiment> experiments = dbCache.getExperiments(ui, true);
		List<Experiment> userExps = new ArrayList<Experiment>(experiments
				.size());
		IAuthorizationService authService = ServiceFactory
				.getAuthorizationService();
		for (Experiment exp : experiments) {
			if (authService.isAuthorized(ui, dbID, PrivilegeLabel.READ, exp.getUniqueid()
					.intValue())) {
				userExps.add(exp);
			}
		}
		// create an ExperimentSelector
		ExperimentSelector expSelector = new ExperimentSelector(userExps);
		expSelector.setSelectedExpID(selectedExpID);
		queryForm.setExpSelector(expSelector);
	}

	/**
	 * validator for the Score Selection screen. At least one score must be
	 * selected from any of the previously selected assessments.
	 * 
	 * @param queryForm
	 *            the query builder form object containing user supplied state
	 *            information for the wizard
	 * @param request
	 *            a <code>HttpServletRequest</code> object
	 * @throws BaseException
	 */
	public void validateSelectScores(AsQueryBuilderForm queryForm,
			HttpServletRequest request) throws BaseException {
		boolean selected = false;
		int aidx = 0;
		for (AssessmentSelectionInfo asi : queryForm.getAssessments()) {
			if (!asi.isSelected()) {
				++aidx;
				continue;
			}

			for (int sidx = 0; sidx < asi.getScores().size(); sidx++) {
				StringBuffer buf = new StringBuffer();
				buf.append("score_").append(aidx).append('_').append(sidx);
				String selValue = request.getParameter(buf.toString());
				if (selValue != null) {
					log.info("selecting " + buf.toString());
					selected = true;
					break;
				}
			}
			if (selected)
				break;
			++aidx;
		}

		if (!selected) {
			throw new BaseException("", "errors.checked");
		}
	}

	/**
	 * validator for assessment selection. At least one assessment needs to
	 * selected to continue.
	 * 
	 * @param queryForm
	 *            the query builder form object containing user supplied state
	 *            information for the wizard
	 * @param request
	 *            a <code>HttpServletRequest</code> object
	 * @throws BaseException
	 */
	public void validateSelectAssessments(AsQueryBuilderForm queryForm,
			HttpServletRequest request) throws BaseException {
		boolean selected = false;
		for (AssessmentSelectionInfo asi : queryForm.getAssessments()) {
			if (asi.isSelected()) {
				selected = true;
				break;
			}
		}
		if (!selected) {
			throw new BaseException("", "errors.checked");
		}
	}

	/**
	 * validator for query submission page. Here depending of the type of query
	 * variable , values are checked for type, ranges are checked for
	 * correctness (e.g. a low bound must be lower than the upper bound in
	 * between type queries).
	 * 
	 * @param queryForm
	 *            the query builder form object containing user supplied state
	 *            information for the wizard
	 * @param request
	 *            a <code>HttpServletRequest</code> object
	 * @throws BaseException
	 */
	public void validateCollectQuery(AsQueryBuilderForm queryForm,
			HttpServletRequest request) throws BaseException {
		for (QueryPartInfo qp : queryForm.getQueryParts()) {
			if (qp.getScoreInfo() == null) {
				// do SubCorticalVar validation
				continue;
			}

			SearchPredicate.Range range = null;
			int spType = QueryUtils.getSearchPredicateType(qp.getScoreInfo()
					.getType());
			Object value = null;
			// if wild card then skip validation
			if (qp.getRhs().trim().equals("*")) {
				continue;
			}
			try {
				value = QueryUtils.convertToType(qp.getRhs(), spType);
			} catch (NumberFormatException nfe) {
				BaseException be = new BaseException("", "errors.notanumber");
				be.setMessageArgs(new Object[] { "value",
						qp.getScoreInfo().getName() });
				throw be;
			}

			if (QueryUtils.isNumeric(spType) && qp.getRangeInfo() != null
					&& qp.getRhs().trim().length() > 0) {
				RangeInfo ri = qp.getRangeInfo();
				if (!ri.wasInRange(qp.getRhs().trim())) {
					BaseException be = new BaseException("",
							"errors.not_in_range");
					be.setMessageArgs(new Object[] { qp.getScoreInfo()
							.getName() });
					throw be;
				}
			}

			if (qp.getLowBound() != null && qp.getLowBound().length() > 0
					&& qp.getUppBound() != null
					&& qp.getUppBound().length() > 0) {
				Number lowBound = null;
				Number uppBound = null;
				try {
					lowBound = (Number) QueryUtils.convertToType(qp
							.getLowBound(), spType);
				} catch (NumberFormatException nfe) {
					BaseException be = new BaseException("",
							"errors.notanumber");
					be.setMessageArgs(new Object[] { "low bound",
							qp.getScoreInfo().getName() });
					throw be;
				}
				try {
					uppBound = (Number) QueryUtils.convertToType(qp
							.getUppBound(), spType);
				} catch (NumberFormatException nfe) {
					BaseException be = new BaseException("",
							"errors.notanumber");
					be.setMessageArgs(new Object[] { "upper bound",
							qp.getScoreInfo().getName() });
					throw be;
				}
				range = new SearchPredicate.Range(lowBound, uppBound);
			}

			if (range != null) {
				if (range.getLowBound() == null || range.getUppBound() == null) {
					BaseException be = new BaseException("",
							"errors.query.missingbetween");
					be.setMessageArgs(new Object[] { qp.getScoreInfo()
							.getName() });
					throw be;
				}

				// do range checking
				if (qp.getRangeInfo() != null) {
					RangeInfo ri = qp.getRangeInfo();
					if (!ri.wasInRange(qp.getLowBound().trim())) {
						BaseException be = new BaseException("",
								"errors.between_not_in_range");
						be.setMessageArgs(new Object[] { "low bound",
								qp.getScoreInfo().getName() });
						throw be;
					}
					if (!ri.wasInRange(qp.getUppBound().trim())) {
						BaseException be = new BaseException("",
								"errors.between_not_in_range");
						be.setMessageArgs(new Object[] { "upper bound",
								qp.getScoreInfo().getName() });
						throw be;
					}
				}

				if (range.getLowBound().doubleValue() >= range.getUppBound()
						.doubleValue()) {
					if (range.getLowBound() == null
							|| range.getUppBound() == null) {
						BaseException be = new BaseException("",
								"errors.invalidbetween");
						be.setMessageArgs(new Object[] { qp.getScoreInfo()
								.getName() });
						throw be;
					}
				}

			} else {
				// no between and no value either
				if (value == null) {
					BaseException be = new BaseException("",
							"errors.query.novalue");
					be.setMessageArgs(new Object[] { qp.getScoreInfo()
							.getName() });
					throw be;
				}
			}

			if (log.isDebugEnabled()) {
				log.debug("Score= " + qp.getScoreInfo().getName());
				log.debug("operator= " + qp.getLogicalOps(qp.getOperator())
						+ " op=" + qp.getOperator());
				log.debug("rhs= " + qp.getRhs());
			}
		}
	}

	/**
	 * 
	 * @param queryForm
	 * @return
	 */
	public static Element prepareQueryForPersistence(
			AsQueryBuilderForm queryForm) {
		Element elem = new Element("query");
		if (queryForm.getExpSelector() != null) {
			Element eiElem = new Element("exp-id");
			eiElem.addContent(String.valueOf(queryForm.getExpSelector()
					.getSelectedExpID()));
			elem.addContent(eiElem);
		}
		for (QueryPartInfo qp : queryForm.getQueryParts()) {
			Element qpElem = qp.toXML();
			elem.addContent(qpElem);
		}
		return elem;
	}

	public static void persistQueryToFile(Element queryElem, String filePath)
			throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		BufferedWriter writer = null;
		try {
			writer = new BufferedWriter(new FileWriter(filePath));
			xout.output(queryElem, writer);
		} finally {
			if (writer != null)
				try {
					writer.close();
				} catch (Exception x) {
				}
		}
	}

	public static void saveJSONQueryToDB(JSONObject js, String description,
			UserInfo ui, IAssessmentService asService) throws Exception {
		log.info(js.toString(2));
		asService.saveUserQuery(ui, js.toString(2), description);
	}

	public static void persistQueryToDB(Element queryElem, String description,
			UserInfo ui, IAssessmentService asService) throws Exception {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		StringWriter sw = new StringWriter();
		xout.output(queryElem, sw);
		log.info(sw.toString());

		asService.saveUserQuery(ui, sw.toString(), description);
	}

	public static List<QueryPartInfo> prepareQueryFromXML(Element queryElem,
			AssessmentSelectionInfoHelper helper) throws Exception {
		List<QueryPartInfo> queryParts = new ArrayList<QueryPartInfo>();
		List<?> childElems = queryElem.getChildren("query-part-info");
		for (Iterator<?> iter = childElems.iterator(); iter.hasNext();) {
			Element qpElem = (Element) iter.next();
			QueryPartInfo qp = QueryPartInfo.initializeFromXML(qpElem, helper);
			queryParts.add(qp);
		}
		return queryParts;
	}

	public static List<QueryPartInfo> prepareQueryFromJSON(String jsonStr,
			AssessmentSelectionInfoHelper helper) throws Exception {
		List<QueryPartInfo> queryParts = new ArrayList<QueryPartInfo>();
		JSONObject js = new JSONObject(jsonStr);
		JSONArray jsArr = js.getJSONArray("asQPIList");
		for (int i = 0; i < jsArr.length(); i++) {
			// FIXME use helper
			QueryPartInfo qp = QueryPartInfo.initializeFromJSON(jsArr
					.getJSONObject(i));
			queryParts.add(qp);
		}
		return queryParts;
	}

	public static List<QueryPartInfo> loadQueryFromFile(String filePath,
			AssessmentSelectionInfoHelper helper) throws Exception {
		SAXBuilder builder = new SAXBuilder(false); // no validation
		Document doc = builder.build(filePath);
		Element queryElem = doc.getRootElement();
		return prepareQueryFromXML(queryElem, helper);
	}

	public static JSONObject loadJSONQueryFromDB(UserInfo ui,
			IAssessmentService asService, int selectedQueryID, String userName)
			throws Exception {
		Storedquery sq = asService.getFullStoredQuery(ui, new BigDecimal(String
				.valueOf(selectedQueryID)), userName);
		if (sq == null) {
			throw new Exception("Stored query with ID " + selectedQueryID
					+ " does not exists!");
		}
		JSONObject js = new JSONObject(sq.getQuerystate());
		return js;
	}

	/**
	 * Loads a saved query with the given query ID from the DB, converts it into
	 * query parts and returns the query parts in a list.
	 * 
	 * @param ui
	 * @param asService
	 * @param selectedQueryID
	 * @param userName
	 * @param helper
	 * @return a <code>StoredQueryDescription</code>
	 * 
	 * @throws java.lang.Exception
	 */
	public static StoredQueryDescription loadQueryFromDB(UserInfo ui,
			IAssessmentService asService, int selectedQueryID, String userName,
			AssessmentSelectionInfoHelper helper) throws Exception {
		SAXBuilder builder = new SAXBuilder(false); // no validation
		Storedquery sq = asService.getFullStoredQuery(ui, new BigDecimal(String
				.valueOf(selectedQueryID)), userName);
		if (sq == null) {
			throw new Exception("Stored query with ID " + selectedQueryID
					+ " does not exists!");
		}
		StoredQueryDescription sqd = new StoredQueryDescription();
		String queryState = sq.getQuerystate();
		if (queryState.startsWith("{")) {
			// JSON
			// FIXME
		} else {
			StringReader sr = new StringReader(queryState);

			Document doc = builder.build(sr);
			Element queryElem = doc.getRootElement();
			Element eiElem = queryElem.getChild("exp-id");
			if (eiElem != null) {
				sqd.setExpID(Integer.parseInt(eiElem.getTextTrim()));
			}
			List<QueryPartInfo> queryParts = prepareQueryFromXML(queryElem,
					helper);
			sqd.queryParts = queryParts;
		}
		return sqd;
	}

	public static void prepareStoredQueryDescriptions(UserInfo ui,
			IAssessmentService asService, String userName,
			AsQueryBuilderForm queryForm) throws Exception {
		List<Storedquery> sqList = asService.getAvailableStoredQueries(ui,
				userName);
		List<StoredQueryInfo> sqiList = new ArrayList<StoredQueryInfo>(sqList
				.size());
		for (Storedquery sq : sqList) {
			StoredQueryInfo sqi = new StoredQueryInfo(sq.getDescription(), sq
					.getUniqueid().intValue());
			sqiList.add(sqi);
		}
		StoredQuerySelector sqs = new StoredQuerySelector(sqiList);
		queryForm.setSavedQuerySelector(sqs);
	}

	/**
	 * Applies the query parts to the assessment, score, derived data selection
	 * pages' internal state and the query builder pages
	 * 
	 * @param dbID
	 * @param ui
	 * @param queryForm
	 * @param helper
	 * @param queryParts
	 * @throws java.lang.Exception
	 */
	public void prepareQueryWizardSelectionState(String dbID, UserInfo ui,
			AsQueryBuilderForm queryForm, AssessmentSelectionInfoHelper helper,
			List<QueryPartInfo> queryParts) throws Exception {
		List<AssessmentSelectionInfo> asiList = queryForm.getAssessments();
		for (Iterator<AssessmentSelectionInfo> iter = asiList.iterator(); iter
				.hasNext();) {
			AssessmentSelectionInfo asi = iter.next();
			AssessmentSelectionInfo savedAsi = helper
					.findAssessmentSelectionInfo(asi.getAssessmentID(), false);
			if (savedAsi == null) {
				// unselect all the scores selected (if any), since this
				// assessment is not used in the stored query
				changeScoresSelection(asi, false);
			} else {
				if (asi.getScores().isEmpty()) {
					List<AsScoreInfo> savedScoreInfos = helper
							.getAllScoreInfosForAssessment(asi
									.getAssessmentID());
					for (AsScoreInfo savedSi : savedScoreInfos) {
						asi.addScore(savedSi);
					}
				} else {
					// already have scores just select the scores in the saved
					// query for this assessment
					for (AsScoreInfo si : asi.getScores()) {
						AsScoreInfo savedSi = helper.findAsScoreInfo(asi
								.getAssessmentID(), si.getName());
						if (savedSi != null) {
							si.setSelected(savedSi.isSelected());
						} else {
							si.setSelected(false);
						}
						if (si.isSelected()) {
							asi.setSelected(true);
						}
					}
				}
				// create score hierarchy
				createTree(asi);
			}
		}

		// now prepare the derived data selections also (if any)

		queryForm.clearSubcorticalVars();
		Map<String, String> savedSubcorticalVarMap = new HashMap<String, String>(
				7);
		for (QueryPartInfo qp : queryParts) {
			if (qp.getSubCorticalVarInfo() != null) {
				if (qp.getSubCorticalVarInfo().getLaterality() == null) {
					savedSubcorticalVarMap.put(qp.getSubCorticalVarInfo()
							.getBrainRegionName(), "n/a");
				} else {
					String regionName = qp.getSubCorticalVarInfo()
							.getBrainRegionName();
					if (regionName.startsWith("Left ")) {
						regionName = regionName.substring(5);
					} else if (regionName.startsWith("Right ")) {
						regionName = regionName.substring(6);
					}

					String laterality = savedSubcorticalVarMap.get(regionName);
					if (laterality == null) {
						savedSubcorticalVarMap.put(regionName, qp
								.getSubCorticalVarInfo().getLaterality());
					} else {
						savedSubcorticalVarMap.put(regionName, "both");
					}
				}
			}
		}
		log
				.info("savedSubcorticalVarMap size="
						+ savedSubcorticalVarMap.size());
		for (Map.Entry<String, String> entry : savedSubcorticalVarMap
				.entrySet()) {
			log.info("regionName=" + entry.getKey() + " laterality="
					+ entry.getValue());
		}

		if (!savedSubcorticalVarMap.isEmpty()) {
			if (prepareSubcorticalVariables(dbID, ui, queryForm)) {
				log.info("prepared subcortical vars (size = "
						+ queryForm.getSubcorticalVars().size() + ")");
				for (SubCorticalVarInfo scvi : queryForm.getSubcorticalVars()) {
					String savedLaterality = savedSubcorticalVarMap.get(scvi
							.getBrainRegionName());

					if (savedLaterality != null) {
						if (!savedLaterality.equals("n/a")) {
							scvi.setLateralityAndLidx(savedLaterality);
						}
						scvi.setSelected(true);
						log.info("scvi.selected=" + scvi.isSelected() + "\n"
								+ scvi.toString());
					} else {
						scvi.setSelected(false);
					}
				}
			}
		}
		// replace the current queryparts list with the saved query parts
		queryForm.clearQueryParts();
		for (QueryPartInfo qp : queryParts) {
			queryForm.addQueryPart(qp);
		}

		/** @todo save the combinators also with the query */
		// now set the combinators
		setCombinators(queryForm);

		log.info("exiting prepareQueryWizardSelectionState");

	}

	class BrainSegmentVarInfo {
		String brainRegionName;
		List<String> lateralities;
		String measurementType;
		String unit;

		public BrainSegmentVarInfo(String brainRegionName,
				List<String> lateralities, String measurementType, String unit) {
			this.brainRegionName = brainRegionName;
			this.lateralities = lateralities;
			this.measurementType = measurementType;
			this.unit = unit;
		}

		public String[] prepLateralityStrings() {
			String[] latStrs = null;
			if (lateralities.size() >= 1 && !lateralities.contains("n/a")) {
				if (lateralities.size() == 2) {
					latStrs = new String[3];
					latStrs[0] = "both";
					latStrs[1] = lateralities.get(0);
					latStrs[2] = lateralities.get(1);
				}
				// System.out.println(">> lateralities " + lateralities.size() +
				// " " + lateralities);
			}
			return latStrs;
		}
	}

}
