package clinical.web.services;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;

import org.jdom.Element;

import clinical.utils.FileUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class CBFImageTypeLookupService implements IPeriodicService {
	private final static long checkInterval = 1800000l; // every half hour
	private String xmlLookupRulesFile;
	private ConcurrentHashMap<String, MatchPatternGroup> lookupMap = new ConcurrentHashMap<String, MatchPatternGroup>();
	private static CBFImageTypeLookupService instance = null;

	/**
	 * intentionally not thread safe. Intended to be called once at the server
	 * startup
	 * 
	 * @param xmlLookupRulesFile
	 * @return
	 * @throws Exception
	 */
	public static CBFImageTypeLookupService getInstance(
			String xmlLookupRulesFile) throws Exception {
		if (instance == null) {
			instance = new CBFImageTypeLookupService(xmlLookupRulesFile);
		}
		return instance;
	}

	/**
	 * intentionally not thread safe. Just returns the already (at startup)
	 * created instance
	 * 
	 * @return
	 */
	public static CBFImageTypeLookupService getInstance() {
		if (instance == null) {
			throw new RuntimeException(
					"CBFImageTypeLookupService is not initialized properly!");
		}
		return instance;
	}

	private CBFImageTypeLookupService(String xmlLookupRulesFile)
			throws Exception {
		this.xmlLookupRulesFile = xmlLookupRulesFile;
		loadAndPopulate();
	}

	private void loadAndPopulate() throws Exception {
		Element rootEl = FileUtils.loadXML(xmlLookupRulesFile);
		List<?> pgEls = rootEl.getChildren("pattern-group");
		for (Object o : pgEls) {
			Element pge = (Element) o;
			List<?> pElems = pge.getChildren("p");
			String group = pge.getAttributeValue("name");
			MatchPatternGroup mpg = lookupMap.get(group);
			if (mpg == null) {
				mpg = new MatchPatternGroup(group);
				lookupMap.put(group, mpg);
				for (Object o1 : pElems) {
					Element pe = (Element) o1;
					int type = getPatternType(pe);
					MatchPattern mp = new MatchPattern(pe.getTextTrim(), type);
					mpg.patternSet.add(mp);
				}
			} else {
				Set<String> pSet = new HashSet<String>();
				List<MatchPattern> toBeAdded = new ArrayList<MatchPattern>(
						pElems.size());
				List<MatchPattern> toBeRemoved = new ArrayList<MatchPattern>(
						pElems.size());
				for (Object o1 : pElems) {
					Element pe = (Element) o1;
					String p = pe.getTextTrim();
					int type = getPatternType(pe);
					MatchPattern mp = new MatchPattern(p, type);
					pSet.add(p);
					if (!mpg.patternSet.contains(mp)) {
						toBeAdded.add(mp);
					}
				}
				for (MatchPattern mp : mpg.patternSet) {
					if (!pSet.contains(mp.pattern)) {
						toBeRemoved.add(mp);
					}
				}

				for (MatchPattern mp : toBeRemoved) {
					mpg.patternSet.remove(mp);
				}

				for (MatchPattern mp : toBeAdded) {
					mpg.patternSet.add(mp);
				}
			}
		}
	}

	protected int getPatternType(Element pe) {
		int type = MatchPattern.CONTAINS;
		if (pe.getAttribute("type") != null) {
			String typeStr = pe.getAttributeValue("type");
			if (typeStr.equals("exact")) {
				type = MatchPattern.EXACT_MATCH;
			} else if (typeStr.equals("contains")) {
				type = MatchPattern.CONTAINS;
			}
		}
		return type;
	}

	public MatchPatternGroup getMatchPatternGroup(String groupName) {
		return lookupMap.get(groupName);
	}
	
	public List<String> getMatchingGroups(String str, boolean caseSensitive) {
		List<String> matchingGroups = new ArrayList<String>(1);
		for(MatchPatternGroup mpg : lookupMap.values()) {
			if (mpg.containsPattern(str, caseSensitive)) {
		          matchingGroups.add(mpg.getGroup());		
			}
		}
		return matchingGroups;
	}

	@Override
	public long getPeriod() {
		return checkInterval;
	}

	@Override
	public void service() throws Exception {
		loadAndPopulate();
	}

	public static class MatchPattern implements Comparable<MatchPattern> {
		String pattern;
		int type;
		public static int EXACT_MATCH = 1;
		public static int CONTAINS = 2;

		public MatchPattern(String pattern, int type) {
			this.pattern = pattern;
			this.type = type;
		}

		public MatchPattern(String pattern) {
			this.pattern = pattern;
			this.type = CONTAINS;
		}

		public String getPattern() {
			return pattern;
		}

		public int getType() {
			return type;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result
					+ ((pattern == null) ? 0 : pattern.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			MatchPattern other = (MatchPattern) obj;
			if (pattern == null) {
				if (other.pattern != null)
					return false;
			} else if (!pattern.equals(other.pattern))
				return false;
			return true;
		}

		@Override
		public int compareTo(MatchPattern o) {
			return pattern.compareTo(o.pattern);
		}
	}

	public static class MatchPatternGroup {
		private String group;
		private ConcurrentSkipListSet<MatchPattern> patternSet = new ConcurrentSkipListSet<MatchPattern>();

		public MatchPatternGroup(String group) {
			this.group = group;
		}

		public boolean hasExactMatch(String str) {
			return patternSet.contains(str);
		}

		public boolean hasNoCaseSensitiveMatch(String str) {
			for (MatchPattern mp : patternSet) {
				if (mp.pattern.equalsIgnoreCase(str)) {
					return true;
				}
			}
			return false;
		}

		public boolean containsString(String str, boolean caseSensitive) {
			if (!caseSensitive) {
				str = str.toLowerCase();
			}
			for (MatchPattern mp : patternSet) {
				String s = mp.pattern;
				if (!caseSensitive) {
					s = s.toLowerCase();
				}
				if (s.indexOf(str) != -1) {
					return true;
				}
			}
			return false;
		}
		
		public boolean containsPattern(String str, boolean caseSensitive) {
			if (str == null || str.length() == 0) {
				return false;
			}
			if (!caseSensitive) {
				str = str.toLowerCase();
			}
			for (MatchPattern mp : patternSet) {
				String s = mp.pattern;
				if (!caseSensitive) {
					s = s.toLowerCase();
				}
				if (str.indexOf(s) != -1) {
					return true;
				}
			}
			return false;
		}

		public String getGroup() {
			return group;
		}
	}
}
