package clinical.web.services;

import gnu.trove.TIntHashSet;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;

import clinical.server.dao.GroupAnalysisDataDAO;
import clinical.server.vo.GroupAnalysisData;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.Jobs;
import clinical.utils.AFNIBrikReader;
import clinical.utils.Assertion;
import clinical.utils.CSVParser;
import clinical.utils.DateTimeUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.vo.CBFProcessReportRec;
import clinical.web.download.Unpacker;
import clinical.web.exception.BaseException;
import clinical.web.vo.CBFROIJobAssociation;
import clinical.web.vo.CBFROIJobAssociation.JobVisitInfo;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.workflow.cbf.AFNIBrik;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CBFROIGroupAnalysisServiceImpl extends AbstractServiceImpl
		implements ICBFROIGroupAnalysisService {
	protected IDBCache dbCache;

	public CBFROIGroupAnalysisServiceImpl(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(CBFROIGroupAnalysisServiceImpl.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	public List<CBFROIJobAssociation> populateAssociationsWithJobProvDetails(
			UserInfo ui, List<CBFROIJobAssociation> finalAssocList)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			return populateJobDirectories(con, finalAssocList);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void prepareROIProcessInputData(UserInfo ui, File inputRootDir,
			List<CBFROIJobAssociation> finalAssocList) throws Exception {

		Assertion.assertTrue(inputRootDir.isDirectory());

		List<ROIProcInputFileRec> rows = new ArrayList<ROIProcInputFileRec>();
		Map<String, Integer> subjectCountMap = new HashMap<String, Integer>();
		Map<String, JobVisitInfo> jviMap = new HashMap<String, JobVisitInfo>();
		TIntHashSet roiNoUnionSet = new TIntHashSet();
		for (CBFROIJobAssociation assoc : finalAssocList) {
			String subjectID = assoc.getSubjectId();
			File subjectDir = new File(inputRootDir, subjectID);
			Integer count = subjectCountMap.get(subjectID);
			if (count == null) {
				count = new Integer(1);
				subjectCountMap.put(subjectID, count);
			} else {
				count = new Integer(count.intValue() + 1);
				subjectCountMap.put(subjectID, count);
			}
			File maskWorkDir = new File(subjectDir, count.toString());
			if (!maskWorkDir.mkdirs()) {
				log.error("Cannot create mask directory: " + maskWorkDir);
				throw new Exception("Cannot create mask directory!");
			}
			File maskDir = new File(maskWorkDir, "Mask");
			if (!maskDir.mkdir()) {
				log.error("Cannot create mask directory: " + maskDir);
				throw new Exception("Cannot create mask directory!");
			}
			List<File> maskFiles = assoc.getRoiMaskBrik()
					.copyTo(maskDir, false);
			File maskBrikFile = null;
			for (File f : maskFiles) {
				if (AFNIBrik.isBrik(f.getName())) {
					maskBrikFile = f;
					break;
				}
			}

			AFNIBrikReader brikReader = new AFNIBrikReader(
					maskBrikFile.getAbsolutePath());
			float[] brikData = brikReader.load();
			TIntHashSet maskValues = AFNIBrikReader
					.getUniqueMaskValues(brikData);
			roiNoUnionSet = AFNIBrikReader.union(roiNoUnionSet, maskValues);

			ROIProcInputFileRec rpifRec = new ROIProcInputFileRec(
					maskWorkDir.getAbsolutePath(), subjectID,
					maskBrikFile.getAbsolutePath());
			rows.add(rpifRec);

			for (JobVisitInfo jvi : assoc.getCandidates()) {
				// File derivedDir = new File( jvi.getDataURI() );
				File jobDir = new File(maskWorkDir, jvi.getJobID());
				if (!jobDir.mkdir()) {
					log.error("Cannot create mask job directory: " + jobDir);
					throw new Exception("Cannot create mask job directory!");
				}
				jviMap.put(jvi.getJobID(), jvi);
				List<File> allFiles = FileUtils.getAllFilesUnderDir(jvi
						.getDataURI());
				for (File f : allFiles) {
					if (f.getName().startsWith("CBF+orig")) {
						File destFile = new File(jobDir, f.getName());
						FileUtils.copyFile(f.getAbsolutePath(),
								destFile.getAbsolutePath());
					}
				}

				// copy also original briks (not used for ROI group analysis

				// File rawVisitDir = new
				// File(jvi.getDataURI()).getParentFile();
				// copyOrigRawBriks(rawVisitDir, jobDir);

				rpifRec.addJobID(jvi.getJobID());

			}
		}

		int[] sortedROINoList = AFNIBrikReader
				.getSortedROINoList(roiNoUnionSet);
		// CBFROILookupService rls = CBFROILookupService.getInstance();

		BufferedWriter out = null;
		File roiLabelCSVFile = new File(inputRootDir, "roi_labels.csv");
		try {
			out = new BufferedWriter(new FileWriter(roiLabelCSVFile));
			for (int roiNo : sortedROINoList) {
			//	String label = rls.getLabel(CBFROILookupService.FREESURFER,
			//			roiNo);
			//	Assertion.assertNotNull(label, "Not a valid ROI number:"
			//			+ roiNo);
				// TODO needs to be more generic
				out.write("Mean_" + roiNo);
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}

		File inputCSVFile = new File(inputRootDir, "input.csv");
		out = null;
		try {
			out = new BufferedWriter(new FileWriter(inputCSVFile));
			for (ROIProcInputFileRec rec : rows) {
				String line = rec.toCSV();
				out.write(line);
				out.newLine();
				out.flush();
			}
		} finally {
			FileUtils.close(out);
		}

		// job visit id association table
		List<JobVisitAssociationRec> jvAssocList = new ArrayList<JobVisitAssociationRec>(
				jviMap.size());
		for (JobVisitInfo jvi : jviMap.values()) {
			jvAssocList.add(new JobVisitAssociationRec(jvi.getJobID(), jvi
					.getExpName(), jvi.getSubjectID(), jvi.getVisitID(), jvi
					.getVisitDate()));
		}

		Collections.sort(jvAssocList, new Comparator<JobVisitAssociationRec>() {
			@Override
			public int compare(JobVisitAssociationRec o1,
					JobVisitAssociationRec o2) {
				int diff = o1.expName.compareTo(o2.expName);
				if (diff == 0) {
					diff = o1.subjectID.compareTo(o2.subjectID);
				}
				if (diff == 0) {
					diff = o1.visitID - o2.visitID;
				}
				return diff;
			}
		});
		File jobVisitAssocCSVFile = new File(inputRootDir,
				"job_visit_associations.csv");
		out = null;
		try {
			out = new BufferedWriter(new FileWriter(jobVisitAssocCSVFile));
			out.write(JobVisitAssociationRec.getHeader());
			out.newLine();
			for (JobVisitAssociationRec rec : jvAssocList) {
				out.write(rec.toCSV());
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}

	}

	@SuppressWarnings("unused")
	private void copyOrigRawBriks(File rawVisitDir, File destDir)
			throws IOException {
		File[] files = rawVisitDir.listFiles();
		File contentsFile = null;
		for (File f : files) {
			if (f.getName().equals("CONTENTS")) {
				contentsFile = f;
				break;
			}
		}
		Assertion.assertNotNull(contentsFile);
		List<String> lines = FileUtils
				.readLines(contentsFile.getAbsolutePath());
		Set<String> brikFilenames = new HashSet<String>();
		for (String line : lines) {
			if (line.indexOf("AFNI-BRIK") != -1) {
				String[] toks = line.split("[\\t ]+");
				brikFilenames.add(toks[0]);
			}
		}
		if (brikFilenames.isEmpty()) {
			return;
		}
		for (File f : files) {
			if (brikFilenames.contains(f.getName())) {
				File destFile = new File(destDir, f.getName());
				FileUtils.copyFile(f.getAbsolutePath(),
						destFile.getAbsolutePath());
			}
		}

	}

	public static class ROIProcInputFileRec {
		private String workDir;
		private String subjectID;
		private String maskBrikFile;
		private List<String> jobIDList = new ArrayList<String>(2);

		public ROIProcInputFileRec(String workDir, String subjectID,
				String maskBrikFile) {
			super();
			this.workDir = workDir;
			this.subjectID = subjectID;
			this.maskBrikFile = maskBrikFile;
		}

		public void addJobID(String jobID) {
			this.jobIDList.add(jobID);
		}

		public String toCSV() {
			StringBuilder sb = new StringBuilder(256);
			sb.append(workDir).append(',');
			sb.append(subjectID).append(',');
			sb.append(maskBrikFile).append(',');
			for (Iterator<String> it = jobIDList.iterator(); it.hasNext();) {
				String jobID = it.next();
				sb.append(jobID);
				if (it.hasNext()) {
					sb.append(',');
				}
			}
			return sb.toString();
		}
	}// ;

	public static class JobVisitAssociationRec {
		private String jobID;
		private String expName;
		private String subjectID;
		private int visitID;
		private String visitDate;

		public JobVisitAssociationRec(String jobID, String expName,
				String subjectID, int visitID, String visitDate) {
			super();
			this.jobID = jobID;
			this.expName = expName;
			this.subjectID = subjectID;
			this.visitID = visitID;
			this.visitDate = visitDate;
		}

		public static String getHeader() {
			return "Job ID,Project Name,Subject ID,Visit ID,Visit Date";
		}

		public String toCSV() {
			StringBuilder sb = new StringBuilder(80);
			sb.append(jobID).append(',');
			sb.append(expName).append(',');
			sb.append(subjectID).append(',');
			sb.append(visitID).append(',');
			sb.append(visitDate);
			return sb.toString();
		}

	}// ;

	private List<CBFROIJobAssociation> populateJobDirectories(Connection con,
			List<CBFROIJobAssociation> finalAssocList) throws Exception {
		List<CBFROIJobAssociation> filteredFinalAssocList = new ArrayList<CBFROIJobAssociation>(
				finalAssocList.size());
		Set<String> jobIDSet = new HashSet<String>();
		for (CBFROIJobAssociation assoc : finalAssocList) {
			if (assoc.getCandidates() != null
					&& !assoc.getCandidates().isEmpty()) {
				for (JobVisitInfo jvi : assoc.getCandidates()) {
					jobIDSet.add(jvi.getJobID());
				}
				filteredFinalAssocList.add(assoc);
			}
		}
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		StringBuilder sb = new StringBuilder(512);
		sb.append("select a.*, b.datauri from Jobs as a, JobProvenance as b ");
		sb.append("where a.uniqueid = b.jobUniqueid and ");
		sb.append("a.jobid in (");
		for (Iterator<String> it = jobIDSet.iterator(); it.hasNext();) {
			String jobID = it.next();
			sb.append("'").append(jobID).append("'");
			if (it.hasNext()) {
				sb.append(',');
			}
		}
		sb.append(')');
		List<?> results = tsp.executeQuery(con, sb.toString());
		Map<String, List<JobProvenanceWrapper>> map = new HashMap<String, List<JobProvenanceWrapper>>();
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Jobs job = (Jobs) row[0];
			JobProvenance jp = (JobProvenance) row[1];
			List<JobProvenanceWrapper> jpwList = map.get(job.getJobid());
			if (jpwList == null) {
				jpwList = new ArrayList<JobProvenanceWrapper>(1);
				map.put(job.getJobid(), jpwList);
			}
			jpwList.add(new JobProvenanceWrapper(job, jp));
		}

		for (CBFROIJobAssociation assoc : filteredFinalAssocList) {
			for (JobVisitInfo jvi : assoc.getCandidates()) {
				List<JobProvenanceWrapper> jpwList = map.get(jvi.getJobID());
				Assertion.assertTrue(jpwList != null && !jpwList.isEmpty());
				if (jpwList.size() == 1) {
					JobProvenanceWrapper jpw = jpwList.get(0);
					// set the data uri and job uniqueid
					jvi.setDataURI(jpw.jp.getDatauri());
					jvi.setJobUniqueID(jpw.job.getUniqueid().longValue());
				} else {
					throw new RuntimeException(
							"Should not happen! More than one provenance record per job!");
				}
			}
		}
		return filteredFinalAssocList;
	}

	static class JobProvenanceWrapper {
		Jobs job;
		JobProvenance jp;

		public JobProvenanceWrapper(Jobs job, JobProvenance jp) {
			super();
			this.job = job;
			this.jp = jp;
		}

		public Jobs getJob() {
			return job;
		}

		public JobProvenance getJp() {
			return jp;
		}
	}// ;

	public static String combineCommmonPrefixWithKey(String commonPrefix,
			String key) {
		// check if any part of common prefix delimited by / is contained at the
		// beginning of the key
		if (countNumOfChar(commonPrefix, '/') <= 2) {
			return commonPrefix + key;
		}
		String[] toks = commonPrefix.split("\\/");
		List<String> parts = new ArrayList<String>(toks.length);
		for (String tok : toks) {
			if (tok.length() > 0) {
				parts.add(tok);
			}
		}

		int endIdx = -1;
		for (int i = 1; i < parts.size(); i++) {
			String prefix = buildPath(parts, i, false);
			if (key.startsWith(prefix)) {
				endIdx = i;
				break;
			}
		}
		if (endIdx > 0) {
			StringBuilder sb = new StringBuilder();
			sb.append('/');
			for (int i = 0; i < endIdx; i++) {
				sb.append(parts.get(i)).append('/');
			}
			return sb.toString() + key;
		}

		// fallback
		return commonPrefix + key;
	}

	public static String buildPath(List<String> parts, int startOffset,
			boolean includeSlashPrefix) {
		StringBuilder sb = new StringBuilder();
		if (includeSlashPrefix) {
			sb.append('/');
		}
		int len = parts.size();
		for (int i = startOffset; i < len; i++) {
			sb.append(parts.get(i)).append('/');
		}

		return sb.toString();
	}

	public static int countNumOfChar(String s, char c) {
		if (s.indexOf(c) == -1) {
			return 0;
		}
		int len = s.length();
		int tot = 0;
		for (int i = 0; i < len; i++) {
			if (s.charAt(i) == c) {
				tot++;
			}
		}
		return tot;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.ICBFROIGroupAnalysisService#handleROIMasksTarBall
	 * (clinical.web.common.UserInfo, java.lang.String, java.lang.String,
	 * java.util.List)
	 */
	@Override
	public List<CBFROIJobAssociation> handleROIMasksTarBall(UserInfo ui,
			String stagingDirRoot, String tarBallName,
			List<CBFProcessReportRec> cbfRecList) throws Exception {
		File archiveFile = new File(stagingDirRoot, tarBallName);

		String destDirName = tarBallName;
		destDirName = destDirName.replaceFirst("\\.\\w+$", "_files");
		File destDir = new File(stagingDirRoot, destDirName);
		if (!destDir.mkdirs()) {
			log.error("handleROIMasksTarBall: Cannot create staging directory!:"
					+ destDir);
			throw new Exception("Cannot create staging directory!");
		}

		Unpacker unpacker = new Unpacker(archiveFile.getAbsolutePath(),
				destDir.getAbsolutePath());

		unpacker.unpack();

		List<CBFROIJobAssociation> assocList = new ArrayList<CBFROIJobAssociation>();

		List<File> allFiles = FileUtils.getAllFilesUnderDir(destDir
				.getAbsolutePath());
		File assocFile = null;
		String destDirPath = destDir.getPath();
		Map<String, AFNIBrik> afniMap = new HashMap<String, AFNIBrik>();
		String commonPrefix = null;
		for (File f : allFiles) {
			String name = f.getName();
			if (AFNIBrik.isBrik(name)) {
				String relPath = f.getAbsolutePath();
				relPath = relPath.replaceFirst("\\.\\w+$", "");
				relPath = relPath.substring(destDirPath.length());
				if (commonPrefix == null) {
					commonPrefix = relPath;
				} else {
					commonPrefix = refineCommonPrefix(commonPrefix, relPath);
				}
				if (commonPrefix.length() == 0) {
					break;
				}
			}
		}

		for (File f : allFiles) {
			String name = f.getName();
			if (AFNIBrik.isAFNIBrikFile(name)) {
				String key = f.getAbsolutePath();
				key = key.replaceFirst("\\.\\w+$", "");
				key = key.substring(destDirPath.length());
				AFNIBrik brik = afniMap.get(key);
				boolean headFile = AFNIBrik.isHeader(name);
				if (brik == null) {
					if (headFile) {
						brik = new AFNIBrik(f, null);
					} else {
						brik = new AFNIBrik(null, f);
					}
					afniMap.put(key, brik);
				} else {
					AFNIBrik newBrik;
					if (headFile) {
						newBrik = new AFNIBrik(f, brik.getBrikFile());
					} else {
						newBrik = new AFNIBrik(brik.getHeaderFile(), f);
					}
					afniMap.put(key, newBrik);
				}
			} else if (name.equals("association.csv") && name.endsWith(".csv")) {
				assocFile = f;
			}
		}
		if (assocFile == null) {
			throw new Exception(
					"No association file 'association.csv' is provided in the tar file!");
		}

		Map<String, List<CBFProcessReportRec>> subjCBFRecMap = new HashMap<String, List<CBFProcessReportRec>>();
		for (CBFProcessReportRec cbfRec : cbfRecList) {
			String subjectID = cbfRec.getSubjectID();
			List<CBFProcessReportRec> list = subjCBFRecMap.get(subjectID);
			if (list == null) {
				list = new ArrayList<CBFProcessReportRec>(5);
				subjCBFRecMap.put(subjectID, list);
			}
			list.add(cbfRec);
		}

		if (assocFile != null) {
			List<AssocTableRec> atrList = readAssociations(assocFile);
			for (AssocTableRec atr : atrList) {
				CBFROIJobAssociation roiJobAssoc = null;
				String key = atr.brikFileRelPath;
				key = key.replaceFirst("\\.\\w+$", "");
				AFNIBrik brik = afniMap.get(key);
				if (brik == null && commonPrefix.length() > 0) {
					String key2 = commonPrefix + key;
					brik = afniMap.get(key2);
					if (brik == null) {
						String key3 = combineCommmonPrefixWithKey(commonPrefix,
								key);
						brik = afniMap.get(key3);
					}
				}
				if (brik != null) {
					roiJobAssoc = new CBFROIJobAssociation(brik, atr.subjectID,
							atr.visitDate, destDirPath);
					assocList.add(roiJobAssoc);
					Assertion.assertNotNull(atr.subjectID);
					List<CBFProcessReportRec> list = subjCBFRecMap
							.get(atr.subjectID);
					if (list != null) {
						for (CBFProcessReportRec cbfRec : list) {
							String visitDate = DateTimeUtils.formatDate(cbfRec
									.getVisitDate());
							if (atr.visitDate != null) {
								Date date = DateTimeUtils.toDate(atr.visitDate);
								if (date != null
										&& cbfRec.getVisitDate()
												.compareTo(date) != 0) {
									// skip the record
									continue;
								}
							}
							String label = cbfRec.getTag();
							int visitID = GenUtils.toInt(cbfRec.getVisitId(),
									-1);
							CBFROIJobAssociation.JobVisitInfo jvi = new JobVisitInfo(
									cbfRec.getJobID(), cbfRec.getSubjectID(),
									visitID, cbfRec.getProjectName(),
									visitDate, label);
							roiJobAssoc.addCandidate(jvi);
						}
					}

				}
			}
		}
		return assocList;
	}

	public List<GroupAnalysisData> addDerivedDataWithJobProvenance(UserInfo ui,
			List<GroupDerivedDataInfo> gddiList, JobProvenanceInfo jpi)
			throws Exception {
		if (gddiList == null || gddiList.isEmpty()) {
			return new ArrayList<GroupAnalysisData>(0);
		}
		IJobProvenanceService jps = ServiceFactory
				.getJobProvenanceService(theDBID);
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			List<GroupAnalysisData> gadList = saveDerivedData(ui, jpi.getJob()
					.getUniqueid(), gddiList, con);
			jps.saveJobProvenance(con, jpi);
			con.commit();
			return gadList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addDerivedDataWithJobProvenance", x,
					true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	private List<GroupAnalysisData> saveDerivedData(UserInfo ui,
			BigDecimal jobUniqueID, List<GroupDerivedDataInfo> gddiList,
			Connection con) throws Exception {
		GroupAnalysisDataDAO dao = DAOFactory
				.createGroupAnalysisDataDAO(theDBID);
		GroupAnalysisData cr = new GroupAnalysisData();
		cr.setJobUniqueid(jobUniqueID);
		List<GroupAnalysisData> existingGadList = dao.find(con, cr);
		Map<String, GroupAnalysisData> existingMap = new HashMap<String, GroupAnalysisData>();
		for (GroupAnalysisData gad : existingGadList) {
			existingMap.put(gad.getDatauri(), gad);
		}
		existingGadList = null;
		List<GroupAnalysisData> gadList = new ArrayList<GroupAnalysisData>(
				gddiList.size());
		for (GroupDerivedDataInfo gddi : gddiList) {
			if (existingMap.containsKey(gddi.getDataURI())) {
				GroupAnalysisData gad = existingMap.get(gddi.getDataURI());
				gadList.add(gad);
				continue;
			}
			GroupAnalysisData gad = addGroupAnalysisData(con, ui, gddi,
					jobUniqueID);
			gadList.add(gad);
		}

		return gadList;
	}

	private GroupAnalysisData addGroupAnalysisData(Connection con, UserInfo ui,
			GroupDerivedDataInfo gddi, BigDecimal jobUniqueID) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		GroupAnalysisDataDAO dao = DAOFactory
				.createGroupAnalysisDataDAO(theDBID);
		GroupAnalysisData gad = new GroupAnalysisData();
		gad.setJobUniqueid(jobUniqueID);
		gad.setModtime(new Date());
		gad.setDatauri(gddi.getDataURI());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.DERIVED_DATA, "uniqueid");
		gad.setUniqueid(uniqueid);

		dao.insert(con, gad);
		return gad;
	}

	private String refineCommonPrefix(String commonPrefix, String relPath) {
		if (relPath.startsWith(commonPrefix)) {
			return commonPrefix;
		}
		char[] carr = commonPrefix.toCharArray();
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < carr.length; i++) {
			if (relPath.charAt(i) == carr[i]) {
				sb.append(carr[i]);
			} else {
				break;
			}
		}
		String s = sb.toString();
		if (!s.endsWith("/")) {
			int idx = s.lastIndexOf('/');
			if (idx == -1) {
				return "";
			} else {
				return s.substring(0, idx + 1);
			}
		}
		return s;
	}

	private List<AssocTableRec> readAssociations(File assocFile)
			throws IOException {
		CSVParser csvParser = new CSVParser();
		csvParser.extractData(assocFile.getAbsolutePath());
		List<List<String>> rows = csvParser.getRows();
		List<AssocTableRec> atrList = new ArrayList<AssocTableRec>(rows.size());
		for (List<String> row : rows) {
			int len = row.size();
			if (len == 0) {
				continue;
			}
			AssocTableRec atr = new AssocTableRec();
			if (len > 0) {
				atr.brikFileRelPath = row.get(0);
			}
			if (len > 1) {
				atr.subjectID = row.get(1);
			}
			if (len > 2) {
				atr.visitDate = row.get(2);
			}
			atrList.add(atr);
		}
		return atrList;
	}

	private static class AssocTableRec {
		String brikFileRelPath;
		String subjectID;
		String visitDate;
	}

}
