package clinical.web.services;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Experiment;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.Jobs;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.vo.CBFProcessReportRec;
import clinical.web.exception.BaseException;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTable;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTableRow;
import clinical.web.vo.CBFROIJobAssociation.JobVisitInfo;
import clinical.web.vo.GroupAnalysisCBFJobInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CBFStandardSpaceGroupAnalysisServiceImpl extends
		AbstractServiceImpl implements ICBFStandardSpaceGroupAnalysisService {
	protected IDBCache dbCache;

	public CBFStandardSpaceGroupAnalysisServiceImpl(String dbID)
			throws BaseException {
		super(dbID);
		log = LogFactory.getLog(CBFStandardSpaceGroupAnalysisServiceImpl.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	/* (non-Javadoc)
	 * @see clinical.web.services.ICBFStandardSpaceGroupAnalysisService#prepareProcessInputData(clinical.web.common.UserInfo, java.io.File, java.util.List)
	 */
	@Override
	public void prepareProcessInputData(UserInfo ui, File inputRootDir,
			List<GroupAnalysisCBFJobInfo> gajiList) throws Exception {
		Assertion.assertTrue(inputRootDir.isDirectory());
		List<StandardSpaceProcInputFileRec> sspifList = new ArrayList<StandardSpaceProcInputFileRec>();
		for (GroupAnalysisCBFJobInfo gaji : gajiList) {
			String subjectID = gaji.getSubjectId();
			String dataURI = gaji.getJvi().getDataURI();
			int idx = dataURI.indexOf(subjectID);
			Assertion.assertTrue(idx != -1);
			String relPart = dataURI.substring(idx);
			String projectDirname = gaji.getJvi().getExpName();
			projectDirname = projectDirname.replaceAll("\\s+", "_");
			File jobDir = new File(inputRootDir, projectDirname);
			jobDir = new File(jobDir, relPart);
			if (!jobDir.mkdirs()) {
				log.error("Cannot create job input directory: " + jobDir);
				throw new Exception("Cannot create job input directory!");
			}

			String site = null;
			String expName = gaji.getJvi().getExpName();
			idx = expName.lastIndexOf('_');
			if (idx != -1) {
				site = expName.substring(idx + 1);
				if (!CBFWFManagementService.isAKnownSite(site)) {
					site = null;
				}
			}
			
			StandardSpaceProcInputFileRec sspif = new StandardSpaceProcInputFileRec(
					subjectID, gaji.getVisitDate(), gaji.getJvi().getJobID(),
					jobDir.getAbsolutePath(), expName, site);
			sspifList.add(sspif);

			List<File> allFiles = FileUtils.getAllFilesUnderDir(gaji.getJvi()
					.getDataURI());
			for (File f : allFiles) {
				String name = f.getName();
				if (f.getName().startsWith("CBF+orig")) {
					File destFile = new File(jobDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				} else if (name.startsWith("anat")
						&& name.indexOf("reg+stripped+orig") != -1) {
					File destFile = new File(jobDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					if (sspif.anatBrikName == null) {
						sspif.anatBrikName = name.replaceFirst("\\.\\w+$", "");
					}
				} else if (name.equals("gm.txt")) {
					File destFile = new File(jobDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		File inputCSVFile = new File(inputRootDir, "input.csv");
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(inputCSVFile));
			for (StandardSpaceProcInputFileRec sspif : sspifList) {
				out.write(sspif.toCSV());
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}

		CSVTable table = loadAssessmentData(ui, gajiList);

		File assessmentCSVFile = new File(inputRootDir, "assessment.csv");
		out = null;
		try {
			out = new BufferedWriter(new FileWriter(assessmentCSVFile));
			out.write(table.header.toCSV());
			out.newLine();
			for (String rowKey : table.rowKeys) {
				CSVTableRow row = table.rowMap.get(rowKey);
				out.write(row.toCSV(table.allScoreLabels));
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}
		
		// sanity check
		List<List<String>> asRows = WFGenUtils.loadCSV(assessmentCSVFile);
		int numCols = asRows.get(0).size();
		for(List<String> row : asRows) {
			if (row.size() != numCols) {
				log.warn("row.size:" + row.size() + " numCols:" + numCols);
				GenUtils.showRow("Bad assessment row", row);
			}
			Assertion.assertTrue(row.size() == numCols);
		}
		
	}

	/* (non-Javadoc)
	 * @see clinical.web.services.ICBFStandardSpaceGroupAnalysisService#loadAssessmentData(clinical.web.common.UserInfo, java.util.List)
	 */
	@Override
	public CSVTable loadAssessmentData(UserInfo ui,
			List<GroupAnalysisCBFJobInfo> gajiList) throws Exception {
		IBatchAssessmentService bas = ServiceFactory
				.getBatchAssessmentService(this.theDBID);
		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		Map<String, Experiment> expMap = new HashMap<String, Experiment>();
		for (Experiment exp : experiments) {
			expMap.put(exp.getName(), exp);
		}
		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>();
		Set<Integer> seenExpIdSet = new HashSet<Integer>();
		for (GroupAnalysisCBFJobInfo gaji : gajiList) {
			String subjectID = gaji.getSubjectId();
			seenExpIdSet.clear();
			Experiment exp = expMap.get(gaji.getJvi().getExpName());
			if (exp != null) {
				Integer expId = new Integer(exp.getUniqueid().intValue());
				List<String> list = expSubjectIDListMap.get(expId);
				if (list == null) {
					list = new LinkedList<String>();
					expSubjectIDListMap.put(expId, list);
				}
				if (!list.contains(subjectID)) {
					list.add(subjectID);
				}
			}
		}
		List<SubjectAsScoreValueSummary> sasvsList = bas
				.getAllAssessmentValues(ui, expSubjectIDListMap);

		CSVTable csvTable = bas.prepCSVTable(ui, sasvsList);
		return csvTable;
	}

	public static class StandardSpaceProcInputFileRec {
		private String workDir;
		private String subjectID;
		private String projectName;
		private String visitDate;
		private String jobID;
		private String site;
		private String anatBrikName;

		public StandardSpaceProcInputFileRec(String subjectID,
				String visitDate, String jobID, String workDir, String projectName, String site) {
			super();
			this.subjectID = subjectID;
			this.visitDate = visitDate;
			this.jobID = jobID;
			this.workDir = workDir;
			this.projectName = projectName;
			this.site = site;
		}

		public String toCSV() {
			StringBuilder sb = new StringBuilder(256);
			sb.append(workDir).append(',');
			sb.append(subjectID).append(',');
			sb.append(projectName).append(',');
			sb.append(visitDate).append(',');
			sb.append(jobID).append(',');
			if (site != null) {
				sb.append(site).append(',');
			} else {
				sb.append(',');
			}
			sb.append(anatBrikName);
			return sb.toString();
		}
	} // ;

	/* (non-Javadoc)
	 * @see clinical.web.services.ICBFStandardSpaceGroupAnalysisService#toGroupAnalysisJobInfoList(clinical.web.common.UserInfo, java.util.List)
	 */
	@Override
	public List<GroupAnalysisCBFJobInfo> toGroupAnalysisJobInfoList(
			UserInfo ui, List<CBFProcessReportRec> cbfRecList) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			return toGroupAnalysisJobInfoList(con, cbfRecList);
		} finally {
			releaseConnection(con, ui);
		}
	}

	/* (non-Javadoc)
	 * @see clinical.web.services.ICBFStandardSpaceGroupAnalysisService#toGroupAnalysisJobInfoList(java.sql.Connection, java.util.List)
	 */
	@Override
	public List<GroupAnalysisCBFJobInfo> toGroupAnalysisJobInfoList(
			Connection con, List<CBFProcessReportRec> cbfRecList)
			throws Exception {

		Set<String> jobIDSet = new HashSet<String>();
		for (CBFProcessReportRec cbfRec : cbfRecList) {
			jobIDSet.add(cbfRec.getJobID());
		}

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(512);
		sb.append("select a.*, b.datauri from Jobs as a, JobProvenance as b ");
		sb.append("where a.uniqueid = b.jobUniqueid and ");
		sb.append("a.jobid in (");
		for (Iterator<String> it = jobIDSet.iterator(); it.hasNext();) {
			String jobID = it.next();
			sb.append("'").append(jobID).append("'");
			if (it.hasNext()) {
				sb.append(',');
			}
		}
		sb.append(')');
		List<?> results = tsp.executeQuery(con, sb.toString());
		Map<String, List<JobProvenanceWrapper>> map = new HashMap<String, List<JobProvenanceWrapper>>();
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Jobs job = (Jobs) row[0];
			JobProvenance jp = (JobProvenance) row[1];
			List<JobProvenanceWrapper> jpwList = map.get(job.getJobid());
			if (jpwList == null) {
				jpwList = new ArrayList<JobProvenanceWrapper>(1);
				map.put(job.getJobid(), jpwList);
			}
			jpwList.add(new JobProvenanceWrapper(job, jp));
		}

		List<GroupAnalysisCBFJobInfo> gajiList = new ArrayList<GroupAnalysisCBFJobInfo>(
				cbfRecList.size());
		for (CBFProcessReportRec cbfRec : cbfRecList) {
			// FIXME
			String rootDir = null;
			String visitDate = DateTimeUtils.formatDate(cbfRec.getVisitDate());
			String label = cbfRec.getTag();
			int visitID = GenUtils.toInt(cbfRec.getVisitId(), -1);
			JobVisitInfo jvi = new JobVisitInfo(cbfRec.getJobID(),
					cbfRec.getSubjectID(), visitID, cbfRec.getProjectName(),
					visitDate, label);
			GroupAnalysisCBFJobInfo gaji = new GroupAnalysisCBFJobInfo(
					cbfRec.getSubjectID(), visitDate, rootDir, jvi);
			gajiList.add(gaji);
		}

		for (GroupAnalysisCBFJobInfo gaji : gajiList) {
			List<JobProvenanceWrapper> jpwList = map.get(gaji.getJvi()
					.getJobID());
			Assertion.assertTrue(jpwList != null && !jpwList.isEmpty());
			if (jpwList.size() == 1) {
				JobProvenanceWrapper jpw = jpwList.get(0);
				// set the data uri and job uniqueid
				gaji.getJvi().setDataURI(jpw.jp.getDatauri());
				gaji.getJvi().setJobUniqueID(jpw.job.getUniqueid().longValue());
			} else {
				throw new RuntimeException(
						"Should not happen! More than one provenance record per job!");
			}
		}

		return gajiList;
	}

	static class JobProvenanceWrapper {
		Jobs job;
		JobProvenance jp;

		public JobProvenanceWrapper(Jobs job, JobProvenance jp) {
			super();
			this.job = job;
			this.jp = jp;
		}

		public Jobs getJob() {
			return job;
		}

		public JobProvenance getJp() {
			return jp;
		}
	}// ;
}
