package clinical.web.services;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import clinical.server.dao.DeriveddataDAO;
import clinical.server.dao.ExpcomponentDAO;
import clinical.server.dao.RawdataDAO;
import clinical.server.vo.Deriveddata;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Rawdata;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: DataArchivalService.java 629 2012-07-18 01:21:58Z bozyurt $
 */
public class DataArchivalService extends AbstractServiceImpl implements
		IDataArchivalService {
	protected String dataRoot;
	protected String archiveRoot;
	public final static String ARCHIVE_ROOT_NAME = ".archive";

	public DataArchivalService(String dbID) throws BaseException {
		super(dbID);

		IAppConfigService configService = ServiceFactory.getAppConfigService();

		this.dataRoot = configService.getParamValue(Constants.CBFBIRN_DATA_ROOT_KEY); // cbfbirn.data.root
		File f = new File(this.dataRoot, ARCHIVE_ROOT_NAME);
		if (!f.exists()) {
			f.mkdir();
		}
		this.archiveRoot = f.getAbsolutePath();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IDataArchivalService#archiveSubject(clinical.web
	 * .common.UserInfo, java.lang.String, java.lang.Integer)
	 */
	public File[] archiveSubject(UserInfo ui, String subjectID, Integer expID)
			throws Exception {
		List<Expcomponent> visits = null;
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			ExpcomponentDAO dao = DAOFactory
					.createExpcomponentDAO(this.theDBID);
			Expcomponent cr = new Expcomponent();

			cr.setSubjectid(subjectID);
			cr.setNcExperimentUniqueid(new BigDecimal(expID));

			visits = dao.find(con, cr);
		} finally {
			releaseConnection(con, ui);
		}
		
		List<File[]> srcDestVisitDirs = new ArrayList<File[]>(visits.size());
		for (Expcomponent visit : visits) {
			int visitID = visit.getComponentid().intValue();
			File[] pair = archiveVisit(ui, subjectID, expID, visitID);
			srcDestVisitDirs.add(pair);
		}

		if (srcDestVisitDirs.isEmpty()) {
			return null;
		}
		File[] pair = srcDestVisitDirs.get(0);
		File[] result = new File[2];
		result[0] = pair[0].getParentFile();
		result[1] = pair[1].getParentFile();
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IDataArchivalService#archiveVisit(clinical.web.
	 * common.UserInfo, java.lang.String, java.lang.Integer, java.lang.Integer)
	 */
	public File[] archiveVisit(UserInfo ui, String subjectID, Integer expID,
			Integer visitID) throws Exception {
		List<String> srcPaths = new ArrayList<String>();
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());

			List<Rawdata> rawDataList = findRawData(con, subjectID, expID,
					visitID);
			List<Deriveddata> derivedDataList = findDerivedData(con, subjectID,
					expID, visitID);

			for (Rawdata rd : rawDataList) {
				populateSourcePaths(srcPaths, rd.getDatauri());
			}

			for (Deriveddata dd : derivedDataList) {
				populateSourcePaths(srcPaths, dd.getDatauri());
			}

		} finally {
			releaseConnection(con, ui);
		}
		if (srcPaths.isEmpty()) {
			log.info("No source directory to archive!");
			return new File[0];
		}

		PathPair pathPair = ensureUniqArchiveVisitDirectory(srcPaths.get(0),
				this.dataRoot); // baseUri => dataRoot
		File contentsFile = new File(this.dataRoot, pathPair.getSrcRelPath());
		contentsFile = new File(contentsFile, "CONTENTS");
		if (contentsFile.exists() && contentsFile.isFile()) {
			srcPaths.add(contentsFile.getAbsolutePath());
		}

		for (String srcPath : srcPaths) {
			String destPath = prepDestPath(srcPath, this.dataRoot,
					pathPair);
			FileUtils.copyFile(srcPath, destPath);
		}

		File srcVisitDir = new File(this.dataRoot, pathPair.getSrcRelPath());
		File[] result = new File[2];
		result[0] = srcVisitDir;
		result[1] = new File(pathPair.getDestPath());
		return result;
	}

	private void populateSourcePaths(List<String> srcPaths, String dataURI) {
		File f = new File(dataURI);
		Assertion.assertTrue(f.exists());
		if (f.isFile()) {
			srcPaths.add(dataURI);
		} else {
			List<File> files = FileUtils.getAllFilesUnderDir(f
					.getAbsolutePath());
			for (File file : files) {
				srcPaths.add(file.getAbsolutePath());
			}
		}
	}

	protected PathPair ensureUniqArchiveVisitDirectory(String srcPath,
			String srcBase) {
		Assertion.assertTrue(srcPath.startsWith(srcBase));

		String srcRelPath = srcPath.substring(srcBase.length());
		if (srcRelPath.startsWith(File.separator)) {
			srcRelPath = srcRelPath.substring(1);
		}

		// find the Visit directory (the top dir as far as uniqueness is
		// concerned
		int lidx = srcRelPath.lastIndexOf("Visit_");
		Assertion.assertTrue(lidx != -1);
		lidx = srcRelPath.indexOf(File.separator, lidx);
		Assertion.assertTrue(lidx != -1);
		String relDirPart = srcRelPath.substring(0, lidx);
		File destDir = new File(this.archiveRoot, relDirPart);
		if (destDir.exists()) {
			int versionNumber = 1;
			do {
				destDir = new File(this.archiveRoot, relDirPart + "."
						+ versionNumber);
				versionNumber++;
			} while (destDir.exists());
		}
		destDir.mkdirs();
		Assertion.assertTrue(destDir.exists());
		return new PathPair(relDirPart, destDir.getAbsolutePath());
	}

	protected String prepDestPath(String srcPath, String srcBase,
			PathPair pathPair) {
		if (srcPath.startsWith(srcBase)) {
			String srcRelPath = srcPath.substring(srcBase.length());
			if (srcRelPath.startsWith(File.separator)) {
				srcRelPath = srcRelPath.substring(1);
			}
			int lidx = srcRelPath.lastIndexOf(File.separator);
			Assertion.assertTrue(lidx != -1);
			String relDirPart = srcRelPath.substring(0, lidx);
			String filename = srcRelPath.substring(lidx + 1);
			Assertion.assertTrue(relDirPart
					.startsWith(pathPair.getSrcRelPath()));
			relDirPart = relDirPart
					.substring(pathPair.getSrcRelPath().length());

			File destDir = new File(pathPair.getDestPath(), relDirPart);

			destDir.mkdirs();
			Assertion.assertTrue(destDir.exists());

			return new File(destDir, filename).getAbsolutePath();
		} else {
			return null;
		}
	}

	protected List<Rawdata> findRawData(Connection con, String subjectID,
			Integer expID, Integer visitID) throws Exception {
		Assertion.assertNotNull(subjectID);
		RawdataDAO dao = DAOFactory.createRawdataDAO(theDBID);
		Rawdata cr = new Rawdata();
		cr.setSubjectid(subjectID);
		if (expID != null) {
			cr.setNcExperimentUniqueid(new BigDecimal(expID));
		}
		if (visitID != null) {
			cr.setComponentid(new BigDecimal(visitID));
		}
		return dao.find(con, cr);
	}

	protected List<Deriveddata> findDerivedData(Connection con,
			String subjectID, Integer expID, Integer visitID) throws Exception {
		Assertion.assertNotNull(subjectID);
		DeriveddataDAO dao = DAOFactory.createDeriveddataDAO(theDBID);
		Deriveddata cr = new Deriveddata();
		cr.setSubjectid(subjectID);
		if (expID != null) {
			cr.setNcExperimentUniqueid(new BigDecimal(expID));
		}
		if (visitID != null) {
			cr.setComponentid(new BigDecimal(visitID));
		}
		return dao.find(con, cr);
	}

	public static class PathPair {
		String srcRelPath;
		String destPath;

		public PathPair(String srcRelPath, String destPath) {
			this.srcRelPath = srcRelPath;
			this.destPath = destPath;
		}

		public String getSrcRelPath() {
			return srcRelPath;
		}

		public String getDestPath() {
			return destPath;
		}
	}// ;
}
