package clinical.web.services;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONObject;

import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentdata;
import clinical.server.vo.Experiment;
import clinical.server.vo.GroupAnalysisData;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.Jobs;
import clinical.server.vo.Storedassessment;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.vo.Factor;
import clinical.web.common.vo.Factor.LevelCount;
import clinical.web.common.vo.GroupCBFReportRec;
import clinical.web.common.vo.GroupCBFReportRec.AugmentedJobVisitInfo;
import clinical.web.common.vo.GroupCBFReportRec.GAResult;
import clinical.web.common.vo.GroupCBFReportRec.ROISummary;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobInfo;
import clinical.web.vo.CBFROIJobAssociation;
import clinical.web.vo.CBFROIJobAssociation.JobVisitInfo;
import clinical.web.vo.FactorRepeatedMeasureInfo;
import clinical.web.vo.GroupAnalysisCBFJobInfo;
import clinical.web.vo.upload.ScoreValue;
import clinical.web.workflow.cbf.group.GroupAnalysisContext;
import clinical.web.workflow.cbf.group.GroupROIAnalysisContext;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class GroupCBFReportQueryService extends AbstractServiceImpl implements
		IGroupCBFReportQueryService {
	private final IDBCache dbCache;

	public GroupCBFReportQueryService(String dbID) throws BaseException {
		super(dbID);
		dbCache = ServiceFactory.getDBCache(this.theDBID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IGroupCBFReportQueryService#getGroupCBFRecords(
	 * clinical.web.common.UserInfo, java.util.List)
	 */
	@Override
	public List<GroupCBFReportRec> getGroupCBFRecords(final UserInfo ui,
			final List<Experiment> allowedExps, Set<String> includedJobTypeSet)
			throws Exception {
		Connection con = null;
		try {
			con = this.pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			List<GroupCBFReportRec> recList = getGroupCBFRecords(ui, con,
					allowedExps, includedJobTypeSet);
			con.commit();
			return recList;
		} finally {
			if (con != null) {
				con.setAutoCommit(true);
			}
			releaseConnection(con, ui);
		}
	}

	public List<GroupAnalysisData> getGroupAnalysisDerivedData(
			final UserInfo ui, final String jobID) throws Exception {
		Connection con = null;
		try {
			con = this.pool.getConnection(ui.getName());
			return getGroupAnalysisDerivedData(con, jobID);
		} finally {
			releaseConnection(con, ui);
		}
	}

	private List<GroupAnalysisData> getGroupAnalysisDerivedData(Connection con,
			String jobID) throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(128);
		sb.append("select a.* from GroupAnalysisData as a, ");
		sb.append("Jobs as b where a.jobUniqueid = b.uniqueid and ");
		sb.append("b.jobid = '").append(jobID).append("'");

		List<?> results = tsp.executeQuery(con, sb.toString());
		List<GroupAnalysisData> gadList = new ArrayList<GroupAnalysisData>(
				results.size());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			GroupAnalysisData gad = (GroupAnalysisData) it.next();
			gadList.add(gad);
		}
		return gadList;
	}

	public List<FactorRepeatedMeasureInfo> getEligibleFactors(
			final UserInfo ui, final List<JobVisitInfo> jviList)
			throws Exception {
		Connection con = null;
		try {
			con = this.pool.getConnection(ui.getName());
			return getEligibleFactors(ui, con, jviList);
		} finally {
			releaseConnection(con, ui);
		}

	}

	List<FactorRepeatedMeasureInfo> getEligibleFactors(final UserInfo ui,
			final Connection con, final List<JobVisitInfo> jviList)
			throws Exception {
		Set<String> subjectIDSet = new HashSet<String>();
		// repeated measure
		Map<String, Integer> subjectID2IdxMap = new HashMap<String, Integer>();
		int i = 0;
		for (JobVisitInfo jvi : jviList) {
			if (!subjectIDSet.contains(jvi.getSubjectID())) {
				subjectID2IdxMap.put(jvi.getSubjectID(), i);
				i++;
			}
			subjectIDSet.add(jvi.getSubjectID());
		}

		List<Assessment> assessments = dbCache.getAssessments(ui, false);
		Map<Integer, Assessment> asMap = new HashMap<Integer, Assessment>();
		for (Assessment as : assessments) {
			asMap.put(as.getUniqueid().intValue(), as);
		}

		List<SubjectVisitAssessment> svaList = getAllCategoricalAssessmentData(
				con, subjectIDSet);
		Map<String, List<SubjectVisitAssessment>> svaMap = new HashMap<String, List<SubjectVisitAssessment>>();
		Set<Integer> uniqueAsIdSet = new HashSet<Integer>();
		// ASSUMPTION: assessments are not more granular than visits.
		for (SubjectVisitAssessment sva : svaList) {
			StringBuilder sb = new StringBuilder();
			sb.append(sva.subjectID).append(':').append(sva.expID).append(':')
					.append(sva.visitID);
			String key = sb.toString();
			List<SubjectVisitAssessment> list = svaMap.get(key);
			if (list == null) {
				list = new LinkedList<SubjectVisitAssessment>();
				svaMap.put(key, list);
			}
			list.add(sva);
			uniqueAsIdSet.add(sva.asID);
		}

		Map<String, Experiment> expMap = dbCache.getExperimentMap(ui, false);
		Map<String, Factor> factorMap = new HashMap<String, Factor>();
		Map<Factor, FactorRepeatedMeasureInfo> frmMap = new HashMap<Factor, FactorRepeatedMeasureInfo>();

		boolean ok = true;
		Set<Experiment> seenExpSet = new HashSet<Experiment>();
		for (JobVisitInfo jvi : jviList) {
			StringBuilder sb = new StringBuilder();

			Experiment exp = expMap.get(jvi.getExpName());
			Assertion.assertNotNull(exp);
			seenExpSet.add(exp);
			sb.append(jvi.getSubjectID()).append(':').append(exp.getUniqueid())
					.append(':').append(jvi.getVisitID());
			String key = sb.toString();
			List<SubjectVisitAssessment> list = svaMap.get(key);
			if (list != null) {
				for (SubjectVisitAssessment sva : list) {
					Assessment assessment = asMap.get(sva.asID);
					Assertion.assertNotNull(assessment);
					for (ScoreValue sv : sva.svList) {
						String factorKey = assessment.getName() + ":"
								+ sv.getName();
						Factor factor = factorMap.get(factorKey);
						if (factor == null) {
							if (sv.getValue() != null) {
								factor = new Factor(assessment.getName(),
										sv.getName());
								factorMap.put(factorKey, factor);
								factor.addLevel(sv.getValue());
								FactorRepeatedMeasureInfo frmi = new FactorRepeatedMeasureInfo(
										factor, subjectID2IdxMap);
								frmMap.put(factor, frmi);
								frmi.add(sv.getValue(), jvi.getSubjectID());
							}
						} else {
							if (sv.getValue() == null) {
								// any missing data invalidates the factor
								// candidate
								Factor removedFactor = factorMap
										.remove(factorKey);
								frmMap.remove(removedFactor);
							} else {
								factor.addLevel(sv.getValue());
								FactorRepeatedMeasureInfo frmi = frmMap
										.get(factor);
								frmi.add(sv.getValue(), jvi.getSubjectID());
							}
						}
					}
				}
			} else {
				ok = false;
				break;
			}
		}// jviList

		if (!ok || factorMap.isEmpty()) {
			return new ArrayList<FactorRepeatedMeasureInfo>(0);
		}

		for (Iterator<Factor> it = factorMap.values().iterator(); it.hasNext();) {
			Factor factor = it.next();
			List<LevelCount> levelSet = factor.getLevelSet();
			if (levelSet.size() < 2) {
				frmMap.remove(factor);
			} else {
				FactorRepeatedMeasureInfo frmi = frmMap.get(factor);
				// in a repeated measure design, each subject must only appear
				// in each level only once
				if (frmi.hasAnyViolations()) {
					frmMap.remove(factor);
				} else if (!frmi.hasAllSubjectsCovered()) {

					frmMap.remove(factor);
				}
			}
		}

		List<FactorRepeatedMeasureInfo> factorList = new ArrayList<FactorRepeatedMeasureInfo>(
				frmMap.values());
		if (factorList.isEmpty()) {
			return factorList;
		}
		if (seenExpSet.size() > 1) {
			Factor factor = new Factor(null, "Project");
			FactorRepeatedMeasureInfo frmi = new FactorRepeatedMeasureInfo(
					factor, subjectID2IdxMap);
			frmMap.put(factor, frmi);
			for (JobVisitInfo jvi : jviList) {
				Experiment exp = expMap.get(jvi.getExpName());
				String expName = exp.getName();
				frmi.add(expName, jvi.getSubjectID());
				factor.addLevel(expName);
			}
			factorList.add(frmi);
		}

		Collections.sort(factorList,
				new Comparator<FactorRepeatedMeasureInfo>() {
					@Override
					public int compare(FactorRepeatedMeasureInfo o1,
							FactorRepeatedMeasureInfo o2) {
						int diff = 0;
						if (o1.getFactor().getAsName() != null
								&& o2.getFactor().getAsName() != null) {
							diff = o1.getFactor().getAsName()
									.compareTo(o2.getFactor().getAsName());
						}
						if (diff == 0) {
							return o1.getFactor().getName()
									.compareTo(o2.getFactor().getName());
						}
						return diff;
					}
				});
		return factorList;
	}

	public List<JobVisitInfo> getAllJobVisitInfos(
			List<GroupAnalysisCBFJobInfo> jiList) {
		List<JobVisitInfo> jviList = new ArrayList<JobVisitInfo>(jiList.size());
		for (GroupAnalysisCBFJobInfo gaji : jiList) {
			jviList.add(gaji.getJvi());
		}
		return jviList;
	}

	List<GroupCBFReportRec> getGroupCBFRecords(final UserInfo ui,
			final Connection con, final List<Experiment> allowedExps,
			Set<String> includedJobTypeSet) throws Exception {
		Set<String> allowedExpNameSet = new HashSet<String>(17);
		Map<String, Experiment> expMap = new HashMap<String, Experiment>();
		for (Experiment exp : allowedExps) {
			allowedExpNameSet.add(exp.getName());
			expMap.put(exp.getName(), exp);
		}

		List<Jobs> jobList = getGroupAnalysisJobs(con, allowedExpNameSet);

		// remove any job record with status REMOVED and has no jobsize implying
		// no output
		for (Iterator<Jobs> it = jobList.iterator(); it.hasNext();) {
			Jobs job = it.next();
			String jobStatus = job.getJobstatus();
			if ((jobStatus.equalsIgnoreCase(JobInfo.REMOVED) && job
					.getJobsize() == null) || job.getJobsize() == null) {
				it.remove();
			}
		}

		if (jobList.isEmpty()) {
			return new ArrayList<GroupCBFReportRec>(0);
		}

		List<Assessment> assessments = dbCache.getAssessments(ui, false);
		Map<String, Assessment> asMap = new HashMap<String, Assessment>();
		for (Assessment as : assessments) {
			asMap.put(as.getName(), as);
		}

		List<JobProvenance> jpList = getJobProvenances(con, jobList);
		Map<BigDecimal, JobProvenance> jpMap = new HashMap<BigDecimal, JobProvenance>();
		for (JobProvenance jp : jpList) {
			jpMap.put(jp.getJobUniqueid(), jp);
		}
		jpList = null;
		List<GroupAnalysisJobWrapper> gajwList = new ArrayList<GroupAnalysisJobWrapper>(
				jobList.size());
		Set<String> subjectIDSet = new HashSet<String>();
		List<Jobs> skippedJobs = new ArrayList<Jobs>();
		for (Jobs job : jobList) {
			JSONObject js = new JSONObject(job.getJobContext());
			String jobType = job.getJobtype();
			if (includedJobTypeSet != null
					&& !includedJobTypeSet.contains(jobType)) {
				skippedJobs.add(job);
				// skip any job not included in the job id set if a set is
				// specified
				continue;
			}

			if (jobType.equals(CBFBIRNConstants.CBF_BASELINE_GROUP_ANALYSIS)
					|| jobType
							.equals(CBFBIRNConstants.CBF_STANDARD_SPACE_ANALYSIS)) {
				GroupAnalysisContext ctx = GroupAnalysisContext.fromJSON(js);

				GroupAnalysisJobWrapper gajw = new GroupAnalysisJobWrapper(job,
						ctx);
				gajwList.add(gajw);
				List<GroupAnalysisCBFJobInfo> jiList = ctx.getJiList();
				for (GroupAnalysisCBFJobInfo gaji : jiList) {
					subjectIDSet.add(gaji.getSubjectId());
				}
			} else {
				GroupROIAnalysisContext ctx = GroupROIAnalysisContext
						.fromJSON(js);
				GroupAnalysisJobWrapper gajw = new GroupAnalysisJobWrapper(job,
						ctx);
				gajwList.add(gajw);
				for (CBFROIJobAssociation assoc : ctx.getAssocList()) {
					for (JobVisitInfo jvi : assoc.getCandidates()) {
						subjectIDSet.add(jvi.getSubjectID());
					}
				}
			}
		}
		// now remove all the skipped jobs if any from the job list
		if (!skippedJobs.isEmpty()) {
			jobList.removeAll(skippedJobs);
		}

		Assessment assessment = asMap.get("Subject Information");
		Assertion.assertNotNull(assessment);
		List<SubjectVisitAssessment> svaList = getAssessmentData(con,
				assessment, subjectIDSet);
		Map<String, SubjectVisitAssessment> svaMap = new HashMap<String, SubjectVisitAssessment>();

		// ASSUMPTION: assessments are not more granular than visits.
		for (SubjectVisitAssessment sva : svaList) {
			StringBuilder sb = new StringBuilder();
			sb.append(sva.subjectID).append(':').append(sva.expID).append(':')
					.append(sva.visitID);
			svaMap.put(sb.toString(), sva);
		}

		List<GroupCBFReportRec> recList = new ArrayList<GroupCBFReportRec>(
				jobList.size());
		List<GroupAnalysisJobWrapper> toBeRemoved = new ArrayList<GroupAnalysisJobWrapper>();
		for (GroupAnalysisJobWrapper gajw : gajwList) {
			boolean nonVisibleExpFound = false;
			List<AugmentedJobVisitInfo> ajviList = null;
			String filterQuery = null;
			if (gajw.getGaContext() != null) {
				GroupAnalysisContext gaContext = gajw.getGaContext();
				List<GroupAnalysisCBFJobInfo> jiList = gaContext.getJiList();
				ajviList = new ArrayList<AugmentedJobVisitInfo>(jiList.size());

				for (GroupAnalysisCBFJobInfo gaji : jiList) {
					JobVisitInfo jvi = gaji.getJvi();
					if (!expMap.containsKey(jvi.getExpName())) {
						nonVisibleExpFound = true;
						break;
					}
					prepAugmentedJobVisitInfo(expMap, svaMap, ajviList, jvi);
				}
				if (nonVisibleExpFound) {
					toBeRemoved.add(gajw);
					continue;
				}
			} else {
				GroupROIAnalysisContext roiContext = gajw.getRoiContext();
				List<CBFROIJobAssociation> assocList = roiContext
						.getAssocList();
				ajviList = new ArrayList<AugmentedJobVisitInfo>(
						assocList.size());
				for (CBFROIJobAssociation crja : assocList) {
					for (JobVisitInfo jvi : crja.getCandidates()) {
						if (!expMap.containsKey(jvi.getExpName())) {
							nonVisibleExpFound = true;
							break;
						}
						prepAugmentedJobVisitInfo(expMap, svaMap, ajviList, jvi);
					}
				}
				if (nonVisibleExpFound) {
					toBeRemoved.add(gajw);
					continue;
				}
			}

			Collections.sort(ajviList, new Comparator<AugmentedJobVisitInfo>() {
				@Override
				public int compare(AugmentedJobVisitInfo o1,
						AugmentedJobVisitInfo o2) {
					int diff = o1.getJvi().getExpName()
							.compareTo(o2.getJvi().getExpName());
					if (diff == 0) {
						diff = o1.getJvi().getSubjectID()
								.compareTo(o2.getJvi().getSubjectID());
						if (diff == 0) {
							diff = o1.getJvi().getVisitID()
									- o2.getJvi().getVisitID();
						}
					}
					return diff;
				}
			});

			JobProvenance jp = jpMap.get(gajw.getJob().getUniqueid());

			if (jp != null) {
				List<GAResult> garList = getGAResultArtifacts(jp);
				ROISummary roiSummary = getROISummary(jp);
				Date jobEndDate = gajw.getJob().getJobenddate();
				GroupCBFReportRec rec = new GroupCBFReportRec(gajw.getJob()
						.getJobid(), gajw.getJob().getUniqueid().intValue(),
						gajw.getJob().getJobtype(), filterQuery, jobEndDate,
						ajviList, garList, roiSummary, jp.getName());
				// for TEST
				if (!garList.isEmpty()) {
					recList.add(rec);
				}
			}
		}
		return recList;
	}

	private List<GAResult> getGAResultArtifacts(JobProvenance jp) {
		File[] files = new File(jp.getDatauri()).listFiles();
		List<GAResult> garList = new ArrayList<GAResult>(10);
		for (File f : files) {
			if (f.isFile()) {
				String name = f.getName();
				if (name.endsWith(".png")) {
					String label = GenUtils.removeSuffix(name);
					GAResult gar = GAResult.newImageResult(label,
							f.getAbsolutePath());
					garList.add(gar);
				} else if (name.endsWith(".txt")) {
					String label = GenUtils.removeSuffix(name);
					GAResult gar = GAResult.newTextFileResult(label,
							f.getAbsolutePath());
					garList.add(gar);
				}
			}
		}
		return garList;
	}

	private ROISummary getROISummary(JobProvenance jp) {
		File[] files = new File(jp.getDatauri()).listFiles(new FileFilter() {
			@Override
			public boolean accept(File pathname) {
				return pathname.getName().equals("path2_summary.csv");
			}
		});
		if (files != null && files.length == 1) {
			try {
				ROISummary roiSummary = new ROISummary(
						files[0].getAbsolutePath());
				return roiSummary;
			} catch (IOException e) {
				log.error("getROISummary", e);
			}
		}
		return null;
	}

	private void prepAugmentedJobVisitInfo(Map<String, Experiment> expMap,
			Map<String, SubjectVisitAssessment> svaMap,
			List<AugmentedJobVisitInfo> ajviList, JobVisitInfo jvi) {
		String gender = null;
		String age = null;
		String condition = null;
		Experiment exp = expMap.get(jvi.getExpName());
		if (exp == null) {
			log.warn("Experiment " + jvi.getExpName() + " is not visible");
			return;
		}

		StringBuilder sb = new StringBuilder();

		sb.append(jvi.getSubjectID()).append(':').append(exp.getUniqueid())
				.append(':').append(jvi.getVisitID());
		String key = sb.toString();
		SubjectVisitAssessment sva = svaMap.get(key);
		if (sva != null) {
			age = sva.getValue("age");
			gender = sva.getValue("gender");
			condition = sva.getValue("diagnosis");
		}
		AugmentedJobVisitInfo ajvi = new AugmentedJobVisitInfo(jvi, gender,
				age, condition);

		ajviList.add(ajvi);
	}

	public static class GroupAnalysisJobWrapper {
		private final Jobs job;
		private final GroupAnalysisContext gaContext;
		private final GroupROIAnalysisContext roiContext;

		public GroupAnalysisJobWrapper(Jobs job, GroupAnalysisContext gaContext) {
			this.job = job;
			this.gaContext = gaContext;
			this.roiContext = null;
		}

		public GroupAnalysisJobWrapper(Jobs job,
				GroupROIAnalysisContext roiContext) {
			this.job = job;
			this.roiContext = roiContext;
			this.gaContext = null;
		}

		public Jobs getJob() {
			return job;
		}

		public GroupAnalysisContext getGaContext() {
			return gaContext;
		}

		public GroupROIAnalysisContext getRoiContext() {
			return roiContext;
		}
	}// ;

	private List<SubjectVisitAssessment> getAllCategoricalAssessmentData(
			final Connection con, final Set<String> subjectIDSet)
			throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qb = new StringBuilder();
		qb.append("select a.scorename, a.textvalue, a.assessmentid, b.* ");
		qb.append("from Assessmentdata as a, Storedassessment as b where ");
		qb.append("a.ncStoredassessmentUniqueid = b.uniqueid and ");
		qb.append("a.scoretype = 'varchar' and ");
		qb.append(" b.subjectid in (");
		for (Iterator<String> it = subjectIDSet.iterator(); it.hasNext();) {
			String subjectID = it.next();
			qb.append("'").append(subjectID).append("'");
			if (it.hasNext()) {
				qb.append(",");
			}
		}
		qb.append(")");

		GenUtils.applyWordWrapping("query:" + qb.toString());
		Map<String, SubjectVisitAssessment> svaMap = new HashMap<String, SubjectVisitAssessment>();
		List<?> results = tsp.executeQuery(con, qb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Assessmentdata ad = (Assessmentdata) row[0];
			Storedassessment sa = (Storedassessment) row[1];

			SubjectVisitAssessment sva = new SubjectVisitAssessment(
					sa.getSubjectid(), sa.getNcExperimentUniqueid().intValue(),
					sa.getComponentid().intValue(), sa.getSegmentid()
							.intValue(), ad.getAssessmentid().intValue());
			StringBuilder sb = new StringBuilder();
			sb.append(sva.subjectID).append(':').append(sva.expID).append(':')
					.append(sva.visitID).append(':').append(sva.asID);
			String key = sb.toString();
			SubjectVisitAssessment sva2 = svaMap.get(key);
			if (sva2 == null) {
				svaMap.put(key, sva);
			} else {
				sva = sva2;
			}
			sva.addScoreValue(ad.getScorename(), ad.getTextvalue());
		}

		return new ArrayList<SubjectVisitAssessment>(svaMap.values());
	}

	private List<SubjectVisitAssessment> getAssessmentData(
			final Connection con, final Assessment assessment,
			Set<String> subjectIDSet) throws Exception {

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qb = new StringBuilder();
		qb.append("select a.scorename, a.textvalue, b.* ");
		qb.append("from Assessmentdata as a, Storedassessment as b where ");
		qb.append("a.ncStoredassessmentUniqueid = b.uniqueid and ");
		qb.append("b.assessmentid = ").append(assessment.getUniqueid());
		qb.append(" and b.subjectid in (");
		for (Iterator<String> it = subjectIDSet.iterator(); it.hasNext();) {
			String subjectID = it.next();
			qb.append("'").append(subjectID).append("'");
			if (it.hasNext()) {
				qb.append(",");
			}
		}
		qb.append(")");

		Map<String, SubjectVisitAssessment> svaMap = new HashMap<String, SubjectVisitAssessment>();
		List<?> results = tsp.executeQuery(con, qb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] row = (Object[]) it.next();
			Assessmentdata ad = (Assessmentdata) row[0];
			Storedassessment sa = (Storedassessment) row[1];

			SubjectVisitAssessment sva = new SubjectVisitAssessment(
					sa.getSubjectid(), sa.getNcExperimentUniqueid().intValue(),
					sa.getComponentid().intValue(), sa.getSegmentid()
							.intValue(), -1);

			String key = sva.getKey();
			SubjectVisitAssessment sva2 = svaMap.get(key);
			if (sva2 == null) {
				svaMap.put(key, sva);
			} else {
				sva = sva2;
			}
			sva.addScoreValue(ad.getScorename(), ad.getTextvalue());
		}

		return new ArrayList<SubjectVisitAssessment>(svaMap.values());

	}

	public static class SubjectVisitAssessment {
		String subjectID;
		int expID;
		int visitID;
		int segmentID;
		int asID;
		List<ScoreValue> svList = new LinkedList<ScoreValue>();

		public SubjectVisitAssessment(String subjectID, int expID, int visitID,
				int segmentID, int asID) {
			super();
			this.subjectID = subjectID;
			this.expID = expID;
			this.visitID = visitID;
			this.segmentID = segmentID;
			this.asID = asID;
		}

		void addScoreValue(String score, String value) {
			svList.add(new ScoreValue(score, value));
		}

		public String getKey() {
			StringBuilder sb = new StringBuilder();
			sb.append(subjectID).append(':').append(expID).append(':');
			sb.append(visitID).append(':').append(segmentID);
			return sb.toString();
		}

		public String getValue(String scoreName) {
			for (ScoreValue sv : svList) {
				if (sv.getName().equals(scoreName)) {
					return sv.getValue();
				}
			}
			return null;
		}
	}

	private List<JobProvenance> getJobProvenances(final Connection con,
			final List<Jobs> jobList) throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qb = new StringBuilder();
		qb.append("select a.* from JobProvenance as a where ");
		qb.append("a.jobUniqueid in (");
		for (Iterator<Jobs> it = jobList.iterator(); it.hasNext();) {
			Jobs jobs = it.next();
			qb.append(jobs.getUniqueid());
			if (it.hasNext())
				qb.append(',');
		}
		qb.append(")");
		List<?> results = tsp.executeQuery(con, qb.toString());
		List<JobProvenance> jpList = new ArrayList<JobProvenance>(
				results.size());
		for (Object o : results) {
			JobProvenance jp = (JobProvenance) o;
			jpList.add(jp);
		}
		return jpList;
	}

	private List<Jobs> getGroupAnalysisJobs(final Connection con,
			final Set<String> allowedExpNameSet) throws Exception {

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder qb = new StringBuilder();
		qb.append("select a.* from Jobs as a where ");
		qb.append("a.jobtype in ('").append(CBFBIRNConstants.CBF_ROI_ANALYSIS)
				.append("','");
		qb.append(CBFBIRNConstants.CBF_STANDARD_SPACE_ANALYSIS).append("','");
		qb.append(CBFBIRNConstants.CBF_BASELINE_GROUP_ANALYSIS);
		qb.append("') and a.jobstatus <> '").append(JobInfo.CANCELED)
				.append("' and a.jobstatus <> '")
				.append(JobInfo.FINISHED_WITH_ERR).append("'");

		List<?> results = tsp.executeQuery(con, qb.toString());
		List<Jobs> jobList = new ArrayList<Jobs>(results.size());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Jobs job = (Jobs) it.next();
			if (allowedExpNameSet.isEmpty()) {
				jobList.add(job);
				continue;
			}
			if (job.getJobContext() == null
					|| job.getJobContext().length() == 0) {
				continue;
			}
			String jobType = job.getJobtype();
			if (jobType.equals(CBFBIRNConstants.CBF_BASELINE_GROUP_ANALYSIS)
					|| jobType
							.equals(CBFBIRNConstants.CBF_STANDARD_SPACE_ANALYSIS)) {
				JSONObject js = new JSONObject(job.getJobContext());
				if (js.has("jiList")) {
					JSONArray jsArr = js.getJSONArray("jiList");
					int len = jsArr.length();
					boolean ok = true;
					for (int i = 0; i < len; i++) {
						JSONObject jviJS = jsArr.getJSONObject(i);
						if (jviJS.has("expName")) {
							String expName = jviJS.getString("expName");
							if (!allowedExpNameSet.contains(expName)) {
								ok = false;
								break;
							}
						}
					}
					if (ok) {
						jobList.add(job);
					}
				}
			} else {
				JSONObject js = new JSONObject(job.getJobContext());
				JSONArray assocJSArr = js.getJSONArray("assocList");
				int len = assocJSArr.length();
				boolean ok = true;
				for (int i = 0; i < len; i++) {
					JSONObject assocJS = assocJSArr.getJSONObject(i);
					if (assocJS.has("candidates")) {
						JSONArray candidates = assocJS
								.getJSONArray("candidates");
						int len2 = candidates.length();
						for (int j = 0; j < len2; j++) {
							JSONObject jviJS = candidates.getJSONObject(j);
							String expName = jviJS.getString("expName");
							if (!allowedExpNameSet.contains(expName)) {
								ok = false;
								break;
							}
						}
					}
					if (!ok) {
						break;
					}
				}
				if (ok) {
					jobList.add(job);
				}
			}
		}
		return jobList;
	}

}
