package clinical.web.services;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Element;

import clinical.utils.FileUtils;
import clinical.web.IHelpService;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: HelpService.java 720 2012-11-14 00:29:03Z bozyurt $
 */
public class HelpService implements IHelpService {
	protected Map<String, Set<HelpMessage>> contextHelpMap = new ConcurrentHashMap<String, Set<HelpMessage>>();
	protected static HelpService instance = null;
	private static Log log = LogFactory.getLog(HelpService.class);
	private String helpMsgRepFile;

	protected HelpService(String helpMsgRepFile) throws IOException {
		super();
		initialize(helpMsgRepFile);
	}

	public synchronized static HelpService getInstance(String helpMsgRepFile)
			throws Exception {
		if (instance == null) {
			instance = new HelpService(helpMsgRepFile);
		}
		return instance;
	}

	public synchronized static HelpService getInstance() throws Exception {
		if (instance == null) {
			throw new Exception("HelpService is not properly initialized!");
		}
		return instance;
	}

	protected synchronized void initialize(String helpMsgRepFile)
			throws IOException {
		File f = new File(helpMsgRepFile);
		if (!f.exists()) {
			log.error("Help repository file does not exists:" + helpMsgRepFile);
			return;
		}

		this.helpMsgRepFile = helpMsgRepFile;

		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(f));
			String line;
			HelpMessage curMsg = null;
			while ((line = in.readLine()) != null) {
				if (line.trim().startsWith("$HELP_REC")
						|| line.trim().startsWith("$HELP_END")) {
					if (curMsg != null && curMsg.getContext() != null
							&& curMsg.getMessageId() != null) {
						Set<HelpMessage> contextMsgSet = contextHelpMap
								.get(curMsg.getContext());
						if (contextMsgSet == null) {
							contextMsgSet = new HashSet<HelpMessage>();
							contextHelpMap.put(curMsg.getContext(),
									contextMsgSet);
						}
						contextMsgSet.add(curMsg);
					}
					curMsg = new HelpMessage();
					continue;
				}

				if (line.trim().startsWith("$HELP_CONTEXT")) {
					curMsg.setContext(extractValue(line));
				} else if (line.trim().startsWith("$HELP_ID")) {
					curMsg.setMessageId(extractValue(line));
				} else {
					curMsg.add2MessageWithNewline(line);
				}
			} // while
		} finally {
			FileUtils.close(in);
		}

	}

	public void update() {
		try {
			Map<String, Set<HelpMessage>> map = readHelpMessages();
			for (String key : map.keySet()) {
				Set<HelpMessage> set = map.get(key);
				contextHelpMap.put(key, set);
			}
		} catch (IOException e) {
			log.error("HelpService update", e);
		}
	}

	private Map<String, Set<HelpMessage>> readHelpMessages() throws IOException {
		File f = new File(helpMsgRepFile);
		if (!f.exists()) {
			log.error("Help repository file does not exists:" + helpMsgRepFile);
			return null;
		}
		Map<String, Set<HelpMessage>> map = new ConcurrentHashMap<String, Set<HelpMessage>>();

		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(f));
			String line = null;
			HelpMessage curMsg = null;
			while ((line = in.readLine()) != null) {
				if (line.trim().startsWith("$HELP_REC")
						|| line.trim().startsWith("$HELP_END")) {
					if (curMsg != null && curMsg.getContext() != null
							&& curMsg.getMessageId() != null) {
						Set<HelpMessage> contextMsgSet = map.get(curMsg
								.getContext());
						if (contextMsgSet == null) {
							contextMsgSet = new HashSet<HelpMessage>();
							map.put(curMsg.getContext(), contextMsgSet);
						}
						contextMsgSet.add(curMsg);
					}
					curMsg = new HelpMessage();
					continue;
				}

				if (line.trim().startsWith("$HELP_CONTEXT")) {
					curMsg.setContext(extractValue(line));
				} else if (line.trim().startsWith("$HELP_ID")) {
					curMsg.setMessageId(extractValue(line));
				} else {
					curMsg.add2MessageWithNewline(line);
				}
			} // while
			return map;
		} finally {
			FileUtils.close(in);
		}
	}

	public static String extractValue(String line) {
		int idx = line.indexOf('=');
		if (idx == -1) {
			return null;
		}
		return line.substring(idx + 1).trim();
	}

	public List<Element> getHelpMessages(String context) throws Exception {
		Set<HelpMessage> contextMsgSet = null;

		contextMsgSet = contextHelpMap.get(context);
		if (contextMsgSet == null) {
			return new ArrayList<Element>(0);
		}
		List<Element> msgElemList = new ArrayList<Element>(contextMsgSet.size());
		for (HelpMessage hm : contextMsgSet) {
			Element elem = new Element("help");
			elem.setAttribute("id", hm.getMessageId());
			elem.addContent(hm.getMessage());
			msgElemList.add(elem);
		}
		return msgElemList;
	}

	public static class HelpMessage {
		String context;
		String messageId;
		StringBuffer messageBuf = new StringBuffer();

		public HelpMessage() {
		}

		public HelpMessage(String context) {
			super();
			this.context = context;
		}

		public String getContext() {
			return context;
		}

		public String getMessage() {
			return messageBuf.toString();
		}

		public String getMessageId() {
			return messageId;
		}

		public void setContext(String context) {
			this.context = context;
		}

		public void add2Message(String line) {
			this.messageBuf.append(line);
		}

		public void add2MessageWithNewline(String line) {
			this.messageBuf.append(line).append("\n");
		}

		public void setMessageId(String messageId) {
			this.messageId = messageId;
		}

		public int hashCode() {
			return messageId.hashCode();
		}

		public boolean equals(Object obj) {
			if (obj == null || !(obj instanceof HelpMessage)) {
				return false;
			}
			return messageId.equals(((HelpMessage) obj).messageId);
		}

	}
}
