package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;

import clinical.server.dao.JobProvParamTypeDAO;
import clinical.server.dao.JobProvenanceDAO;
import clinical.server.dao.JobProvenanceParamDAO;
import clinical.server.dao.JobsDAO;
import clinical.server.vo.JobProvParamType;
import clinical.server.vo.JobProvenance;
import clinical.server.vo.JobProvenanceParam;
import clinical.server.vo.Jobs;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IJobManagementService;
import clinical.web.ISequenceHelper;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.Operator;
import clinical.web.common.query.ProvenanceQueryBuilder;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.JobProvenanceValueSummary;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class JobProvenanceManServiceImpl extends AbstractServiceImpl implements
		IJobProvenanceService {
	protected IDBCache dbCache;
	protected ISequenceHelper seqHelper;

	public JobProvenanceManServiceImpl(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(JobProvenanceManServiceImpl.class);
		dbCache = ServiceFactory.getDBCache(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(dbID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobProvenanceService#deleteJobProvenance(clinical
	 * .web.common.UserInfo, java.util.List)
	 */
	public void deleteJobProvenance(UserInfo ui, List<JobProvenanceInfo> jpiList)
			throws Exception {
		if (jpiList == null || jpiList.isEmpty()) {
			return;
		}
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			List<BigDecimal> jobUniqueIDList = new ArrayList<BigDecimal>(
					jpiList.size());
			for (JobProvenanceInfo jpi : jpiList) {
				jobUniqueIDList.add(jpi.getJob().getUniqueid());
			}
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.uniqueid from JobProvenance as a "
					+ "where a.jobUniqueid in (");
			boolean first = true;
			for (BigDecimal id : jobUniqueIDList) {
				if (!first) {
					sb.append(',');
				}
				sb.append(id);
				first = false;
			}
			sb.append(')');
			log.info("deleteJobProvenance: query:" + sb.toString());
			List<?> results = tsp.executeQuery(con, sb.toString());

			JobProvenanceDAO jpDAO = DAOFactory
					.createJobProvenanceDAO(this.theDBID);
			JobProvenanceParamDAO paramDAO = DAOFactory
					.createJobProvenanceParamDAO(theDBID);
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				JobProvenance jp = (JobProvenance) it.next();
				JobProvenanceParam cr = new JobProvenanceParam();
				cr.setJobProvId(jp.getUniqueid());
				paramDAO.delete(con, cr);
				jpDAO.delete(con, jp);
			}
			con.commit();
		} catch (Exception x) {
			log.error("Error in deleteJobProvenance", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Jobs getTheJob(UserInfo ui, String jobID, String jobUser)
			throws Exception {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(theDBID);
		return jms.getTheJob(ui, jobUser, jobID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobProvenanceService#saveJobProvenance(clinical
	 * .web.common.UserInfo, clinical.web.vo.JobProvenanceInfo)
	 */
	public void saveJobProvenance(UserInfo ui, JobProvenanceInfo jpi)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			saveJobProvenance(con, jpi);
			con.commit();
		} catch (Exception x) {
			log.error("Error in saveJobProvenance", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addUpdateProvenanceParams(UserInfo ui, Integer jobUniqueID,
			List<JobProvenanceParamInfo> jpiList) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			addUpdateProvenanceParams(con, jobUniqueID, jpiList);
			con.commit();
		} catch (Exception x) {
			log.error("Error in addUpdateProvenanceParams", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addUpdateProvenanceParams(Connection con, Integer jobUniqueID,
			List<JobProvenanceParamInfo> jpiList) throws Exception {
		JobProvenance jpCR = new JobProvenance();
		JobProvenanceDAO jpDAO = DAOFactory
				.createJobProvenanceDAO(this.theDBID);
		JobProvenanceParamDAO paramDAO = DAOFactory
				.createJobProvenanceParamDAO(theDBID);

		jpCR.setJobUniqueid(new BigDecimal(jobUniqueID));
		List<JobProvenance> jpList = jpDAO.find(con, jpCR);
		Assertion.assertTrue(jpList.size() == 1);
		JobProvenance theJP = jpList.get(0);
		JobProvenanceParam jppCR = new JobProvenanceParam();
		jppCR.setJobProvId(theJP.getUniqueid());
		List<JobProvenanceParam> jppList = paramDAO.find(con, jppCR);
		Map<String, JobProvenanceParam> jppMap = new HashMap<String, JobProvenanceParam>(
				23);
		for (JobProvenanceParam jpp : jppList) {
			jppMap.put(jpp.getName(), jpp);
		}
		List<JobProvenanceParamInfo> newOnes = new ArrayList<JobProvenanceParamInfo>(
				jppList.size());
		for (JobProvenanceParamInfo jpi : jpiList) {
			JobProvenanceParam jpp = jppMap.get(jpi.getName());
			if (jpp == null) {
				newOnes.add(jpi);
			} else {
				if (!jpp.getValue().equals(jpi.getValue())) {
					JobProvenanceParam cr = new JobProvenanceParam();
					cr.setUniqueid(jpp.getUniqueid());
					JobProvenanceParam bean = new JobProvenanceParam();
					bean.setValue(jpi.getValue());
					paramDAO.update(con, bean, cr);
				}
			}
		}
		if (!newOnes.isEmpty()) {
			for (JobProvenanceParamInfo jpi : newOnes) {
				JobProvenanceParam jpp = new JobProvenanceParam();
				jpp.setName(jpi.getName());
				jpp.setValue(jpi.getValue());
				jpp.setJobProvId(theJP.getUniqueid());
				jpp.setUniqueid(seqHelper.getNextUID(con,
						Constants.JOB_PROVENANCE_PARAM_DB_TABLE, "uniqueid"));
				paramDAO.insert(con, jpp);
			}
			updateJobProvParamTypes(con, newOnes);
		}
	}

	public void updateJobProvParamTypes(Connection con,
			List<JobProvenanceParamInfo> jpiList) throws Exception {
		JobProvParamTypeDAO dao = DAOFactory
				.createJobProvParamTypeDAO(this.theDBID);
		List<JobProvParamType> list = dao.find(con, new JobProvParamType());
		List<JobProvenanceParamInfo> newOnes = new ArrayList<JobProvenanceParamInfo>(
				5);
		Map<String, JobProvParamType> jpptMap = new HashMap<String, JobProvParamType>(
				17);
		for (JobProvParamType jppt : list) {
			jpptMap.put(jppt.getName(), jppt);
		}
		for (JobProvenanceParamInfo jpi : jpiList) {
			if (!jpptMap.containsKey(jpi.getName())) {
				newOnes.add(jpi);
			}
		}
		if (!newOnes.isEmpty()) {
			for (JobProvenanceParamInfo jpi : newOnes) {
				JobProvParamType jppt = new JobProvParamType();
				BigDecimal uniqueid = this.seqHelper.getNextUID(con,
						"nc_job_prov_param_type", "uniqueid");
				jppt.setUniqueid(uniqueid);
				jppt.setName(jpi.getName());
				jppt.setDataType(jpi.getTypeFromValue().toString());

				dao.insert(con, jppt);
			}
		}
	}

	public void saveJobProvenance(Connection con, JobProvenanceInfo jpi)
			throws Exception {
		JobProvenance jp = new JobProvenance();
		JobProvenanceDAO jpDAO = DAOFactory
				.createJobProvenanceDAO(this.theDBID);
		JobProvenanceParamDAO paramDAO = DAOFactory
				.createJobProvenanceParamDAO(theDBID);

		jp.setName(jpi.getName());
		jp.setDescription(jpi.getDescription());
		jp.setDatauri(jpi.getDataURI());
		if (jpi.getModTime() != null) {
			jp.setModtime(jpi.getModTime());
		} else {
			jp.setModtime(new Date());
		}
		jp.setJobUniqueid(jpi.getJob().getUniqueid());
		BigDecimal jpUniqueID = seqHelper.getNextUID(con,
				Constants.JOB_PROVENANCE_DB_TABLE, "uniqueid");
		jp.setUniqueid(jpUniqueID);

		jpDAO.insert(con, jp);
		for (JobProvenanceParamInfo pi : jpi.getParams()) {
			JobProvenanceParam jpp = new JobProvenanceParam();
			jpp.setName(pi.getName());
			jpp.setValue(pi.getValue());
			jpp.setJobProvId(jpUniqueID);
			jpp.setUniqueid(seqHelper.getNextUID(con,
					Constants.JOB_PROVENANCE_PARAM_DB_TABLE, "uniqueid"));
			paramDAO.insert(con, jpp);
		}
	}

	public static boolean isJobValid(String jobStatus) {
		return (jobStatus.equals(JobInfo.FINISHED) || jobStatus
				.equals(JobInfo.REMOVED));
	}

	public List<JobProvenanceInfo> findJobProvenances(Connection con,
			List<Jobs> jobsList) throws Exception {
		if (jobsList.isEmpty()) {
			return new ArrayList<JobProvenanceInfo>(0);
		}
		List<?> results = null;
		if (jobsList.size() > 20) {
			Set<BigDecimal> jobIDSet = new HashSet<BigDecimal>();
			for (Jobs job : jobsList) {
				jobIDSet.add(job.getUniqueid());
			}
			JobProvenanceDAO dao = DAOFactory
					.createJobProvenanceDAO(this.theDBID);
			results = dao.find(con, new JobProvenance());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				JobProvenance jp = (JobProvenance) it.next();
				if (!jobIDSet.contains(jp.getJobUniqueid())) {
					it.remove();
				}
			}
		} else {
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(256);
			sb.append("select a.* from JobProvenance as a ");
			sb.append("where a.jobUniqueid in (");
			boolean first = true;
			for (Jobs job : jobsList) {
				if (!first) {
					sb.append(',');
				}
				sb.append(job.getUniqueid().intValue());
				first = false;
			}
			sb.append(')');
			log.info("findJobProvenances: query:" + sb.toString());
			results = tsp.executeQuery(con, sb.toString());
		}
		if (results.isEmpty()) {
			return new ArrayList<JobProvenanceInfo>(0);
		}
		Map<BigDecimal, Jobs> jobMap = new HashMap<BigDecimal, Jobs>();
		for (Jobs job : jobsList) {
			jobMap.put(job.getUniqueid(), job);
		}

		Map<BigDecimal, JobProvenanceInfo> jpiMap = new HashMap<BigDecimal, JobProvenanceInfo>(
				7);
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			JobProvenance jp = (JobProvenance) it.next();
			Jobs theJob = jobMap.get(jp.getJobUniqueid());
			Assertion.assertNotNull(theJob);
			JobProvenanceInfo jpi = new JobProvenanceInfo(jp.getName(),
					jp.getDescription(), jp.getDatauri(), jp.getModtime(),
					theJob);
			jpiMap.put(jp.getUniqueid(), jpi);
		}

		if (jpiMap.size() > 20) {
			JobProvenanceParamDAO jppDAO = DAOFactory
					.createJobProvenanceParamDAO(theDBID);
			results = jppDAO.find(con, new JobProvenanceParam());
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				JobProvenanceParam jpp = (JobProvenanceParam) it.next();
				if (!jpiMap.containsKey(jpp.getJobProvId())) {
					it.remove();
				}
			}

		} else {
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.* from JobProvenanceParam as a where ");
			sb.append("a.jobProvId in (");
			boolean first = true;
			for (BigDecimal jpId : jpiMap.keySet()) {
				if (!first) {
					sb.append(',');
				}
				sb.append(jpId);
				first = false;
			}
			sb.append(')');
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			results = tsp.executeQuery(con, sb.toString());
		}
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			JobProvenanceParam param = (JobProvenanceParam) it.next();
			JobProvenanceInfo jpi = jpiMap.get(param.getJobProvId());
			Assertion.assertNotNull(jpi);
			JobProvenanceParamInfo pi = new JobProvenanceParamInfo(
					param.getName(), param.getValue());
			jpi.addParam(pi);
		}
		List<JobProvenanceInfo> jpiList = new ArrayList<JobProvenanceInfo>(
				jpiMap.values());

		return jpiList;
	}

	public List<JobProvenanceInfo> findJobProvenances(UserInfo ui, int expId,
			String subjectID, int visitId) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select distinct b.* from VisitJob as a, Jobs as b");
			sb.append(" where a.jobUniqueId = b.uniqueid");
			sb.append(" and a.expId = ").append(expId);
			sb.append(" and a.subjectid = '").append(subjectID).append("' ");
			sb.append("and a.componentId = ").append(visitId);
			List<?> results = tsp.executeQuery(con, sb.toString());
			if (results.isEmpty()) {
				con.commit();
				return null;
			}
			Map<BigDecimal, Jobs> jobMap = new HashMap<BigDecimal, Jobs>(11);
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Jobs job = (Jobs) it.next();
				if (isJobValid(job.getJobstatus())) {
					jobMap.put(job.getUniqueid(), job);
				}
			}
			results = null;

			sb = new StringBuilder(128);

			// no outer join support in TSQLProcessor
			sb.append("select a.* from JobProvenance as a ");
			sb.append("where a.jobUniqueid in (");
			boolean first = true;
			for (BigDecimal jobUniqueID : jobMap.keySet()) {
				if (!first) {
					sb.append(',');
				}
				sb.append(jobUniqueID);
				first = false;
			}
			sb.append(')');
			log.info("findJobProvenances: query:" + sb.toString());
			results = tsp.executeQuery(con, sb.toString());
			if (results.isEmpty()) {
				con.commit();
				return new ArrayList<JobProvenanceInfo>(0);
			}

			Map<BigDecimal, JobProvenanceInfo> jpiMap = new HashMap<BigDecimal, JobProvenanceInfo>(
					7);
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				JobProvenance jp = (JobProvenance) it.next();
				Jobs theJob = jobMap.get(jp.getJobUniqueid());
				Assertion.assertNotNull(theJob);
				JobProvenanceInfo jpi = new JobProvenanceInfo(jp.getName(),
						jp.getDescription(), jp.getDatauri(), jp.getModtime(),
						theJob);
				jpiMap.put(jp.getUniqueid(), jpi);
			}

			sb = new StringBuilder(128);
			sb.append("select a.* from JobProvenanceParam as a where ");
			sb.append("a.jobProvId in (");
			first = true;
			for (BigDecimal jpId : jpiMap.keySet()) {
				if (!first) {
					sb.append(',');
				}
				sb.append(jpId);
				first = false;
			}
			sb.append(')');
			results = tsp.executeQuery(con, sb.toString());

			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				JobProvenanceParam param = (JobProvenanceParam) it.next();
				JobProvenanceInfo jpi = jpiMap.get(param.getJobProvId());
				Assertion.assertNotNull(jpi);
				JobProvenanceParamInfo pi = new JobProvenanceParamInfo(
						param.getName(), param.getValue());
				jpi.addParam(pi);
			}
			con.commit();

			List<JobProvenanceInfo> jpiList = new ArrayList<JobProvenanceInfo>(
					jpiMap.values());
			Collections.sort(jpiList, new Comparator<JobProvenanceInfo>() {
				@Override
				public int compare(JobProvenanceInfo o1, JobProvenanceInfo o2) {
					return o1.getModTime().compareTo(o2.getModTime());
				}
			});
			return jpiList;
		} catch (Exception x) {
			log.error("Error in findJobProvenances", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public JobProvenance findJobProvenanceByJobID(UserInfo ui, String jobID)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.* from JobProvenance as a, ");
			sb.append("Jobs as b where a.jobUniqueid = b.uniqueid and ");
			sb.append("b.jobid ='").append(jobID).append("'");
			List<?> results = tsp.executeQuery(con, sb.toString());
			if (results.isEmpty()) {
				return null;
			}
			JobProvenance jp = (JobProvenance) results.get(0);
			return jp;
		} finally {
			releaseConnection(con, ui);
		}
	}
	

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * clinical.web.services.IJobProvenanceService#findJobProvenancesByJobUniqueID
	 * (clinical.web.common.UserInfo, java.math.BigDecimal)
	 */
	public List<JobProvenanceInfo> findJobProvenancesByJobUniqueID(UserInfo ui,
			BigDecimal jobUniqueID) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
			StringBuilder sb = new StringBuilder(128);
			sb.append("select a.*, b.* from JobProvenance as a, ");
			sb.append("JobProvenanceParam as b where ");
			sb.append("a.uniqueid = b.jobProvId " + "and a.jobUniqueid =");
			sb.append(jobUniqueID);
			log.info("findJobProvenancesByJobID: query:" + sb.toString());
			List<?> results = tsp.executeQuery(con, sb.toString());
			if (results.isEmpty()) {
				con.commit();
				return null;
			}
			Jobs cr = new Jobs();
			cr.setUniqueid(jobUniqueID);
			JobsDAO dao = DAOFactory.createJobsDAO(theDBID);
			List<Jobs> jobsList = dao.find(con, cr);
			if (jobsList.isEmpty()) {
				throw new Exception(
						"Inconsistent database state! No Job record for the corresponding JobProvenance records!");
			}
			Jobs theJob = jobsList.get(0);

			Map<BigDecimal, JobProvenanceInfo> jpiMap = new HashMap<BigDecimal, JobProvenanceInfo>(
					7);
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Object[] row = (Object[]) it.next();
				JobProvenance jp = (JobProvenance) row[0];
				JobProvenanceParam param = (JobProvenanceParam) row[1];
				JobProvenanceInfo jpi = jpiMap.get(jp.getUniqueid());
				if (jpi == null) {
					jpi = new JobProvenanceInfo(jp.getName(),
							jp.getDescription(), jp.getDatauri(),
							jp.getModtime(), theJob);
					jpiMap.put(jp.getUniqueid(), jpi);
				}
				JobProvenanceParamInfo pi = new JobProvenanceParamInfo(
						param.getName(), param.getValue());
				jpi.addParam(pi);
			}
			con.commit();
			List<JobProvenanceInfo> jpiList = new ArrayList<JobProvenanceInfo>(
					jpiMap.values());
			Collections.sort(jpiList, new Comparator<JobProvenanceInfo>() {
				@Override
				public int compare(JobProvenanceInfo o1, JobProvenanceInfo o2) {
					return o1.getModTime().compareTo(o2.getModTime());
				}
			});
			return jpiList;
		} catch (Exception x) {
			log.error("Error in findJobProvenancesByJobID", x);
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback:", e);
			}
			throw x;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<JobProvenanceValueSummary> queryForJobProvenance(UserInfo ui,
			Operator rootOp, List<Integer> experimentIds) throws Exception {
		List<JobProvenanceValueSummary> resultList = new LinkedList<JobProvenanceValueSummary>();
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		Map<String, JobProvParamType> paramTypeMap = new HashMap<String, JobProvParamType>(
				17);
		try {
			con = pool.getConnection(ui.getName());

			List<JobProvParamType> paramTypeList = getProvParamTypeList(con);
			for (JobProvParamType pt : paramTypeList) {
				paramTypeMap.put(pt.getName(), pt);
			}
			paramTypeList = null;

			ProvenanceQueryBuilder pqb = new ProvenanceQueryBuilder(sqlDialect,
					experimentIds);
			pqb.visit(rootOp);
			String query = pqb.getQuery();
			log.info("queryForJobProvenance = "
					+ GenUtils.applyWordWrapping(query));
			st = con.createStatement();
			rs = st.executeQuery(query);

			while (rs.next()) {
				JobProvenanceValueSummary vs = prepareJobProvenanceValueSummary(
						rs, paramTypeMap);
				resultList.add(vs);
			}
		} finally {
			DBUtils.close(st, rs);
			releaseConnection(con, ui);
		}
		return resultList;
	}

	List<JobProvParamType> getProvParamTypeList(Connection con)
			throws Exception {
		JobProvParamTypeDAO dao = DAOFactory
				.createJobProvParamTypeDAO(this.theDBID);
		return dao.find(con, new JobProvParamType());
	}

	private JobProvenanceValueSummary prepareJobProvenanceValueSummary(
			ResultSet rs, Map<String, JobProvParamType> paramTypeMap)
			throws SQLException {
		String subjectID = rs.getString(1);
		String value = rs.getString(2);
		String varName = rs.getString(3);
		Integer jobUniqueId = getValue(rs, 4);
		Integer expID = getValue(rs, 5);
		Integer visitID = getValue(rs, 6);
		JobProvenanceValueSummary vs = new JobProvenanceValueSummary();
		vs.setSubjectID(subjectID);
		vs.setName(varName);
		vs.setValue(value);
		vs.setJobUniqueId(jobUniqueId);
		vs.setExperimentID(expID);
		vs.setVisitID(visitID);
		JobProvParamType pt = paramTypeMap.get(varName);

		clinical.web.vo.JobProvenanceInfo.JobProvenanceParamType resultType = clinical.web.vo.JobProvenanceInfo.JobProvenanceParamType
				.valueOf(pt.getDataType());
		vs.setResultType(resultType);

		return vs;
	}

	public static Integer getValue(ResultSet rs, int columIndex)
			throws SQLException {
		int x = rs.getInt(columIndex);
		if (rs.wasNull()) {
			return null;
		} else {
			return new Integer(x);
		}
	}
}
