package clinical.web.vo;

import java.io.Serializable;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.web.workflow.cbf.AFNIBrik;
import clinical.web.workflow.common.JSONUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class CBFROIJobAssociation implements Serializable {
	private static final long serialVersionUID = 1L;
	private AFNIBrik roiMaskBrik;
	private String subjectId;
	private String visitDate;
	private String rootDir;

	private List<JobVisitInfo> candidates;

	public CBFROIJobAssociation(AFNIBrik roiMaskBrik, String subjectId,
			String visitDate, String rootDir) {
		super();
		this.roiMaskBrik = roiMaskBrik;
		this.subjectId = subjectId;
		this.visitDate = visitDate;
		this.rootDir = rootDir;
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		JSONUtils.addJSONAFNIBrik("roiMaskBrik", roiMaskBrik, js);
		js.put("subjectId", subjectId);
		JSONUtils.addStringField(js, "visitDate", visitDate, "");
		JSONUtils.addStringField(js, "rootDir", rootDir, "");
		if (candidates != null && !candidates.isEmpty()) {
			JSONArray jsArr = new JSONArray();
			js.put("candidates", jsArr);
			for (JobVisitInfo jvi : candidates) {
				jsArr.put(jvi.toJSON());
			}
		}
		return js;
	}

	public static CBFROIJobAssociation fromJSON(JSONObject js)
			throws JSONException, ParseException {
		String subjectId = js.getString("subjectId");
		String visitDate = JSONUtils.getString(js, "visitDate");
		String rootDir = JSONUtils.getString(js, "rootDir");
		AFNIBrik roiMaskBrik = JSONUtils.getAFNIBrik("roiMaskBrik", js);
		CBFROIJobAssociation assoc = new CBFROIJobAssociation(roiMaskBrik,
				subjectId, visitDate, rootDir);

		if (js.has("candidates")) {
			JSONArray jsArr = js.getJSONArray("candidates");
			int len = jsArr.length();
			for(int i = 0; i < len; i++) {
				JSONObject jviJS = jsArr.getJSONObject(i);
				assoc.addCandidate(  JobVisitInfo.fromJSON(jviJS) );
			}
		}
		return assoc;
	}

	public String getSubjectId() {
		return subjectId;
	}

	public String getVisitDate() {
		return visitDate;
	}

	public String getRootDir() {
		return rootDir;
	}

	public AFNIBrik getRoiMaskBrik() {
		return roiMaskBrik;
	}

	public List<JobVisitInfo> getCandidates() {
		return candidates;
	}

	public String getRelativeBrikPath() {
		String brikFile = roiMaskBrik.getBrikFile().getAbsolutePath();
		String relPath = brikFile.substring(rootDir.length());
		if (relPath.startsWith("/")) {
			relPath = relPath.substring(1);
		}
		return relPath;
	}

	public boolean hasCandidates() {
		return candidates != null && !candidates.isEmpty();
	}

	public void addCandidate(JobVisitInfo jvi) {
		if (candidates == null) {
			candidates = new ArrayList<JobVisitInfo>(5);
		}
		candidates.add(jvi);
	}

	public static class JobVisitInfo implements Serializable {
		private static final long serialVersionUID = 1L;
		private String jobID;
		private String subjectID;
		private int visitID;
		private String expName;
		private String visitDate;
		private String label;
		//
		private String dataURI;
		private long jobUniqueID = -1l;

		public JobVisitInfo(String jobID, String subjectID, int visitID,
				String expName, String visitDate, String label) {
			super();
			this.jobID = jobID;
			this.subjectID = subjectID;
			this.visitID = visitID;
			this.expName = expName;
			this.visitDate = visitDate;
			this.label = label;
		}

		public JSONObject toJSON() throws JSONException {
			JSONObject js = new JSONObject();
			js.put("jobID", jobID);
			js.put("subjectID", subjectID);
			js.put("visitID", visitID);
			js.put("expName", expName);
			js.put("visitDate", visitDate);
			js.put("label", label);
			JSONUtils.addStringField(js, "dataURI", dataURI, "");
			js.put("jobUniqueID", jobUniqueID);
			return js;
		}

		public static JobVisitInfo fromJSON(JSONObject js) throws JSONException {
			String jobID = js.getString("jobID");
			String subjectID = js.getString("subjectID");
			int visitID = js.getInt("visitID");
			String expName = js.getString("expName");
			String visitDate = js.getString("visitDate");
			String label = js.getString("label");
			JobVisitInfo jvi = new JobVisitInfo(jobID, subjectID, visitID,
					expName, visitDate, label);
			jvi.dataURI = JSONUtils.getString(js, "dataURI");
			jvi.jobUniqueID = js.getLong("jobUniqueID");
			return jvi;
		}

		public String getVisitDate() {
			return visitDate;
		}

		public String getLabel() {
			return label;
		}

		public String getJobID() {
			return jobID;
		}

		public String getSubjectID() {
			return subjectID;
		}

		public int getVisitID() {
			return visitID;
		}

		public String getExpName() {
			return expName;
		}

		@Override
		public String toString() {
			return "JobVisitInfo [jobID=" + jobID + ", subjectID=" + subjectID
					+ ", visitID=" + visitID + ", expName=" + expName + "]";
		}

		public String getDataURI() {
			return dataURI;
		}

		public void setDataURI(String dataURI) {
			this.dataURI = dataURI;
		}

		public long getJobUniqueID() {
			return jobUniqueID;
		}

		public void setJobUniqueID(long jobUniqueID) {
			this.jobUniqueID = jobUniqueID;
		}

	}// ;

	@Override
	public String toString() {
		return "CBFROIJobAssociation [roiMaskBrik=" + roiMaskBrik
				+ ", subjectId=" + subjectId + ", visitDate=" + visitDate
				+ ", rootDir=" + rootDir + "\n, candidates=" + candidates + "]";
	}

}
