package clinical.web.workflow.cbf;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobEventListener;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobEvent;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.JobInfo;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobScheduler;
import clinical.web.scheduler.JobSchedulerException;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractCBFProcessingJob.java 816 2013-07-03 01:13:11Z bozyurt
 *          $
 */
public abstract class AbstractCBFProcessingJob implements IJob {
	protected static final int USER_ACTION_MAIL = 1;
	protected static final int JOB_FINISHED_MAIL = 2;
	protected String id;
	protected UserInfo ui;
	protected String description;
	protected String dbID;
	protected boolean canceled = false;
	protected CBFWFContext context;
	protected VisitProcessInfo vpi;
	protected String templateDir;
	protected String matlabDir;
	protected String resultTarFile;
	protected String emailTo;
	private String status;
	protected SimpleDateFormat sdf = new SimpleDateFormat("MM_dd_yyyy");
	protected List<IJobEventListener> listeners = new ArrayList<IJobEventListener>(
			1);
	protected Log log = LogFactory.getLog(AbstractCBFProcessingJob.class);

	public AbstractCBFProcessingJob() {
		super();
	}

	public synchronized boolean isCanceled() {
		return canceled;
	}

	public synchronized void cancel() {
		canceled = true;
	}

	public int getNumberOfStages() {
		return 2;
	}

	protected void prep4Recovery() {
		List<VisitProcessInfo> vpiList = new ArrayList<VisitProcessInfo>(1);
		vpiList.add(this.vpi);
		context.setVpiList(vpiList);
	}

	protected void sendMessage(String msg) {
		JobEvent event = new JobEvent(this, msg);
		// assumption only JobScheduler listens for events and does not remove
		// itself as a listener later
		for (IJobEventListener listener : listeners) {
			listener.statusChanged(event);
		}
	}

	public String getDbID() {
		return context.getDbID();
	}

	public String getDescription() {
		return description;
	}

	public String getID() {
		return id;
	}

	public String[] getResultsFiles() {
		String[] rfArr = new String[1];
		rfArr[0] = this.resultTarFile;
		return rfArr;
	}

	public String getType() {
		return CBFBIRNConstants.CBF_PROCESSING_WORKFLOW;
	}

	public String getUser() {
		return ui.getPerceivedName();
	}

	public UserInfo getUserInfo() {
		return ui;
	}

	public int getDurationType() {
		return LONG;
	}

	public void addJobEventListener(IJobEventListener listener) {
		listeners.add(listener);
	}

	protected void handleEmail(int messageType) throws Exception {
		System.out.println("handling email...");
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		String emailHost = configService.getParamValue("email.host");
		String emailUser = configService.getParamValue("email.user");
		String emailPwd = configService.getParamValue("email.pwd");
		String emailFrom = configService.getParamValue("email.from");
		boolean canSendEmail =  GenUtils.hasEnoughInfoForEmail(emailHost, emailFrom);

		canSendEmail &= GenUtils.isValidEmail(emailFrom);

		String str = configService.getParamValue("cbfbirn.send.job.email");
		if ((str != null) && str.toLowerCase().equals("false")) {
			canSendEmail = false;
		}

		if (canSendEmail) {
			VisitInfo vi = context.getViList().get(0);
			String subjectID = vi.getSubjectID();

			EmailHelper helper = new EmailHelper(emailHost, emailUser,
					emailPwd, emailFrom);
			String toEmail = this.emailTo;
			System.out.println("handleEmail:: toEmail:" + toEmail);
			if (toEmail == null) {
				log.error("Cannot send email email was null!");
				return;
			}
			if (messageType == USER_ACTION_MAIL) {
				String subject = "Your CBF Process needs your input";
				Map<String, String> paramMap = new HashMap<String, String>();
				paramMap.put("email", toEmail);
				paramMap.put("username", ui.getName());
				paramMap.put("jobId", this.id);
				paramMap.put("subjectID", subjectID);

				try {
					helper.sendEmail(toEmail, subject, paramMap,
							this.templateDir, "user_action_needed.vm", false);
				} catch (Throwable t) {
					log.error("sendEmail", t);
				}
			} else {
				String subject = "Your CBF Process for subject " + subjectID
						+ " has finished";
				Map<String, String> paramMap = new HashMap<String, String>();
				paramMap.put("email", toEmail);
				paramMap.put("username", ui.getName());
				paramMap.put("jobId", this.id);
				paramMap.put("subjectID", subjectID);
				try {
					helper.sendEmail(toEmail, subject, paramMap,
							this.templateDir, "cbf_wf_finished.vm", false);
				} catch (Throwable t) {
					log.error("sendEmail", t);
				}
			}
		}
	}

	@Override
	public IJobFactory getJobFactory() {
		return new CBFProcessingJobFactory();
	}

	public synchronized String getStatus() {
		return status;
	}

	public synchronized void setStatus(String status) {
		this.status = status;
	}

	@Override
	public void cleanup() {
		log.info("job cleanup");
		cleanupWorkingDir();
	}

	@Override
	public abstract void execute(int stageId) throws JobException;

	@Override
	public String getContextAsJSON() {
		try {
			JSONObject json = context.toJSON();
			return json.toString();
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	@Override
	public List<JobVisitContext> getJobVisitContextList() {
		List<JobVisitContext> jvcList = new ArrayList<JobVisitContext>(1);
		for (VisitInfo vi : context.getViList()) {
			JobVisitContext jvc = new JobVisitContext(vi.getExpId(),
					vi.getVisitId(), vi.getSubjectID());
			jvcList.add(jvc);
		}
		return jvcList;
	}

	@Override
	public void saveResults(File file) throws JobException {
		// no op
	}

	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}

	@Override
	public void shutdown() {
		listeners = null;
	}

	/**
	 * cleanup cache directory
	 * 
	 * @param jobID
	 */
	protected void cleanupCache(String jobID) {
		try {
			JobScheduler.getInstance().cleanupCache(jobID);
		} catch (JobSchedulerException e) {
			log.error("cleanupCache", e);
		}
	}

	protected void cleanupWorkingDir() {
		boolean doClean = true;
		String curStatus = getStatus();
		if (curStatus == null) {
			doClean = false;
		} else {
			if (!curStatus.equals(JobInfo.CANCELED)
					&& !curStatus.equals(JobInfo.FINISHED)
					&& !curStatus.equals(JobInfo.FINISHED_WITH_ERR)
					&& !curStatus.equals(JobInfo.REMOVED)) {
				doClean = false;
			}
		}
		try {
			IAppConfigService configService = ServiceFactory
					.getAppConfigService();
			String value = configService.getParamValue("cbfbirn.clean.workdir");
			if (value != null && value.equalsIgnoreCase("false")) {
				doClean = false;
			}
		} catch (BaseException e) {
			log.error("cleanupWorkingDir", e);
		}
		if (doClean) {
			String cacheDir = context.getCacheDir();
			log.info("removing temporary working directory contents (excluding download tarballs):"
					+ cacheDir);
			File wd = new File(cacheDir);
			File[] immediateChildren = wd.listFiles();
			for (File f : immediateChildren) {
				if (f.isDirectory()) {
					FileUtils.deleteRecursively(f);
				}
			}
		}
	}

	public static class CBFProcessingJobFactory implements IJobFactory {
		@Override
		public IJob create(JobRecord jr) throws JobException {
			if (jr == null) {
				return null;
			}
			String ctxStr = jr.getContext();
			if (ctxStr == null || ctxStr.trim().length() == 0) {
				return null;
			}
			try {
				JSONObject js = new JSONObject(jr.getContext());
				if (js.has("vpiList")) {
					JSONArray jsArr = js.getJSONArray("vpiList");
					if (jsArr.length() > 0) {
						CBFWFContext ctx = CBFWFContext.fromJSON(js);
						// FIXME one visit per job processing
						JSONObject vpiJS = jsArr.getJSONObject(0);
						VisitProcessInfo vpi = VisitProcessInfo
								.initializeFromJSON(vpiJS);
						UserInfo jobUI = new UserInfo(jr.getUser(), null, null);
                        return new CBFProcessingJob(
                                jr.getJobID(), jobUI, jr.getDescription(), ctx,
                                ctx.getTemplateDir(), ctx.getMatlabDir(),
                                ctx.getEmailTo(), vpi);
					}
				}

			} catch (Throwable t) {
				throw new JobException(t);
			}
			return null;
		}
	}// ;

}
