package clinical.web.workflow.cbf;

import static clinical.web.CBFBIRNConstants.AFNI_DIR_NAME;
import static clinical.web.CBFBIRNConstants.ANATOMICAL;
import static clinical.web.CBFBIRNConstants.CSF;
import static clinical.web.CBFBIRNConstants.FIELD_MAP;
import static clinical.web.CBFBIRNConstants.MINCON;
import static clinical.web.CBFBIRNConstants.PCASL_8MP;
import static clinical.web.CBFBIRNConstants.PCASL_8MP_CSF;
import static clinical.web.CBFBIRNConstants.MATLAB_EXEC;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.CBFImageTypeDecider;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.CBFBirnHelper;
import clinical.web.services.CBFImageTypeLookupService;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFProcJobHelper.java 794 2013-04-09 23:34:08Z bozyurt $
 */
public final class CBFProcJobHelper extends AbstractCBFProcJobHelper {
	private SegmentInfo fieldMap1SI;
	private SegmentInfo fieldMap2SI;
	protected static Log _log = LogFactory.getLog(CBFProcJobHelper.class);

	public CBFProcJobHelper(UserInfo ui, CBFWFContext context,
			String templateDir) throws BaseException {
		super(ui, context, templateDir);
	}

	// only used for testing
	CBFProcJobHelper(UserInfo ui, CBFWFContext context, String templateDir,
			boolean verbose) throws BaseException {
		super();
		this.context = context;
		this.ui = ui;
		this.templateDir = templateDir;
	}

	private void makeDir(File aDir) throws Exception {
		if (!aDir.mkdir()) {
			_log.error("cannot create dir:" + aDir);
			throw new Exception(
					"A unrecovervable problem occured during CBF job processing!");
		}
	}

	public VisitProcessInfo prepare4Job(File matlabDirFile) throws Exception {
		File afniDIR = new File(this.context.getCacheDir(), AFNI_DIR_NAME);
		makeDir(afniDIR);
		File outDir = new File(this.context.getCacheDir(), "matlab_out");
		makeDir(outDir);
		File fmDir = new File(this.context.getCacheDir(), "fm");
		makeDir(fmDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		makeDir(resultsDir);

		File senseDir = new File(this.context.getCacheDir(), "sense");
		makeDir(senseDir);

		File skullDir = new File(this.context.getCacheDir(), "skull_strip");
		makeDir(skullDir);

		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		makeDir(intermediateDir);

		List<SegmentInfo> si4AFNIList = getSegmentsNeedingAFNIConversion();
		handleAFNIConversions(si4AFNIList, afniDIR);

		List<FieldMapInfo> fmList = null;
		List<SenseInfo> seList = null;
		if (this.context.isDoB0Correction()) {
			determineFieldMapSegments();

			List<SegmentInfo> si4FMList = getSegmentsNeedingFieldMapCorrection();
			fmList = prepare4FieldMaps(si4FMList, fmDir);

		} else if (this.context.isDoSenseval()) {
			List<SegmentInfo> si4SenseList = getSegmentsNeedingSenseval();
			seList = prepare4Sense(si4SenseList, senseDir);
		}

		List<BrikWrapper> cbfBIList = new ArrayList<BrikWrapper>();
		for (BrikWrapper bi : this.brikInfoList) {
			if (CBFBirnHelper.isCBFProcessable(bi.getSi().getProtocolId())) {
				cbfBIList.add(bi);
			}
		}
		for (BrikWrapper bi : this.existingBrikInfoList) {
			if (CBFBirnHelper.isCBFProcessable(bi.getSi().getProtocolId())) {
				File newBrikFile = copyBrikFilesTo(bi.getBrikFile(), afniDIR);
				bi.setBrikFile(newBrikFile);
				cbfBIList.add(bi);
			} else {
				// still needs to copy them to working afni directory so that
				// matlab processes can find them
				copyBrikFilesTo(bi.getBrikFile(), afniDIR);
			}
		}

		List<CBFProcessInfo> cpiList = prepare4CBFProcessing(cbfBIList, outDir,
				matlabDirFile);

		VisitProcessInfo vpi = null;
		if (seList != null) {
			vpi = new VisitProcessInfo(cpiList);
			vpi.setSeList(seList);
		} else {
			vpi = new VisitProcessInfo(cpiList, fmList);
		}

		// FIXME orig (cleanupPreviousDerivedData)
		// cleanupPreviousDerivedData();
		return vpi;
	}

	private List<SenseInfo> prepare4Sense(List<SegmentInfo> si4SenseList,
			File outputDir) {
		List<SenseInfo> seList = new ArrayList<SenseInfo>(5);
		for (SegmentInfo si : si4SenseList) {
			File pfile = new File(si.getShi().getImages().get(0));
			String prefix = pfile.getName();
			// remove any suffix
			prefix = prefix.replaceFirst("\\.[^\\.]+$", "");
			prefix = CBFBIRNConstants.SENSE_PREFIX + prefix;
			SenseInfo se = new SenseInfo(pfile, prefix, outputDir, si);
			seList.add(se);
		}
		return seList;
	}

	private List<SegmentInfo> getSegmentsNeedingSenseval() {
		List<SegmentInfo> si4SenseList = new ArrayList<SegmentInfo>(5);
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.PFILE)) {
				if (si.getProtocolId().equals(CSF)
						|| CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
					si4SenseList.add(si);
				}
			}
		}
		return si4SenseList;
	}

	public void updateCPIListWithPreviousSenseWFStep(VisitProcessInfo vpi,
			WFStep senseStep) throws Exception {
		if (vpi.getSeList().isEmpty()) {
			return;
		}
		Map<String, SenseInfo> seMap = new HashMap<String, SenseInfo>(11);
		for (SenseInfo se : vpi.getSeList()) {
			String prefix = se.getPrefix();
			seMap.put(prefix, se);
		}
		File afniDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		AFNIBrik csfTcatBrik = null;
		for (AFNIBrik brik : senseStep.getBriks()) {
			String brikName = brik.getBrikName();
			if (brikName.startsWith(CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX)) {
				List<File> brikFiles = brik.copyTo(afniDIR, false);
				csfTcatBrik = AFNIBrik.create(brikFiles);
			} else {
				String prefix = getPrefix(brik.getBrikFile());
				int idx = prefix.indexOf(CBFBIRNConstants.TCAT_PATTERN);
				if (idx != -1) {
					prefix = prefix.substring(0, idx);
				}
				if ((idx = prefix.indexOf('+')) != -1) {
					prefix = prefix.substring(0, idx);
				}

				SenseInfo se = seMap.get(prefix);
				Assertion.assertNotNull(se);
				List<File> senseOutFiles = brik.copyTo(afniDIR, false);
				for (File f : senseOutFiles) {
					se.addSenseOutBrikFile(f);
				}
			}
		}
		// 22x docsf=1 (embedded CSF) support
		updateCPIListWithSenseFromExisting(vpi, csfTcatBrik);
	}

	public void updateCPIListWithSenseFromExisting(VisitProcessInfo vpi,
			AFNIBrik csfTcatBrik) throws Exception {
		if (vpi.getSeList().isEmpty()) {
			return;
		}
		Map<String, SenseInfo> seMap = new HashMap<String, SenseInfo>(37);
		SenseInfo csfSE = null;
		for (SenseInfo se : vpi.getSeList()) {
			if (se.getSenseOutBrikFiles() != null
					&& !se.getSenseOutBrikFiles().isEmpty()) {
				String prefix = se.getSenseOutBrikFiles().get(0).getName();
				int idx = prefix.indexOf('+');
				if (idx != -1) {
					prefix = prefix.substring(0, idx);
				}
				idx = prefix.indexOf("tcat");
				if (idx != -1) {
					prefix = prefix.substring(0, idx);
				}
				System.out.println("prefix:" + prefix);
				seMap.put(prefix, se);
				if (se.getPfileSI().getProtocolId().equals(CSF)) {
					csfSE = se;
				}
			}
		}

		Set<SenseInfo> usedSet = new HashSet<SenseInfo>();
		Set<CBFProcessInfo> senseAffectedCPISet = new HashSet<CBFProcessInfo>(7);

		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			if (csfSE != null) {
				System.out.println("updating cpi " + cpi.getCbfSI().getName()
						+ " CSF via SENSE version");

				cpi.csfBrik = csfSE.getSenseOutBrikFile();
				cpi.csfSI = csfSE.getPfileSI();
				usedSet.add(csfSE);
			}

			File cbfBrikFile = cpi.getCbfBrik();
			String prefix = getPrefix(cbfBrikFile);
			int idx = prefix.indexOf('+');
			prefix = prefix.substring(0, idx);
			if (!prefix.startsWith(CBFBIRNConstants.SENSE_PREFIX)) { // "sense_"
				prefix = CBFBIRNConstants.SENSE_PREFIX + prefix; // sense_
				// also remove any 'brik'
				if (prefix.indexOf("brik") != -1) {
					prefix = prefix.replaceFirst("brik", "");
				}
				System.out.println("prefix for SENSE cbf:" + prefix);
			}
			SenseInfo se = seMap.get(prefix);
			if (se != null) {
				cpi.cbfBrik = se.getSenseOutBrikFile();
				cpi.cbfSI = se.getPfileSI();
				usedSet.add(se);
				senseAffectedCPISet.add(cpi);
				System.out.println("SENSE cpi:" + cpi.toString());
			} else {
				System.out.println("No match! SENSE cpi:" + cpi.toString());
			}
		}

		File outDir = new File(this.context.getCacheDir(), "matlab_out");

		for (CBFProcessInfo cpi : senseAffectedCPISet) {
			SegmentInfo aslSI = cpi.getCbfSI();
			if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(aslSI
					.getProtocolId())) {
				Assertion.assertNotNull(csfTcatBrik);

				cpi.csfBrik = csfTcatBrik.getBrikFile(); // newCPI.getCsfBrik();
				cpi.csfSI = cpi.getCbfSI();

				// replace cpi ASL brik file with the tcat version
				// cpi.cbfBrik = newCPI.getCbfBrik();
			}
		}

		for (SenseInfo se : seMap.values()) {
			if (usedSet.contains(se)) {
				continue;
			}
			if (CBFBirnHelper.isCBFProcessable(se.getPfileSI().getProtocolId())) {
				// for GE 22x docsf = 1
				if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(se
						.getPfileSI().getProtocolId())) {
					Assertion.assertNotNull(csfTcatBrik);
					csfSE = new SenseInfo(se.getPfile(), se.getPrefix(),
							se.getOutputDir(), se.getPfileSI());
					csfSE.addSenseOutBrikFile(csfTcatBrik.getBrikFile());

				}
				CBFProcessInfo cpi = prepareSense4CBFProcessing(se, csfSE,
						outDir);
				vpi.getCpiList().add(cpi);
			}
		}
	}

	public void updateCPIListWithSense(VisitProcessInfo vpi, File matlabDirFile)
			throws Exception {
		if (vpi.getSeList().isEmpty()) {
			return;
		}
		Map<String, SenseInfo> seMap = new HashMap<String, SenseInfo>(37);
		SenseInfo csfSE = null;
		for (SenseInfo se : vpi.getSeList()) {
			if (se.getSenseOutBrikFiles() != null
					&& !se.getSenseOutBrikFiles().isEmpty()) {
				String prefix = se.getSenseOutBrikFiles().get(0).getName();
				prefix = prefix.replaceFirst("\\.\\w+$", "");
				System.out.println("prefix:" + prefix);
				seMap.put(prefix, se);
				if (se.getPfileSI().getProtocolId().equals(CSF)) {
					csfSE = se;
				}
			}
		}

		Set<SenseInfo> usedSet = new HashSet<SenseInfo>();
		Set<CBFProcessInfo> senseAffectedCPISet = new HashSet<CBFProcessInfo>(7);

		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			if (csfSE != null) {
				System.out.println("updating cpi " + cpi.getCbfSI().getName()
						+ " CSF via SENSE version");

				cpi.csfBrik = csfSE.getSenseOutBrikFile();
				cpi.csfSI = csfSE.getPfileSI();
				usedSet.add(csfSE);
			}

			File cbfBrikFile = cpi.getCbfBrik();
			String prefix = getPrefix(cbfBrikFile);
			if (!prefix.startsWith(CBFBIRNConstants.SENSE_PREFIX)) { // "sense_"
				prefix = CBFBIRNConstants.SENSE_PREFIX + prefix; // sense_
				// also remove any 'brik'
				if (prefix.indexOf("brik") != -1) {
					prefix = prefix.replaceFirst("brik", "");
				}
				System.out.println("prefix for SENSE cbf:" + prefix);
			}
			SenseInfo se = seMap.get(prefix);
			if (se != null) {
				cpi.cbfBrik = se.getSenseOutBrikFile();
				cpi.cbfSI = se.getPfileSI();
				usedSet.add(se);
				senseAffectedCPISet.add(cpi);
				System.out.println("SENSE cpi:" + cpi.toString());
			} else {
				System.out.println("No match! SENSE cpi:" + cpi.toString());
			}
		}

		File outDir = new File(this.context.getCacheDir(), "matlab_out");

		for (CBFProcessInfo cpi : senseAffectedCPISet) {
			SegmentInfo aslSI = cpi.getCbfSI();
			if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(aslSI
					.getProtocolId())) {
				BrikWrapper bi = new BrikWrapper(cpi.getCbfBrik(), aslSI);
				CBFProcessInfo newCPI = handleBrikSplitting(bi, matlabDirFile,
						outDir);

				cpi.csfBrik = newCPI.getCsfBrik();
				cpi.csfSI = cpi.getCbfSI();

				// replace cpi ASL brik file with the tcat version
				cpi.cbfBrik = newCPI.getCbfBrik();
			}
		}

		for (SenseInfo se : seMap.values()) {
			if (usedSet.contains(se)) {
				continue;
			}
			if (CBFBirnHelper.isCBFProcessable(se.getPfileSI().getProtocolId())) {
				// for GE 22x docsf = 1
				if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(se
						.getPfileSI().getProtocolId())) {
					BrikWrapper bi = new BrikWrapper(se.getSenseOutBrikFile(),
							se.getPfileSI());
					CBFProcessInfo newCPI = handleBrikSplitting(bi,
							matlabDirFile, outDir);
					csfSE = new SenseInfo(se.getPfile(), se.getPrefix(),
							se.getOutputDir(), se.getPfileSI());
					csfSE.addSenseOutBrikFile(newCPI.getCsfBrik());
					// replace the embedded ASL dbl brik with tcat version
					se.replaceSenseOutBrikFiles(newCPI.getCbfBrik());
				}
				CBFProcessInfo cpi = prepareSense4CBFProcessing(se, csfSE,
						outDir);
				vpi.getCpiList().add(cpi);
			}
		}
	}

	public void updateCPIListWithPreviousFieldMapWFStep(VisitProcessInfo vpi,
			WFStep fmStep) throws Exception {
		if (vpi.getFmList().isEmpty()) {
			return;
		}
		Map<String, FieldMapInfo> fmiMap = new HashMap<String, FieldMapInfo>(11);
		for (FieldMapInfo fmi : vpi.getFmList()) {
			String pFile = fmi.getPfile().getName();
			String prefix = pFile.replaceFirst("\\.\\w+$", "");
			if (!prefix.endsWith("brik")) {
				prefix = prefix + "brik";
			}
			System.out
					.println("updateCPIListWithPreviousFieldMapWFStep prefix:"
							+ prefix);
			fmiMap.put(prefix, fmi);
		}
		// copy over fieldmap corrected briks from the original run with
		// fieldmap to afni dir and add them to the corresponding FieldMapInfo
		// objects as output files
		File afniDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		AFNIBrik csfTcatBrik = null;
		for (AFNIBrik brik : fmStep.getBriks()) {
			String brikName = brik.getBrikName();
			if (brikName.startsWith(CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX)) {
				List<File> brikFiles = brik.copyTo(afniDIR, false);
				csfTcatBrik = AFNIBrik.create(brikFiles);
			} else {
				int idx = brikName
						.indexOf(CBFBIRNConstants.FIELDMAP_BRIK_PATTERN);
				String prefix = brikName.substring(0, idx);
				FieldMapInfo fmi = fmiMap.get(prefix);
				Assertion.assertNotNull(fmi);
				List<File> fmOutFiles = brik.copyTo(afniDIR, false);
				for (File f : fmOutFiles) {
					fmi.addFmOutBrikFile(f);
				}
			}
		}

		updateCPIListWithFieldMapCorFromExisting(vpi, csfTcatBrik);
	}

	public void updateCPIListWithFieldMapCorFromExisting(VisitProcessInfo vpi,
			AFNIBrik csfTcatBrik) throws Exception {
		if (vpi.getFmList().isEmpty()) {
			return;
		}
		Map<String, FieldMapInfo> fmiMap = new HashMap<String, FieldMapInfo>(11);
		FieldMapInfo csfFMI = null;
		FieldMapInfo minconFMI = null;
		for (FieldMapInfo fmi : vpi.getFmList()) {
			if (fmi.getFmOutBrikFiles() != null
					&& !fmi.getFmOutBrikFiles().isEmpty()) {
				String prefix = fmi.getFmOutBrikFiles().get(0).getName();
				prefix = prefix.replaceFirst("\\.\\w+$", "");
				int idx = prefix
						.indexOf(CBFBIRNConstants.FIELDMAP_BRIK_PATTERN); // "+dbl"
				prefix = prefix.substring(0, idx);
				System.out.println("prefix:" + prefix);
				fmiMap.put(prefix, fmi);
				if (fmi.getPfileSI().getProtocolId().equals(MINCON)) {
					minconFMI = fmi;
				} else if (fmi.getPfileSI().getProtocolId().equals(CSF)) {
					csfFMI = fmi;
				}
			}
		}

		Set<FieldMapInfo> usedSet = new HashSet<FieldMapInfo>();
		Set<CBFProcessInfo> fmAffectedCPISet = new HashSet<CBFProcessInfo>(7);

		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			if (csfFMI != null) {
				cpi.csfBrik = csfFMI.getFmOutBrikFile();
				cpi.csfSI = csfFMI.getPfileSI();
				usedSet.add(csfFMI);
			}

			if (minconFMI != null) {
				cpi.minconBrik = minconFMI.getFmOutBrikFile();
				cpi.minconSI = minconFMI.getPfileSI();
				usedSet.add(minconFMI);
			}

			File cbfBrikFile = cpi.getCbfBrik();
			String prefix = getPrefix(cbfBrikFile);
			FieldMapInfo fmi = fmiMap.get(prefix);
			if (fmi != null) {
				cpi.cbfBrik = fmi.getFmOutBrikFile();
				cpi.cbfSI = fmi.getPfileSI();
				fmAffectedCPISet.add(cpi);
				usedSet.add(fmi);
			}
		}

		for (CBFProcessInfo cpi : fmAffectedCPISet) {
			SegmentInfo aslSI = cpi.getCbfSI();
			if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(aslSI
					.getProtocolId())) {
				Assertion.assertNotNull(csfTcatBrik);

				cpi.csfBrik = csfTcatBrik.getBrikFile(); // newCPI.getCsfBrik();
				cpi.csfSI = cpi.getCbfSI();
				// replace cpi ASL brik file with the tcat version
				// cpi.cbfBrik = newCPI.getCbfBrik();
			}
		}

		File outDir = new File(this.context.getCacheDir(), "matlab_out");
		for (FieldMapInfo fmi : fmiMap.values()) {
			if (usedSet.contains(fmi)) {
				continue;
			}
			if (CBFBirnHelper
					.isCBFProcessable(fmi.getPfileSI().getProtocolId())) {
				// for GE 22x docsf = 1
				if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(fmi
						.getPfileSI().getProtocolId())) {
					Assertion.assertNotNull(csfTcatBrik);
					csfFMI = new FieldMapInfo(fmi.getPfile(),
							fmi.getFieldMap1DicomDir(),
							fmi.getFieldMap2DicomDir(), fmi.getPrefix(),
							fmi.getOutputDir(), fmi.getPfileSI());
					csfFMI.addFmOutBrikFile(csfTcatBrik.getBrikFile());
				}

				CBFProcessInfo cpi = prepareFMI4CBFProcessing(fmi, csfFMI,
						minconFMI, outDir);
				vpi.getCpiList().add(cpi);
			}
		}
	}

	public void updateCPIListWithFieldMapCorrection(VisitProcessInfo vpi,
			File matlabDirFile) throws Exception {
		if (vpi.getFmList().isEmpty()) {
			return;
		}
		Map<String, FieldMapInfo> fmiMap = new HashMap<String, FieldMapInfo>(11);
		FieldMapInfo csfFMI = null;
		FieldMapInfo minconFMI = null;
		for (FieldMapInfo fmi : vpi.getFmList()) {
			if (fmi.getFmOutBrikFiles() != null
					&& !fmi.getFmOutBrikFiles().isEmpty()) {
				String prefix = fmi.getFmOutBrikFiles().get(0).getName();
				prefix = prefix.replaceFirst("\\.\\w+$", "");
				int idx = prefix
						.indexOf(CBFBIRNConstants.FIELDMAP_BRIK_PATTERN); // "+dbl"
				prefix = prefix.substring(0, idx);
				System.out.println("prefix:" + prefix);
				fmiMap.put(prefix, fmi);
				if (fmi.getPfileSI().getProtocolId().equals(MINCON)) {
					minconFMI = fmi;
				} else if (fmi.getPfileSI().getProtocolId().equals(CSF)) {
					csfFMI = fmi;
				}
			}
		}

		Set<FieldMapInfo> usedSet = new HashSet<FieldMapInfo>();
		Set<CBFProcessInfo> fmAffectedCPISet = new HashSet<CBFProcessInfo>(7);

		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			if (csfFMI != null) {
				cpi.csfBrik = csfFMI.getFmOutBrikFile();
				cpi.csfSI = csfFMI.getPfileSI();
				usedSet.add(csfFMI);
			}

			if (minconFMI != null) {
				cpi.minconBrik = minconFMI.getFmOutBrikFile();
				cpi.minconSI = minconFMI.getPfileSI();
				usedSet.add(minconFMI);
			}

			File cbfBrikFile = cpi.getCbfBrik();
			String prefix = getPrefix(cbfBrikFile);
			FieldMapInfo fmi = fmiMap.get(prefix);
			if (fmi != null) {
				cpi.cbfBrik = fmi.getFmOutBrikFile();
				cpi.cbfSI = fmi.getPfileSI();
				fmAffectedCPISet.add(cpi);
				usedSet.add(fmi);
			}
		}

		File outDir = new File(this.context.getCacheDir(), "matlab_out");

		for (CBFProcessInfo cpi : fmAffectedCPISet) {
			SegmentInfo aslSI = cpi.getCbfSI();
			if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(aslSI
					.getProtocolId())) {
				BrikWrapper bi = new BrikWrapper(cpi.getCbfBrik(), aslSI);
				CBFProcessInfo newCPI = handleBrikSplitting(bi, matlabDirFile,
						outDir);

				cpi.csfBrik = newCPI.getCsfBrik();
				cpi.csfSI = cpi.getCbfSI();
				// replace cpi ASL brik file with the tcat version
				cpi.cbfBrik = newCPI.getCbfBrik();
			}
		}

		for (FieldMapInfo fmi : fmiMap.values()) {
			if (usedSet.contains(fmi)) {
				continue;
			}
			if (CBFBirnHelper
					.isCBFProcessable(fmi.getPfileSI().getProtocolId())) {
				// for GE 22x docsf = 1
				if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(fmi
						.getPfileSI().getProtocolId())) {
					BrikWrapper bi = new BrikWrapper(fmi.getFmOutBrikFile(),
							fmi.getPfileSI());
					CBFProcessInfo newCPI = handleBrikSplitting(bi,
							matlabDirFile, outDir);
					csfFMI = new FieldMapInfo(fmi.getPfile(),
							fmi.getFieldMap1DicomDir(),
							fmi.getFieldMap2DicomDir(), fmi.getPrefix(),
							fmi.getOutputDir(), fmi.getPfileSI());
					csfFMI.addFmOutBrikFile(newCPI.getCsfBrik());
					// replace the embedded ASL dbl brik with tcat version
					fmi.replaceFmOutFiles(newCPI.getCbfBrik());
				}

				CBFProcessInfo cpi = prepareFMI4CBFProcessing(fmi, csfFMI,
						minconFMI, outDir);
				vpi.getCpiList().add(cpi);
			}
		}
	}

	public void handleSENSEProcessing(SenseInfo se, File matlabDir)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		ctx.put("matlabdir", matlabDir);
		ctx.put("pfile", se.getPfile());
		ctx.put("prefix", se.getPrefix());
		ctx.put("outdir", se.getOutputDir());

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("sense.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(se.getOutputDir(), "sense.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		executor.setErrorStreamReliable(false);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File afniDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		File[] files = se.getOutputDir().listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.startsWith(se.getPrefix()) && hasAFNIExtension(name)) {
				File afniFile = new File(afniDIR, f.getName());
				if (!FileUtils.moveTo(f, afniFile)) {
					throw new Exception("Cannot move file:" + f.getName());
				}
				se.addSenseOutBrikFile(afniFile);
			}
		}
		cleanDirectory(se.getOutputDir());
	}

	public void handleFieldMapProcessing(FieldMapInfo fmi, File matlabDir,
			boolean doRegistration) throws Exception {
		if (isCanceled()) {
			return;
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		String doRegStr = doRegistration ? "1" : "0";
		ctx.put("matlabdir", matlabDir);
		ctx.put("pfile", fmi.getPfile());
		ctx.put("fm1", fmi.getFieldMap1DicomDir());
		ctx.put("fm2", fmi.getFieldMap2DicomDir());
		ctx.put("prefix", fmi.getPrefix());
		ctx.put("outdir", fmi.getOutputDir());
		ctx.put("doReg", doRegStr);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("fm.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(fmi.getOutputDir(), "fm.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		executor.setErrorStreamReliable(false);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File afniDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		File[] files = fmi.getOutputDir().listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.startsWith(fmi.getPrefix()) && hasAFNIExtension(name)) {
				File afniFile = new File(afniDIR, f.getName());
				if (!FileUtils.moveTo(f, afniFile)) {
					throw new Exception("Cannot move file:" + f.getName());
				}
				if (name.indexOf(CBFBIRNConstants.FIELDMAP_BRIK_PATTERN) != -1) { // "+dbl"
					fmi.addFmOutBrikFile(afniFile);
				} else {
					fmi.addOutBrikFile(afniFile);
				}
			}
		}
		cleanDirectory(fmi.getOutputDir());
	}

	protected CBFProcessInfo prepareSense4CBFProcessing(SenseInfo se,
			SenseInfo csfSE, File outDir) throws Exception {
		boolean hasAnat = hasAnatomical();
		BrikWrapper anatBI = null;
		if (hasAnat) {
			anatBI = findBrikInfo(ANATOMICAL);
		}
		File csfBrikFile;
		File minconBrikFile;
		SegmentInfo csfSI;
		SegmentInfo minconSI;
		CBFProcessInfo cpi;

		BrikWrapper csfBI = findBrikInfo(CSF);
		if (csfSE != null) {
			csfBrikFile = csfSE.getSenseOutBrikFile();
			csfSI = csfSE.getPfileSI();
		} else {
			csfBrikFile = csfBI.getBrikFile();
			csfSI = csfBI.getSi();
		}

		BrikWrapper minconBI = findBrikInfo(MINCON);
		minconBrikFile = minconBI.getBrikFile();
		minconSI = minconBI.getSi();

		File cbfBrikFile = se.getSenseOutBrikFile();
		if (hasAnat) {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					anatBI.getBrikFile(), se.getPfileSI(), csfSI, minconSI,
					anatBI.getSi(), outDir);
		} else {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					se.getPfileSI(), csfSI, minconSI, outDir);
		}
		return cpi;
	}

	protected CBFProcessInfo prepareFMI4CBFProcessing(FieldMapInfo fmi,
			FieldMapInfo csfFMI, FieldMapInfo minconFMI, File outDir)
			throws Exception {
		boolean hasAnat = hasAnatomical();
		BrikWrapper anatBI = null;
		if (hasAnat) {
			anatBI = findBrikInfo(ANATOMICAL);
		}
		File csfBrikFile;
		File minconBrikFile;
		SegmentInfo csfSI;
		SegmentInfo minconSI;
		CBFProcessInfo cpi;

		BrikWrapper csfBI = findBrikInfo(CSF);
		if (csfFMI != null && csfBI == null) {
			csfBrikFile = csfFMI.getFmOutBrikFile();
			csfSI = csfFMI.getPfileSI();
		} else {
			csfBrikFile = csfBI.getBrikFile();
			csfSI = csfBI.getSi();
		}

		BrikWrapper minconBI = findBrikInfo(MINCON);
		if (minconFMI != null && minconBI == null) {
			minconBrikFile = minconFMI.getFmOutBrikFile();
			minconSI = minconFMI.getPfileSI();
		} else {
			// minconBI can only be null if there is a corresponding minconFMI
			minconBrikFile = minconBI.getBrikFile();
			minconSI = minconBI.getSi();
		}

		File cbfBrikFile = fmi.getFmOutBrikFile();
		if (hasAnat) {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					anatBI.getBrikFile(), fmi.getPfileSI(), csfSI, minconSI,
					anatBI.getSi(), outDir);
		} else {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					fmi.getPfileSI(), csfSI, minconSI, outDir);
		}
		return cpi;
	}

	private List<CBFProcessInfo> prepare4CBFProcessing(
			List<BrikWrapper> biList, File outDir, File matlabDirFile)
			throws Exception {
		List<CBFProcessInfo> cpiList = new ArrayList<CBFProcessInfo>();
		boolean hasAnat = hasAnatomical();
		BrikWrapper minconBI = findBrikInfo(MINCON);
		BrikWrapper csfBI = findBrikInfo(CSF);
		BrikWrapper anatBI = null;
		if (hasAnat) {
			anatBI = findBrikInfo(ANATOMICAL);
		}

		// 22x support
		for (BrikWrapper bi : biList) {
			SegmentInfo si = bi.getSi();
			if (CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				if (CBFBirnHelper.isCBFProcessableAndEmbeddedCSF(si
						.getProtocolId())) {
					if (!this.context.isDoB0Correction()
							&& !this.context.isDoSenseval()) {
						// run gencsf
						CBFProcessInfo newCPI = handleBrikSplitting(bi,
								matlabDirFile, outDir);
						CBFProcessInfo cpi;
						if (hasAnat) {
							cpi = new CBFProcessInfo(newCPI.getCbfBrik(),
									newCPI.getCsfBrik(),
									minconBI.getBrikFile(),
									anatBI.getBrikFile(), bi.getSi(),
									newCPI.getCsfSI(), minconBI.getSi(),
									anatBI.getSi(), outDir);
						} else {
							cpi = new CBFProcessInfo(newCPI.getCbfBrik(),
									newCPI.getCsfBrik(),
									minconBI.getBrikFile(), bi.getSi(),
									newCPI.getCsfSI(), minconBI.getSi(), outDir);
						}
						cpiList.add(cpi);
					} else {
						// CSF data is embedded in ASL brik and not available
						// currently, will be available after fieldmap or SENSE
						// preprocessing
						CBFProcessInfo cpi;
						if (hasAnat) {
							cpi = new CBFProcessInfo(bi.getBrikFile(), null,
									minconBI.getBrikFile(),
									anatBI.getBrikFile(), bi.getSi(), null,
									minconBI.getSi(), anatBI.getSi(), outDir);
						} else {
							cpi = new CBFProcessInfo(bi.getBrikFile(), null,
									minconBI.getBrikFile(), bi.getSi(), null,
									minconBI.getSi(), outDir);
						}
						cpiList.add(cpi);
					}
				} else if (CBFBirnHelper.is3DPCASL(si.getProtocolId())) { 
					// IBO 07/14/2015
					CBFProcessInfo cpi = null;
					if (hasAnat) {
						cpi = new CBFProcessInfo(bi.getBrikFile(), null, null,
								anatBI.getBrikFile(), bi.getSi(), null, null,
								anatBI.getSi(), outDir);
					} else {
						cpi = new CBFProcessInfo(bi.getBrikFile(), null, null,
								bi.getSi(), null, null, outDir);
					}
					cpiList.add(cpi);
				} else {
					CBFProcessInfo cpi = null;
					if (hasAnat) {
						cpi = new CBFProcessInfo(bi.getBrikFile(),
								csfBI.getBrikFile(), minconBI.getBrikFile(),
								anatBI.getBrikFile(), bi.getSi(),
								csfBI.getSi(), minconBI.getSi(),
								anatBI.getSi(), outDir);
					} else {
						cpi = new CBFProcessInfo(bi.getBrikFile(),
								csfBI.getBrikFile(), minconBI.getBrikFile(),
								bi.getSi(), csfBI.getSi(), minconBI.getSi(),
								outDir);
					}
					cpiList.add(cpi);
				}
			}
		}

		return cpiList;
	}

	private CBFProcessInfo handleBrikSplitting(BrikWrapper bi,
			File matlabDirFile, File outDir) throws Exception {
		copyBrikFilesTo(bi.getBrikFile(), outDir);
		File newBrikFile = new File(outDir, bi.getBrikFile().getName());
		Assertion.assertTrue(newBrikFile.exists());
		CBFProcessInfo cpi = new CBFProcessInfo(newBrikFile, null, null,
				bi.getSi(), null, null, outDir);
		return handleEmbeddedCSFBrikSplit(cpi, matlabDirFile, bi.getBrikFile()
				.getParentFile());

	}

	private List<FieldMapInfo> prepare4FieldMaps(List<SegmentInfo> si4FMList,
			File outDir) {

		List<FieldMapInfo> fmiList = new ArrayList<FieldMapInfo>();
		if (fieldMap1SI == null || fieldMap2SI == null) {
			_log.warn("Both field maps are needed for field map correction");
			return fmiList;
		}
		for (SegmentInfo si : si4FMList) {

			File pfile = new File(si.getShi().getImages().get(0));
			String prefix = pfile.getName();
			// remove any suffix
			prefix = prefix.replaceFirst("\\.[^\\.]+$", "");

			File fm1DicomDir = new File(this.fieldMap1SI.getShi().getImages()
					.get(0));
			File fm2DicomDir = new File(this.fieldMap2SI.getShi().getImages()
					.get(0));
			FieldMapInfo fmi = new FieldMapInfo(pfile, fm1DicomDir,
					fm2DicomDir, prefix, outDir, si);
			fmiList.add(fmi);

		}
		return fmiList;
	}

	private void handleAFNIConversions(List<SegmentInfo> si4AFNIList,
			File outDir) throws Exception {
		if (isCanceled()) {
			return;
		}
		CBFImageTypeLookupService service = ServiceFactory
				.getCBFImageTypeLookupService();
		CBFImageTypeDecider decider = new CBFImageTypeDecider(service);

		for (SegmentInfo si : si4AFNIList) {
			File imageFile = new File(si.getShi().getImages().get(0));
			String prefix = imageFile.getName();
			// remove any suffix
			prefix = prefix.replaceFirst("\\.[^\\.]+$", "");
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)) {
				String dicomDir = si.getShi().getImages().get(0);
				if (si.getProtocolId().equals(FIELD_MAP)) {
					String seriesName = getSeriesName(dicomDir);
					System.out.println("seriesName:" + seriesName);
					String dicomDirName = new File(dicomDir).getName();

					int fieldMapType = decider.getFieldMapNo(seriesName,
							dicomDirName);
					if (fieldMapType == CBFImageTypeDecider.FIELD_MAP1) {
						this.fieldMap1SI = si;
					} else if (fieldMapType == CBFImageTypeDecider.FIELD_MAP2) {
						this.fieldMap2SI = si;
					} else if (fieldMapType == CBFImageTypeDecider.FIELD_MAP_GRASS) {
						this.fieldMap1SI = si;
						this.fieldMap2SI = si;
					} else {
						System.err.println("Not a recognized field map series "
								+ "description:" + seriesName);
					}
					/*
					 * if (seriesName.equals(CBFBIRNConstants.FM_TE1_DESC)) {
					 * this.fieldMap1SI = si; } else if
					 * (seriesName.equals(CBFBIRNConstants.FM_TE2_DESC)) {
					 * this.fieldMap2SI = si; } else if (seriesName
					 * .equals(CBFBIRNConstants.FM_GRASS_DESC)) {
					 * this.fieldMap1SI = si; this.fieldMap2SI = si; } else {
					 * System.err
					 * .println("Not a recognized field map series description:"
					 * + seriesName); }
					 */
				} else if (si.getProtocolId().equals(ANATOMICAL)) {
					prefix = "anat";
				}
				System.out.println("DICOM 2 AFNI conversion:" + dicomDir
						+ "\nprefix:" + prefix);
				File brikFile = convertDICOMSeries2AFNI(new File(dicomDir),
						outDir, prefix);

				brikInfoList.add(new BrikWrapper(brikFile, si));

			} else if (si.getShi().getImageType()
					.equals(SeriesHeaderInfo.PFILE)) {
				String pfile = si.getShi().getImages().get(0);
				boolean use8Ph = si.getProtocolId().equals(PCASL_8MP)
						|| si.getProtocolId().equals(PCASL_8MP_CSF);

				System.out.println("PFILE 2 AFNI conversion:" + pfile
						+ "\nuse8Ph:" + use8Ph);

				File brikFile = convertPFILE2AFNI(new File(pfile), outDir,
						prefix, use8Ph);

				brikInfoList.add(new BrikWrapper(brikFile, si));
			} else {
				throw new RuntimeException("Not a supported image type:"
						+ si.getShi().getImageType());
			}
		}
	}

	public List<Deriveddata> saveDerivedData(VisitProcessInfo vpi)
			throws Exception {
		return super.saveDerivedData(vpi, fieldMap1SI, fieldMap2SI);
	}

	private List<SegmentInfo> getSegmentsNeedingFieldMapCorrection() {
		List<SegmentInfo> si4FMList = new ArrayList<SegmentInfo>();
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.PFILE)) {
				if (si.getProtocolId().equals(MINCON)
						|| si.getProtocolId().equals(CSF)
						|| CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
					si4FMList.add(si);
				}
			}
		}
		return si4FMList;
	}

	private void determineFieldMapSegments() throws Exception {
		VisitInfo visitInfo = context.getViList().get(0);
		CBFImageTypeLookupService service = ServiceFactory
				.getCBFImageTypeLookupService();
		CBFImageTypeDecider decider = new CBFImageTypeDecider(service);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getProtocolId().equals(FIELD_MAP)
					&& si.getShi().getImageType()
							.equals(SeriesHeaderInfo.DICOM)) {
				String dicomDir = si.getShi().getImages().get(0);
				String seriesName = getSeriesName(dicomDir);
				System.out.println("seriesName:" + seriesName);
				String dicomDirName = new File(dicomDir).getName();

				int fieldMapType = decider.getFieldMapNo(seriesName,
						dicomDirName);
				if (fieldMapType == CBFImageTypeDecider.FIELD_MAP1) {
					this.fieldMap1SI = si;
				} else if (fieldMapType == CBFImageTypeDecider.FIELD_MAP2) {
					this.fieldMap2SI = si;
				} else if (fieldMapType == CBFImageTypeDecider.FIELD_MAP_GRASS) {
					this.fieldMap1SI = si;
					this.fieldMap2SI = si;
				} else {
					System.err.println("Not a recognized field map series "
							+ "description:" + seriesName);
				}
				/*
				 * if (seriesName.equals(CBFBIRNConstants.FM_TE1_DESC)) {
				 * this.fieldMap1SI = si; } else if
				 * (seriesName.equals(CBFBIRNConstants.FM_TE2_DESC)) {
				 * this.fieldMap2SI = si; } else if
				 * (seriesName.equals(CBFBIRNConstants.FM_GRASS_DESC)) {
				 * this.fieldMap1SI = si; this.fieldMap2SI = si; } else {
				 * System.err
				 * .println("Not a recognized field map series description:" +
				 * seriesName); }
				 */
			}
		}
	}

	private File convertPFILE2AFNI(File srcPFile, File destDir, String prefix,
			boolean use8Ph) throws CBFException {
		try {
			Executor executor = new Executor("/usr/local/bin/gr_wrapper.pl",
					true);
			executor.setErrorStreamReliable(false);
			StringBuilder sb = new StringBuilder(200);
			sb.append(srcPFile.getAbsolutePath()).append(' ');
			sb.append(prefix).append(' ');
			sb.append(destDir.getAbsolutePath());
			if (use8Ph) {
				sb.append(" -use-8ph");
			}
			String cmdLine = sb.toString();
			System.out.println("gr_wrapper.pl " + cmdLine);

			File brikFile = null;

			executor.execute(cmdLine);

			File[] files = destDir.listFiles();
			List<File> afniFiles = new ArrayList<File>(2);
			for (File f : files) {
				if (f.getName().startsWith(prefix)) {
					afniFiles.add(f);
					if (f.getName().endsWith(".BRIK")) {
						brikFile = f;
					}
				}
			}
			if (afniFiles.isEmpty()
					|| (afniFiles.size() != 2 && afniFiles.size() != 4)) {
				throw new Exception(
						"An error occured during pfile to AFNI conversion:"
								+ srcPFile.getName());
			}
			if (afniFiles.size() > 2) {
				brikFile = null;
				for (File f : afniFiles) {
					if (f.getName().indexOf("_e01") != -1) {
						if (f.getName().startsWith(prefix)
								&& f.getName().endsWith(".BRIK")) {
							brikFile = f;
						}
					} else {
						f.delete();
					}
				}
			}
			Assertion.assertNotNull(brikFile);

			return brikFile;
		} catch (Throwable t) {
			t.printStackTrace();
			throw new CBFException("convertPFILE2AFNI", t);
		}
	}

	public static void testit() throws Exception {
		File serFile = new File(System.getProperty("user.home"),
				"CBFWFContext.ser");
		CBFWFContext ctx = (CBFWFContext) FileUtils.deserialize(serFile
				.getAbsolutePath());
		UserInfo ui = new UserInfo("dshin", null, null);

		CBFProcJobHelper helper = new CBFProcJobHelper(ui, ctx, null, false);

		VisitProcessInfo vpi = helper.prepare4Job(null);

		File vpiSerFile = new File(System.getProperty("user.home"), "vpi.ser");
		FileUtils.serialize(vpi, vpiSerFile.getAbsolutePath());
	}

	public static void testFieldMap() throws Exception {
		String homeDir = System.getProperty("user.home");
		File serFile = new File(homeDir, "CBFWFContext.ser");

		CBFWFContext ctx = (CBFWFContext) FileUtils.deserialize(serFile
				.getAbsolutePath());
		ctx.setDoB0Correction(true);
		UserInfo ui = new UserInfo("dshin", null, null);

		File templateDir = new File(homeDir, "dev/java/clinical/conf/wf");
		CBFProcJobHelper helper = new CBFProcJobHelper(ui, ctx,
				templateDir.getAbsolutePath(), false);

		setEnvironment(helper);

		VisitProcessInfo vpi = null;

		vpi = helper.prepare4Job(null);

		File vpiSerFile = new File(homeDir, "vpi.ser");
		FileUtils.serialize(vpi, vpiSerFile.getAbsolutePath());

		vpi = (VisitProcessInfo) FileUtils.deserialize(vpiSerFile
				.getAbsolutePath());

		System.out.println(vpi.fmList);
		File matlabDir = new File("/data/matlab/birncbf");
		for (FieldMapInfo fmi : vpi.getFmList()) {
			helper.handleFieldMapProcessing(fmi, matlabDir, true);
			break;
		}

		FileUtils.serialize(vpi, vpiSerFile.getAbsolutePath());

	}

	public static void testCBFRun() throws Exception {
		String homeDir = System.getProperty("user.home");
		File serFile = new File(homeDir, "CBFWFContext.ser");

		CBFWFContext ctx = (CBFWFContext) FileUtils.deserialize(serFile
				.getAbsolutePath());
		UserInfo ui = new UserInfo("dshin", null, null);

		File templateDir = new File(homeDir, "dev/java/clinical/conf/wf");
		CBFProcJobHelper helper = new CBFProcJobHelper(ui, ctx,
				templateDir.getAbsolutePath(), false);

		setEnvironment(helper);

		File vpiSerFile = new File(homeDir, "vpi.ser");
		VisitProcessInfo vpi = (VisitProcessInfo) FileUtils
				.deserialize(vpiSerFile.getAbsolutePath());

		File matlabDir = new File(homeDir, "matlab/birncbf");
		helper.handleCBFProcessingStep1(vpi.getCpiList().get(0), matlabDir,
				ctx.isDoPartialVolumeCorrection(), ctx.getAlignment(),
				ctx.getCsfMethod(), ctx.isDoFabber());
	}

	public static void testCBFRun2() throws Exception {
		String homeDir = System.getProperty("user.home");
		File serFile = new File(homeDir, "CBFWFContext.ser");

		CBFWFContext ctx = (CBFWFContext) FileUtils.deserialize(serFile
				.getAbsolutePath());
		UserInfo ui = new UserInfo("dshin", null, null);

		File templateDir = new File(homeDir, "dev/java/clinical/conf/wf");
		CBFProcJobHelper helper = new CBFProcJobHelper(ui, ctx,
				templateDir.getAbsolutePath(), false);

		helper.setEnvPath("/usr/local/bin:/bin:/usr/bin:/data/apps/fsl/bin:/data/apps/afni/bin:.");

		File vpiSerFile = new File(homeDir, "vpi.ser");
		VisitProcessInfo vpi = (VisitProcessInfo) FileUtils
				.deserialize(vpiSerFile.getAbsolutePath());

		File matlabDir = new File(homeDir, "matlab/birncbf");
		CBFProcessInfo cpi = vpi.getCpiList().get(0);
		helper.handleCBFProcessingStep1(cpi, matlabDir,
				ctx.isDoPartialVolumeCorrection(), ctx.getAlignment(),
				ctx.getCsfMethod(), ctx.isDoFabber());
		File maskFile = new File(cpi.getOutputDir().getParent(), "mask.txt");
		Assertion.assertTrue(maskFile.exists());
		cpi.setBinaryMaskFile(maskFile);

		File sliceNoFile = new File(cpi.getOutputDir().getParent(),
				"slice_no.txt");
		Assertion.assertTrue(sliceNoFile.exists());

		cpi.setSelectedSliceFile(sliceNoFile);

		helper.handleCBFProcessingStep2(cpi, matlabDir, true, 0.6,
				ctx.isDoPartialVolumeCorrection(), CBFWFContext.AUTOMATIC, 0.0,
				false);
	}

	public static void main(String[] args) throws Exception {
		// testit();

		// testCBFRun();
		testFieldMap();
	}

}
