package clinical.web.workflow.cbf;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobException;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.upload.VisitInfo;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SiemensCBFProcessingJob.java 786 2013-03-20 02:21:25Z bozyurt $
 */

public class SiemensCBFProcessingJob extends AbstractCBFProcessingJob {
	private SiemensCBFProcJobHelper helper;
	private volatile boolean shortcut = false;

	public SiemensCBFProcessingJob(String id, UserInfo ui, String description,
			CBFWFContext context, String templateDir, String matlabDir,
			String emailTo, VisitProcessInfo vpi) throws BaseException {
		super();
		log = LogFactory.getLog(SiemensCBFProcessingJob.class);
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.context = context;
		this.templateDir = templateDir;
		this.matlabDir = matlabDir;
		this.emailTo = emailTo;
		this.helper = new SiemensCBFProcJobHelper(ui, context, templateDir);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();

		this.vpi = vpi;
	}

	public SiemensCBFProcessingJob(String id, UserInfo ui, String description,
			CBFWFContext context, String templateDir, String matlabDir,
			String emailTo) throws BaseException {
		this(id, ui, description, context, templateDir, matlabDir, emailTo,
				null);
	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			if (context.isUseAutomatedSegmentation() || context.isUseLTMethod()) {
				handleAutoCSF();
			} else {
				handleProcessing(stageId);
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new JobException(e);
		}
	}

	private Map<String, WFStep> findWFSteps2Skip() throws Exception {
		List<JobProvenanceInfo> jpiList = helper.getExistingWFResults();
		JobProvenanceInfo theJPI = MultipleWFHelper.prepJobProvenanceInfo("",
				"", context);
		Map<String, WFStep> wfStepMap = MultipleWFHelper.findWFSteps2Skip(
				theJPI, jpiList);
		return wfStepMap;
	}

	protected void handleAutoCSF() throws Exception {
		IAppConfigService configService = ServiceFactory.getAppConfigService();
		prepareEnvironment(configService);

		Map<String, WFStep> wfStepMap = findWFSteps2Skip();
		if (wfStepMap.containsKey(CBFBIRNConstants.KEY_SMOOTHING)) {
			handleOptimizedProcessing(wfStepMap);
			return;
		}

		File matlabDirFile = new File(matlabDir);
		sendMessage("preparing for Siemens CBF processing");
		this.vpi = helper.prepare4Job();
		if (isCanceled()) {
			return;
		}

		if (context.isDoB0Correction()) {
			if (vpi.getSiemensFmList() != null) {
				boolean forcePrep = true;
				for (SiemensFieldMapInfo fmi : vpi.getSiemensFmList()) {
					helper.handleFieldMapProcessing(fmi, forcePrep);
					if (isCanceled()) {
						return;
					}
				}
			}
			if (isCanceled()) {
				return;
			}
			helper.updateCPIListWithFieldMapCorrection(vpi);
			if (isCanceled()) {
				return;
			}
			Assertion.assertTrue(!vpi.getCpiList().isEmpty());
			for (int i = 0; i < vpi.getCpiList().size(); i++) {
				CBFProcessInfo cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingAutomated(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isUseLTMethod(),
						context.getGaussianFilterParam(), context.isDoFabber());
			}
		} else {
			if (isCanceled()) {
				return;
			}
			Assertion.assertTrue(!vpi.getCpiList().isEmpty());
			for (int i = 0; i < vpi.getCpiList().size(); i++) {
				CBFProcessInfo cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingAutomated(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isUseLTMethod(),
						context.getGaussianFilterParam(), context.isDoFabber());
			}
		}
		// save derived data and create a bundle of results for download
		postprocess();
	}

	protected void handleProcessing(int stageId) throws Exception {
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		prepareEnvironment(configService);

		Map<String, WFStep> wfStepMap = findWFSteps2Skip();
		if (wfStepMap.containsKey(CBFBIRNConstants.KEY_SMOOTHING)) {
			handleOptimizedProcessing(wfStepMap);
			return;
		}

		File matlabDirFile = new File(matlabDir);
		if (stageId == 1) {
			if (isCanceled()) {
				return;
			}
			sendMessage("preparing for Siemens CBF processing");
			this.vpi = helper.prepare4Job();
			if (isCanceled()) {
				return;
			}
			if (context.isDoB0Correction()) {
				if (vpi.getSiemensFmList() != null) {
					boolean forcePrep = true;
					for (SiemensFieldMapInfo fmi : vpi.getSiemensFmList()) {
						if (isCanceled()) {
							return;
						}
						helper.handleFieldMapProcessing(fmi, forcePrep);
						if (isCanceled()) {
							return;
						}
					}
				}
				helper.updateCPIListWithFieldMapCorrection(vpi);
				if (isCanceled()) {
					return;
				}
				Assertion.assertTrue(!vpi.getCpiList().isEmpty());
				CBFProcessInfo cpi = vpi.getCpiList().get(0);
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isDoFabber());
				// also set vpi for recovery
				prep4Recovery();

				handleEmail(USER_ACTION_MAIL);
			} else {
				// no field map
				if (isCanceled()) {
					return;
				}
				Assertion.assertTrue(!vpi.getCpiList().isEmpty());

				CBFProcessInfo cpi = vpi.getCpiList().get(0);
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoB0Correction(), context.getAlignment(),
						context.getCsfMethod(), context.isDoFabber());
				// also set vpi for recovery
				prep4Recovery();
				handleEmail(USER_ACTION_MAIL);
			}
		} else {
			// stage 2
			String cacheDir = context.getCacheDir();
			File maskFile = new File(cacheDir, "mask.txt");
			File sliceNoFile = new File(cacheDir, "slice_no.txt");
			if (!maskFile.exists()) {
				// multiple slice annotation
				maskFile = new File(cacheDir);
			}
			Assertion.assertTrue(sliceNoFile.isFile());
			CBFProcessInfo cpi = vpi.getCpiList().get(0);
			cpi.setBinaryMaskFile(maskFile);
			cpi.setSelectedSliceFile(sliceNoFile);

			if (isCanceled()) {
				return;
			}

			sendMessage("processing (Step 2) " + cpi.getCbfSI().getProtocolId());
			helper.handleCBFProcessingStep2(cpi, matlabDirFile,
					context.isDoSkullStripping(), context.getGmThreshold(),
					context.isDoPartialVolumeCorrection(),
					context.getCsfMethod(), context.getGaussianFilterParam(),
					context.isDoFabber());

			// the rest of the segments/briks do not need user intervention
			for (int i = 1; i < vpi.getCpiList().size(); i++) {
				cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isDoFabber());
				cpi.setBinaryMaskFile(maskFile);
				cpi.setSelectedSliceFile(sliceNoFile);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing (Step 2) "
						+ cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep2(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getCsfMethod(),
						context.getGaussianFilterParam(), context.isDoFabber());
			}
			// save derived data and create a bundle of results for download
			postprocess();
		} // else stage 2
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		helper.setEnvPath(envPath);
		helper.addEnvironmentVar("FSLDIR",
				configService.getParamValue("FSLDIR"));
		helper.addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		helper.addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		helper.addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		helper.addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		helper.addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	/**
	 * skips all the processing till the final step
	 * 
	 * @param wfStepMap
	 * @throws Exception
	 */
	void handleOptimizedProcessing(Map<String, WFStep> wfStepMap)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		this.shortcut = true;
		WFStep skipStep = wfStepMap.get(CBFBIRNConstants.KEY_SMOOTHING);
		File matlabDirFile = new File(matlabDir);
		sendMessage("preparing for Siemens CBF processing");
		this.vpi = helper.prepare4Job();
		if (isCanceled()) {
			return;
		}
		Assertion.assertTrue(!vpi.getCpiList().isEmpty());
		CBFProcessInfo cpi = vpi.getCpiList().get(0);
		sendMessage("processing " + cpi.getCbfSI().getProtocolId());
		if (context.isDoB0Correction()) {
			if (vpi.getSiemensFmList() != null) {
				boolean forcePrep = true;
				for (SiemensFieldMapInfo fmi : vpi.getSiemensFmList()) {
					if (isCanceled()) {
						return;
					}
					helper.handleFieldMapProcessing(fmi, forcePrep);
					if (isCanceled()) {
						return;
					}
				}
			}
			helper.updateCPIListWithFieldMapCorrection(vpi);
			if (isCanceled()) {
				return;
			}
		}
		// all but last step is same, so use results from a matching job
		helper.handleCBFProcessingUsingResultFrom(cpi, skipStep.getJpi(),
				matlabDirFile, false);
		if (isCanceled()) {
			return;
		}
		// save derived data and create a bundle of results for download
		postprocess();
	}

	protected void postprocess() throws Exception {
		if (isCanceled()) {
			return;
		}
		// derived data persistence
		List<Deriveddata> ddList = helper.saveDerivedData(this.vpi);

		// bundle the results for download
		List<Deriveddata> downloadDDList = new ArrayList<Deriveddata>(10);
		for (Deriveddata dd : ddList) {
			if (dd.getDatauri().endsWith(".mat")
					|| dd.getDatauri().endsWith(".png")
					|| dd.getDatauri().indexOf("intermediate") != -1
					|| dd.getDatauri().indexOf("CBF+orig") != -1
					|| dd.getDatauri().indexOf("process_summary.txt") != -1
					|| dd.getDatauri().indexOf("cbfbirn_path1_summary.txt") != -1) {
				downloadDDList.add(dd);
			}
		}

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();
		List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
		String srcRootDir = null;
		for (Deriveddata dd : downloadDDList) {
			String datauri = dd.getDatauri();
			if (srcRootDir == null) {
				srcRootDir = WFGenUtils.extractRootDir(datauri, subjectID);
			}
			String relativePath = FileUtils
					.getRelativePath(srcRootDir, datauri);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}
		// final packaging

		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();
		if (isCanceled()) {
			return;
		}

		handleEmail(JOB_FINISHED_MAIL);
	}

	@Override
	public int getNumberOfStages() {
		if (this.context.isUseAutomatedSegmentation() || this.shortcut) {
			return 1;
		} else {
			return 2;
		}
	}

	@Override
	public synchronized void cancel() {
		super.cancel();
		if (helper != null) {
			helper.setCanceled(true);
		}
	}

}
