package clinical.web.workflow.dti;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.Deriveddata;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobEventListener;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.vo.upload.VisitInfo;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: DTIProcessingJob.java 825 2013-07-30 21:16:55Z bozyurt $
 */
public class DTIProcessingJob implements IJob {
	protected String id;
	protected UserInfo ui;
	protected String description;
	protected String dbID;
	protected boolean canceled = false;
	protected DTIWFContext context;
	protected String resultTarFile;
	protected String emailTo;
	protected String templateDir;
	protected DTIProcJobHelper helper;
	protected DTIVisitProcessInfo dvpi;
	protected String status;
	SimpleDateFormat sdf = new SimpleDateFormat("MM_dd_yyyy");
	private Log log = LogFactory.getLog(DTIProcessingJob.class);

	public DTIProcessingJob(String id, UserInfo ui, String description,
			DTIWFContext context, String templateDir, String emailTo)
			throws BaseException {
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.context = context;
		this.templateDir = templateDir;
		this.emailTo = emailTo;

		this.helper = new DTIProcJobHelper(ui, context, templateDir);
		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate());
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();

	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			log.info("starting DTI job with id:" + id + " ...");
			handleProcessing();
			log.info("finished DTI job with id:" + id + ".");
		} catch (Exception x) {
			x.printStackTrace();
			throw new JobException(x);
		}
	}

	protected void handleProcessing() throws Exception {
		this.dvpi = helper.prepare4Job();
		if (isCanceled()) {
			return;
		}
		
		// TODO handle actual processing by sending data and monitoring processing finish
		
		// derived data persistence
		List<Deriveddata> ddList = helper.saveDerivedData(this.dvpi);
		
		// bundle the results for download
		List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
		
		List<Deriveddata> downloadDDList = new ArrayList<Deriveddata>(10);
		// do file filtering (if any)
		for (Deriveddata dd : ddList) {
			downloadDDList.add(dd);
		}
		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();
		
		String bundleName = subjectID + "_Visit_" + sdf.format(vi.getVisitDate());
		String srcRootDir = null;
		for(Deriveddata dd : downloadDDList) {
			String datauri = dd.getDatauri();
			if (srcRootDir == null) {
				srcRootDir = WFGenUtils.extractRootDir(datauri, subjectID);
			}
			String relativePath = FileUtils.getRelativePath(srcRootDir, datauri);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}
		
		Packager p = new Packager(context.getCacheDir(), bundleName, FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();
		// send job finished email
		handleEmail();
	}

	@Override
	public void addJobEventListener(IJobEventListener listener) {
		// no op
	}

	@Override
	public synchronized void cancel() {
		canceled = true;
	}

	@Override
	public String getContextAsJSON() {
		try {
			JSONObject json = context.toJSON();
			return json.toString();
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	@Override
	public String getDbID() {
		return context.getDbID();
	}

	@Override
	public String getDescription() {
		return this.description;
	}

	@Override
	public int getDurationType() {
		// short in the sense of not heavy load (long duration otherwise)
		return LONG;
	}

	@Override
	public String getID() {
		return id;
	}

	@Override
	public List<JobVisitContext> getJobVisitContextList() {
		List<JobVisitContext> jvcList = new ArrayList<JobVisitContext>(1);
		for (VisitInfo vi : context.getViList()) {
			JobVisitContext jvc = new JobVisitContext(vi.getExpId(), vi
					.getVisitId(), vi.getSubjectID());
			jvcList.add(jvc);
		}
		return jvcList;
	}

	@Override
	public int getNumberOfStages() {
		return 1;
	}

	@Override
	public String[] getResultsFiles() {
		String[] rfArr = new String[1];
		rfArr[0] = this.resultTarFile;
		return rfArr;
	}

	@Override
	public String getType() {
		return "DTI Processing Workflow";
	}

	@Override
	public String getUser() {
		return ui.getPerceivedName();
	}

	@Override
	public UserInfo getUserInfo() {
		return ui;
	}

	@Override
	public synchronized boolean isCanceled() {
		return canceled;
	}

	@Override
	public void saveResults(File file) throws JobException {
		// no op
	}

	@Override
	public void shutdown() {
		// no op
	}

	// helper methods
	protected void handleEmail() throws Exception {
		System.out.println("handling email...");
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		String emailHost = configService.getParamValue("email.host");
		String emailUser = configService.getParamValue("email.user");
		String emailPwd = configService.getParamValue("email.pwd");
		String emailFrom = configService.getParamValue("email.from");
		boolean canSendEmail = GenUtils.hasEnoughInfoForEmail(emailHost, emailFrom);
		
		String str = configService.getParamValue("cbfbirn.send.job.email");
		if (str != null && str.toLowerCase().equals(false)) {
			canSendEmail = false;
		}

		if (canSendEmail) {
			VisitInfo vi = context.getViList().get(0);
			String subjectID = vi.getSubjectID();

			EmailHelper helper = new EmailHelper(emailHost, emailUser,
					emailPwd, emailFrom);
			String toEmail = this.emailTo;
			System.out.println("handleEmail:: toEmail:" + toEmail);
			if (toEmail == null) {
				log.error("Cannot send email to:" + toEmail);
				return;
			}

			String subject = "Your DTI process has finished";
			Map<String, String> paramMap = new HashMap<String, String>();
			paramMap.put("email", toEmail);
			paramMap.put("username", ui.getName());
			paramMap.put("jobId", this.id);
			paramMap.put("subjectID", subjectID);

			helper.sendEmail(toEmail, subject, paramMap, this.templateDir,
					"dti_finished.vm", false);
		}
	}

	@Override
	public IJobFactory getJobFactory() {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public synchronized String getStatus() {
		return status;
	}
	
	@Override
	public synchronized void setStatus(String status) {
        this.status = status;		
	}
	
	@Override
	public void cleanup() {
		// no op
	}
	
	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}
}
