% generalized canonical correlation analysis in NPAIRS framework (gCCA)
%
% Description:
%  It is a multivariate method based on generalized canonical
%  correlation analysis (gCCA) to maximize SPM reproducibility without
%  adopting any model for the hemodynamic response or other temporal brain
%  responses. The underlying assumption is that there are multiple subjects
%  that share an unknown spatial response (or spatial map) to the common
%  experimental manipulation but may show different temporal responses to
%  external stimulation. For each subject, our gCCA explores a broad,
%  potentially inconsistent range of temporal responses in fMRI time-series
%  space while maximizing the mean of correlation coefficients between
%  the pair-wise spatial maps of the subjects.
% =========================================================================
% Syntax:
%   Data_FileName          : A cell array, each cell is an string
%                                  containing an fMRI nifti file names
%   MaskFile               : An string contain the name and the path
%                                  of Mask Nifti File.
%   num_resample           : number of random resampling of the data
%                                  in the split-half resampling framework.
%   num_pc                 : Number of priciple component in
%                                dimension reduction step used in main gCCA
%                                num_pc=0  ---> Determine the number of pc,
%                                based on leave-one out cross validation 
%
%                                (This option would be very slow very slow)
%                                num_pc=-1 Laplacian dimension reduction
%                                (See Minka T, "Automatic choice of
%                                dimensionality for PCA", In 13th Annual
%                                Conference on Neural Information
%                                Processing Systems, 2000).
%                                num_pc=-2 No dimension reduction
%                                0<num_pc<1---> Runs dimension reduction
%                                               and keeps 100num_pc
%                                               perecent of energy of
%                                                each subject
%                                num_pc>=1 ---> Runs dimension reduction
%                                               and keeps num_pc
%                                               priciple components
%
%   Outputdirectory        : The path of output z-map files
%   r                      : is a (num_resample x num_out_pc)
%                                   matrix containing reproducibility
%                                   measure for each split and each maps
%   var_mode(Optional)     : variance estimation mode
%                                 0: voxel-wise variance estimation
%                                 1:pooled variance estimation (NPAIRS-Like)
%                                 2: Both Strategies (default)
%  WeightsFile(Optional)   : If the WeightsFile provided, the program
%                            maxmizes the reproducibility where the voxels in WeightsFile~=0,
%                            and minimizes the variance for voxels
%                            in WeightsFile=0

%==========================================================================
% Output files:
% 1) reproducibility_(xx)_(nn)subjects_(pp).mat
%                   xx: PV (Z-map acquired using pooled variance estimation)
%                       VV (Z-map acquired using voxel based variance estimation)
%                   nn: Number of subjects used in the analysis
%                   pp: num_pc*100 (num_pc is the input parameter)
%
%              Content:
%                      A MAT file will be saved after the split-half
%                      resampling is performed. This Mat file contains
%                      four variables:
%
%                      Avect_avg: Subject x Time points x num_pc
%                                 It contains the average of temporal
%                                 response of each subject over all
%                                 split-half resamplings.
%                      A_full:    A cell array, each component is
%                                 the temporal responses of each
%                                 subject, when we run the gCCA on
%                                 full data.
%                                (without split-half resmapling)
%                     Zm_full:   z-map acquired from full data.
%
% 2) Group_Z_Map_(xx)_(nn)subjects_(pp).nii
%                   xx: PV (Z-map acquired using pooled variance estimation)
%                       VV (Z-map acquired using voxel based variance estimation)
%                   nn: Number of subjects used in the analysis
%                   pp: num_pc*100 (num_pc is the input parameter)
%
%              Content:
%                       It is Nifti file, containing all the z-maps
%                       acquired using gCCA in NPAIRS framework.
%
%
%==========================================================================
% Example:
% <Please be aware the data is not included in the package, this is only an
% example how to use the package>
%
% Input_Data = {'Healthy_Subject1.nii','Patient_Subjects2.nii',...
%               'Patient_Subject1.nii'};
% r = gCCA(Filenames,'Mask.nii',4,10,'./results_matlab');
%==========================================================================
% Reference:
% For more details see:
%   Afshin-Pour, B., Hossein-Zadeh, G. A., Strother, S. C.,
%   Soltanian-Zadeh,H. (2012). Enhancing reproducibility of fMRI
%   statistical maps using generalized canonical correlation analysis
%   in NPAIRS framework. NeuroImage.
%
% Babak Afshin-Pour (bafshinpour@research.baycrest.org)
% Date: 05-May-2014

%==========================================================================
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%     gCCA is distributed in the hope that it will be useful,
%     but WITHOUT ANY WARRANTY; without even the implied warranty of
%     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%     GNU General Public License for more details.
%
%     You should have received a copy of the GNU General Public License
%     along with gCCA.  If not, see <http://www.gnu.org/licenses/>.



function [r,RV1,RV2] = gCCA(Data_FileName,MaskFile,num_resample,num_pc,outputdirectory,var_mode,WeightsFile)


[inOctave,version_str] = in_octave();
display(['Running gCCA under ' version_str '...']);

pathstr = fileparts(which('gCCA'));
addpath([pathstr '/NIFTI_20110921/']);

% Reading Mask file
nii_mask = load_untouch_nii(MaskFile);
inbrain = find(nii_mask.img);

if nargin==7
    nii_weights = load_untouch_nii(WeightsFile);
    Weights = nii_weights.img(inbrain)';
else
    Weights = [];
end

if isnumeric(num_pc)
    if numel(num_pc)>1
        temp_img = nii_mask.img;
        temp_img(:) = 0;
        for i = 1:size(num_pc,1)
            temp_img(num_pc(i,1),num_pc(i,2),num_pc(i,3)) = 1;
        end
        
        seed_ind   = find(temp_img);
        seed_index = find(ismember(inbrain,seed_ind));
    end
end

if nargin<6
    var_mode = 2;
end

% Get numer of subjects or runs
num_subject = length(Data_FileName);
num_voxel   = length(inbrain);

if num_subject<4
    error('more than 3 subjects are needed to run split-half resampling + gCCA');
end

% if nargin<7
%     data_block_size = 0;
% end
% Reading fMRI Data of each subject (or run) and normalize each time-series
% to its standard deviation and remove mean of each scans
% blk_count = 0;

num_time_points = zeros(1,num_subject);

if ischar(Data_FileName{1})
    for i = 1:length(Data_FileName)
        nii = load_untouch_nii(Data_FileName{i});
        [r,c,k,t] = size(nii.img);
        Ts  = double(reshape(nii.img,r*c*k,t));
        %    if data_block_size==0
        Data{i} = Ts(inbrain,:)';
        Data{i} = Data{i} - repmat(mean(Data{i},2),1,size(Data{i},2));
        Data{i} = Data{i} ./ repmat(std(Data{i},0,2),1,size(Data{i},2));
        Data{i} = Data{i} - repmat(mean(Data{i},1),size(Data{i},1),1);
        Data{i}(:,(~isfinite(sum(Data{i},1)))) = 0;
        num_time_points(i) = t;
    end
else
    Data = Data_FileName;
    
    for i = 1:length(Data)
        num_time_points(i) = size(Data{i},1);
    end
end

% generate the list of all possible split half subjects
divider = min(16,num_subject);
step = ceil(num_subject/divider);
list_subject = 1:num_subject;
A = [];
for i = 1:step-1
    xnum_subject = list_subject((i-1)*divider+1:min(i*divider,num_subject));
    A  =   [A nchoosek(xnum_subject,round(length(xnum_subject)/2))];
end
xnum_subject = list_subject((step-1)*divider+1:min(step*divider,num_subject));
temp  = nchoosek(xnum_subject,round(length(xnum_subject)/2));
if ~isempty(A)
    temp2 = repmat(temp,ceil(size(A,1)/size(temp,1)),1);
    A = [A temp2(1:size(A,1),:)];
else
    A = temp;
end
if ((num_subject/2) == fix(num_subject/2))
    index1     = randperm(fix(size(A,1)/2));
else
    if (num_resample<fix(size(A,1)/2))
        index1 = randperm(fix(size(A,1)/2));
    else
        index1 = randperm(size(A,1));
    end
end
num_resample = min(num_resample,length(index1));

% Only for test:
% index1 = [27 1 13 28];
% Number of subjects

sub_total    = 1:num_subject;



if ~isempty(outputdirectory)
    if ((outputdirectory(end)~='/') && (outputdirectory(end)~='\'))
        outputdirectory(end+1)='/';
    end
end
if ~exist(outputdirectory,'dir')
    mkdir(outputdirectory);
end

% Initialization
num_out_pc = inf;

rData = Data;
if isnumeric(num_pc)
    if (numel(num_pc)==1)
        dim_reduction_name = sprintf('%05d',round(num_pc*100));
        if (num_pc>0)
            for i = 1:length(Data)
                [rData{i},rTrans{i}] = reduce_dimension(Data{i},num_pc);
            end
            
            [A_full,Zm_full] = Generalized_CCA(rData,Weights,0,rTrans);
        elseif (num_pc==0)
            
            
            [rfull_Data,rfull_Trans] = Optimal_Number_of_Pcs2(Data);
            [A_full,Zm_full] = Generalized_CCA(rfull_Data,Weights,0,rfull_Trans);
            rData      = Data;
            rTrans     = rfull_Trans;
            
        elseif (num_pc==-1)
            for i = 1:length(Data)
                [rData{i},rTrans{i}] = reduce_dimension_ppca(Data{i});
            end
            [A_full,Zm_full] = Generalized_CCA(rData,Weights,0,rTrans);
        else
            for sc = 1:length(Data)
                rTrans{sc}     = eye(size(Data{sc},1));
                rData{sc}      = Data{sc};
            end
            [A_full,Zm_full] = Generalized_CCA(rData,Weights,0,rTrans);
        end
    else
        format_str = [repmat('%+.5f ',1,length(seed_index)) '\n'];
        FILE = fopen([outputdirectory 'seed.txt'],'w');
        for sc = 1:length(Data)
            rTrans{sc}     = Data{sc}(:,seed_index);
            rData{sc}      = rTrans{sc}'*Data{sc};
            fprintf(FILE,format_str,rTrans{sc}');
        end
        
        fclose(FILE);
        clear Data  Trans
        [A_full,Zm_full] = Generalized_CCA(rData,Weights,0,rTrans);
        dim_reduction_name = '0Seed';
    end
else
    dim_reduction_name = '0User';
    for i = 1:length(Data)
        [rData{i},rTrans{i}] = num_pc(Data{i});
    end
    [A_full,Zm_full] = Generalized_CCA(rData,Weights,0,rTrans);
end


for sc = 1:length(rData)
    num_pcs_per_data(sc) = size(rData{sc},1);
end


AVect  = zeros(num_resample,num_subject,max(num_time_points),min(num_pcs_per_data));
%DVect  = zeros(num_resample,num_subject,max(num_time_points),min(num_pcs_per_data));
%Zm     = zeros(num_resample,min(num_pcs_per_data),num_voxel);
Zm1 = zeros(num_resample,min(num_pcs_per_data),num_voxel);
Zm2 = zeros(num_resample,min(num_pcs_per_data),num_voxel);

r        = zeros(num_resample,min(num_pcs_per_data));
mean_Z   = zeros(min(num_pcs_per_data),num_voxel);
mean_Npx = zeros(min(num_pcs_per_data),num_voxel);
mean_Smx = zeros(min(num_pcs_per_data),num_voxel);

% save([outputdirectory 'reproducibility_' sprintf('%02d',num_subject) '.mat'],'A_full','Zm_full','-append');
% return
% Split-Half Resampling Loop
for mres = 1:num_resample
    
    
    train = A(index1(mres),:);
    count_v = 0;
    for vv = train
        count_v = count_v + 1;
        Y1{count_v} = rData{vv};
        Trans_out1{count_v}  =  rTrans{vv};
    end
    
    train = setdiff(sub_total,A(index1(mres),:));
    count_v = 0;
    for vv = train
        count_v = count_v + 1;
        Y2{count_v}          =  rData{vv};
        Trans_out2{count_v}  =  rTrans{vv};
    end
    
    if isnumeric(num_pc)
        if numel(num_pc)==1
            if num_pc == 0
                [YOut1,Trans_out1] = Optimal_Number_of_Pcs2(Y1);
                [YOut2,Trans_out2] = Optimal_Number_of_Pcs2(Y2);
                
                [A1_in,Zm1_in] = Generalized_CCA(YOut1,Weights,0,Trans_out1);
                [A2_in,Zm2_in] = Generalized_CCA(YOut2,Weights,0,Trans_out2);
                
            else
                % Running Generalized CCA
                [A1_in,Zm1_in] = Generalized_CCA(Y1,Weights,0,Trans_out1);
                [A2_in,Zm2_in] = Generalized_CCA(Y2,Weights,0,Trans_out2);
            end
            display(sprintf('npcs=%d, Split No. = %d,',num_pc,mres));
        else
            % seed cordinated cordinates
            
            [A1_in,Zm1_in] = Generalized_CCA(Y1,Weights,0,Trans_out1);
            [A2_in,Zm2_in] = Generalized_CCA(Y2,Weights,0,Trans_out2);
            display(sprintf('Seedvoxel based analysis, Split No. = %d,',mres));
            
        end
    else
        % Running Generalized CCA
        [A1_in,Zm1_in] = Generalized_CCA(Y1,Weights,0,Trans_out1);
        [A2_in,Zm2_in] = Generalized_CCA(Y2,Weights,0,Trans_out2);
        display(sprintf('User defined dimension reduction, Split No. = %d,',mres));
        
    end
    
    
    
    
    xsubnum1 = A(index1(mres),:);
    xsubnum2 = setdiff(sub_total,A(index1(mres),:));
    
    [A1_temp,Zm1_temp] = sort_eigenimages_withfulldata(A_full,Zm_full,A1_in,Zm1_in,xsubnum1);
    [A2_temp,Zm2_temp] = sort_eigenimages_withfulldata(A_full,Zm_full,A2_in,Zm2_in,xsubnum2);
    
    num_out_pc_new = min(size(Zm1_temp,1),size(Zm2_temp,2));
    z1_temp = (Zm1_temp - repmat(mean(Zm1_temp,2),1,num_voxel));
    z1_temp = z1_temp ./ repmat(std(z1_temp,0,2),1,num_voxel);
    z2_temp = (Zm2_temp - repmat(mean(Zm2_temp,2),1,num_voxel));
    z2_temp = z2_temp ./ repmat(std(z2_temp,0,2),1,num_voxel);
    r_temp  = (sum(z1_temp.*z2_temp,2)./sqrt((sum(z1_temp.^2,2).*sum(z2_temp.^2,2))));
    Zm_temp = (z1_temp + z2_temp)./sqrt(2*repmat(1-r_temp,1,num_voxel));
    
    %     for i = 1:num_out_pc_new
    %         z1_temp = (Zm1_temp(i,:) - mean(Zm1_temp(i,:)))/std(Zm1_temp(i,:));
    %         z2_temp = (Zm2_temp(i,:) - mean(Zm2_temp(i,:)))/std(Zm2_temp(i,:));
    %         temp = corrcoef(z1_temp,z2_temp);
    %         r_temp(i)    = temp(1,2);
    %         Zm_temp(i,:) = (z1_temp + z2_temp)/sqrt(2*(1-r_temp(i)));
    %     end
    
    A_temp = zeros(max(size(A1_temp),size(A2_temp)));
    [rr,cc,kk] = size(A1_temp);
    A_temp(1:rr,1:cc,1:kk) = A_temp(1:rr,1:cc,1:kk) + A1_temp(1:rr,1:cc,1:kk);
    [rr,cc,kk] = size(A2_temp);
    A_temp(1:rr,1:cc,1:kk) = A_temp(1:rr,1:cc,1:kk) + A2_temp(1:rr,1:cc,1:kk);
    
    %[A_temp,Zm_temp,r_temp,Zm1_temp,Zm2_temp] = sort_eigenimages(A1_in,Zm1_in,xsubnum1,A2_in,Zm2_in,xsubnum2);
    if num_out_pc_new<num_out_pc
        num_out_pc = num_out_pc_new;
    end
    
    AVect(mres,:,:,1:num_out_pc)       = A_temp;
    %Zm(mres,1:num_out_pc,:)            = Zm_temp(1:num_out_pc,:);
    r(mres,1:num_out_pc)               = r_temp(1:num_out_pc)';
    mean_Z(1:num_out_pc,:) = mean_Z(1:num_out_pc,:) + Zm_temp(1:num_out_pc,:);
    
    mean_Smx(1:num_out_pc,:) = mean_Smx(1:num_out_pc,:) +  (Zm1_temp(1:num_out_pc,:) + Zm2_temp(1:num_out_pc,:))/2;
    mean_Npx(1:num_out_pc,:) = mean_Npx(1:num_out_pc,:) + ((Zm1_temp(1:num_out_pc,:) - Zm2_temp(1:num_out_pc,:))/2).^2;
    
    
    Zm1(mres,1:num_out_pc,:)           = Zm1_temp(1:num_out_pc,:);
    Zm2(mres,1:num_out_pc,:)           = Zm2_temp(1:num_out_pc,:);
    
    
    %     DVect_temp = calc_temporal_response(Data,Zm_temp);
    %     DVect(mres,:,:,:)  = DVect_temp;
    
    
    
end
mean_Z   = mean_Z/num_resample;
mean_Npx = mean_Npx/num_resample;
mean_Smx = mean_Smx/num_resample;


% Zm  = Zm(:,1:num_out_pc,:);
% Zm1 = Zm1(:,1:num_out_pc,:);
% Zm2 = Zm2(:,1:num_out_pc,:);

r   = r(:,1:num_out_pc);

RV1 = zeros(num_resample,4,7);
RV2 = zeros(num_resample,4,7);

% prc_count = 0;
% for prc_th = 5:5:20
%     prc_r    = prctile(r,prc_th);
%     prc_count = prc_count + 1;
%     Th_count = 0;
%     for  Threshold = 0:0.05:0.3
%         Th_count = Th_count + 1;
%         ind = prc_r>Threshold;
%         for mres = 1:num_resample
%             
%             zm1_temp = squeeze(Zm1(mres,ind,:));
%             zm2_temp = squeeze(Zm2(mres,ind,:));
%             RV1(mres,prc_count,Th_count) = rv_coef(zm1_temp,zm2_temp);
%             
%             zm1_temp = (zm1_temp - repmat(mean(zm1_temp,2),1,num_voxel));
%             zm1_temp = zm1_temp ./ repmat(std(zm1_temp,0,2),1,num_voxel);
%             zm2_temp = (zm2_temp - repmat(mean(zm2_temp,2),1,num_voxel));
%             zm2_temp = zm2_temp ./ repmat(std(zm2_temp,0,2),1,num_voxel);
%             
%             RV2(mres,prc_count,Th_count) = rv_coef(zm1_temp,zm2_temp);
%         end
%     end
% end

nii = nii_mask;
nii.hdr.dime.datatype = 16;
[xx,yy,zz] = size(nii.img);
nii.img = zeros(xx,yy,zz,num_out_pc);
nii.hdr.dime.dim = [3 xx yy zz num_out_pc 1 1 1];

[num_resample,num_subject,time_points,num_pc_temp] = size(AVect);
AVect_avg = zeros(num_subject,time_points,num_out_pc);

if ((var_mode==0) || (var_mode==2))
    %     Smx = (Zm1 + Zm2)/2;
    %     Npx = ((Zm1 - Zm2)/2).^2;
    for pc = 1:num_out_pc
        %[m,n,o,AVect_avg(:,:,pc)] = max_corr_mean_4(squeeze(Smx(:,pc,:)),squeeze(Smx(:,pc,:)),squeeze(Npx(:,pc,:)),AVect(:,:,:,pc));
        
        n = squeeze(mean_Smx(pc,:));
        o = squeeze(mean_Npx(pc,:));
        AVect_avg(:,:,pc) = squeeze(mean(AVect(:,:,:,pc),1));
        
        z = zeros(xx,yy,zz);
        z(inbrain) = n./sqrt(o);
        nii.img(:,:,:,pc) = z;
    end
    try
        save_untouch_nii(nii,[outputdirectory 'Group_Z_Map_VV_' sprintf('%02d',num_subject) 'subjects'  '_' dim_reduction_name '.nii']);
    catch
        warning(['Output nifti file could not be saved! It is saved in current path as ' date '_' 'Group_Z_Map_VV_' sprintf('%02d',num_subject) 'subjects' '_' dim_reduction_name '.nii']);
        save_untouch_nii(nii,[date '_' 'Group_Z_Map_VV_' sprintf('%02d',num_subject) 'subjects'  '_' dim_reduction_name '.nii']);
    end
end

if ((var_mode==1) || (var_mode==2))
    for pc = 1:num_out_pc
        %[m,n,o,AVect_avg(:,:,pc)] = max_corr_mean_4(squeeze(Zm(:,pc,:)),squeeze(Zm(:,pc,:)),squeeze(Zm(:,pc,:)),AVect(:,:,:,pc));
        z = zeros(xx,yy,zz);
        z(inbrain) = squeeze(mean_Z(pc,:));
        nii.img(:,:,:,pc) = z;
        AVect_avg(:,:,pc) = squeeze(mean(AVect(:,:,:,pc),1));
    end
    try
        save_untouch_nii(nii,[outputdirectory 'Group_Z_Map_PV_' sprintf('%02d',num_subject) 'subjects'  '_' dim_reduction_name '.nii']);
    catch
        warning(['Output nifti file could not be saved! It is saved in current path as ' date '_' 'Group_Z_Map_PV_' sprintf('%02d',num_subject) 'subjects' '_' dim_reduction_name '.nii']);
        save_untouch_nii(nii,[date '_' 'Group_Z_Map_PV_' sprintf('%02d',num_subject) 'subjects' '_' dim_reduction_name '.nii']);
    end
end

try
    save([outputdirectory 'reproducibility_' sprintf('%02d',num_subject) '_' dim_reduction_name '.mat'],'Data_FileName','r','RV1','RV2','AVect_avg','A_full','Zm_full','AVect','mean_Z');
catch
    warning(['Output reproducibility mat file could not be saved! It is saved in current path as ' date 'reproducibility_' sprintf('%02d',num_subject) '_' dim_reduction_name '.mat']);
    save([date 'reproducibility_' sprintf('%02d',num_subject) '_' dim_reduction_name '.mat'],'Data_FileName','r','RV1','RV2','AVect_avg','A_full','Zm_full','AVect','mean_Z');
end


% for sub = 1:num_subject
%     nii.img(:) = 0;
%     for pc = 1:num_out_pc
%
%         X = squeeze(AVect_avg(num_subject,:,pc))';
%         X = X - mean(X);
%         Y = Data{sub};
%         Y = Y - repmat(mean(Y),size(Y,1),1);
%
%         beta    = inv(X'*X)*X'*Y;
%         error   = Y - X*beta;
%         std_err = sqrt(sum(error.^2));
%         t       = beta./std_err;
%
%
%         Z          = zeros(xx,yy,zz);
%         Z(inbrain) = t;
%         nii.img(:,:,:,pc) = Z;
%     end
%
%     [pathstr,name] = fileparts(Data_FileName{sub});
%     save_untouch_nii(nii,sprintf([outputdirectory '%s_SingleSubject_T_Map.nii'],name));
%
% end

display('===== Results saved!======');

end

function [rTs,Trans] = reduce_dimension(Ts,q)

[p,n]=size(Ts);

C = cov(Ts');
[U,S,V] = svd(Ts,'econ');
dd = diag(S);
[dds,index] = sort(dd,'descend');
if fix(q)==q
    ind   = index(1:q);
    
else
    pp    = cumsum(dds)/sum(dds);
    ind_temp   = find(pp>q);
    ind = index(1:ind_temp(1)-1);
end
S2    = S(ind,ind);
Trans = U(:,ind)*inv(S2);
rTs   = V(:,ind)';
end

function [rTs,Trans] = reduce_dimension_ppca(Ts)

[p,n]=size(Ts);

C = cov(Ts');

[V,D]= eig(C);
dd = diag(D);
[dds,index] = sort(dd,'descend');

q = laplace_pca([],dds,n,p);

Trans = V(:,index(1:q));
rTs   = Trans'*Ts;

end

function [inOctave,version_str] = in_octave()
try
    ver_num = OCTAVE_VERSION;
    inOctave = 1;
    version_str = ['OCTAVE ' ver_num];
catch
    inOctave = 0;
    version_str  = ['MATLAB ' version];
end
end

% function DVect = calc_temporal_response(Data,Ztemp)
%
% for i = 1:length(Data)
%     len(i) = size(Data{i},1);
% end
%
% Dvect = zeros(length(Data),max(len),size(Ztemp,1));
% for sub_num = 1:length(Data)
%     for i = 1:len(sub_num)
%         [A,B,r] = canoncorr(Data{sub_num}(i,:)',Ztemp');
%         DVect(sub_num,i,1:size(Ztemp,1)) = B(1:size(Ztemp,1));
%     end
% end
% end

function RV = rv_coef(Zm1,Zm2)

c12 = Zm1*Zm2';
C12 = trace(c12*c12');

c11 = Zm1*Zm1';
C11 = trace(c11*c11');
c22 = Zm2*Zm2';
C22 = trace(c22*c22');

RV = C12/sqrt(C11*C22);


end