function [bestMu,idx,GMM,SNN,KNN] = clusterISC(X,k,featureSet,resultPath)

% This function performs a clustering stage in FuSeISC.
%
% Inputs:
%
% X -- input data set of size N by D, where N is the number of samples and D is dimension. 
% In the input data X, features should be organized so that features 1,3,5,7...,end-1 correspond to ISC variability features 
% and features 2,4,6,8,...,end correspond to mean ISC features.
%
% k -- neighborhood size k. Size should be roughly equal to the number of data points in smallest cluster of interest.
%
% featureSet - this denotes which features will be used. Options are:
% 1 = use all features
% 2 = use all features, but weight variability features
% 3 = use only mean ISC features
% 4 = use only variability features
%
% resultsPath -- pathname to a folder where k-nearest neighbor graph will be saved. 
% The saving of the graph avoids constructing k-NN multiple times when results values are evaluated  for different k values.
%
% Outputs:
%
% bestMu -- final cluster centers corresponding to the means of the final GMM.
% idx -- cluster indices.
% GMM -- all parameters of the final GMM
% SNN -- centroids and parameters provided by the SNN clustering stage.
% KNN -- k-NN list.
%
% EXAMPLE USAGE:
%
% compileSNN_mex; % first compile necessary c-functions.
% X = randn(4000,10); % generate some data to test the algorithm
% k = 20; % set neighborhood size k
% featureSet = 1; % use all the original features of the data
% resultPath = '/cs/fs2/home/jukkaupp/fuseISCresults/'; % save k-NN graph in this folder
% [bestMu,idx,GMM,SNN,KNN] = clusterISC(X,k,featureSet,resultPath); % run clustering.
%
% Jukka-Pekka Kauppi
% University of Helsinki, Department of Computer Science
% 31.10.2014
%

warning off
mkdir(resultPath)
warning on

% return meanISC features, variability features, or both:
data = returnFeatures(X,featureSet);

% set result path:
%resultPath = '/cs/fs2/home/jukkaupp/fuseResults';

if exist(resultPath) == 0
    mkdir(resultPath)
end

disp(['Feature set = ' num2str(featureSet)])
disp(['Neighborhood size k = ' num2str(k)])

if k > 200 && k <= 300
   kk = 300;
elseif k > 300 & k <= 500
   kk = 500;
elseif k > 100 & k <= 200
   kk = 200;
elseif k < 100
    kk = 100;
else
   kk = k;
end
if strcmp(resultPath(end),'/') || strcmp(resultPath(end),'\')
   resultPath(end) = [];
end
fn = ([resultPath '/KNNG_k' num2str(kk) '_featureSet' num2str(featureSet) '.mat']);

if exist(fn) == 2
   load(fn,'KNN')
else
   KNN = constructKNNgraph(data,kk);
   V=version;
   if strcmp(V(1),'7')
      save(fn,'KNN')
   else
       save(fn,'KNN','-v7.3')
   end
end

% run SNN clustering with automatic graph selection:
SNN = runSNNclustering(data,k,KNN);

SNN.ctrsExemp = X(SNN.clustCent,:);
centroids = SNN.mu;

[bestMu,idx,GMM] = estimateGMM(data,centroids);
features = data;
