function [bestMu,idx,GMM,SNN,KNN] = clusterISC(X,k,featureSet,resultPath,ForceDiag,Regularize,overwrite_all,overwrite_GMM)

% This function performs a clustering stage in FuSeISC.
%
% Inputs:
%
% X - input data set of size N by D, where N is the number of samples and D is dimension. 
% In the input data X, features should be organized so that features 1,3,5,7...,end-1 correspond to ISC variability features 
% and features 2,4,6,8,...,end correspond to mean ISC features.
% k - neighborhood size k. Size should be roughly equal to the number of data points in smallest cluster of interest.
% featureSet - this denotes which features will be used. Options are:
% 1 = use all features
% 2 = use all features, but weight variability features
% 3 = use only mean ISC features
% 4 = use only variability features
%
% resultsPath - pathname to a folder where k-nearest neighbor graph will be saved. 
% The saving of the graph avoids constructing k-NN multiple times when results values are evaluated for different k values.
% 
% ForceDiag - use diagonal covariance (ForceDiag = 1) or full covariance
% (ForceDiag = 0).
% 
% Regularize - use regularization to avoid singular covariance matrices
% (Regularize = 1), do not regularize (Regularize = 0).
%
% Outputs:
%
% bestMu - final cluster centers corresponding to the means of the final GMM.
% idx - cluster indices.
% GMM - all parameters of the final GMM
% SNN - centroids and parameters provided by the SNN clustering stage.
% KNN - k-NN list.
%
% EXAMPLE USAGE:
%
% compileSNN_mex; % first compile necessary c-functions.
% X = randn(4000,10); % generate some data to test the algorithm
% k = 20; % set neighborhood size k
% featureSet = 1; % use all the original features of the data
% resultPath = '/cs/fs2/home/jukkaupp/fuseISCresults/'; % save k-NN graph in this folder
% ForceDiag = 0;
% Regularize = 0;
% [bestMu,idx,GMM,SNN,KNN] = clusterISC(X,k,featureSet,resultPath,ForceDiag,Regularize); % run clustering.
% 
% See also:
% ESTIMATEGMM
% RUNFUSE
% CONSTRUCTKNNGRAPH
% RUNSNNCLUSTERING

% Jukka-Pekka Kauppi
% University of Helsinki, Department of Computer Science
% 31.10.2014
disp('Starting FuSeISC process...')
disp(datestr(now))

% return meanISC features, variability features, or both:
data = returnFeatures(X,featureSet);

if exist(resultPath) == 0
    mkdir(resultPath)
end

FN{1} = 'ISC mean and variability features chosen.';
FN{2} = 'ISC mean and weighted variability features chosen.';
FN{3} = 'Only ISC mean features chosen.';
FN{4} = 'Only ISC variability features chosen.';
disp(FN{featureSet})
disp(['Neighborhood size k = ' num2str(k) ' chosen.'])

if k > 200 && k <= 300
   kk = 300;
elseif k > 300 & k <= 500
   kk = 500;
elseif k > 100 & k <= 200
   kk = 200;
elseif k < 100
    kk = 100;
else
   kk = k;
end
if strcmp(resultPath(end),'/') || strcmp(resultPath(end),'\')
   resultPath(end) = [];
end
fn = ([resultPath '/KNNG_k' num2str(kk) '_featureSet' num2str(featureSet) '.mat']);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% K-NN graph construction
V = version;

% load KNN-graph (if it exists) or construct and save KNN-graph
% (if it does not exist):
%first test if larger KNN-graph exists already
knn_files = dir([resultPath '/KNNG_k*_featureSet' num2str(featureSet) '.mat']);
if ~isempty(knn_files)
    for l = 1:length(knn_files)
        tmp = knn_files(l).name;
        tmp_ = strfind(tmp,'_');
        kkx = str2double(tmp(tmp_(1)+2:tmp_(2)-1));
        if kkx > kk
            disp('Found kNN-graph with larger k than currently needed, using the larger graph...')
            fn = ([resultPath '/KNNG_k' num2str(kkx) '_featureSet' num2str(featureSet) '.mat']);
            kk = kkx;
        end
    end
end
disp(datestr(now))
if ( exist(fn,'file') == 2 ) && ( overwrite_all == 0 )
   disp('Existing kNN-struct found, loading...')
   load(fn,'KNN')
else
   disp('Constructing kNN-graph...')    
   KNN = constructKNNgraph(data,kk);
   disp('Saving KNN-struct...')    
   if strcmp(V(1),'7')
      save(fn,'KNN')
   else
       save(fn,'KNN','-v7.3')
   end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SNN clustering phase:
disp(datestr(now))
fn2 = ([resultPath '/SNN_k' num2str(k) '_featureSet' num2str(featureSet) '.mat']);
if exist(fn2) == 2 && ( overwrite_all == 0 )
   disp('Existing SNN-struct found, loading...')
   load(fn2,'SNN') % load SNN clustering results
else
    % If not yet computed, run SNN clustering with automatic graph selection:
    disp('Run criterion-based SNN clustering to estimate an initial Gaussian mixture model...')
    SNN = runSNNclustering(data,k,KNN);
    SNN.ctrsExemp = X(SNN.clustCent,:);
    disp('Saving SNN-struct...')
    if strcmp(V(1),'7')
       save(fn2,'SNN')
    else
        save(fn2,'SNN','-v7.3')
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate Gaussian mixture model:
disp(datestr(now))
%disp('Estimate a Gaussian mixture model...')
[bestMu,idx,GMM] = estimateGMM(data,SNN.mu,ForceDiag,Regularize,resultPath,k,featureSet,overwrite_GMM);

disp(datestr(now))
