function [Fuse,R] = loadSegmentationResults(Params,minClusterSize,k)

% This function loads and extracts FuSeISC results.
%
% Inputs: 
% Params - a parameter struct of the project
% minClusterSize - remove subclusters whose size is
% less than "minClusterSize" voxels. When minClusterSize = 0, no subclusters are
% removed. Default is minClusterSize = 100.
% k - k-value used to build k-nearest neighbor graph. The selection of k
% affects the number of segments: the larger the k, the smaller the
% number of segments. Note: This parameter is only needed, when FuSe- has been 
% performed using multiple k-values, i.e. Params.PublicParams.FuSe_multiK = true.
%
% Outputs:
%
% Fuse - a structure containing main segmentation information. 
% Use this function with the function plot_segment. The Fuse-struct includes 
% the following fields:
%
% "map": contains a segmentation map as a 3D-indexed image in 
% anatomical space (x_Dim * y_Dim * z_Dim). 
% Segments are ordered in the increasing order of segment stability 
% or, relative variability. Relative variability is defined as: 
% relVar = average_ISCvar_across_clips/abs(average_ISCmean_across_clips)
%
% "mean_features": contains ISC mean -values of each segment
% for individual clips. This is matrix of size numSegments * numClips.
% Sorting of the segments has been done in terms of cluster stability.
%
% "var_features": contains ISC variability -values of each segment
% for individual clips. This is matrix of size numSegments * numClips.
% Sorting of the segments has been done in terms of cluster stability.
%
% "brain_MNI": anatomical brain (MNI) of size (x_Dim * y_Dim * z_Dim)
%
% "params": parameters of the ISC-project and Fuse segmentation.
%
% R - a struct containg other segmentation-related information. Use this
% struct with the function plotSegmentationResults.
% 
% Example for visualization of the FuSe-results:
% Plot ISC information of the clusters:
%  minClusterSize = 200;
%  k = 200;
%  [Fuse,R] = loadSegmentationResults(Params,minClusterSize,k)
%  nrClips = size(Fuse.mean_features,2);
%  figure
%  subplot(3,1,1);
%  bar(Fuse.mean_features);ylabel('ISC mean')
%  for m = 1:nrClips;Leg{m} = ['Clip' num2str(m-1)];end
%  xlabel('segment index');legend(Leg)
%  subplot(3,1,2);
%  bar(Fuse.var_features);ylabel('ISC variability')
%  xlabel('segment index');legend(Leg)
%  subplot(3,1,3);
%  bar(Fuse.var_features./abs(Fuse.mean_features));
%  ylabel([{'ISC relative'};{'variability'}])
%  xlabel('segment index');legend(Leg) 
%  % plot segmentation image of 5 most stable cluster:
%  slices_z = [10:10:80]; % choose axial slice index for visualization
%  for cluster_index = 1:5
%     plot_segment(Fuse,cluster_index,slices_z)
%  end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
 

if nargin < 1 | nargin > 3
    error('Number of inputs must be 1, 2, or 3!')
end
if nargin < 2
    minClusterSize = 100;
    warning(['minClusterSize not provided by user, subclusters of size 100 and less are removed by default.'])
end
if nargin < 3
    k = [];
end
flag = license('test','image_toolbox');
if ~flag
    warning('No Image processing toolbox found, cannot analyze subclusters and perform min cluster size removal!')
    find_subclusters = 0;
else
    [flag,mess] = license('checkout','image_toolbox');
    if ~flag
        find_subclusters = 0;
        disp(mess)
        warning('Image processing toolbox checkout not successful, cannot analyze subclusters and perform min cluster size removal!')
    else
        find_subclusters = 1;
    end
end

disp('Loading FuSeISC results...')
sort_descend = 0; % sorting order of relative variance
Params2 = Params;
if ~isempty(k)
    Params2.PublicParams.FuSe_multiK = 0;
    Params2.PublicParams.FuSe_K = k;
end
[Clust,X] = load_files(Params2);

droppedClusts = setdiff(1:max(Clust.idx),Clust.idx);
if ~isempty(droppedClusts)
    idx = Clust.idx;
    un = unique(idx);
    idx2 = zeros(size(idx));
    for m = 1:length(un)
        idx2(idx == un(m)) = m;
    end
    Clust.idx = idx2;
    Clust.GMM.idx = idx2;
    Clust.GMM.mu(droppedClusts,:) = [];
    Clust.GMM.PComponents(droppedClusts) = [];
    Clust.GMM.Sigmas(:,:,droppedClusts) = [];
    Clust.bestMu(droppedClusts,:) = [];
end

%[Di,Me,E,sME,sortedClusterIndex,xx] = load_clusters_GMM(Clust,sort_descend);
[Di,rel_variance,E,rel_variance_sorted,rel_variance_sorted_index,xx,mean_feat_sorted,var_feat_sorted] = load_clusters_GMM(Clust,sort_descend);

[brain,mask,atlasCort,atlasSub,AffMat,Labels] = loadAtlases(Params);
% Extract cluster information:
R = postProcessClusters_ver3(Clust,mask,atlasCort,atlasSub,Labels,...
    minClusterSize,AffMat,rel_variance_sorted_index,find_subclusters);


R.Clust = Clust;
R.rel_variance = rel_variance;
R.rel_variance_sorted = rel_variance_sorted;
R.rel_variance_sorted_index = rel_variance_sorted_index;
R.atlasCort = atlasCort;
R.atlasSub = atlasSub;
R.brain = brain;
R.mask = mask;
R.AffMat = AffMat;
R.Labels = Labels;
R.Di = Di;
R.E = E;
R.k = R.Clust.SNN.k;
R.minClusterSize = minClusterSize;

% create segmentation map:
cluster_centroids = R.Clust.bestMu;
nrClusters = length(R.rel_variance_sorted_index);
segmentationMap = zeros(size(R.brain.img));
for mm = 1:nrClusters
    sub_clusts = R.subclusters{R.rel_variance_sorted_index(mm)};
    for nn = 1:length(sub_clusts)
        segmentationMap(sub_clusts{nn}) = mm;
    end
end

Fuse.map = segmentationMap;
Fuse.mean_features = mean_feat_sorted; 
Fuse.var_features = var_feat_sorted;
Fuse.brain_MNI = brain;
Fuse.params.k = R.k;
Fuse.params.minClusterSize = minClusterSize;
Fuse.params.Params = Params;


function [Di,rel_variance,E,rel_variance_sorted,rel_variance_sorted_index,...
    xx,mean_feat,var_feat] = load_clusters_GMM(Clust,sort_descend)

Di = [];
for m = 1:size(Clust.GMM.Sigmas,3)
    if size(Clust.GMM.Sigmas,1) == 1
        Di(:,m) = Clust.GMM.Sigmas(:,:,m);
    else
        Di(:,m) = diag(Clust.GMM.Sigmas(:,:,m));
    end
end

E = sqrt(Di');
data = Clust.GMM.mu;

mean_feat = Clust.GMM.mu(:,2:2:end);
var_feat = Clust.GMM.mu(:,1:2:end);
score = var_feat./abs(mean_feat);
rel_variance = mean(score,2);

% if sort_descend = 1, sort clusters according to relative variance (ascending order):
if sort_descend
    [rel_variance_sorted,rel_variance_sorted_index] = sort(rel_variance,'descend');
else
    [rel_variance_sorted,rel_variance_sorted_index] = sort(rel_variance,'ascend');
end
Di = Di(:,rel_variance_sorted_index);
E = E(rel_variance_sorted_index,:);
clear xx
xx=0;

mean_feat = mean_feat( rel_variance_sorted_index, : );
var_feat = var_feat( rel_variance_sorted_index, : );



