"""
Read and construct design matrix.

Author: Chao Huang (chaohuang.stat@gmail.com)
Last update: 2017-08-14
"""

import numpy as np
from scipy.stats import boxcox

"""
installed all the libraries above
"""


def preprocess(coord_data, img_data, var_matrix, var_type):
    """
        Read and construct design matrix.

        :param
            coord_data (matrix): un-normalized coordinate matrix (n_v*d)
            img_data (matrix): un-transformed image matrix (n*n_v*l)
            var_matrix (matrix): un-normalized design matrix (n*(p-1))
            var_type (vector): covariate type in var_matrix (0-discrete; 1-continuous)
        :return
            y_design (matrix): box-cox transformed image matrix (n*n_v*l)
            x_design (matrix): normalized design matrix (n*p)
            n_coord_data (matrix): normalized coordinate matrix (n_v*d)
    """

    y_design = img_data * 0
    for mii in range(y_design.shape[2]):
        # box-cox transformation on response
        data_tmp = np.reshape(img_data[:, :, 0], (img_data.shape[0]*img_data.shape[1], 1))
        data_boxcox = boxcox(data_tmp)[0]
        y_design[:, :, mii] = np.reshape(data_boxcox, (img_data.shape[0], img_data.shape[1]))

    n, p = var_matrix.shape
    if n < p:
        mat = var_matrix.T
    else:
        mat = var_matrix
    n, p = mat.shape

    mat_new = np.zeros((n, p))

    for kk in range(p):
        if var_type[kk] == 1:
            mat_new[:, kk] = (mat[:, kk] - np.mean(mat[:, kk]))/np.std(mat[:, kk])
        else:
            mat_new[:, kk] = mat[:, kk]
    const = np.ones((n, 1))
    x_design = np.hstack((const, mat_new))

    c_coord = np.mean(coord_data, axis=0)  # find the center of all coordinates
    coord_data = coord_data-c_coord     # move all the coordinates to be centered at 0
    coord_norm = np.sqrt(np.sum(coord_data**2, axis=1))
    coord_scale = np.max(coord_norm)   # find the largest radius of all coordinates
    n_coord_data = coord_data/coord_scale   # shrinkage all the coordinates into a unit ball

    return y_design, x_design, n_coord_data
