"""
Local linear kernel smoothing on MVCM in FGWAS.

Author: Chao Huang (chaohuang.stat@gmail.com)
Last update: 2017-09-18
"""

import numpy as np
import statsmodels.nonparametric.api as nparam

"""
installed all the libraries above
"""


def mvcm_1d(coord_data, y_design, hat_mat):
    """
        Local linear kernel smoothing on MVCM (1d) in FGWAS.

        :param
            coord_data (matrix): common coordinate matrix (l*d)
            y_design (matrix): imaging response data (response matrix, n*l*m)
            hat_mat (matrix): hat matrix (n*n)
        :return
            h_opt (matrix): optimal bandwidth (d*m)
            smy_design (matrix): smoothed image response data (n*l*m)
            resy_design (matrix): estimated residual matrix (n*l*m)
            efit_eta (matrix): estimated eta matrix (n*l*m)
    """

    # Set up
    n, l, m = y_design.shape
    smy_design = y_design * 0
    efity_design = y_design * 0
    resy_design = y_design * 0
    efit_eta = y_design * 0
    h_opt = np.zeros(shape=(1, m))

    for mii in range(m):
        y_avg = np.mean(y_design[:, :, mii], axis=0)
        model_bw = nparam.KernelReg(endog=[y_avg], exog=[coord_data], var_type='c', bw='aic')
        bw_opt = model_bw.bw
        h_opt[0, mii] = bw_opt
        print("the optimal bandwidth for the j-th image measurement is ", h_opt[0, mii])
        for nii in range(n):
            y_ii = y_design[nii, :, mii]
            model_y = nparam.KernelReg(endog=[y_ii], exog=[coord_data], var_type='c', bw=bw_opt)
            smy_design[nii, :, mii] = model_y.fit()[0]
        efity_design[:, :, mii] = np.dot(hat_mat, smy_design[:, :, mii])
        resy_design[:, :, mii] = y_design[:, :, mii] - efity_design[:, :, mii]
        for nii in range(n):
            resy_ii = resy_design[nii, :, mii]
            model_resy = nparam.KernelReg(endog=[resy_ii], exog=[coord_data], var_type='c', bw=bw_opt)
            efit_eta[nii, :, mii] = model_resy.fit()[0]

    return h_opt, smy_design, resy_design, efit_eta


def mvcm_2d(coord_data, y_design, hat_mat):
        """
            Local linear kernel smoothing on MVCM (2d) in FGWAS.

            :param
                coord_data (matrix): common coordinate matrix (l*d)
                y_design (matrix): imaging response data (response matrix, n*l*m)
                hat_mat (matrix): hat matrix (n*n)
            :return
                h_opt (matrix): optimal bandwidth (d*m)
                smy_design (matrix): smoothed image response data (n*l*m)
                resy_design (matrix): estimated residual matrix (n*l*m)
                efit_eta (matrix): estimated eta matrix (n*l*m)
        """

        # Set up
        n, l, m = y_design.shape
        smy_design = y_design * 0
        efity_design = y_design * 0
        resy_design = y_design * 0
        efit_eta = y_design * 0
        h_opt = np.zeros((2, m))

        for mii in range(m):
            y_avg = np.mean(y_design[:, :, mii], axis=0)
            coord_1 = coord_data[:, 0]
            coord_2 = coord_data[:, 1]
            model_bw = nparam.KernelReg(endog=[y_avg], exog=[coord_1, coord_2], var_type='cc', bw='aic')
            bw_opt = model_bw.bw
            h_opt[:, mii] = bw_opt
            print("the optimal bandwidth for the j-th image measurement is ", h_opt[:, mii])
            for nii in range(n):
                y_ii = y_design[nii, :, mii]
                model_y = nparam.KernelReg(endog=[y_ii], exog=[coord_1, coord_2], var_type='cc', bw=bw_opt)
                smy_design[nii, :, mii] = model_y.fit()[0]
            efity_design[:, :, mii] = np.dot(hat_mat, smy_design[:, :, mii])
            resy_design[:, :, mii] = y_design[:, :, mii] - efity_design[:, :, mii]
            for nii in range(n):
                resy_ii = resy_design[nii, :, mii]
                model_resy = nparam.KernelReg(endog=[resy_ii], exog=[coord_1, coord_2], var_type='cc', bw=bw_opt)
                efit_eta[nii, :, mii] = model_resy.fit()[0]

        return h_opt, smy_design, resy_design, efit_eta


def mvcm_3d(coord_data, y_design, hat_mat):
    """
        Local linear kernel smoothing on MVCM (1d) in FGWAS.

        :param
            coord_data (matrix): common coordinate matrix (l*d)
            y_design (matrix): imaging response data (response matrix, n*l*m)
            hat_mat (matrix): hat matrix (n*n)
        :return
            h_opt (matrix): optimal bandwidth (d*m)
            smy_design (matrix): smoothed image response data (n*l*m)
            resy_design (matrix): estimated residual matrix (n*l*m)
            efit_eta (matrix): estimated eta matrix (n*l*m)
    """

    # Set up
    n, l, m = y_design.shape
    smy_design = y_design * 0
    efity_design = y_design * 0
    resy_design = y_design * 0
    efit_eta = y_design * 0
    h_opt = np.zeros((3, m))

    for mii in range(m):
        y_avg = np.mean(y_design[:, :, mii], axis=0)
        coord_1 = coord_data[:, 0]
        coord_2 = coord_data[:, 1]
        coord_3 = coord_data[:, 2]
        model_bw = nparam.KernelReg(endog=[y_avg], exog=[coord_1, coord_2, coord_3], var_type='ccc', bw='aic')
        bw_opt = model_bw.bw
        h_opt[:, mii] = bw_opt
        print("the optimal bandwidth for the j-th image measurement is ", h_opt[:, mii])
        for nii in range(n):
            y_ii = y_design[nii, :, mii]
            model_y = nparam.KernelReg(endog=[y_ii], exog=[coord_1, coord_2, coord_3], var_type='ccc', bw=bw_opt)
            smy_design[nii, :, mii] = model_y.fit()[0]
        efity_design[:, :, mii] = np.dot(hat_mat, smy_design[:, :, mii])
        resy_design[:, :, mii] = y_design[:, :, mii] - efity_design[:, :, mii]
        for nii in range(n):
            resy_ii = resy_design[nii, :, mii]
            model_resy = nparam.KernelReg(endog=[resy_ii], exog=[coord_1, coord_2, coord_3], var_type='ccc', bw=bw_opt)
            efit_eta[nii, :, mii] = model_resy.fit()[0]

    return h_opt, smy_design, resy_design, efit_eta
