function dki_preprocessing(d0)
%DKI_PREPROCESSING performs coregistration and averages things for dke
%
%Coregistration is performed on all b0 images with spm using the b0 image 
%preceeding the first set of diffusion weighted images as the target image
%to be registered to. 
%
%Averaging is performed on all images found to have the same b-value and
%diffusion encoding vector (including the b0 images). 
%
%d0 is the path to a directory containing:
%
%   1. Individual nifti images (.nii) for each image in the dki dataset
%   2. Gradient table (.bvec) n x 3 matrix with each row corresponding to
%      the nth image in the dki dataset
%   3. B-value table (.bval) n x 1 matrix with each row corresponding to
%      the nth image in the dki dataset
%
%The d0 directory can be generated using MRICRON's dcm2nii gui
%
%NOTE: DKI_PREPROCESSING requires SPM (tested with SPM8). 
%
%Author: Russell Glenn
%Medical University of South Carolina

%Check for parallel computing
try if matlabpool('size')==0; matlabpool open; fprintf('\n'); end; 
catch ME
    ME.message;
end 

avg_flag = 1; %1: b0 and repeated dwis 0: don't average anything 
dT = 1; %degree threshold for gradient vectors to be consiered the same

%CHECK FILES IN D0
D0 = dir(d0); 
x = cell(size(D0)); [x{:}] = deal(D0(:).name); %file names in D0; 

bval_i = find(cellfun(@(x)~isempty(strfind(x,'.bval')),x),1); %.bval file
bvec_i = find(cellfun(@(x)~isempty(strfind(x,'.bvec')),x),1); %.bvec file
img_i = find(cellfun(@(x)~isempty(strfind(x,'.nii')),x));     %.nii files

if isempty(bval_i); error('Could not find .bval file'); end
if isempty(bvec_i); error('Could not find .bvec file'); end

bt = dlmread(fullfile(d0,x{bval_i})); if size(bt,1)<size(bt,2); bt = bt'; end %b-values
gt = dlmread(fullfile(d0,x{bvec_i})); if size(gt,1)<size(gt,2); gt = gt'; end %gradient directions

if length(img_i)~=length(bt)||length(img_i)~=size(gt,1); 
    error('Number of image volumes found does not batch bval or bvecs files'); 
end

%COREG B0 IMAGES: Note ~ b0 preceeding first dwi is used as the refernce 
hdr = spm_vol(fullfile(d0,x{img_i(1)}));  %template header
b0_idx = find(bt==0); %index for b0 images in x
dwi_idx = find(bt>0); %index for dwi images in x


if length(b0_idx)<=1; %Nothing to do. Just make 4D nifti
    img = zeros(hdr.dim(1),hdr.dim(2),hdr.dim(3),length(img_i)); 
    for i = 1:length(img_i); img(:,:,:,i) = spm_read_vols(spm_vol(fullfile(d0,x{img_i(i)}))); end
    make_4D_nii(hdr,img,'dki_4D'); 
else
    b0_idx = [b0_idx;length(img_i)+1]; %Add placeholder
    ref_i = find(b0_idx(1:end-1)+1~=b0_idx(2:end),1,'first'); %Target image index in b0_idx
    fn_target = fullfile(d0,x{img_i(b0_idx(ref_i))}); %Target image for coregistration
    d_coreg = fullfile(d0,'coreg'); %Temp folder
    if ~isdir(d_coreg); mkdir(d_coreg); end
    parfor i = 1:length(b0_idx)-1
        if i~=ref_i %coregister and copy subsequent dwis
            fn_source = fullfile(d0,x{img_i(b0_idx(i))}); %source image to coregister to target
            fn_other = fn_source; %other images to copy
            for j = b0_idx(i)+1:b0_idx(i+1)-1
                fn_other = str2mat(fn_other,fullfile(d0,x{img_i(j)})); 
            end
            coreg_DKI(fn_source,fn_target,fn_other,d_coreg)
        else %Copy files to temp d_coreg
            for j = b0_idx(i):b0_idx(i+1)-1
                copyfile(fullfile(d0,x{img_i(j)}),fullfile(d_coreg,x{img_i(j)}),'f'); 
            end
        end
    end    
end    

if avg_flag 
    fprintf('Averaging Images...\n\n')
    b0 = 0; %Average b0s
    for i = 1:length(b0_idx)-1; 
        b0 = b0 + spm_read_vols(spm_vol(fullfile(d_coreg,x{img_i(b0_idx(i))})))./(length(b0_idx)-1); 
    end
    
    n = 1; 
    dwi_n = zeros(size(dwi_idx)); 
    while ~prod(dwi_n)
        ni = find(dwi_n==0,1,'first'); 
        dwi_n(bt(dwi_idx(ni))==bt(dwi_idx)&real(acosd(gt(dwi_idx(ni,:),:)*gt(dwi_idx,:)'))'<dT)=ni; 
        n = n+1;   
    end   
    
    %Initialize outputs
    bt_out = 0; 
    gt_out = [0 0 0];     
    img_out = zeros(hdr.dim(1),hdr.dim(2),hdr.dim(3),max(dwi_n)+1); 
    img_out(:,:,:,1) = b0; clear b0; 
    
    for i = 1:max(dwi_n); 
        ii = 0;  
        for j = find(dwi_n==i)'; 
            ii = ii+spm_read_vols(spm_vol(fullfile(d_coreg,x{img_i(dwi_idx(j))})))./sum(dwi_n==i); 
        end
        img_out(:,:,:,i+1)=ii;
        bt_out = [bt_out;bt(dwi_idx(find(dwi_n==i,1,'first')))]; 
        gt_out = [gt_out;gt(dwi_idx(find(dwi_n==i,1,'first')),:)];  
    end
    
else
    img_out = zeros(hdr.dim(1),hdr.dim(2),hdr.dim(3),length(img_i));
    for i = 1:length(img_i); 
        img_out(:,:,:,i) = spm_read_vols(spm_vol(fullfile(d_coreg,x{img_i(i)}))); 
    end
    bt_out = bt; 
    gt_out = gt; 
end

rmdir(d_coreg,'s')

%WRITE OUTPUTS
dout = fullfile(d0,'output'); 
if~isdir(dout); mkdir(dout); end; 

fprintf('Saving files to\n\t%s\n\n',dout)
outname = 'dki_4D.nii'; 
make_4D_nii(hdr,img_out,outname)
movefile(fullfile(d0,outname),fullfile(dout,outname),'f')

[~,f,e] = fileparts(fullfile(d0,x{bval_i})); 
fid = fopen(fullfile(dout,[f,'_new',e]),'w'); 
fprintf(fid,'%d\n',bt_out); 
fclose(fid); 

[~,f,e] = fileparts(fullfile(d0,x{bvec_i})); 
fid = fopen(fullfile(dout,[f,'_new',e]),'w'); 
fprintf(fid,'%10.8f\t%10.8f\t%10.8f\n',gt_out'); 
fclose(fid); 

%Gradietn table(s) for DKE ~ check Uniqueness of gradient table for each b-value
b = []; for i = dwi_idx'; if~ismember(bt(i),b);b = [b,bt(i)]; end; end
gt_b = cell(size(b)); 
for i = 1:length(b); gt_b{i} = gt(bt==b(i),:); end
unq = 0; 
g1 = gt_b{1}; 
for i = 2:length(b)
    if unq==0
        gi = gt_b{i}; 
        if size(g1,1)~=size(gi,1); unq=1; 
        else            
            deg = zeros(size(gi,1),1); 
            for j = 1:size(gi,1); deg(j) = acosd(g1(j,:)*gi(j,:)'); end
            if sum(deg>dT);unq=1;end
        end
    end
end

[~,f,e] = fileparts(fullfile(d0,x{bvec_i})); 
if~unq    
    fid = fopen(fullfile(dout,[f,'_dke',e]),'w'); 
    fprintf(fid,'%10.8f\t%10.8f\t%10.8f\n',g1'); 
    fclose(fid);
else
    for i = 1:length(b)
        fid = fopen(fullfile(dout,[f,sprintf('_dke_d',b(i)),e]),'w'); 
        fprintf(fid,'%10.8f\t%10.8f\t%10.8f\n',g1'); 
        fclose(fid);
    end
end