function dNBS(varargin)
%NBS Opens the NBS graphical user interface and delivers user inputs to 
%dNBSrun when run button pressed. A user interface to dNBSrun.  
%
%   NBS opens the NBS graphical user interface
%
%   NBS(FILENAME) opens the graphical user interface and attempts to load 
%   the user inputs specified in a Matlab file called FILENAME. FILENAME
%   must contain an nbs structure. 
%
%   An nbs structure contains the following structures:
%       nbs.GLM:        Specification of the general linear model
%                       See dNBSglm for details
%       nbs.NBS         Results 
%                       See dNBSview for details
%       nbs.STATS       Statistical thresholds and information 
%                       See dNBSstats for details
%       nbs.UI          User inputs 
%                       See dNBSrun for details
%    
%   See also dNBSrun 
%
%   Copyright (C) 2012
% 
%   This program is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
% 
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
% 
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%   azalesky@unimelb.edu.au

%Declare the nbs structure global to avoid passing between NBS and dNBSrun
%12.12.2019
%Redifne Gui for NBS Directed Toolbox (MvG)
global nbs

%Try to load results
try nbs=load(varargin{1});
    nbs=nbs.nbs;
    UI=nbs.UI;
catch 
    %Otherwise, intialise UI based on previous run 
    if exist('history-dNBS.mat','file')
        UI=load('history-dNBS.mat');
        UI=UI.UI;
    else
        %If no previous run, initialise UI to the schizophrenia example
    	UI.method.ui='Run NBS'; 
        UI.design.ui=['Example_Data',filesep,'designMatrix.txt'];
        UI.contrast.ui='[-1,1]'; 
        UI.thresh.ui='[0.1, 0.1, 3]';
        UI.test.ui='t-test';
        UI.matrices.ui=['Example_Data',filesep,'matrices',filesep,'subject01.txt'];
        UI.node_coor.ui=['Example_Data',filesep,'COG.txt'];                         
        UI.node_label.ui=['Example_Data',filesep,'nodeLabels.txt'];                          
        UI.perms.ui='5000';
        UI.alpha.ui='0.05';
        UI.size.ui='Extent';  
        UI.exchange.ui=''; 
    end
end

%Open the GUI 
open_ui(UI); 

                          
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Callback for radio button
%Set text in run button to correspond with the radio button pressed 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function radio_call(varargin)
    S=varargin{3};   
    %Load icons
    Fnbs=load(['icons',filesep,'dnbs.mat']);
    Fnbs=Fnbs.F; 
    Fspc=load(['icons',filesep,'spc.mat']);
    Fspc=Fspc.F; 
    Ffdr=load(['icons',filesep,'fdr.mat']);
    Ffdr=Ffdr.F;  
    set(S.NBS.run,'Enable','off','BackgroundColor',[0.65 0.65 0.65]);
    set(S.STATS.thresh.edit,'Visible','on');
    set(S.STATS.thresh.text,'Visible','on');
    set(S.ADV.size.edit,'Visible','on');
    set(S.ADV.size.text,'Visible','on');
    
    ind=get(S.ADV.nbs.edit,'value');
    if ind==1
        set(S.NBS.run,'UserData','Run NBS');
        set(S.NBS.run,'cdata',Fnbs.cdata);
        set(S.NBS.run,'Enable','on');
        if ispc
            set(S.NBS.run,'BackgroundColor',[0.9 0.9 0.9]);
        end
    elseif ind==2
        set(S.NBS.run,'UserData','Run SPC');
        set(S.NBS.run,'cdata',Fspc.cdata);
        tmp=get(S.OUT.ls,'string');
        set(S.OUT.ls,'string',[{'SPC not available in v1.2'};tmp]);
    elseif ind==3
        set(S.NBS.run,'UserData','Run FDR');
        set(S.NBS.run,'cdata',Ffdr.cdata);
        set(S.NBS.run,'Enable','on');
        if ispc
            set(S.NBS.run,'BackgroundColor',[0.9 0.9 0.9]);
        end
        set(S.STATS.thresh.edit,'Visible','off');
        set(S.STATS.thresh.text,'Visible','off');
        set(S.ADV.size.edit,'Visible','off');
        set(S.ADV.size.text,'Visible','off');
    end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Callback for file open
%Load file and put path and file into edit box 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function load_file(varargin)
    h_edit=varargin{3};
    [filename,pathname]=uigetfile({'*.mat;*.txt','MAT-files (*.mat), Text Files (*.txt)'},'Select text (.txt) or Matlab (*.mat)  file');
    tmp=1;
    try tmp = all(filename==0) && all(pathname==0); catch end
    if tmp 
    else
        set(h_edit,'string',[pathname,filename]); 
    end
 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Callback for run button
%Gather UI and run using the gathered UI
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function get_ui(varargin)

     S=varargin{3}; 
    %Reset all colors to black
    set(S.DATA.matrices.text,'ForegroundColor','black');
    set(S.STATS.design.text,'ForegroundColor','black');
    set(S.STATS.contrast.text,'ForegroundColor','black');
    set(S.STATS.thresh.text,'ForegroundColor','black');
    set(S.STATS.test.text,'ForegroundColor','black');
    set(S.ADV.perms.text,'ForegroundColor','black');
    set(S.ADV.alpha.text,'ForegroundColor','black');
    set(S.ADV.size.text,'ForegroundColor','black');
    set(S.DATA.node_coor.text,'ForegroundColor','black');
    set(S.ADV.exchange.text,'ForegroundColor','black');
    set(S.DATA.node_label.text,'ForegroundColor','black');

    UI.method.ui=get(S.NBS.run,'UserData');
    UI.design.ui=get(S.STATS.design.edit,'string');
    UI.contrast.ui=get(S.STATS.contrast.edit,'string');
    UI.thresh.ui=get(S.STATS.thresh.edit,'string');
    ind=get(S.STATS.test.edit,'value');
    tmp=get(S.STATS.test.edit,'string');
    UI.test.ui=strtrim(tmp(ind,:));
    
    UI.matrices.ui=get(S.DATA.matrices.edit,'string');
    UI.node_coor.ui=get(S.DATA.node_coor.edit,'string');                         
    UI.node_label.ui=get(S.DATA.node_label.edit,'string');                          
    UI.perms.ui=get(S.ADV.perms.edit,'string');
    UI.alpha.ui=get(S.ADV.alpha.edit,'string');
    ind=get(S.ADV.size.edit,'value');
    tmp=get(S.ADV.size.edit,'string');
    UI.size.ui=strtrim(tmp(ind,:));
    UI.exchange.ui=get(S.ADV.exchange.edit,'string');
    %Save UI 
    save('history-dNBS.mat','UI')
    
    set(S.NBS.run,'Enable','off');
    %%%%%%%%%%%%
    %Run NBS
    dNBSrun(UI,S); 
    %%%%%%%%%%%%
    set(S.NBS.run,'Enable','on');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Set up GUI
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function open_ui(UI)
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Main figure
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    sz=get(0,'ScreenSize');
    lenx=0.6;
    leny=0.8;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
    S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Directed NBS v1.0.2',...
        'numbertitle','off',...
        'resize','off');
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Output panel
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    S.OUT.panel=uipanel('Parent',S.fh,...
        'units','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'Title','',...
        'FontSize',1,...
        'FontWeight','Bold',...
        'BorderType','none',...
        'Position',[0.02 0.02 1-0.04 0.2250]);
     
    %MvG Change Display Print Out
    S.OUT.ls = uicontrol('Parent',S.OUT.panel,...
                  'style','list',...
                 'unit','normalized',...
                 'position',[0 0 0.75 0.95],...
                 'min',0,'max',2,...
                 'fontsize',10,...
                 'FontName','FixedWidth',...
                 'BackgroundColor','white',...
                 'Value',[],...
                 'String','Ready...|Directed NBS v1.0.2 Copyright (C) 2020|Directed NBS is inspired by the Matlab Code provided from Matt Hyett|and the GUI of NBS1.2 Copyright(C)2012 provided from Prof. Dr. Andrew Zalesky.|Directed NBS is provided from: Maximilianvon Gellhorn|All versions of Directed NBS comes with ABSOLUTELY NO WARRANTY (as does NBS1.2).|This is free software, distributed under the GNU General Public Licence.||Refer to the software as suggested under about dNBS in the help menu.'); 
 
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %RUN BUTTON
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    S.NBS.run=uicontrol('Parent',S.OUT.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.79 0.18 0.175 0.63 ],...
        'fontsize',15,...
        'Enable','off',...
        'HorizontalAlignment','left',...
        'BackgroundColor',[0.9,0.9,0.9],...
        'String','');       
    
    %Remove highlighting from listbox
    set(S.OUT.ls,'Callback',{@bdfcn_ls,S});   
    
%     %NBS Icon
%     F=load(['icons',filesep,'nbs.mat']);
%     F=F.F; 
% 
%       %Create a temporary figure (invisible) and axes for the NBS icon
%       %See \icons\NBSicon.m
%       TMP.fh=figure('units','centimeters',...
%                     'position',[3 3 13.5 14],...
%                     'name','NBS v1.2',...
%                     'numbertitle','off',...
%                     'resize','off',...
%                     'visible','off');
% 
%       width_waitbar=0.4+0.1;
%       TMP.ax=axes('Parent',TMP.fh,'units','centimeters',...
%                   'Color',[0.8 0.8 0.8],...
%                   'position',[9.8+0.5+width_waitbar 0.5 2.7-0.5 2],...
%                   'box','off',...
%                   'Visible','off');
%     
%       imshow(F.cdata,'Parent',TMP.ax);
%      
%       %%Get axes and its children
%       Ax=get(TMP.fh,'children');
%       copyobj(Ax,S.fh); 

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %STATS panel
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    S.STATS.panel=uipanel('Parent',S.fh,...
        'units','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'Title','Statistical Model',...
        'FontSize',12,...
        'FontWeight','Bold',...
        'BorderType','BeveledIn',...
        'Position',[0.02 0.7550 1-0.04 0.2250]);
    
  
    S.STATS.design.text=uicontrol('Parent',S.STATS.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 0.55+0.2 0.4 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Design Matrix:');
    
    
    
    S.STATS.design.edit=uicontrol('Parent',S.STATS.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.02 0.55-0.03 0.6 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    S.STATS.design.button=uicontrol('Parent',S.STATS.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.62 0.55-0.03 0.05 0.23 ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.STATS.design.edit});
    
    S.STATS.contrast.text=uicontrol('Parent',S.STATS.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 0.08+0.2 0.4 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Contrast:');
    
    S.STATS.contrast.edit=uicontrol('Parent',S.STATS.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.02 0.08 0.6 0.23  ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    S.STATS.contrast.button=uicontrol('Parent',S.STATS.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.62 0.08 0.05 0.23 ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.STATS.contrast.edit});
    

    S.STATS.thresh.text=uicontrol('Parent',S.STATS.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.7 0.08+0.2 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Threshold:');

    
    S.STATS.thresh.edit=uicontrol('Parent',S.STATS.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.7 0.08 0.12 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    
    S.STATS.test.text=uicontrol('Parent',S.STATS.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.7 0.55+0.2 0.24 0.2],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Statistical Test:');
    
    %Only t-test and f-test supported for Diracted NBS
    S.STATS.test.edit=uicontrol('Parent',S.STATS.panel,...
        'style','popupmenu',...
        'unit','normalized',...
        'position',[0.7 0.55 0.2 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','t-test|F-test');%'One Sample|t-test|F-test');
    

    S.STATS.help=uicontrol('Parent',S.STATS.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[1-0.1-0.02 0.08 0.1 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Help');
    
     set(S.STATS.help,'callback',{@help_stats}); 

    
%      S.STATS.view=uicontrol('Parent',S.STATS.panel,...
%         'style','pushbutton',...
%         'unit','normalized',...
%         'position',[0.2 0.55+0.22 0.15 0.15],...
%         'fontsize',8,...
%         'HorizontalAlignment','left',...
%         'String','View Design');   
%     
%      set(S.STATS.view,'callback',{@view_model,S});                  

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %Data panel
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    S.DATA.panel=uipanel('Parent',S.fh,...
        'units','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'Title','Data',...
        'FontSize',12,...
        'FontWeight','Bold',...
        'BorderType','BeveledIn',...
        'Position',[0.02 0.51 1-0.04 0.2250]);
    
    origin_matrices=0.78;
    S.DATA.matrices.text=uicontrol('Parent',S.DATA.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_matrices-0.01 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Connectivity Matrices:');
    
    
    S.DATA.matrices.edit=uicontrol('Parent',S.DATA.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.32 origin_matrices 0.6 0.23],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');

    S.DATA.matrices.button=uicontrol('Parent',S.DATA.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.6+0.32 origin_matrices 0.05 0.23 ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.DATA.matrices.edit});
    

    origin_node_coor=0.45;
    S.DATA.node_coor.text=uicontrol('Parent',S.DATA.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_node_coor-0.01 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Node Coordinates (MNI):');
    
    S.DATA.node_coor.edit=uicontrol('Parent',S.DATA.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.32 origin_node_coor 0.6 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    S.DATA.node_coor.button=uicontrol('Parent',S.DATA.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.6+0.32 origin_node_coor 0.05 0.23 ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.DATA.node_coor.edit});
   
    origin_node_labels=0.1;
    S.DATA.node_label.text=uicontrol('Parent',S.DATA.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_node_labels-0.01 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Node Labels [opt]:');
    
    S.DATA.node_label.edit=uicontrol('Parent',S.DATA.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.32 origin_node_labels 0.45 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    S.DATA.node_coor.button=uicontrol('Parent',S.DATA.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.45+0.32 origin_node_labels 0.05 0.23 ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.DATA.node_label.edit});
    
    S.DATA.help=uicontrol('Parent',S.DATA.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[1-0.1-0.02 0.08 0.1 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Help');
    
      set(S.DATA.help,'callback',{@help_data}); 

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %ADVANCED panel
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
    S.ADV.panel=uipanel('Parent',S.fh,...
        'units','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'Title','Advanced Settings',...
        'FontSize',12,...
        'FontWeight','Bold',...
        'BorderType','BeveledIn',...
        'Position',[0.02 0.2650 1-0.04 0.2250]);

    origin_exchange=0.75;
    S.ADV.exchange.text=uicontrol('Parent',S.ADV.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_exchange-0.01 0.34 0.2  ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Exchange Blocks [opt]:');
    
    S.ADV.exchange.edit=uicontrol('Parent',S.ADV.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.32 origin_exchange 0.6 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    S.STATS.exchange.button=uicontrol('Parent',S.ADV.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[0.6+0.32 origin_exchange 0.05 0.23  ],...
        'CData',iconRead(['icons',filesep,'opendoc.mat']),...
        'callback',{@load_file,S.ADV.exchange.edit});
    
    
    origin_perms=0.41;
    S.ADV.perms.text=uicontrol('Parent',S.ADV.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_perms-0.01 0.3 0.2],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Permutations:');
    
    
    S.ADV.perms.edit=uicontrol('Parent',S.ADV.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.2+0.02 origin_perms 0.1 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','');
    
    
    origin_alpha=0.1;
    S.ADV.alpha.text=uicontrol('Parent',S.ADV.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.02 origin_alpha-0.01 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Significance:');
    
    
    
    S.ADV.alpha.edit=uicontrol('Parent',S.ADV.panel,...
        'style','edit',...
        'unit','normalized',...
        'position',[0.02+0.2 origin_alpha 0.1 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','0.05');
    
   
    origin_size=origin_alpha;
    S.ADV.size.text=uicontrol('Parent',S.ADV.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.4 origin_size 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Component Size:');
    
    S.ADV.size.edit=uicontrol('Parent',S.ADV.panel,...
        'style','popupmenu',...
        'unit','normalized',...
        'position',[0.64 origin_size+0.05 0.15 0.18 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Extent|Intensity');
    
    origin_nbs=origin_perms;
    S.ADV.nbs.text=uicontrol('Parent',S.ADV.panel,...
        'style','text',...
        'unit','normalized',...
        'BackgroundColor',[0.8 0.8 0.8],...
        'position',[0.4 origin_nbs 0.3 0.2 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Method:');
    
    % MvG Only NBS
    S.ADV.nbs.edit=uicontrol('Parent',S.ADV.panel,...
        'style','popupmenu',...
        'unit','normalized',...
        'position',[0.55 origin_nbs+0.05 0.42 0.18 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Directed Network-Based Statistic (dNBS)');
    
    S.ADV.help=uicontrol('Parent',S.ADV.panel,...
        'style','pushbutton',...
        'unit','normalized',...
        'position',[1-0.1-0.02 0.08 0.1 0.23 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Help');
    
    set(S.ADV.help,'callback',{@help_advanced}); 
   
       
    %Set callback for run button 
    set(S.NBS.run,'callback',{@get_ui,S}); 
    
    %Set callback for method popupmenu
    set(S.ADV.nbs.edit,'callback',{@radio_call,S});
 
    %Initialise UI 
    if strcmp(UI.method.ui,'Run NBS')
        set(S.ADV.nbs.edit,'value',1);
    elseif strcmp(UI.method.ui,'Run SPC')
        set(S.ADV.nbs.edit,'value',2);
    elseif strcmp(UI.method.ui,'Run FDR')
        set(S.ADV.nbs.edit,'value',3);
    end
    radio_call([],[],S); 
    set(S.STATS.design.edit,'string',UI.design.ui);
    set(S.STATS.contrast.edit,'string',UI.contrast.ui);
    set(S.STATS.thresh.edit,'string',UI.thresh.ui);
    if strcmp(UI.test.ui,'One Sample')
        set(S.STATS.test.edit,'value',1);
    elseif strcmp(UI.test.ui,'t-test')
%     if strcmp(UI.test.ui,'t-test')
        set(S.STATS.test.edit,'value',1);%changed Numbers only 2 Test
    elseif strcmp(UI.test.ui,'F-test')
        set(S.STATS.test.edit,'value',2);%changed Numbers only 2 Test
    end
    set(S.DATA.matrices.edit,'string',UI.matrices.ui);
    set(S.DATA.node_coor.edit,'string',UI.node_coor.ui);                         
    set(S.DATA.node_label.edit,'string',UI.node_label.ui);                          
    set(S.ADV.perms.edit,'string',UI.perms.ui);
    set(S.ADV.alpha.edit,'string',UI.alpha.ui);
    if strcmp(UI.size.ui,'Extent')
        set(S.ADV.size.edit,'value',1);
    elseif strcmp(UI.size.ui,'Intensity')
        set(S.ADV.size.edit,'value',2);
    end
    set(S.ADV.exchange.edit,'string',UI.exchange.ui); 
        
    %Main menu
    S.menu_file=uimenu(S.fh,'Label','File');
    S.menu_file_sub(1)=uimenu(S.menu_file,'Label','Load previous',...
                              'Callback',{@load_nbs,S});
    S.menu_file_sub(2)=uimenu(S.menu_file,'Label','Save current',...
                              'Callback',{@save_nbs,S});                     
    S.menu_file_sub(3)=uimenu(S.menu_file,'Label','Exit',...
                              'Callback',{@exit_nbs,S});
                          
    S.menu_view=uimenu(S.fh,'Label','View');
    S.menu_view_sub(1)=uimenu(S.menu_view,'Label','dNBSview',...
                               'Callback',{@dNBSview,S});
    S.menu_view_sub(1)=uimenu(S.menu_view,'Label','View Design Matrix',...
                               'Callback',{@view_model,S});

    S.menu_help=uimenu(S.fh,'Label','Help');
    S.menu_help_sub(1)=uimenu(S.menu_help,'Label','About dNBS',...
                              'Callback',{@load_about,S});
    S.menu_help_sub(2)=uimenu(S.menu_help,'Label','Acknowledgements',...
                              'Callback',{@load_ack,S});
    S.menu_help_sub(3)=uimenu(S.menu_help,'Label','Examples',...
                              'Callback',{@load_examples,S});
    
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Disable highlight in listbox
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function bdfcn_ls(varargin)
    S = varargin{3};
    set(S.OUT.ls,'Value',[]);
 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%STATS help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_stats(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;

    S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Help on Statistical Model',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','edit',...
        'units','normalized',...
        'enable','inactive',...
        'Max',2,'Min',0,...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[1-0.02-0.15 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_stats.txt'],'%s','delimiter','\n'),...
        'fontsize',12);
    set(S.close,'callback',{@close_help,S}); 
    
    
    S.ttest=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.02 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','t-test',...
       'callback',{@help_ttest,S} );
        
       
    S.rep=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.20 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Paired',...
        'callback',{@help_rep,S} );
    
   S.anova=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.38 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','ANOVA',...
        'callback',{@help_anova,S});
    
   S.ancova=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.56 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
         'String','ANCOVA',...
         'callback',{@help_ancova,S});
     
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%ttest help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_ttest(varargin)    
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','t-test example',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','text',...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_ttest.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Repeated measures help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_rep(varargin)    
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Repeated measures example',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','text',...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_repeated_measures.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});   

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%ANOVA help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_anova(varargin)    
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','ANOVA example',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','text',...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_anova.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});   
  
 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%ANCOVA help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_ancova(varargin)    
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','ANCOVA example',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','text',...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_ancova.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});     
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%DATA help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_data(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Help on Data',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','edit',...
        'enable','inactive',...
        'Max',2,'Min',0,...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_data.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S}); 
   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%ADVANCED SETTINGS help 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function help_advanced(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Help on Advanced Settings',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','edit',...
        'units','normalized',...
        'enable','inactive',...
        'Max',2,'Min',0,...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read help file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'help_advanced.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S}); 
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%About    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function load_about(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','About dNBS',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','edit',...
        'units','normalized',...
        'enable','inactive',...
        'Max',2,'Min',0,...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read about file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'about.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S}); 
    
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Acknowledgements
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function load_ack(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.4;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Acknowledgements',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','text',...
        'units','normalized',...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.15 1-0.04 1-0.15-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read about file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.1 0.09 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'acknowledgements.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});  

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Examples
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function load_examples(varargin)
    sz=get(0,'ScreenSize');
    lenx=0.5;
    leny=0.7;
    ox=0.5-lenx/2;
    oy=0.5-leny/2;
      S.fh = figure('units','normalized',...
        'position',[ox oy lenx leny],...
        'menubar','none',...
        'name','Examples',...
        'numbertitle','off',...
        'resize','off');
    
    S.txt=uicontrol('Parent',S.fh,...
        'style','edit',...
        'units','normalized',...
        'enable','inactive',...
        'Max',2,'Min',0,...
        'BackgroundColor',[1 1 1],...
        'position',[0.02 0.1 1-0.04 1-0.1-0.02],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Could not read about file!');
    
    S.close=uicontrol('Parent',S.fh,...
        'style','pushbutton',...
        'units','normalized',...
        'position',[0.5-0.15/2 0.02 0.15 0.05 ],...
        'fontsize',12,...
        'HorizontalAlignment','left',...
        'String','Close');
    
    set(S.txt,'string',textread(['help',filesep,'examples.txt'],'%s','delimiter','\n'));
    set(S.close,'callback',{@close_help,S});    


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Close help box
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function close_help(varargin)
    S=varargin{3};
    delete(S.fh);
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Exit
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function exit_nbs(varargin)
    S=varargin{3};
    delete(S.fh); 
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Load
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function load_nbs(varargin)
    global nbs
    S=varargin{3};
    [filename,pathname]=uigetfile('*.mat','Select Matlab (*.mat) file containing previous settings and results');
    if ~isequal(filename,0)
        ok=1; 
        try nbs=load([pathname,filename]); 
        catch; str='File could not be opened...'; ok=0; end
        if ok
            try nbs=nbs.nbs; UI=nbs.UI;
            catch; str='File does not contain valid nbs structure...'; ok=0; end
        end
        if ok
            %Reset to black
            set(S.DATA.matrices.text,'ForegroundColor','black');
            set(S.STATS.design.text,'ForegroundColor','black');
            set(S.STATS.contrast.text,'ForegroundColor','black');
            set(S.STATS.thresh.text,'ForegroundColor','black');
            set(S.STATS.test.text,'ForegroundColor','black');
            set(S.ADV.perms.text,'ForegroundColor','black');
            set(S.ADV.alpha.text,'ForegroundColor','black');
            set(S.ADV.size.text,'ForegroundColor','black');
            set(S.DATA.node_coor.text,'ForegroundColor','black');
            set(S.ADV.exchange.text,'ForegroundColor','black');
            set(S.DATA.node_label.text,'ForegroundColor','black');
            
            if strcmp(UI.method.ui,'Run NBS')
                set(S.ADV.nbs.edit,'value',1);
                radio_call([],[],S);
            elseif strcmp(UI.method.ui,'Run SPC')
                set(S.ADV.nbs.edit,'value',2);
                radio_call([],[],S);
            elseif strcmp(UI.method.ui,'Run FDR')
                set(S.ADV.nbs.edit,'value',3);
                radio_call([],[],S);
            else
                str='UI is corrupted...'; ok=0;
            end
            set(S.STATS.design.edit,'string',UI.design.ui);
            set(S.STATS.contrast.edit,'string',UI.contrast.ui);
            set(S.STATS.thresh.edit,'string',UI.thresh.ui);
            if strcmp(UI.test.ui,'One Sample')
                set(S.STATS.test.edit,'value',1);
            elseif strcmp(UI.test.ui,'t-test')
                set(S.STATS.test.edit,'value',2);
            elseif strcmp(UI.test.ui,'F-test')
                set(S.STATS.test.edit,'value',3);
            else
                str='UI is corrupted...'; ok=0; 
            end
            set(S.DATA.matrices.edit,'string',UI.matrices.ui);
            set(S.DATA.node_coor.edit,'string',UI.node_coor.ui);
            set(S.DATA.node_label.edit,'string',UI.node_label.ui);
            set(S.ADV.perms.edit,'string',UI.perms.ui);
            set(S.ADV.alpha.edit,'string',UI.alpha.ui);
            if strcmp(UI.size.ui,'Extent')
                set(S.ADV.size.edit,'value',1);
            elseif strcmp(UI.size.ui,'Intensity')
                set(S.ADV.size.edit,'value',2);
            else
                str='UI is corrupted...'; ok=0; 
            end
            set(S.ADV.exchange.edit,'string',UI.exchange.ui);
        end
        if ok
            str='Previous settings and results opened...';
        end
        tmp=get(S.OUT.ls,'string');
        set(S.OUT.ls,'string',[{str};tmp]);
    end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Save
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function save_nbs(varargin)
    global nbs
    S=varargin{3};
    [filename,pathname]=uiputfile('*.mat','Select Matlab (*.mat) file to save current settings and results');
    if ~isequal(filename,0)
        if ~isempty(nbs)
            try save([pathname,filename],'nbs');
                str='File saved...';
            catch 
                str='File could not be saved...'; 
            end
        else
            str='Nothing to save...';
        end
        tmp=get(S.OUT.ls,'string');
        set(S.OUT.ls,'string',[{str};tmp]); 
    end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%dNBSview
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function dNBSview(varargin)
    global nbs
    S=varargin{3};
    if isfield(nbs,'NBS')
        dNBSview(nbs.NBS);
    else
        dNBSview; 
    end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%View design matrix and contrast in a seperate figure
%Callback for View Model button
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function view_model(varargin)
    S=varargin{3};
    design_ui=get(S.STATS.design.edit,'string');
    contrast_ui=get(S.STATS.contrast.edit,'string');
    ind=get(S.STATS.test.edit,'value');
    tmp=get(S.STATS.test.edit,'string');
    test_ui=strtrim(tmp(ind,:));
    %Design matrix
    [X,Xok,Npred]=read_design(design_ui); 
    %Contrast
    [contrast,Cok]=read_contrast(contrast_ui,Npred);
    if Cok && Xok
        %Make figure
        sz=get(0,'ScreenSize');
        lenx=530;
        leny=580;
        ox=sz(3)/2-lenx/2;
        oy=sz(4)/2-leny/2;
        H.fh = figure('units','pixels',...
              'position',[ox oy lenx leny+50],...
              'Color',[1 1 1],...
              'menubar','none',...
              'name','Design Matrix & Contrast',...
              'numbertitle','off',...
              'resize','off');
        H.close=uicontrol('Parent',H.fh,...
                          'style','pushbutton',...
                          'units','pixels',...
                          'position',[430 590 70 25 ],...
                          'fontsize',12,...
                          'HorizontalAlignment','left',...
                          'String','Close');
        H.text=uicontrol('Parent',H.fh,...
                          'Style','text',...
                          'BackgroundColor',[1 1 1],...
                         'Position',[lenx/2-100/2 590, 100, 20],....
                         'String',test_ui,...
                          'ForeGroundColor','blue',...
                          'Fontweight','bold',...
                          'Fontsize',14); 
        H.ax_design = axes('units','pixels',...
                           'position',[50 190 lenx-100 leny-220 ],...
                           'box','off');
        H.ax_contrast = axes('units','pixels',...
                             'position',[50 40 lenx-100, 190-40-50],...
                             'box','off'); 
        H.design = imagesc(X,'Parent',H.ax_design);  % Display the image.
        %axis(H.ax_design,'image'); 
        grid(H.ax_design,'on'); 
        set(H.ax_design,'TickDir','out');
        %set(H.ax_design,'xtick',[],'ytick',[],'box','on','Visible','on');
        set(get(H.ax_design,'YLabel'),'String','Connectivity Matrices','fontsize',12);
        set(get(H.ax_design,'title'),'string','Design Matrix','color','blue','fontweight','bold','fontsize',12); 
        set(H.ax_design,'XTick',[1:size(X,2)],'YTick',[1:size(X,1)]); 
        set(H.ax_design,'XTickLabel',{''});
        if size(X,1)>30
            %Don't show row label if more than 30 rows
            set(H.ax_design,'YTickLabel',{''});
        end  
        colormap(H.ax_design,'gray');
        
        H.contrast=imagesc(contrast,'Parent',H.ax_contrast);  % Display the image.
       % axis(H.ax_contrast,'image'); 
        grid(H.ax_contrast,'on'); 
        set(get(H.ax_contrast,'title'),'string','Contrast','color','blue','fontweight','bold','fontsize',12);
        set(get(H.ax_contrast,'XLabel'),'String','Predictors','fontsize',12);
        set(H.ax_contrast,'XTick',[1:size(X,2)],'YTick',[1:size(X,1)]); 
        set(H.ax_contrast,'YTickLabel',{''});
        colormap(H.ax_contrast,'gray');
        set(H.close,'callback',{@close_help,H});  
    else
        if ~Xok
            msg={'Stop: Design Matrix not found or inconsistent'};
        else
            msg={'Stop: Contrast not found or inconsistent'};
        end
        %Attempt to print result of error checking to listbox. If this fails, print
        % to screen
        try tmp=get(S.OUT.ls,'string'); set(S.OUT.ls,'string',[msg;tmp]); drawnow;
        catch
            fprintf('%s\n',msg); 
        end
    end
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Read design matrix
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [X,ok,Npred]=read_design(Name)
ok=1;
data=readUI(Name);
if ~isempty(data)
    [nr,nc,ns]=size(data);
    if nr>0 && nc>0 && ns==1 && isnumeric(data) 
        X=data; 
    else
        ok=0; X=[];
    end
else
    ok=0; X=[];
end
clear data
if ok==1
    %Number of predictors
    Npred=nc;
else
    Npred=0;
end    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Read contrast
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [contrast,ok]=read_contrast(Name,Npred)
ok=1; 
data=readUI(Name);
if ~isempty(data)
    [nr,nc,ns]=size(data); 
    if nr==1 && nc==Npred && ns==1 && isnumeric(data) 
        contrast=data; 
    else
        ok=0; contrast=[];
    end
else
    ok=0; contrast=[];
end            
        
       