function [data,voxels,contrasts,labels]=spm_crossvalidation_extract(maskname,roiname,spmname)
% SPM_CROSSVALIDATION_EXTRACT extracts BOLD data from subject-specific
% cross-validation mask file. 
%
% After running SPM_CROSSVALIDATION use
%    data=SPM_CROSSVALIDATION_EXTRACT;
% to extract cross-validated data using the resulting subject-specific masks.
%
% Steps:
%    1) select a cross-validation mask file (*.img file created using SPM_CROSSVALIDATION)
%    2) select ROI file(s) (this file(s) should contain a priori ROI definitions,
%    the data from each subject-specific mask will be aggregated across
%    each ROI; you can leave this empty if you prefer to aggregated across
%    ALL of the voxels in the subject-specific masks)
%    3) select second-level SPM.mat file (typically the same second-level
%    SPM.mat file used in the SMP_CROSSVALIDATION step)
%
% The matrix data (of size number-of-subjects by number-of-regions) will contain 
% the cross-validated data of the selected second-level analysis (SPM.xY volumes) 
% aggregated across each ROI. 
%
% see also SPM_CROSSVALIDATION
%

% alfnie@gmail.com 06/10

if nargin<1
    maskname=spm_select(1,'.*\.img','Select cross-validation mask file');
end
[maskpath,maskname,maskext]=fileparts(maskname);
if nargin<2||strcmp(roiname,'?'),
    toptions={'Extract from:','','All voxels','Fixed number of clusters','All voxels within user-defined ROI file'};
    answ=questdlg(toptions{:},'All voxels');
    if isempty(answ), return; end
    switch(answ)
        case(toptions{1}), roiname=[];
        case(toptions{2}), roiname=inputdlg({'Number of clusters'},'',1,{'1'}); roiname=str2num(roiname{1});
        case(toptions{3}), roiname=spm_select(inf,'\.tal$|\.img$|\.nii$','Select ROI file(s)');
    end
end
if nargin<3
    if ~isempty(dir(fullfile(maskpath,[maskname,'.cv']))),
        load(fullfile(maskpath,[maskname,'.cv']),'SPM','-mat');
        spmname={fullfile(SPM.swd,'SPM.mat')};
    else spmname='';end
    spmname=spm_select(1,'SPM.*\.mat$','Select second-level analysis file',spmname);
end

%load(fullfile(maskpath,[maskname,'.cv']),'SPM','-mat');
load(spmname,'SPM');
N=size(SPM.xX.X,1);
ni=nifti(maskname);
Nm=[ni.dat.dim 1 1 1 1];
Nm=Nm(4);

data=[];voxels=[];
if ~isfield(SPM.xY,'P')
    SPM.xY.P=cellfun(@(a,b)[a,',',b],{SPM.xY.VY.fname},cellfun(@(x)num2str(x(1)),{SPM.xY.VY.n},'uni',0),'uni',0);
end
if isempty(roiname) % extract average across ALL suprathrehsold voxels
    for n=1:N,
        m=1+mod(n-1,Nm);
        disp(['Extracting data from subject ',num2str(m)]);
        [data(n,:),labels]=rex(SPM.xY.P{n},fullfile(maskpath,[maskname,maskext,',',num2str(m)]),'level','rois');
        voxels(n,:)=rex(SPM.xY.P{n},fullfile(maskpath,[maskname,maskext,',',num2str(m)]),'summary_measure','size','level','rois');
    end
elseif ischar(roiname) % extract average across all suprathreshold voxels within each ROI defined in the file 'roiname'
    for n=1:N,
        m=1+mod(n-1,Nm);
        disp(['Extracting data from subject ',num2str(m)]);
        [data(n,:),labels]=rex(SPM.xY.P{n},roiname,'level','clusters','conjunction_mask',fullfile(maskpath,[maskname,maskext,',',num2str(m)]));
        voxels(n,:)=rex(SPM.xY.P{n},roiname,'summary_measure','size','level','clusters','conjunction_mask',fullfile(maskpath,[maskname,maskext,',',num2str(m)]));
    end
elseif isnumeric(roiname) % extract average across all suprathreshold voxels within each of the N largest clusters
    for n=1:N,
        m=1+mod(n-1,Nm);
        disp(['Extracting data from subject ',num2str(m)]);
        [tdata{n},labels]=rex(SPM.xY.P{n},fullfile(maskpath,[maskname,maskext,',',num2str(m)]),'level','clusters');
        tvoxels{n}=rex(SPM.xY.P{n},fullfile(maskpath,[maskname,maskext,',',num2str(m)]),'summary_measure','size','level','clusters');
    end
    data=cellfun(@(x)[x(1:min(numel(x),roiname)) nan(1,max(0,roiname-numel(x)))],tdata,'uni',0); % selects N ('roiname') largest clusters
    data=cell2mat(data(:));
    voxels=cellfun(@(x)[x(1:min(numel(x),roiname)) nan(1,max(0,roiname-numel(x)))],tvoxels,'uni',0);
    voxels=cell2mat(voxels(:));
else
    error('incorrect field roiname (third input argument should be empty, a number, or a filename)');
end

KWY   = spm_filter(SPM.xX.K,SPM.xX.W*data);
beta  = SPM.xX.pKX*KWY;
clear contrasts;
for ncon=1:numel(SPM.xCon),
    contrasts(ncon)=struct('name',SPM.xCon(ncon).name,'STAT',SPM.xCon(ncon).STAT,'values',SPM.xCon(ncon).c'*beta);
    if size(data,2)==1, disp(['Contrast ''',SPM.xCon(ncon).name,''' cross-validated value(s) : ',num2str(contrasts(ncon).values(:)')]); end
end

if ~nargout
    assignin('base','CV_DATA',data);
    assignin('base','CV_CONTRASTS',contrasts);
    assignin('base','CV_VOXELS',voxels);
    assignin('base','CV_LABELS',labels);
    disp('Output stored in workspace''s variable CV_DATA, CV_CONTRASTS, CV_VOXELS, and CV_LABELS');
end
