#include <iostream>
#include <typeinfo>
#include <exception> 

#include "cbicaCmdParser.h"
#include "cbicaITKSafeImageIO.h"
#include "cbicaLogging.h"

#include "itkDiscreteGaussianImageFilter.h"
#include "itkCannyEdgeDetectionImageFilter.h"

int main(int argc, char *argv[])
{ 
  // structure the command line interface
  auto parser = cbica::CmdParser(argc, argv, "ITK_CPP11");
  parser.addRequiredParameter("i", "input", cbica::Parameter::FILE, ".nii.gz", "The input file");
  parser.addRequiredParameter("o", "output", cbica::Parameter::FILE, ".nii.gz", "File to write output");
  parser.addRequiredParameter("var", "variance", cbica::Parameter::FLOAT, "no range defined", "The sigma value to use for hessian filter", "It is measured in the units of image spacing.");
  parser.addRequiredParameter("k", "kernelWidth", cbica::Parameter::FLOAT, "no range defined", "The sigma value to use for hessian filter", "It is measured in the units of image spacing.");
  
  // parameters to get from the command line
  std::string inputFileName, referenceFileName, outputFileName, toWrite, varianceValue, kernelWidthValue;

  if ((argc < 1) || (parser.isPresent("u")))
  {
    parser.echoUsage();
    return EXIT_SUCCESS;
  }

  if (parser.isPresent("h"))
  {
    parser.echoHelp();
    return EXIT_SUCCESS;
  }

  if (parser.isPresent("v"))
  {
    parser.echoVersion();
    return EXIT_SUCCESS;
  }

  // checking for required parameters 
  parser.getParameterValue("i", inputFileName);
  parser.getParameterValue("o", outputFileName);
  parser.getParameterValue("var", varianceValue);
  parser.getParameterValue("k", kernelWidthValue);
  
  try // exception handling
  {
    // define the data types to be used 
    using ImageType = itk::Image< float, 3 >; // primitive image of pixel type float and 3 dimensions
	
    auto inputImage = cbica::ReadImage< ImageType >(inputFileName); // use cbica built-in function to read image in safe manner

    std::cout << "Input Image read\n";
	
    //
    // all Image Pointers in ITK are "smart pointers" (https://msdn.microsoft.com/en-us/library/hh279674.aspx) - basically, their allocation and deallocation is handled by the compiler, thus making the program free of memory and resource leaks and ensures exception-safety. best part: nothing extra needs to be done by developer for this
    auto gaussianFilter = itk::DiscreteGaussianImageFilter< ImageType, ImageType >::New(); // https://itk.org/Doxygen/html/classitk_1_1DiscreteGaussianImageFilter.html
    gaussianFilter->SetInput( inputImage );
    gaussianFilter->SetVariance( std::atof(varianceValue.c_str()) );
    gaussianFilter->SetMaximumKernelWidth( std::atof(kernelWidthValue.c_str()) );
    gaussianFilter->Update();
    //
  	
    std::cout << "Smoothing finished\n";
  
    //
    auto cannyFilter = itk::CannyEdgeDetectionImageFilter< ImageType, ImageType >::New(); // https://itk.org/Doxygen/html/classitk_1_1CannyEdgeDetectionImageFilter.html
    cannyFilter->SetInput( gaussianFilter->GetOutput() );
    cannyFilter->Update();
    //
  	
    std::cout << "Edge detection finished\n";

    cbica::WriteImage< ImageType >(cannyFilter->GetOutput(), outputFileName);

    std::cout << "Successfully finished\n");
    return EXIT_SUCCESS;
  }
  catch (itk::ExceptionObject &e)
  {
    std::cerr << "Exception thrown: " + std::string(e.what()) << "\n";
    return EXIT_FAILURE;
  }
}
