#pragma once

#include <iostream>
#include <iterator>
#include <stdlib.h>
#include <string>
#include <algorithm>
#include <exception>
#include <typeinfo>
#include <stdexcept>
#include <stdio.h>
#include <numeric>
#include <functional>
#include <vector> 
#include <sstream>

#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkScalarImageToRunLengthFeaturesFilter.h"
#include "itkHistogramToTextureFeaturesFilter.h"
#include "itkLabelImageToShapeLabelMapFilter.h"
#include "itkVectorImage.h"
#include "itkScalarImageToCooccurrenceMatrixFilter.h"
#include "itkRegionOfInterestImageFilter.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkVectorContainer.h"
#include "itkStatisticsImageFilter.h"
#include "itkImageToHistogramFilter.h"
#include "itkShapeLabelObject.h"
#include "itkConnectedComponentImageFilter.h"
#include "itkEnhancedScalarImageToSizeZoneFeaturesFilter.h"
#include "itkMinimumMaximumImageCalculator.h"
#include "itkEnhancedScalarImageToNeighbourhoodGreyLevelDifferenceFeaturesFilter.h"
#include "itkMinimumMaximumImageCalculator.h"
#include "CAPTk.h"

typedef ImageTypeFloat3D::OffsetType OffsetType;
typedef itk::VectorContainer< unsigned char, OffsetType > OffsetVector;
typedef itk::Statistics::ScalarImageToCooccurrenceMatrixFilter<ImageTypeFloat3D>Image2CoOccuranceType;
typedef Image2CoOccuranceType::HistogramType HistogramType;
typedef itk::Statistics::HistogramToTextureFeaturesFilter<HistogramType> Hist2FeaturesType1;

class TextureFeatures
{
public:
  enum Types
  {
    Custom,
    Torso,
    Neuro
  };


  //! Default Constructor
  TextureFeatures();

  //! Default Destructor
  ~TextureFeatures();


  /**
  \brief Calculate intensity features
  */
  template <class TImageType = ImageTypeFloat3D>
  void Intensity_features(typename TImageType::Pointer image, typename TImageType::Pointer mask, 
    std::vector< std::tuple<std::string, std::string, float>>* featurevec);

  /**
  \brief Calculate Texture Features

  This function computes features showcases the texture of a given image. this function uses ITK ScalarImageToTextureFeaturesFilter.
  The texture features have proven to be useful in image classification for biological and medical imaging.
  This function computes the texture features of a specified ROI(region of interest) got through the mask provided as an input para.
  The features that are calculated are contrast, correlation, energy, entropy, haralick correlation, cluster shde etc.
  The user can set the number of bins if required. The deafult value used is 16.

  \param image The input Image on which to calculate the features
  \param mask The image specificying the roi
  \param min Minimum intensity in given roi
  \param max Maximum intensity in given roi
  \param offsetvector vector with 13 offset direction
  \param featurename Vector holing name of the features
  \param feature A vector holding features of each offset direction

  */
  template <class TImageType = ImageTypeFloat3D, typename Offsets = OffsetVector>
  void calculateTextureFeatures(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max, Offsets *offset, 
    std::vector< std::tuple<std::string, std::string, float>>* featurevec)
  {
    //glcm_featureNames.push_back("dummyVal");
    //glcm_features.push_back(vector<double>(1, 1));
    //return;
    //https://itk.org/Wiki/ITK/Examples/Statistics/TextureFeatures RK
    //glcm_featureNames
    typename TImageType::Pointer inertia = TImageType::New();
    typename TImageType::Pointer correlation = TImageType::New();
    typename TImageType::Pointer energy = TImageType::New();
    typename TImageType::Pointer entropy = TImageType::New();
    inertia->SetRegions(image->GetLargestPossibleRegion());
    correlation->SetRegions(image->GetLargestPossibleRegion());
    energy->SetRegions(image->GetLargestPossibleRegion());
    entropy->SetRegions(image->GetLargestPossibleRegion());
    typename Offsets::Pointer offsets = Offsets::New();
    offsets = offset;

    typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
    std::vector< typename TImageType::IndexType> index_vec;
    //   std::vector<double> nonzero_pix;
    IteratorType inputIt(mask, mask->GetLargestPossibleRegion());

    for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
    {
      if (mask->GetPixel(inputIt.GetIndex()) != 0)
      {
        index_vec.push_back(inputIt.GetIndex());
        nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));

      }
    }

    for (unsigned int i = 0; i < offsets->size(); i++)
    {
      Image2CoOccuranceType::Pointer glcmGenerator = Image2CoOccuranceType::New();
      glcmGenerator->SetOffset(offsets->at(i));
      glcmGenerator->SetNumberOfBinsPerAxis(16); //reasonable number of bins
      glcmGenerator->SetPixelValueMinMax(min, max); //for input UCHAR pixel type
      glcmGenerator->SetMaskImage(mask);
      glcmGenerator->SetInput(image);
      Hist2FeaturesType1::Pointer featureCalc = Hist2FeaturesType1::New();
      glcmGenerator->Update();
      featureCalc->SetInput(glcmGenerator->GetOutput());
      featureCalc->Update();
      // std::cout << "correlation" << offsets->at(i) << featureCalc->GetFeature(Hist2FeaturesType1::Correlation);
      // std::cout << "energy" << featureCalc->GetFeature(Hist2FeaturesType1::Energy);
      // std::cout << "contrast" << featureCalc->GetFeature(Hist2FeaturesType1::Inertia);
      // std::cout << "entropy" << featureCalc->GetFeature(Hist2FeaturesType1::Entropy);

      //  std::cout << offsets->GetElement(0).operator[](0);

     
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "correlation" + std::to_string(i), featureCalc->GetFeature(Hist2FeaturesType1::Correlation)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "energy" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::Energy)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "contrast" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::Inertia)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "entropy" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::Entropy)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "inversediffmom" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::InverseDifferenceMoment)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture","Clustershade" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::ClusterShade)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "ClusterProminence" + std::to_string(i) , featureCalc->GetFeature(Hist2FeaturesType1::ClusterProminence)));
     featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "haralickCorrelation" + std::to_string(i) ,featureCalc->GetFeature(Hist2FeaturesType1::HaralickCorrelation)));

   /*   featurevector["correlation" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::Correlation);
      featurevector["energy" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::Energy);
      featurevector["contrast" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::Inertia);
      featurevector["entropy" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::Entropy);
      featurevector["inversediffmom" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::InverseDifferenceMoment);
      featurevector["Clustershade" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::ClusterShade);
      featurevector["ClusterProminence" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::ClusterProminence);
      featurevector["haralickCorrelation" + std::to_string(i)] = featureCalc->GetFeature(Hist2FeaturesType1::HaralickCorrelation);*/


    }

 }

  template <class TImageType = ImageTypeFloat3D, typename Offsets = OffsetVector >
  void calculateRunLength(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max,
    Offsets *offset, std::vector< std::tuple<std::string, std::string, float>>* featurevec);

  template <class TImageType = ImageTypeFloat3D >
  void ShapeFeatures(typename TImageType::Pointer mask, std::vector< std::tuple<std::string, std::string, float>> *featurevec);

  void HistogramFeatures(std::vector<double> &intensities, int start, int interval, int end, std::vector< std::tuple<std::string, std::string, float>>* featurevec);
  bool m_errormsg =FALSE;
  std::vector<double> nonzero_pix;

  template<class TImageType = ImageTypeFloat3D>
  void CalculateGrayLevelSizeZoneFeatures(typename TImageType::Pointer itkImage, typename TImageType::Pointer maskImage, std::vector< std::tuple<std::string, std::string, float>>* featurevec
    )
  {
    typedef TImageType ImageType;
    //typedef TImageType MaskType;
    typedef itk::Statistics::EnhancedScalarImageToSizeZoneFeaturesFilter<ImageType> FilterType;
    typedef itk::MinimumMaximumImageCalculator<ImageType> MinMaxComputerType;
    typedef typename FilterType::SizeZoneFeaturesFilterType TextureFilterType;

    typename FilterType::Pointer filter = FilterType::New();

    typename FilterType::OffsetVector::Pointer newOffset = FilterType::OffsetVector::New();
    auto oldOffsets = filter->GetOffsets();
    auto oldOffsetsIterator = oldOffsets->Begin();
    while (oldOffsetsIterator != oldOffsets->End())
    {
      bool continueOuterLoop = false;
      typename FilterType::OffsetType offset = oldOffsetsIterator->Value();
      for (unsigned int i = 0; i < TImageType::ImageDimension; ++i)
      {
        if (/*params.m_Direction == i + 2 &&*/ offset[i] != 0)
        {
          continueOuterLoop = true;
        }
      }
      /*if (params.m_Direction == 1)
      {
      offset[0] = 0;
      offset[1] = 0;
      offset[2] = 1;
      newOffset->push_back(offset);
      break;
      }*/

      oldOffsetsIterator++;
      if (continueOuterLoop)
        continue;
      newOffset->push_back(offset);
    }
    filter->SetOffsets(newOffset);


    // All features are required
    typename FilterType::FeatureNameVectorPointer requestedFeatures = FilterType::FeatureNameVector::New();
    requestedFeatures->push_back(TextureFilterType::SmallZoneEmphasis);
    requestedFeatures->push_back(TextureFilterType::LargeZoneEmphasis);
    requestedFeatures->push_back(TextureFilterType::GreyLevelNonuniformity);
    requestedFeatures->push_back(TextureFilterType::SizeZoneNonuniformity);
    requestedFeatures->push_back(TextureFilterType::LowGreyLevelZoneEmphasis);
    requestedFeatures->push_back(TextureFilterType::HighGreyLevelZoneEmphasis);
    requestedFeatures->push_back(TextureFilterType::SmallZoneLowGreyLevelEmphasis);
    requestedFeatures->push_back(TextureFilterType::SmallZoneHighGreyLevelEmphasis);
    requestedFeatures->push_back(TextureFilterType::LargeZoneLowGreyLevelEmphasis);
    requestedFeatures->push_back(TextureFilterType::LargeZoneHighGreyLevelEmphasis);
    requestedFeatures->push_back(TextureFilterType::ZonePercentage);
    requestedFeatures->push_back(TextureFilterType::GreyLevelVariance);
    requestedFeatures->push_back(TextureFilterType::SizeZoneVariance);
    requestedFeatures->push_back(TextureFilterType::ZoneEntropy);

    typename MinMaxComputerType::Pointer minMaxComputer = MinMaxComputerType::New();
    minMaxComputer->SetImage(itkImage);
    minMaxComputer->Compute();

    filter->SetInput(itkImage);
    filter->SetMaskImage(maskImage);
    filter->SetRequestedFeatures(requestedFeatures);
    int rangeOfPixels = /*params.m_Range*/256;
    /*if (rangeOfPixels < 2)
    rangeOfPixels = 256;

    if (params.m_UseCtRange)
    {
    filter->SetPixelValueMinMax((TPixel)(-1024.5),(TPixel)(3096.5));
    filter->SetNumberOfBinsPerAxis(3096.5+1024.5);
    } else*/
    {
      filter->SetPixelValueMinMax(minMaxComputer->GetMinimum(), minMaxComputer->GetMaximum());
      filter->SetNumberOfBinsPerAxis(rangeOfPixels);
    }

    filter->SetDistanceValueMinMax(0, rangeOfPixels);

    filter->Update();

    auto featureMeans = filter->GetFeatureMeans();
    auto featureStd = filter->GetFeatureStandardDeviations();

    std::ostringstream  ss;
    ss << rangeOfPixels;
    std::string strRange = ss.str();
    for (std::size_t i = 0; i < featureMeans->size(); ++i)
    {
      switch (i)
      {
      case TextureFilterType::SmallZoneEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "SmallZoneEmphasis Means", featureMeans->ElementAt(i)));
        
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") SmallZoneEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::LargeZoneEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "LargeZoneEmphasis Means", featureMeans->ElementAt(i)));

       // featurevector["LargeZoneEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") LargeZoneEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::GreyLevelNonuniformity:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "GreyLevelNonuniformity Means", featureMeans->ElementAt(i)));

       // featurevector["GreyLevelNonuniformity Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") GreyLevelNonuniformity Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::SizeZoneNonuniformity:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "SizeZoneNonuniformity Means", featureMeans->ElementAt(i)));

       // featurevector["SizeZoneNonuniformity Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") SizeZoneNonuniformity Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::LowGreyLevelZoneEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "LowGreyLevelZoneEmphasis Means", featureMeans->ElementAt(i)));

       // featurevector["LowGreyLevelZoneEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") LowGreyLevelZoneEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::HighGreyLevelZoneEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "HighGreyLevelZoneEmphasis Means", featureMeans->ElementAt(i)));

       // featurevector["HighGreyLevelZoneEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") HighGreyLevelZoneEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::SmallZoneLowGreyLevelEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "SmallZoneLowGreyLevelEmphasis Means", featureMeans->ElementAt(i)));

        //featurevector["SmallZoneLowGreyLevelEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") SmallZoneLowGreyLevelEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::SmallZoneHighGreyLevelEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "SmallZoneHighGreyLevelEmphasis Means", featureMeans->ElementAt(i)));

        //featurevector["SmallZoneHighGreyLevelEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") SmallZoneHighGreyLevelEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::LargeZoneLowGreyLevelEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "LargeZoneLowGreyLevelEmphasis Means", featureMeans->ElementAt(i)));

        //featurevector["LargeZoneLowGreyLevelEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") LargeZoneLowGreyLevelEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::LargeZoneHighGreyLevelEmphasis:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "LargeZoneHighGreyLevelEmphasis Means", featureMeans->ElementAt(i)));

        //featurevector["LargeZoneHighGreyLevelEmphasis Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") LargeZoneHighGreyLevelEmphasis Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::ZonePercentage:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "ZonePercentage Means", featureMeans->ElementAt(i)));

        //featurevector["ZonePercentage Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") ZonePercentage Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::GreyLevelVariance:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "GreyLevelVariance Means", featureMeans->ElementAt(i)));

        //featurevector["GreyLevelVariance Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") GreyLevelVariance Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::SizeZoneVariance:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "SizeZoneVariance Means", featureMeans->ElementAt(i)));

        //featurevector["SizeZoneVariance Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") SizeZoneVariance Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::ZoneEntropy:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "ZoneEntropy Means", featureMeans->ElementAt(i)));
        //featurevector["ZoneEntropy Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("SizeZone ("+ strRange+") ZoneEntropy Means",featureMeans->ElementAt(i)));
        break;
      default:
        break;
      }
    }
  }

 
  template<class TImageType = ImageTypeFloat3D>
  void CalculateGrayLevelNeighbourhoodGreyLevelDifferenceFeatures(typename TImageType::Pointer itkImage,
    typename TImageType::Pointer maskImage, std::vector< std::tuple<std::string, std::string, float>>* featurevec)
  {
    typedef TImageType ImageType;
    //typedef TImageType MaskType;
    typedef itk::Statistics::EnhancedScalarImageToNeighbourhoodGreyLevelDifferenceFeaturesFilter<ImageType> FilterType;
    typedef itk::MinimumMaximumImageCalculator<ImageType> MinMaxComputerType;
    typedef typename FilterType::NeighbourhoodGreyLevelDifferenceFeaturesFilterType TextureFilterType;

    typename FilterType::Pointer filter = FilterType::New();

    typename FilterType::OffsetVector::Pointer newOffset = FilterType::OffsetVector::New();
    auto oldOffsets = filter->GetOffsets();
    auto oldOffsetsIterator = oldOffsets->Begin();
    while (oldOffsetsIterator != oldOffsets->End())
    {
      bool continueOuterLoop = false;
      typename FilterType::OffsetType offset = oldOffsetsIterator->Value();
      for (unsigned int i = 0; i < TImageType::ImageDimension; ++i)
      {
        if (/*params.m_Direction == i + 2 &&*/ offset[i] != 0)
        {
          continueOuterLoop = true;
        }
      }
      /*if (params.m_Direction == 1)
      {
      offset[0] = 0;
      offset[1] = 0;
      offset[2] = 1;
      newOffset->push_back(offset);
      break;
      }*/

      oldOffsetsIterator++;
      if (continueOuterLoop)
        continue;
      newOffset->push_back(offset);
    }
    filter->SetOffsets(newOffset);


    // All features are required
    typename FilterType::FeatureNameVectorPointer requestedFeatures = FilterType::FeatureNameVector::New();
    requestedFeatures->push_back(TextureFilterType::Coarseness);
    requestedFeatures->push_back(TextureFilterType::Contrast);
    requestedFeatures->push_back(TextureFilterType::Busyness);
    requestedFeatures->push_back(TextureFilterType::Complexity);
    requestedFeatures->push_back(TextureFilterType::Strength);

    typename MinMaxComputerType::Pointer minMaxComputer = MinMaxComputerType::New();
    minMaxComputer->SetImage(itkImage);
    minMaxComputer->Compute();

    filter->SetInput(itkImage);
    filter->SetMaskImage(maskImage);
    filter->SetRequestedFeatures(requestedFeatures);
    int rangeOfPixels = /*params.m_Range*/256;
    /*if (rangeOfPixels < 2)
    rangeOfPixels = 256;

    if (params.m_UseCtRange)
    {
    filter->SetPixelValueMinMax((TPixel)(-1024.5),(TPixel)(3096.5));
    filter->SetNumberOfBinsPerAxis(3096.5+1024.5);
    } else*/
    {
      filter->SetPixelValueMinMax(minMaxComputer->GetMinimum(), minMaxComputer->GetMaximum());
      filter->SetNumberOfBinsPerAxis(rangeOfPixels);
    }

    filter->SetDistanceValueMinMax(0, rangeOfPixels);

    filter->Update();

    auto featureMeans = filter->GetFeatureMeans();
    auto featureStd = filter->GetFeatureStandardDeviations();

    std::ostringstream  ss;
    ss << rangeOfPixels;
    std::string strRange = ss.str();
    for (std::size_t i = 0; i < featureMeans->size(); ++i)
    {
      //MITK_WARN << "Writing output " << i;
      switch (i)
      {
      case TextureFilterType::Coarseness:
       // featurevector["Coarseness Means"] = featureMeans->ElementAt(i);
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "Coarseness Means", featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::Contrast:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "Contrast Means", featureMeans->ElementAt(i)));

       // featurevector["Contrast Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("NeighbourhoodGreyLevelDifference ("+ strRange+") Contrast Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::Busyness:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "Busyness Means", featureMeans->ElementAt(i)));

     //   featurevector["Busyness Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("NeighbourhoodGreyLevelDifference ("+ strRange+") Busyness Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::Complexity:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "Complexity Means", featureMeans->ElementAt(i)));

       // featurevector["Complexity Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("NeighbourhoodGreyLevelDifference ("+ strRange+") Complexity Means",featureMeans->ElementAt(i)));
        break;
      case TextureFilterType::Strength:
        featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", "Strength Means", featureMeans->ElementAt(i)));

       // featurevector["Strength Means"] = featureMeans->ElementAt(i);
        //featureList.push_back(std::make_pair("NeighbourhoodGreyLevelDifference ("+ strRange+") Strength Means",featureMeans->ElementAt(i)));
        break;
      default:
        break;
      }
    }
  }

};

template <class TImageType>
void TextureFeatures::Intensity_features(typename TImageType::Pointer image, typename TImageType::Pointer maskin,
  std::vector< std::tuple<std::string, std::string, float>>* featurevec)
{
  typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
  std::vector< typename TImageType::IndexType> index_vec;

  IteratorType inputIt(maskin, maskin->GetLargestPossibleRegion());

  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  {
    if (maskin->GetPixel(inputIt.GetIndex()) != 0)
    {
      index_vec.push_back(inputIt.GetIndex());
      nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));
    }
  }
  if (nonzero_pix.empty())
  {
    m_errormsg = TRUE;
    return;
  }

  //if (nonzero_pix.size() == 0)
  //{
  //  IteratorType inputIt(image, image->GetLargestPossibleRegion());
  //  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  //  {
  //    if (image->GetPixel(inputIt.GetIndex()) != 0)
  //    {
  //      index_vec.push_back(inputIt.GetIndex());
  //      nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));
  //    }
  //  }
  //}

  double min = *min_element(nonzero_pix.begin(), nonzero_pix.end());
  //std::cout << "Min value: " << min;
  double max = *max_element(nonzero_pix.begin(), nonzero_pix.end());
  //std::cout << "Max value: " << max;

  double sum = std::accumulate(nonzero_pix.begin(), nonzero_pix.end(), 0.0);
  double mean = sum / nonzero_pix.size();

  std::vector<double> diff(nonzero_pix.size());
  std::transform(nonzero_pix.begin(), nonzero_pix.end(), diff.begin(),
    std::bind2nd(std::minus<double>(), mean));
  double sq_sum = std::inner_product(diff.begin(), diff.end(), diff.begin(), 0.0);
  double stdev = std::sqrt(sq_sum / nonzero_pix.size());

  double voxvol = image->GetSpacing().GetElement(0)*image->GetSpacing().GetElement(1)*image->GetSpacing().GetElement(2);
  double volume = nonzero_pix.size() * voxvol;

  featurevec->push_back(std::tuple<std::string, std::string, float>("intensity", "min_intensity", min));
  featurevec->push_back(std::tuple<std::string, std::string, float>("intensity", "max_intensity", max));
  featurevec->push_back(std::tuple<std::string, std::string, float>("intensity", "mean_intensity", mean));
  featurevec->push_back(std::tuple<std::string, std::string, float>("intensity", "std", stdev));
  featurevec->push_back(std::tuple<std::string, std::string, float>("intensity", "volume", volume));

  return;
}


template <class TImageType, typename Offsets>
void TextureFeatures::calculateRunLength(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max,
  Offsets *offset, std::vector< std::tuple<std::string, std::string, float>>* featurevec)
{
  //typedef itk::VectorContainer< unsigned char, OffsetType > OffsetVector;
  typename Offsets::Pointer offsets = Offsets::New();
  offsets = offset;

  typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
  std::vector< typename TImageType::IndexType> index_vec;
  //std::vector<double> nonzero_pix;
  IteratorType inputIt(mask, mask->GetLargestPossibleRegion());

  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  {
    if (mask->GetPixel(inputIt.GetIndex()) != 0)
    {
      index_vec.push_back(inputIt.GetIndex());
      nonzero_pix.push_back(inputIt.Get());
    }
  }
  typedef itk::Statistics::DenseFrequencyContainer2 HistogramFrequencyContainerType;
  typedef itk::Statistics::ScalarImageToRunLengthFeaturesFilter
    <ImageTypeFloat3D, HistogramFrequencyContainerType> RunLengthFilterType;
  typedef  RunLengthFilterType::RunLengthMatrixFilterType RunLengthMatrixGenerator;
  typedef  RunLengthFilterType::RunLengthFeaturesFilterType RunLengthFeatures;
  RunLengthMatrixGenerator::Pointer matrix_generator = RunLengthMatrixGenerator::New();
  typedef RunLengthFilterType::RunLengthFeaturesFilterType    RunLengthFeatureName;
  typedef RunLengthFeatureName::RunLengthFeatureName InternalRunLengthFeatureName;
  RunLengthFilterType::FeatureNameVectorPointer requestedFeatures = RunLengthFilterType::FeatureNameVector::New();

  requestedFeatures->push_back(RunLengthFeatureName::ShortRunEmphasis);
  requestedFeatures->push_back(RunLengthFeatureName::LongRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::GreyLevelNonuniformity);
  requestedFeatures->push_back(
    RunLengthFeatureName::RunLengthNonuniformity);
  requestedFeatures->push_back(
    RunLengthFeatureName::LowGreyLevelRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::HighGreyLevelRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::ShortRunLowGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::ShortRunHighGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::LongRunLowGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::LongRunHighGreyLevelEmphasis);
  std::vector <std::string> featurename;
  featurename.push_back("SRE"); featurename.push_back("LRE");
  featurename.push_back("GLN"); featurename.push_back("RLN");
  featurename.push_back("LGRE"); featurename.push_back("HGRE");
  featurename.push_back("SRLGE");  featurename.push_back("SRHGE");
  featurename.push_back("LRLGE");  featurename.push_back("LRHGE");

  // texFilter->SetRequestedFeatures(requestedFeatures);
  // texFilter->Update();
  // FeatureValueVectorPointer means, stds;
  //  means = texFilter->GetFeatureMeans();
  // stds = texFilter->GetFeatureStandardDeviations();

  OffsetVector::ConstIterator offsetIt;
  size_t offsetNum = 0, featureNum = 0;
  //size_t numOffsets = offsets->size();
  //size_t numFeatures = requestedFeatures->size();
  //double **features;
  //features = new double *[numOffsets];

  //for (size_t i = 0; i < numOffsets; i++)
  //{
  //  features[i] = new double[numFeatures];
  //}

  for (offsetIt = offsets->Begin();
    offsetIt != offsets->End(); offsetIt++, offsetNum++)
  {
    matrix_generator->SetOffset(offsetIt.Value());
    matrix_generator->SetMaskImage(mask);
    matrix_generator->SetInput(image);    
    matrix_generator->SetInsidePixelValue(1);
    matrix_generator->SetPixelValueMinMax(min, max);
    matrix_generator->SetDistanceValueMinMax(0, 4); // TOCHECK - why is this only between 0-4? SP
    matrix_generator->SetNumberOfBinsPerAxis(10); // TOCHECK - needs to be statistically significant
    matrix_generator->Update();

    RunLengthFeatures::Pointer runLengthMatrixCalculator = RunLengthFeatures::New();
    runLengthMatrixCalculator->SetInput(matrix_generator->GetOutput());
    runLengthMatrixCalculator->Update();
    RunLengthFilterType::FeatureNameVector::ConstIterator fnameIt;
    std::ostringstream ss;
    featureNum = 0;
    for (fnameIt = requestedFeatures->Begin(); fnameIt != requestedFeatures->End(); fnameIt++, featureNum++)
    {
      ss << offsetNum;
      featurevec->push_back(std::tuple<std::string, std::string, double>("Texture", featurename[featureNum] + ss.str(),
        runLengthMatrixCalculator->GetFeature((InternalRunLengthFeatureName)fnameIt.Value())));

      //featurevector[featurename[featureNum] + ss.str()] = // TOCHECK: instead of using 'featurename.at(fnameIt.Value())', might use 'featurename[fnameIt.Value()]'; it is quicker
      //  runLengthMatrixCalculator->GetFeature((InternalRunLengthFeatureName)fnameIt.Value());
    }
  }
}


template <class TImageType>
void TextureFeatures::ShapeFeatures(typename TImageType::Pointer mask, std::vector<std::tuple<std::string,std::string,float>>* featurevec)
{
  typedef  short LabelType;
  typedef itk::Image< LabelType, 3 > OutputImageType;
  typedef itk::ShapeLabelObject< LabelType, 3 > ShapeLabelObjectType;
  typedef itk::LabelMap< ShapeLabelObjectType > LabelMapType;

  typedef itk::ConnectedComponentImageFilter < TImageType, OutputImageType > ConnectedComponentImageFilterType;
  typedef itk::LabelImageToShapeLabelMapFilter < OutputImageType, LabelMapType > I2LType;
  typename ConnectedComponentImageFilterType::Pointer connected = ConnectedComponentImageFilterType::New();
  connected->SetInput(mask);
  connected->Update();


  I2LType::Pointer i2l = I2LType::New();
  i2l->SetInput(connected->GetOutput());
  i2l->SetComputePerimeter(true);
  i2l->Update();
  LabelMapType::Pointer labelMap = i2l->GetOutput();
  //std::cout << " has " << labelMap->GetNumberOfLabelObjects() << " labels." << std::endl;
  ShapeLabelObjectType::Pointer labelObject = labelMap->GetNthLabelObject(0);

  //std::cout << labelObject->GetPrincipalMoments() << labelObject->GetElongation() <<
  //labelObject->GetPerimeter() << labelObject->GetRoundness() << labelObject->GetFlatness();

  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "no_of_pix", labelObject->GetNumberOfPixels()));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "principleMoment_1", labelObject->GetPrincipalMoments().operator[](0)));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "principleMoment_2", labelObject->GetPrincipalMoments().operator[](1)));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "principleMoment_3", labelObject->GetPrincipalMoments().operator[](2)));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "Elongation", labelObject->GetElongation()));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "Perimeter", labelObject->GetPerimeter()));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "Roundness", labelObject->GetRoundness()));
  featurevec->push_back(std::tuple<std::string, std::string, double>("Shape", "Flatness", labelObject->GetFlatness()));

  //FileToWorkWith.open(filename, std::fstream::in | std::fstream::out | std::fstream::app);
}
