#ifndef __itkVectorImageCentralDifferenceFunction_h
#define __itkVectorImageCentralDifferenceFunction_h

#include "itkImageFunction.h"
#include "itkVariableSizeMatrix.h"
#include "itkImageBase.h"
#include "itkVariableLengthVector.h"

namespace itk
{

 
/**
 * \class VectorCentralDifferenceImageFunction
 * \brief Calculate the derivative by central differencing.
 *
 * This class is templated over the input image type and
 * the coordinate representation type (e.g. float or double).
 *
 * Possible improvements:
 * - the use of Neighborhood operators may improve efficiency.
 *
 * \author Tom Vercauteren, INRIA & Mauna Kea Technologies
 *
 * \ingroup ImageFunctions
 */
template <
  class TInputImage, 
  class TCoordRep = double >
class ITK_EXPORT VectorImageCentralDifferenceFunction :
  public ImageFunction< TInputImage, 
                        VariableSizeMatrix<double>, \
                        TCoordRep >
{
public:
  /**typedef typename TInputImage::PixelType InputPixelType;**/
  
  typedef VariableLengthVector<double> InputPixelType; 
  typedef VariableSizeMatrix<double> MatrixType;     
  
 
  /** Dimension underlying input image. */
  itkStaticConstMacro(ImageDimension, unsigned int,
                      TInputImage::ImageDimension);
		      
  itkSetMacro (VectorLength, int);
  itkGetMacro (VectorLength, int);		      

  /** Standard class typedefs. */
  typedef VectorImageCentralDifferenceFunction Self;
  typedef ImageFunction<TInputImage,
                        VariableSizeMatrix<double>,  
                        TCoordRep>       Superclass;
  typedef SmartPointer<Self>             Pointer;
  typedef SmartPointer<const Self>       ConstPointer;
  
  /** Run-time type information (and related methods). */
  itkTypeMacro(VectorImageCentralDifferenceFunction, ImageFunction);

  /** Method for creation through the object factory. */
  itkNewMacro(Self);
  
  
  

  /** InputImageType typedef support. */
  typedef TInputImage InputImageType;

  /** OutputType typdef support. */
  typedef typename Superclass::OutputType OutputType;

  /** Index typedef support. */
  typedef typename Superclass::IndexType IndexType;
  
  /** ContinuousIndex typedef support. */
  typedef typename Superclass::ContinuousIndexType ContinuousIndexType;

  /** Point typedef support. */
  typedef typename Superclass::PointType PointType;

  /** Evalulate the image derivative by central differencing at specified index.
   *
   *  No bounds checking is done.
   *  The point is assume to lie within the image buffer.
   *
   *  ImageFunction::IsInsideBuffer() can be used to check bounds before
   * calling the method. */
  virtual OutputType EvaluateAtIndex( const IndexType& index ) const;
  
  /** Evalulate the image derivative by central differencing at non-integer 
   *  positions.
   *
   *  No bounds checking is done.
   *  The point is assume to lie within the image buffer.
   *
   *  ImageFunction::IsInsideBuffer() can be used to check bounds before
   * calling the method. */
   virtual OutputType Evaluate( const PointType& point ) const
    { 
    IndexType index;
    this->ConvertPointToNearestIndex( point, index );
    return this->EvaluateAtIndex( index ); 
    }
  virtual OutputType EvaluateAtContinuousIndex( 
    const ContinuousIndexType& cindex ) const
    { 
    IndexType index;
    this->ConvertContinuousIndexToNearestIndex( cindex, index );
    return this->EvaluateAtIndex( index ); 
    }
  
protected:
  VectorImageCentralDifferenceFunction();
  ~VectorImageCentralDifferenceFunction(){};
  void PrintSelf(std::ostream& os, Indent indent) const;

private:
  VectorImageCentralDifferenceFunction( const Self& ); //purposely not implemented
  void operator=( const Self& ); //purposely not implemented
  int m_VectorLength;

};

} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkVectorImageCentralDifferenceFunction.txx"
#endif

#endif
