/**
 * This is a small tool that shows how to use the diffeomorphic demons algorithm.
 * The user can choose if diffeomorphic, additive or compositive demons should be used.
 * The user can also choose the type of demons forces, or other parameters;
 *
 * \author Tom Vercauteren, INRIA & Mauna Kea Technologies
 */

#include "itkMultiResolutionPDEDeformableRegistration2.h"
#include "itkFastSymmetricForcesDemonsRegistrationFilter.h"
#include "itkDiffeomorphicDemonsRegistrationFilter.h"
#include "itkWarpImageFilter.h"
#include "itkCommand.h"
#include "itkWarpJacobianDeterminantFilter.h"
#include "itkMinimumMaximumImageCalculator.h"
#include "itkWarpHarmonicEnergyCalculator.h"
#include "itkGridForwardWarpImageFilter.h"
#include "itkVectorCentralDifferenceImageFunction.h"
#include "itkVector.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkHistogramMatchingImageFilter.h"

#include "itkTransformFileReader.h"
#include "itkTransformToDeformationFieldFilter.h"

#include <iostream>
#include "DemonsRegistrationCLP.h"


template <class TPixel=float, unsigned int VImageDimension=3>
class CommandIterationUpdate : public itk::Command 
{
public:
   typedef  CommandIterationUpdate   Self;
   typedef  itk::Command             Superclass;
   typedef  itk::SmartPointer<Self>  Pointer;

   typedef itk::Image< TPixel, VImageDimension > InternalImageType;
   typedef itk::Vector< TPixel, VImageDimension >    VectorPixelType;
   typedef itk::Image<  VectorPixelType, VImageDimension > DeformationFieldType;

   typedef itk::DiffeomorphicDemonsRegistrationFilter<
      InternalImageType,
      InternalImageType,
      DeformationFieldType>   DiffeomorphicDemonsRegistrationFilterType;

   typedef itk::FastSymmetricForcesDemonsRegistrationFilter<
      InternalImageType,
      InternalImageType,
      DeformationFieldType>   FastSymmetricForcesDemonsRegistrationFilterType;

   typedef itk::MultiResolutionPDEDeformableRegistration2<
      InternalImageType, InternalImageType,
      DeformationFieldType, TPixel >   MultiResRegistrationFilterType;

   typedef itk::WarpJacobianDeterminantFilter<
      DeformationFieldType, InternalImageType> JacobianFilterType;
   
   typedef itk::MinimumMaximumImageCalculator<InternalImageType> MinMaxFilterType;

   typedef itk::WarpHarmonicEnergyCalculator<DeformationFieldType>
      HarmonicEnergyCalculatorType;

   typedef itk::VectorCentralDifferenceImageFunction<DeformationFieldType>
      WarpGradientCalculatorType;

   typedef typename WarpGradientCalculatorType::OutputType WarpGradientType;
   
   itkNewMacro( Self );

private:
   std::ofstream m_Fid;
   bool m_headerwritten;
   typename JacobianFilterType::Pointer m_JacobianFilter;
   typename MinMaxFilterType::Pointer m_Minmaxfilter;
   typename HarmonicEnergyCalculatorType::Pointer m_HarmonicEnergyCalculator;
   typename DeformationFieldType::ConstPointer m_TrueField;
   typename WarpGradientCalculatorType::Pointer m_TrueWarpGradientCalculator;
   typename WarpGradientCalculatorType::Pointer m_CompWarpGradientCalculator;

public:
   void SetTrueField(const DeformationFieldType * truefield)
   {
      m_TrueField = truefield;

      m_TrueWarpGradientCalculator = WarpGradientCalculatorType::New();
      m_TrueWarpGradientCalculator->SetInputImage( m_TrueField );

      m_CompWarpGradientCalculator =  WarpGradientCalculatorType::New();
   }
   
   void Execute(itk::Object *caller, const itk::EventObject & event)
   {
      Execute( (const itk::Object *)caller, event);
   }

   void Execute(const itk::Object * object, const itk::EventObject & event)
   {
      if( !(itk::IterationEvent().CheckEvent( &event )) )
      {
         return;
      }

      typename DeformationFieldType::ConstPointer deffield = 0;
      unsigned int iter = -1;
      double metricbefore = -1.0;
      
      if ( const DiffeomorphicDemonsRegistrationFilterType * filter = 
           dynamic_cast< const DiffeomorphicDemonsRegistrationFilterType * >( object ) )
      {
         iter = filter->GetElapsedIterations() - 1;
         metricbefore = filter->GetMetric();
         deffield = const_cast<DiffeomorphicDemonsRegistrationFilterType *>
            (filter)->GetDeformationField();
      }
      else if ( const FastSymmetricForcesDemonsRegistrationFilterType * filter = 
           dynamic_cast< const FastSymmetricForcesDemonsRegistrationFilterType * >( object ) )
      {
         iter = filter->GetElapsedIterations() - 1;
         metricbefore = filter->GetMetric();
         deffield = const_cast<FastSymmetricForcesDemonsRegistrationFilterType *>
            (filter)->GetDeformationField();
      }
      else if ( const MultiResRegistrationFilterType * multiresfilter = 
           dynamic_cast< const MultiResRegistrationFilterType * >( object ) )
      {
         std::cout<<"Finished Multi-resolution iteration :"<<multiresfilter->GetCurrentLevel()-1<<std::endl;
         std::cout<<"=============================="<<std::endl<<std::endl;
      }
      else
      {
         return;
      }

      if (deffield)
      {
         std::cout<<iter<<": MSE "<<metricbefore<<" - ";

         double fieldDist = -1.0;
         double fieldGradDist = -1.0;
         double tmp;
         if (m_TrueField)
         {
            typedef itk::ImageRegionConstIteratorWithIndex<DeformationFieldType>
               FieldIteratorType;
            FieldIteratorType currIter(
               deffield, deffield->GetLargestPossibleRegion() );
            FieldIteratorType trueIter(
               m_TrueField, deffield->GetLargestPossibleRegion() );

            m_CompWarpGradientCalculator->SetInputImage( deffield );

            fieldDist = 0.0;
            fieldGradDist = 0.0;
            for ( currIter.GoToBegin(), trueIter.GoToBegin();
                  ! currIter.IsAtEnd(); ++currIter, ++trueIter )
            {
               fieldDist += (currIter.Value() - trueIter.Value()).GetSquaredNorm();

               // No need to add Id matrix here as we do a substraction
               tmp = (
                  ( m_CompWarpGradientCalculator->EvaluateAtIndex(currIter.GetIndex())
                    -m_TrueWarpGradientCalculator->EvaluateAtIndex(trueIter.GetIndex())
                     ).GetVnlMatrix() ).frobenius_norm();
               fieldGradDist += tmp*tmp;
            }
            fieldDist = sqrt( fieldDist/ (double)(
                     deffield->GetLargestPossibleRegion().GetNumberOfPixels()) );
            fieldGradDist = sqrt( fieldGradDist/ (double)(
                     deffield->GetLargestPossibleRegion().GetNumberOfPixels()) );
            
            std::cout<<"d(.,true) "<<fieldDist<<" - ";
            std::cout<<"d(.,Jac(true)) "<<fieldGradDist<<" - ";
         }
         
         m_HarmonicEnergyCalculator->SetImage( deffield );
         m_HarmonicEnergyCalculator->Compute();
         const double harmonicEnergy
            = m_HarmonicEnergyCalculator->GetHarmonicEnergy();
         std::cout<<"harmo. "<<harmonicEnergy<<" - ";

         
         m_JacobianFilter->SetInput( deffield );
         m_JacobianFilter->UpdateLargestPossibleRegion();

        
         const unsigned int numPix = m_JacobianFilter->
            GetOutput()->GetLargestPossibleRegion().GetNumberOfPixels();
         
         TPixel* pix_start = m_JacobianFilter->GetOutput()->GetBufferPointer();
         TPixel* pix_end = pix_start + numPix;

         TPixel* jac_ptr;

         // Get percentage of det(Jac) below 0
         unsigned int jacBelowZero(0u);
         for (jac_ptr=pix_start; jac_ptr!=pix_end; ++jac_ptr)
         {
            if ( *jac_ptr<=0.0 ) ++jacBelowZero;
         }
         const double jacBelowZeroPrc = static_cast<double>(jacBelowZero)
            / static_cast<double>(numPix);
         

         // Get min an max jac
         const double minJac = *(std::min_element (pix_start, pix_end));
         const double maxJac = *(std::max_element (pix_start, pix_end));

         // Get some quantiles
         // We don't need the jacobian image
         // we can modify/sort it in place
         jac_ptr = pix_start + static_cast<unsigned int>(0.002*numPix);
         std::nth_element(pix_start, jac_ptr, pix_end);
         const double Q002 = *jac_ptr;

         jac_ptr = pix_start + static_cast<unsigned int>(0.01*numPix);
         std::nth_element(pix_start, jac_ptr, pix_end);
         const double Q01 = *jac_ptr;

         jac_ptr = pix_start + static_cast<unsigned int>(0.99*numPix);
         std::nth_element(pix_start, jac_ptr, pix_end);
         const double Q99 = *jac_ptr;

         jac_ptr = pix_start + static_cast<unsigned int>(0.998*numPix);
         std::nth_element(pix_start, jac_ptr, pix_end);
         const double Q998 = *jac_ptr;
         

         std::cout<<"max|Jac| "<<maxJac<<" - "
                  <<"min|Jac| "<<minJac<<" - "
                  <<"ratio(|Jac|<=0) "<<jacBelowZeroPrc<<std::endl;
         
         

         if (this->m_Fid.is_open())
         {
            if (! m_headerwritten)
            {
               this->m_Fid<<"Iteration"
                          <<", MSE before"
                          <<", Harmonic energy"
                          <<", min|Jac|"
                          <<", 0.2% |Jac|"
                          <<", 01% |Jac|"
                          <<", 99% |Jac|"
                          <<", 99.8% |Jac|"
                          <<", max|Jac|"
                          <<", ratio(|Jac|<=0)";
               
               if (m_TrueField)
               {
                  this->m_Fid<<", dist(warp,true warp)"
                             <<", dist(Jac,true Jac)";
               }
               
               this->m_Fid<<std::endl;
               
               m_headerwritten = true;
            }
            
            this->m_Fid<<iter
                       <<", "<<metricbefore
                       <<", "<<harmonicEnergy
                       <<", "<<minJac
                       <<", "<<Q002
                       <<", "<<Q01
                       <<", "<<Q99
                       <<", "<<Q998
                       <<", "<<maxJac
                       <<", "<<jacBelowZeroPrc;

            if (m_TrueField)
            {
               this->m_Fid<<", "<<fieldDist
                          <<", "<<fieldGradDist;
            }
            
            this->m_Fid<<std::endl;
         }
      }
   }
   
protected:   
   CommandIterationUpdate() :
      m_Fid( "metricvalues.csv" ),
      m_headerwritten(false)
   {
      m_JacobianFilter = JacobianFilterType::New();
      m_JacobianFilter->SetUseImageSpacing( true );
      m_JacobianFilter->ReleaseDataFlagOn();
      
      m_Minmaxfilter = MinMaxFilterType::New();

      m_HarmonicEnergyCalculator = HarmonicEnergyCalculatorType::New();

      m_TrueField = 0;
      m_TrueWarpGradientCalculator = 0;
      m_CompWarpGradientCalculator = 0;
   };

   ~CommandIterationUpdate()
   {
      this->m_Fid.close();
   }
};




int main( int argc, char *argv[] )
{
  
  
  PARSE_ARGS;
   
  std::cout << "Moving Image: " <<  MovingImageFilename << std::endl; 
  std::cout << "Fixed Image: " <<  FixedImageFilename << std::endl; 
  //std::cout << "Input Deformation Field:" <<  InputFieldFilename << std::endl; 
  std::cout << "Input Transform Filename: " << InputXformFilename << std::endl;
  std::cout << "Output Image Filename: " << OutputImageFilename << std::endl; 
  std::cout << "Output Deformation Field: " << OutputFieldFilename <<std::endl;
  std::cout << "True Field Filename: " << TrueFieldFilename <<std::endl; 
  std::cout << "Number of Levels: " << NumLevels <<std::endl;
  std::cout << "Iterations[1]: " << Iterations[0];
  std::cout << "Iterations[2]: " << Iterations[1];
  std::cout << "Iterations[3]: " << Iterations[2]<< std::endl;
  std::cout << "Sigma Def: " << SigmaDef <<std::endl;
  std::cout << "Sigma Up: "  << SigmaUp <<std::endl;
  std::cout << "Max Step Length: " << MaxLength<<std::endl;
  std::cout << "update Rule: " <<UpdateRule <<std::endl;
  std::cout << "Gradient Type: " << Gradient<<std::endl;
  //std::cout << "Use Histogram Matching: " << UseHistogram <<std::endl;
  std::cout << "Verbosity: " << Verbosity <<std::endl;
  
  
 
 const    unsigned int    Dimension = 3;
 typedef float PixelType;
 typedef itk::Image< PixelType, Dimension >  ImageType;
 typedef itk::Vector< float, Dimension >    VectorPixelType;
 typedef itk::Image< VectorPixelType, Dimension > DeformationFieldType;

  
 // Set up the file readers
 typedef itk::ImageFileReader< ImageType > FixedImageReaderType;
 typedef itk::ImageFileReader< ImageType > MovingImageReaderType;
 //typedef itk::ImageFileReader< DeformationFieldType > FieldReaderType;
 typedef itk::TransformFileReader TransformReaderType;
 
 typedef itk::Vector<unsigned int, Dimension> IterationType; 
 IterationType numIterations;

 for (int i =0; i<3; i++)
 {
   numIterations[i] = static_cast<unsigned int>(Iterations[i]);
 }

 ImageType::Pointer fixedImage = ImageType::New();
 ImageType::Pointer movingImage = ImageType::New();
 DeformationFieldType::Pointer inputDefField = DeformationFieldType::New();
 
 FixedImageReaderType::Pointer  fixedImageReader  = FixedImageReaderType::New();
 MovingImageReaderType::Pointer movingImageReader = MovingImageReaderType::New();
 
 fixedImageReader->SetFileName( FixedImageFilename );
 movingImageReader->SetFileName( MovingImageFilename );
 
 
 
 // Update the reader
   try
   {
      std::cout << " Reading Fixed and Moving Images ....." << std::endl;
      fixedImageReader->Update();
      movingImageReader->Update();
     
   }
   catch( itk::ExceptionObject& err )
   {
      std::cout << "Could not read one of the input images." << std::endl;
      std::cout << err << std::endl;
      exit( EXIT_FAILURE );
   }
   
/* if (InputFieldFilename.length()> 0)
 {
      // Set up the file readers
      FieldReaderType::Pointer fieldReader = FieldReaderType::New();
      fieldReader->SetFileName(  InputFieldFilename );
      
      // Update the reader
      try
      {
         std::cout << " Reading Initial Deformation Field ....." << std::endl;
	 fieldReader->Update();
      }
      catch( itk::ExceptionObject& err )
      {
         std::cout << "Could not read the input field." << std::endl;
         std::cout << err << std::endl;
         exit( EXIT_FAILURE );
      }*/
      //DeformationFieldType::Pointer inputDefField = fieldReader->GetOutput();
   //   inputDefField->DisconnectPipeline();
 // }
   if ( InputXformFilename.length()> 0)
  {
   TransformReaderType::Pointer transformReader = TransformReaderType::New(); 
   transformReader->SetFileName(  InputXformFilename );
      
      // Update the reader
      try
      {
         std::cout << " Reading Initial Transform ....." << std::endl;
	 transformReader->Update();
	 
      }
      catch( itk::ExceptionObject& err )
      {
         std::cout << "Could not read the input transform." << std::endl;
         std::cout << err << std::endl;
         exit( EXIT_FAILURE );
      }
      
     typedef TransformReaderType::TransformType BaseTransformType;
     BaseTransformType* baseTrsf(0);

     const TransformReaderType::TransformListType* trsflistptr = transformReader->GetTransformList();
      if ( trsflistptr->empty() )
      {
         std::cout << "Could not read the input transform." << std::endl;
         exit( EXIT_FAILURE );
      }
      else if (trsflistptr->size()>1 )
      {
         std::cout << "The input transform file contains more than one transform, we use the first one." << std::endl;
         exit( EXIT_FAILURE );
      }

      baseTrsf = trsflistptr->front();
      if ( !baseTrsf )
      {
         std::cout << "Could not read the input transform." << std::endl;
         exit( EXIT_FAILURE );
      }
      

      // Set up the TransformToDeformationFieldFilter
      typedef itk::TransformToDeformationFieldFilter <DeformationFieldType> FieldGeneratorType;
      typedef FieldGeneratorType::TransformType TransformType;

      TransformType* trsf = dynamic_cast<TransformType*>(baseTrsf);
      if ( !trsf )
      {
         std::cout << "Could not cast input transform to a usable transform." << std::endl;
         exit( EXIT_FAILURE );
      }

     FieldGeneratorType::Pointer fieldGenerator = FieldGeneratorType::New();
      fieldGenerator->SetTransform( trsf );
      fieldGenerator->SetOutputRegion( fixedImageReader->GetOutput()->GetRequestedRegion());
      fieldGenerator->SetOutputSpacing(fixedImageReader->GetOutput()->GetSpacing());
      fieldGenerator->SetOutputOrigin( fixedImageReader->GetOutput()->GetOrigin());
      fieldGenerator->SetOutputDirection( fixedImageReader->GetOutput()->GetDirection());

      // Update the fieldGenerator
      try
      {
         fieldGenerator->Update();
      }
      catch( itk::ExceptionObject& err )
      {
         std::cout << "Could not generate the input field." << std::endl;
         std::cout << err << std::endl;
         exit( EXIT_FAILURE );
      }

     inputDefField = fieldGenerator->GetOutput();
     inputDefField->DisconnectPipeline();
   }
   

     std::cout << " Use Histogram Matching ....." << std::endl;
     
     typedef itk::HistogramMatchingImageFilter  <ImageType, ImageType> MatchingFilterType;
     MatchingFilterType::Pointer matcher = MatchingFilterType::New();

     matcher->SetInput( movingImageReader->GetOutput() );
     matcher->SetReferenceImage( fixedImageReader->GetOutput() );

     matcher->SetNumberOfHistogramLevels( 1024 );
     matcher->SetNumberOfMatchPoints( 7 );
     matcher->ThresholdAtMeanIntensityOn();

     // Update the matcher
      try
      {
         matcher->Update();
      }
      catch( itk::ExceptionObject& err )
      {
         std::cout << "Could not match the input images." << std::endl;
         std::cout << err << std::endl;
         exit( EXIT_FAILURE );
      }

      movingImage = matcher->GetOutput();
      movingImage->DisconnectPipeline();
      
      fixedImage = fixedImageReader->GetOutput();
      fixedImage->DisconnectPipeline();
  
  
  // Set up the demons filter
   typedef itk::PDEDeformableRegistrationFilter < ImageType, ImageType, DeformationFieldType>   BaseRegistrationFilterType;
   BaseRegistrationFilterType::Pointer filter =   BaseRegistrationFilterType::New();
   
    if (UpdateRule == 0)
    { 
    
    std::cout << "Use Diffeomorphic Registration" << std::endl;
     typedef itk::DiffeomorphicDemonsRegistrationFilter < ImageType, ImageType, DeformationFieldType>
         ActualRegistrationFilterType;
     typedef  ActualRegistrationFilterType::GradientType GradientType;    
     ActualRegistrationFilterType::Pointer actualfilter = ActualRegistrationFilterType::New();
     actualfilter->SetMaximumUpdateStepLength( MaxLength );
     actualfilter->SetUseGradientType( static_cast<GradientType>(Gradient) );
     filter = actualfilter;
     
     
    }
    
    else if (UpdateRule == 1)
    { 
     // s <- s + u (ITK basic implementation)
      typedef itk::FastSymmetricForcesDemonsRegistrationFilter   < ImageType, ImageType, DeformationFieldType>
         ActualRegistrationFilterType;
      typedef ActualRegistrationFilterType::GradientType GradientType;
      ActualRegistrationFilterType::Pointer actualfilter  = ActualRegistrationFilterType::New();
      actualfilter->SetMaximumUpdateStepLength( MaxLength );
      actualfilter->SetUseGradientType( static_cast<GradientType>(Gradient) );
      filter = actualfilter;   
    }
    
    else if (UpdateRule ==2)
    {
    // s <- s o (Id + u) (Diffeomorphic demons)
    // This is simply a crude diffeomorphic demons
    // where the exponential is computed in 0 iteration

    typedef itk::DiffeomorphicDemonsRegistrationFilter  < ImageType, ImageType, DeformationFieldType>
         ActualRegistrationFilterType;
    typedef ActualRegistrationFilterType::GradientType GradientType;     
    ActualRegistrationFilterType::Pointer actualfilter = ActualRegistrationFilterType::New();
    actualfilter->SetMaximumUpdateStepLength( MaxLength );
    actualfilter->SetUseGradientType( static_cast<GradientType>(Gradient) );
    actualfilter->UseFirstOrderExpOn();
    filter = actualfilter;
    }
    
    else
    {
      std::cout << "Unsupported update rule." << std::endl;
      exit( EXIT_FAILURE );
    }
    
   if ( SigmaDef > 0.1)
   {
      std::cout << " Smoothing is on ....." << std::endl;
      filter->SmoothDeformationFieldOn();
      filter->SetStandardDeviations( SigmaDef );
   }
   else
   {
      filter->SmoothDeformationFieldOff();
   }

   if ( SigmaUp > 0.1 )
   {
     std::cout << " Smoothing at update....." << std::endl;
      filter->SmoothUpdateFieldOn();
      filter->SetUpdateFieldStandardDeviations( SigmaUp );
   }
   else
   {
      filter->SmoothUpdateFieldOff();
   } 
   
   if ( Verbosity > 0 )
   {
   CommandIterationUpdate<PixelType, Dimension>::Pointer multiresobserver =
   CommandIterationUpdate<PixelType, Dimension>::New();
     filter->AddObserver( itk::IterationEvent(), multiresobserver );
   }
 
  
  // Set up the multi-resolution filter
   typedef itk::MultiResolutionPDEDeformableRegistration2< ImageType, ImageType, DeformationFieldType, PixelType >   MultiResRegistrationFilterType;
   MultiResRegistrationFilterType::Pointer multires = MultiResRegistrationFilterType::New();
   multires->SetRegistrationFilter( filter );
   multires->SetNumberOfLevels( NumLevels );
   multires->SetNumberOfIterations( &numIterations[0] );
   std::cout << " Iterations  : "<< numIterations[0] << std::endl;
   multires->SetFixedImage( fixedImage );
   multires->SetMovingImage( movingImage );
   multires->SetInitialDeformationField( inputDefField );
   
  if ( Verbosity > 0 )
   {
      // Create the Command observer and register it with the registration filter.
      CommandIterationUpdate<PixelType, Dimension>::Pointer multiresobserver1 =
         CommandIterationUpdate<PixelType, Dimension>::New();
      multires->AddObserver( itk::IterationEvent(), multiresobserver1 );
  } 
   
  // Compute the deformation field
   try
   {
      std::cout << " Computing the Deformation Field ....." << std::endl;
      multires->UpdateLargestPossibleRegion();
     
   }
   catch( itk::ExceptionObject& err )
   {
      std::cout << "Unexpected error." << std::endl;
      std::cout << err << std::endl;
      exit( EXIT_FAILURE );
   }

   // The outputs
  DeformationFieldType::Pointer defField = multires->GetOutput();
  defField->DisconnectPipeline();
  
 // warp the result
  typedef itk::WarpImageFilter < ImageType, ImageType, DeformationFieldType >  WarperType;
  WarperType::Pointer warper = WarperType::New();
  warper->SetInput( movingImage );
  warper->SetOutputSpacing( fixedImage->GetSpacing() );
  warper->SetOutputOrigin( fixedImage->GetOrigin() );
  warper->SetOutputDirection(fixedImage->GetDirection());
  warper->SetDeformationField( defField ); 
  
  // Write warped image out to file
   typedef PixelType OutputPixelType;
   typedef itk::Image< OutputPixelType, Dimension > OutputImageType;
   typedef itk::CastImageFilter
      < ImageType, OutputImageType > CastFilterType;
   typedef itk::ImageFileWriter< OutputImageType >  WriterType;
   
   WriterType::Pointer      writer =  WriterType::New();
   CastFilterType::Pointer  caster =  CastFilterType::New();
   writer->SetFileName( OutputImageFilename );
   caster->SetInput( warper->GetOutput() );
   writer->SetInput( caster->GetOutput()   );
   //writer->SetUseCompression( true );
   
   try
   {
      writer->Update();
      std::cout << " Writing output resampled image ....." << std::endl;
   }
   catch( itk::ExceptionObject& err )
   {
      std::cout << "Unexpected error." << std::endl;
      std::cout << err << std::endl;
      exit( EXIT_FAILURE );
   }
   
  // Write deformation field
   if (OutputFieldFilename.length() > 0)
   {
      // Write the deformation field as an image of vectors.
      // Note that the file format used for writing the deformation field must be
      // capable of representing multiple components per pixel. This is the case
      // for the MetaImage and VTK file formats for example.
      typedef itk::ImageFileWriter< DeformationFieldType > FieldWriterType;
      FieldWriterType::Pointer fieldWriter = FieldWriterType::New();
      fieldWriter->SetFileName( OutputFieldFilename );
      fieldWriter->SetInput( defField );
      //fieldWriter->SetUseCompression( true );
      
      try
      {
         fieldWriter->Update();
	 std::cout << " Writing output Deformation Field ....." << std::endl;
      }
      catch( itk::ExceptionObject& err )
      {
         std::cout << "Unexpected error." << std::endl;
         std::cout << err << std::endl;
         exit( EXIT_FAILURE );
      }
   }
   
         

  return 0;
}
   
