/*=========================================================================

 Program:   BRAINS (Brain Research: Analysis of Images, Networks, and Systems)
 Module:    $RCSfile: $
 Language:  TCL
 Date:      $Date: 2006/03/29 14:53:40 $
 Version:   $Revision: 1.9 $
 
   Copyright (c) Iowa Mental Health Clinical Research Center. All rights reserved.
   See BRAINSCopyright.txt or http://www.psychiatry.uiowa.edu/HTML/Copyright.html 
   for details.
 
      This software is distributed WITHOUT ANY WARRANTY; without even 
      the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
      PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkDwiToVectorImageFilter_txx
#define __itkDwiToVectorImageFilter_txx

#include <iostream>
#include <vector>

#include <itkImageRegionIterator.h>
#include <itkImageRegionConstIterator.h>
#include <itkOrientImageFilter.h>
#include <itkCastImageFilter.h>
#include <itkIOCommon.h>
#include <itkMetaDataObject.h>
#include <itkProgressAccumulator.h>


#include "itkDwiToVectorImageFilter.h"



namespace itk
{

template <class TInputImage, class TOutputImage>
DwiToVectorImageFilter<TInputImage, TOutputImage>
::DwiToVectorImageFilter()
{  	
  m_NumberDtiDirections = 1;	  
  m_RotateGradients = false;
}



template <class TInputImage, class TOutputImage>
void
DwiToVectorImageFilter<TInputImage, TOutputImage>
::SetDiffusionDirections ( MatrixType diffusionDirections )
{
  m_DiffusionDirections = diffusionDirections;
}




	
template <class TInputImage, class TOutputImage>
void
DwiToVectorImageFilter<TInputImage, TOutputImage>
::Update ( )
{

  InputImageRegionType region3D	= m_Input->GetLargestPossibleRegion();
  InputImageIndexType	 index3D	= region3D.GetIndex();
  InputImageSizeType	 size3D		= region3D.GetSize();  
  
  std::cout << size3D[2] << std::endl;
  const int numberOfSlices = size3D[2]/(m_NumberDtiDirections);
  size3D[2] = numberOfSlices;
  
  
  region3D.SetIndex(index3D);
  region3D.SetSize(size3D);
  
  
  
  typedef itk::Image<InputImagePixelType, InputImageDimension> TempImageType;
  typedef typename TempImageType::Pointer 	TempImagePointer;
  TempImagePointer tmpImage =  TempImageType::New();
  tmpImage->SetRegions( region3D );
  tmpImage->SetSpacing( m_Input->GetSpacing() );
  tmpImage->SetOrigin( m_Input->GetOrigin() );
  tmpImage->SetDirection( m_Input->GetDirection() );
  tmpImage->Allocate();
  
  std::cout << "step 1"<<std::endl;
  
  typedef itk::Image<OutputImageValueType, OutputImageDimension> VectorIndexImageType;
  typedef typename VectorIndexImageType::Pointer 	VectorImageIndexPointer;
  VectorImageIndexPointer vectorIndexImage;
  
  IteratorType tempIt( tmpImage, tmpImage->GetRequestedRegion() );
  ConstIteratorType inputIt( m_Input, m_Input->GetRequestedRegion() );
  OutputImagePixelType vectorImagePixel ( m_NumberDtiDirections );
  
  /* Should Progress Information be added ???? */
  int tmpIndex = 0;
  inputIt.GoToBegin();
  
   std::cout << "step 2"<<std::endl;
  
  while ( !inputIt.IsAtEnd() )
  {
    /* Replace with Extract Image Region Filter */
    for (tempIt.GoToBegin(); !tempIt.IsAtEnd(); ++tempIt, ++inputIt)
    {  
    tempIt.Set( inputIt.Value() );       
    }

    typedef CastImageFilter<InputImageType, VectorIndexImageType> CastFilterType;
    typedef typename CastFilterType::Pointer 	CastFilterPointer;
    CastFilterPointer CastImageFilter = CastFilterType::New();
    CastImageFilter->SetInput( tmpImage );
    CastImageFilter->Update( );
    vectorIndexImage = CastImageFilter->GetOutput(); 
    vectorIndexImage->SetDirection( m_Input->GetDirection() );
 
    
    if ( tmpIndex == 0)
      {
      std::cout << "Allocate Image: " << vectorIndexImage->GetLargestPossibleRegion() << std::endl;
      m_Output = OutputImageType::New();
      m_Output->SetRegions( vectorIndexImage->GetLargestPossibleRegion() );
      m_Output->SetSpacing( vectorIndexImage->GetSpacing() );
      m_Output->SetOrigin( vectorIndexImage->GetOrigin() );
      m_Output->SetDirection( vectorIndexImage->GetDirection() );
      m_Output->SetVectorLength( m_NumberDtiDirections );
      m_Output->Allocate();
      }
    
    
    typedef ImageRegionConstIterator<VectorIndexImageType>  ConstVectorIndexIteratorType; 
    ConstVectorIndexIteratorType vectorIndexIt( vectorIndexImage, vectorIndexImage->GetRequestedRegion() );
    VectorIteratorType outputIt( m_Output, m_Output->GetRequestedRegion() );
    for (outputIt.GoToBegin(),vectorIndexIt.GoToBegin(); !outputIt.IsAtEnd(); ++outputIt, ++vectorIndexIt)
      {
      vectorImagePixel = outputIt.Get( );
      vectorImagePixel[tmpIndex] = vectorIndexIt.Value();
      outputIt.Set( vectorImagePixel );
  
      }
    tmpIndex++;   
     
  } 
  
   std::cout << "b"<<std::endl;
  SetMetaDataHeader();
  
}


template <class TInputImage, class TOutputImage>
void
DwiToVectorImageFilter<TInputImage, TOutputImage>
::SetMetaDataHeader ( )
{
  itk::MetaDataDictionary meta;
  
  std::string NrrdValue;
  NrrdValue = "cell";
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_centerings[0]",NrrdValue);  
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_centerings[1]",NrrdValue);
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_centerings[2]",NrrdValue);
  NrrdValue = "none";
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_centerings[3]",NrrdValue);
  
  NrrdValue = "space";
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_kinds[0]",NrrdValue);  
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_kinds[1]",NrrdValue);
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_kinds[2]",NrrdValue);  
  NrrdValue = "list";
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_kinds[3]",NrrdValue);
  
  NrrdValue = "mm";
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_space units[0]",NrrdValue);
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_space units[1]",NrrdValue);
  itk::EncapsulateMetaData<std::string>(meta,"NRRD_space units[2]",NrrdValue);
  
  double spacing = (m_Output->GetSpacing())[2];
  itk::EncapsulateMetaData<double>(meta,"NRRD_thicknesses[2]",spacing);
  
  
  std::vector<std::vector<double> > msrFrame(3);
  for (unsigned int saxi=0; saxi < 3; saxi++) 
    {
    msrFrame[saxi].resize(3);
    for (unsigned int saxj=0; saxj < 3; saxj++)
      {
      msrFrame[saxi][saxj] = 0.0;
      }
    }
  msrFrame[0][0] = 1.0; msrFrame[1][1] = 1.0; msrFrame[2][2] = 1.0;   
  itk::EncapsulateMetaData<std::vector<std::vector<double> > >(meta,"NRRD_measurement frame",msrFrame);      
  
  
  NrrdValue = "DWMRI";
  itk::EncapsulateMetaData<std::string>(meta,"modality",NrrdValue);
  
  char tmpStr[64]; 
  sprintf(tmpStr,"%18.15lf", m_BValue);
  NrrdValue = tmpStr;
  itk::EncapsulateMetaData<std::string>(meta, "DWMRI_b-value", NrrdValue);
  
  /* We should apply direction cosines to gradient directions if requested by the user */
  for (int i=0;i<m_NumberDtiDirections;i++)
    {
    NrrdValue.clear( );
    
    /* Rotate Diffusion Directions */ 
    vnl_vector<double> curGradientDirection(3);
    for (int k=0;k<3;k++)
      {
      curGradientDirection[k] = m_DiffusionDirections[i][k];
      }
    
    if (m_RotateGradients)
      {
      curGradientDirection = m_Input->GetDirection().GetVnlMatrix() * curGradientDirection;
      }

    for (int k=0;k<3;k++)
      {
      sprintf(tmpStr," %18.15lf", curGradientDirection[k]);
      NrrdValue += tmpStr;
      }
    sprintf(tmpStr,"DWMRI_gradient_%04d", i);
    itk::EncapsulateMetaData<std::string>(meta, tmpStr, NrrdValue);
    }
  
  m_Output->SetMetaDataDictionary(meta);  
}

}// end namespace itk
#endif
