

#ifndef __itkVectorImageInterpolateFunction_h
#define __itkVectorImageInterpolateFunction_h

#include "itkImageFunction.h"
#include "itkVariableLengthVector.h"


namespace itk
{

template <
class TInputImage,
class TCoordRep = float,
class TPixelType = ITK_TYPENAME TInputImage::PixelType
>
class ITK_EXPORT VectorImageInterpolateFunction : 
  public ImageFunction<
    TInputImage, 
    VariableLengthVector< ITK_TYPENAME NumericTraits<typename TPixelType::ValueType>::RealType>,     
    TCoordRep > 
{
public:
  
 /* VectorImageInterpolateImageFunction is the base for all ImageFunctions that
 * interpolates image with variable length vector pixel types. This function outputs
 * a return value of type VariableLengthVector<double>.
 *
 * This class is templated input image type and the coordinate
 * representation type.
 
   This class can only be used for images with variableLengthVector. 
   We have implemented it using itk::VectorImage */

  
  
  /** Dimension underlying input image. */
  itkStaticConstMacro(ImageDimension, unsigned int,
                      TInputImage::ImageDimension);

  /** Standard class typedefs. */
  typedef VectorImageInterpolateFunction Self;
  typedef ImageFunction<TInputImage,
   VariableLengthVector<double>, TCoordRep > Superclass;
  typedef SmartPointer<Self> Pointer;
  typedef SmartPointer<const Self>  ConstPointer;
  
  /** Run-time type information (and related methods). */
  itkTypeMacro(VectorImageInterpolateFunction, ImageFunction);

  /** InputImageType typedef support. */
  typedef typename Superclass::InputImageType InputImageType;
  typedef typename InputImageType::PixelType  PixelType;
  typedef typename PixelType::ValueType       ValueType;
  typedef typename NumericTraits<ValueType>::RealType  RealType;
    

  /** Point typedef support. */
  typedef typename Superclass::PointType PointType;

  /** Index typedef support. */
  typedef typename Superclass::IndexType IndexType;

  /** ContinuousIndex typedef support. */
  typedef typename Superclass::ContinuousIndexType ContinuousIndexType;

  /** Output type is VariableLengthVector<RealType>. */
  typedef typename Superclass::OutputType OutputType;

  /** CoordRep typedef support. */
  typedef TCoordRep CoordRepType;

  virtual OutputType Evaluate( const PointType& point ) const
    {
    ContinuousIndexType index;
    this->GetInputImage()->TransformPhysicalPointToContinuousIndex( point, index );
    return ( this->EvaluateAtContinuousIndex( index ) );
    }

  /** Interpolate the image at a continuous index position
   *
   * Returns the interpolated image intensity at a 
   * specified index position. No bounds checking is done.
   * The point is assume to lie within the image buffer.
   *
   * Subclasses must override this method.
   *
   * ImageFunction::IsInsideBuffer() can be used to check bounds before
   * calling the method. */
  virtual OutputType EvaluateAtContinuousIndex( 
    const ContinuousIndexType & index ) const = 0;

  /** Interpolate the image at an index position.
   * Simply returns the image value at the
   * specified index position. No bounds checking is done.
   * The point is assume to lie within the image buffer.
   *
   * ImageFunction::IsInsideBuffer() can be used to check bounds before
   * calling the method. */
  virtual OutputType EvaluateAtIndex( const IndexType & index ) const
    {
    unsigned int Dim = this->GetInputImage()->GetVectorLength();
    OutputType output(Dim);
    PixelType input = this->GetInputImage()->GetPixel( index );
   
    for( unsigned int k = 0; k < Dim; k++ )
      {
      output[k] = static_cast<double>( input[k] );
      }
    return ( output );
    }

protected:
  VectorImageInterpolateFunction() {}
  ~VectorImageInterpolateFunction() {}
  void PrintSelf(std::ostream& os, Indent indent) const
    { Superclass::PrintSelf( os, indent ); }

private:
  VectorImageInterpolateFunction(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

};

} // end namespace itk

#endif


