//  Software Guide : BeginLatex
//
//  Probably the most common representation of datasets in clinical
//  applications is the one that uses sets of DICOM slices in order to compose
//  tridimensional images. This is the case for CT, MRI and PET scanners. It is
//  very common therefore for image analysists to have to process volumetric
//  images that are stored in the form of a set of DICOM files belonging to a
//  common DICOM series. 
//
//  The following example illustrates how to use ITK functionalities in order
//  to read a DICOM series into a volume and then save this volume in another
//  file format.
//
//  Software Guide : EndLatex 


#include "itkImageSeriesReader.h"
#include "itkDICOMImageIO2.h"
#include "itkDICOMSeriesFileNames.h"
#include "itkImageFileWriter.h"
#include <itkGDCMImageIO.h>
#include <itkGDCMSeriesFileNames.h>
#include "itkOrientImageFilter.h"
#include "Dicom2nrrd_T1CLP.h"



int main( int argc, char* argv[] )
{

PARSE_ARGS;
  std::cout << "Input Volume (Dicom Dir) " << InputVolume << std::endl;
  std::cout << "Output Volume (nifti/nrrd) " << OutputVolume << std::endl;
  

  typedef signed short PixelType;
  const unsigned int dimension = 3;
  typedef itk::Image<PixelType,dimension>		DicomSeriesImageType;
  typedef DicomSeriesImageType::Pointer			DicomSeriesImagePointer;
  DicomSeriesImagePointer DicomImage;  
  

  
  itk::GDCMSeriesFileNames::Pointer FileNameGenerator = itk::GDCMSeriesFileNames::New();
  FileNameGenerator->SetUseSeriesDetails( false );
  FileNameGenerator->SetDirectory( InputVolume );

  typedef const std::vector<std::string>			ContainerType;
  const ContainerType & seriesUIDs = FileNameGenerator->GetSeriesUIDs();


  typedef itk::ImageSeriesReader< DicomSeriesImageType >     ReaderType;
  ReaderType::Pointer reader = ReaderType::New();
  itk::GDCMImageIO::Pointer dicomIO = itk::GDCMImageIO::New();		

  reader->SetFileNames( FileNameGenerator->GetFileNames( seriesUIDs[0] ) );
  reader->SetImageIO( dicomIO );			  
  try
  {
    reader->Update();
  }
  catch (itk::ExceptionObject &ex)
  {
    std::cout << ex << std::endl;
    throw;
  }
  DicomImage	= reader->GetOutput();  
  std::cout << DicomImage << std::endl;
  
  //ORIENT IMAGE
    typedef itk::OrientImageFilter< DicomSeriesImageType, DicomSeriesImageType > FilterType;
    FilterType::Pointer filter = FilterType::New();
    filter->SetInput( DicomImage );
    filter->UseImageDirectionOn();
    filter->SetDesiredCoordinateOrientation(itk::SpatialOrientation::ITK_COORDINATE_ORIENTATION_RPI);
   
    filter->Update();
   
    DicomSeriesImageType::Pointer image = filter->GetOutput();
  

    typedef itk::ImageFileWriter< DicomSeriesImageType > WriterType;
    WriterType::Pointer writer = WriterType::New();

    std::cout  << "Writing the image as " << std::endl << std::endl;
    std::cout  << OutputVolume << std::endl << std::endl;

    writer->SetFileName( OutputVolume );
    
    writer->SetInput( image );
    std::cout << image <<std::endl;

    try
      {
      writer->Update();
      }
    catch (itk::ExceptionObject &ex)
      {
      std::cout << ex;
      return EXIT_FAILURE;
      }
  

  return EXIT_SUCCESS;

}
