/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkTensorReorientImageFilter.txx,v $
  Language:  C++
  Date:      $Date: 2005/07/27 15:21:12 $
  Version:   $Revision: 1.25 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkTensorReorientImageFilter_txx
#define __itkTensorReorientImageFilter_txx
#include "itkTensorReorientImageFilter.h"

#include "itkImageRegionIterator.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkNumericTraits.h"
#include "itkProgressReporter.h"
#include "itkMatrix.h"
#include "itkDiffusionTensor3D.h"
#include "vnl/vnl_math.h"
#include "vnl/vnl_matrix.h"
#include "itkNeighborhoodAlgorithm.h"
#include "itkConstantBoundaryCondition.h"
#include <vnl/algo/vnl_svd.h>



namespace itk
{

/**
 * Default constructor.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::TensorReorientImageFilter()
{
  // Setup the number of required inputs
  this->SetNumberOfRequiredInputs( 2 );  
  
  // Setup default values
  m_OutputSpacing.Fill( 1.0 );
  m_OutputOrigin.Fill( 0.0 );
  
  for(int i=0;i<6;i++)
  {
	  m_EdgePaddingValue[i] = 0.0;// NumericTraits<PixelType>::Zero;
  }
  for (int i = 0; i < ImageDimension; i++)
    {
    m_NeighborhoodRadius[i] = 1; // radius of neighborhood we will use
    }
  
  // Setup default interpolator
  typename DefaultInterpolatorType::Pointer interp =
    DefaultInterpolatorType::New();

  m_Interpolator = 
    static_cast<InterpolatorType*>( interp.GetPointer() );

}

/**
 * Standard PrintSelf method.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::PrintSelf(std::ostream& os, Indent indent) const
{

  Superclass::PrintSelf(os, indent);

  os << indent << "OutputSpacing: " << m_OutputSpacing << std::endl;;
  os << indent << "OutputOrigin: " << m_OutputOrigin << std::endl;
  os << indent << "EdgePaddingValue: "
     << static_cast<typename NumericTraits<PixelType>::PrintType>(m_EdgePaddingValue)
     << std::endl;
  os << indent << "Interpolator: " << m_Interpolator.GetPointer() << std::endl;
  
}


/**
 * Set the output image spacing.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetOutputSpacing(
  const double* spacing)
{
  SpacingType s(spacing);
  this->SetOutputSpacing( s );
}


/**
 * Set the output image origin.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetOutputOrigin(
  const double* origin)
{
  PointType p(origin);
  this->SetOutputOrigin(p);
}



/**
 * Set deformation field as Inputs[1] for this ProcessObject.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetDeformationField(
  const DeformationFieldType * field )
{
  // const cast is needed because the pipeline is not const-correct.
  DeformationFieldType * input =  
       const_cast< DeformationFieldType * >( field );
  this->ProcessObject::SetNthInput( 1, input );
}


/**
 * Return a pointer to the deformation field.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
typename TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::DeformationFieldType *
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::GetDeformationField(void)
{
  return static_cast<DeformationFieldType *>
    ( this->ProcessObject::GetInput( 1 ));
}


/**
 * Setup state of filter before multi-threading.
 * InterpolatorType::SetInputImage is not thread-safe and hence
 * has to be setup before ThreadedGenerateData
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::BeforeThreadedGenerateData()
{
  
  if( !m_Interpolator )
    {
    itkExceptionMacro(<< "Interpolator not set");
    }

  // Connect input image to interpolator
  m_Interpolator->SetInputImage( this->GetInput() );

}

/**
 * Setup state of filter after multi-threading.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::AfterThreadedGenerateData()
{ 
 
  // Disconnect input image from interpolator
  m_Interpolator->SetInputImage( NULL );

}


/**
 * Compute the output for the region specified by outputRegionForThread.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::ThreadedGenerateData(
  const OutputImageRegionType& outputRegionForThread,
  int threadId )
{
  
  InputImageConstPointer inputPtr = this->GetInput();
  OutputImagePointer outputPtr = this->GetOutput();  
  DeformationFieldPointer fieldPtr = this->GetDeformationField();

  // support progress methods/callbacks
  ProgressReporter progress(this, threadId, outputRegionForThread.GetNumberOfPixels());
  
  // iterator for the output image
  ImageRegionIteratorWithIndex<OutputImageType> outputIt(
    outputPtr, outputRegionForThread );

  // iterator for the deformation field
  ImageRegionIterator<DeformationFieldType> fieldIt(
    fieldPtr, outputRegionForThread );
    
  IndexType index;
  PointType point;
  DisplacementType displacement;
  vnlMatrixType J;
  
    // Define Boundary condition  
    boundaryConditionType cbc;    
    ConstNeighborhoodIteratorType
    bit( m_NeighborhoodRadius, fieldPtr, outputRegionForThread);
    
    cbc.SetConstant(static_cast<VectorType>(0.0));
    bit.OverrideBoundaryCondition(&cbc);
    bit.GoToBegin(); 
     
   
    while ( ! bit.IsAtEnd() )
     {
     
   index = outputIt.GetIndex();
  
   
   outputPtr->TransformIndexToPhysicalPoint( index, point );

   displacement = fieldIt.Get();               
 
    // compute the required input image point
    for(unsigned int j = 0; j < ImageDimension; j++ )
      {
      point[j] += displacement[j];
      }
      
      for (int i = 0; i < 3; ++i)
      {
      for (int j = 0; j < 3; ++j)
        {
       
	 J(j,i) =  0.5 * (bit.GetPrevious(i)[j]-bit.GetNext(i)[j] );
     
        }
      }
      
    
      
    
      vnlMatrixType iden;
      iden.set_identity(); 
      // Compute Singluar value Decompostion of the Jacobian to find the Rotation Matrix. 
      vnl_svd<CompType> svd(J + iden);     
      MatrixType  rotationMatrix(svd.U() * svd.V().transpose());
      
      
      
     
    
      if( m_Interpolator->IsInsideBuffer( point ) )
      {
               
          

    // get the interpolated value
  
      
      typedef typename InterpolatorType::OutputType OutputType;
      const OutputType interpolatedValue =m_Interpolator->Evaluate(point);
      //std::cout << "interpolated point= " << interpolatedValue << std::endl;
      PixelType value;
      for(unsigned int k =0; k<6; k++)
      {
         value[k] = static_cast<ValueType>(interpolatedValue[k]);
        //std::cout << "Value : " << value << std::endl;
      }
     
    
	     
	     MatrixType tensorD;
                tensorD(0,0)= value[0];  		tensorD(0,1)= value[1];		tensorD(0,2)= value[2]; 
		tensorD(1,0)= value[1]; 		tensorD(1,1)= value[3];		tensorD(1,2)= value[4];
		tensorD(2,0)= value[2];		        tensorD(2,1)= value[4];		tensorD(2,2)= value[5];	
		
		
	       
		vnlMatrixType OutMatrix = rotationMatrix*tensorD*rotationMatrix.GetTranspose();
		

	    PixelType outputValue;
		
		outputValue[0] = OutMatrix(0,0);
		outputValue[1] = OutMatrix(0,1);
		outputValue[2] = OutMatrix(0,2);
		outputValue[3] = OutMatrix(1,1);
		outputValue[4] = OutMatrix(1,2);
		outputValue[5] = OutMatrix(2,2);	
	
      outputIt.Set( outputValue );
       }
    else
      {
      outputIt.Set( m_EdgePaddingValue );
      }   
    ++outputIt;
    ++fieldIt; 
    ++bit;
    progress.CompletedPixel();
    }

}


template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::GenerateInputRequestedRegion()
{

  // call the superclass's implementation
  Superclass::GenerateInputRequestedRegion();

  // request the largest possible region for the input image
  InputImagePointer inputPtr = 
    const_cast< InputImageType * >( this->GetInput() );

  if( inputPtr )
    {
    inputPtr->SetRequestedRegionToLargestPossibleRegion();
    }

  // just propagate up the output requested region for the 
  // deformation field.
  DeformationFieldPointer fieldPtr = this->GetDeformationField();
  OutputImagePointer outputPtr = this->GetOutput();
  if( fieldPtr )
    {
    fieldPtr->SetRequestedRegion( outputPtr->GetRequestedRegion() );
    }

}


template <class TInputImage,class TOutputImage,class TDeformationField>
void
TensorReorientImageFilter<TInputImage,TOutputImage,TDeformationField>
::GenerateOutputInformation()
{


  // call the superclass's implementation of this method
  Superclass::GenerateOutputInformation();

  OutputImagePointer outputPtr = this->GetOutput();
  
  typename TOutputImage::RegionType outputLargestPossibleRegion;
  outputLargestPossibleRegion.SetSize( m_Size );
  //outputLargestPossibleRegion.SetIndex( m_OutputStartIndex );
  outputPtr->SetLargestPossibleRegion( outputLargestPossibleRegion );

  outputPtr->SetSpacing( m_OutputSpacing );
  outputPtr->SetOrigin( m_OutputOrigin );

}


} // end namespace itk

#endif
 
