/*=========================================================================

 Program:   BRAINS (Brain Research: Analysis of Images, Networks, and Systems)
 Module:    $RCSfile: $
 Language:  C++
 Date:      $Date: 2006/03/29 14:53:40 $
 Version:   $Revision: 1.9 $

   Copyright (c) Iowa Mental Health Clinical Research Center. All rights reserved.
   See BRAINSCopyright.txt or http://www.psychiatry.uiowa.edu/HTML/Copyright.html
   for details.

      This software is distributed WITHOUT ANY WARRANTY; without even
      the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
      PURPOSE.  See the above copyright notices for more information.

  Forked from Dartmouth2nrrd.cxx from the NAMIC Sandbox

=========================================================================*/

#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#include <vector>
#include <string>
#include <algorithm>
#include <iostream>

#include <metaCommand.h>
#include <itkImage.h>
#include <itkVectorImage.h>
#include <itkImageSeriesReader.h>
#include <itkGDCMImageIO.h>
#include <itkGDCMSeriesFileNames.h>
#include <itkExceptionObject.h>
#include <itkImageFileWriter.h>

#include "itkGtractParameterIO.h"
#include "itkGtractImageIO.h"
#include "itkDwiToVectorImageFilter.h"

#include "Dicom2NrrdCLP.h"

int main(int argc, char* argv[])
{

  PARSE_ARGS;

  bool debug=true;
  if (debug)
    {
    std::cout << "Dicom Directory: " <<  dicomDirectory << std::endl;
    std::cout << "DTI Parameter File: " << dtiParameterFile <<std::endl;
    std::cout << "Rotate Gradients: " << rotateGradients <<std::endl;
    std::cout << "Flip flags (Z, Y, X): " << flipZ << ", " << flipY << ", " << flipX <<std::endl;
    std::cout << "Output Nrrd Volume: " <<  outputVolume << std::endl << std::endl;
    }

  bool violated=false;
  if (dicomDirectory.size() == 0) { violated = true; std::cout << "  --dicomDirectory Required! "  << std::endl; }
  if (dtiParameterFile.size() == 0) { violated = true; std::cout << "  --dtiParameterFile Required! "  << std::endl; }
  if (outputVolume.size() == 0) { violated = true; std::cout << "  --outputVolume Required! "  << std::endl; }
  if (violated) exit(1);

  typedef signed short PixelType;
  const unsigned int dimension = 3;
  typedef itk::Image<PixelType,dimension>		DicomSeriesImageType;
  typedef DicomSeriesImageType::Pointer			DicomSeriesImagePointer;
  DicomSeriesImagePointer DicomImage;

  typedef signed short VectorPixelType;
  //typedef itk::VariableLengthVector<signed short> VectorPixelType;
  typedef itk::VectorImage<VectorPixelType,dimension>	NrrdImageType;
  typedef NrrdImageType::Pointer				        NrrdImagePointer;
  NrrdImagePointer DwiImage;



  itk::GDCMSeriesFileNames::Pointer FileNameGenerator = itk::GDCMSeriesFileNames::New();
  FileNameGenerator->SetUseSeriesDetails( false );
  FileNameGenerator->SetDirectory( dicomDirectory );

  typedef const std::vector<std::string>			ContainerType;
  const ContainerType & seriesUIDs = FileNameGenerator->GetSeriesUIDs();


  typedef itk::ImageSeriesReader< DicomSeriesImageType >     ReaderType;
  ReaderType::Pointer reader = ReaderType::New();
  itk::GDCMImageIO::Pointer dicomIO = itk::GDCMImageIO::New();		

  reader->SetFileNames( FileNameGenerator->GetFileNames( seriesUIDs[0] ) );
  reader->SetImageIO( dicomIO );			
  try
  {
    reader->Update();
  }
  catch (itk::ExceptionObject &ex)
  {
    std::cout << ex << std::endl;
    throw;
  }
  DicomImage	= reader->GetOutput();
  //std::cout << DicomImage << std::endl;

  /* Keep this until GDCM changes are implemented */
  itk::GtractParameterIO::Pointer parameterIO =  itk::GtractParameterIO::New();
  parameterIO->SetFileName( dtiParameterFile );
  try
  {
    parameterIO->Update( );
  }
  catch (itk::ExceptionObject &ex)
  {
    std::cout << ex << std::endl;
    throw;
  }

  int NumberOfDirections = parameterIO->GetNumberOfDirections( );

  DicomSeriesImageType::SizeType	size = DicomImage->GetLargestPossibleRegion().GetSize();
  if ( size[2]  % NumberOfDirections != 0)
  {
    std::cout << "Error: Parameter file does not match the image " << std::endl;
    return (1);				
  }

  /* Update itk::DwiToVectorImageFilter to eliminate image reorient
      and add rotation of gradient directions if requested by the user */

  typedef itk::DwiToVectorImageFilter< DicomSeriesImageType, NrrdImageType > FormatDicomFilterType;
  FormatDicomFilterType::Pointer FormatDicomFilter = FormatDicomFilterType::New();
  FormatDicomFilter->SetInput( DicomImage );
  FormatDicomFilter->SetNumberDtiDirections( NumberOfDirections );
  FormatDicomFilter->SetFlipZ( flipZ );
  FormatDicomFilter->SetFlipY( flipY );
  FormatDicomFilter->SetFlipX( flipX );
  FormatDicomFilter->SetDiffusionDirections( parameterIO->GetDirections( ) );
  FormatDicomFilter->SetBValue( parameterIO->GetBvalue( ) );
  if ( rotateGradients == "on" )
    {
    FormatDicomFilter->SetRotateGradients( true );
    }
  else
    {
    FormatDicomFilter->SetRotateGradients( false );
    }

  try
    {
    FormatDicomFilter->Update( );
    }
  catch (itk::ExceptionObject &ex)
    {
    std::cout << ex << std::endl;
    throw;
    }

  DwiImage = FormatDicomFilter->GetOutput( );


  typedef itk::ImageFileWriter<NrrdImageType> WriterType;
  WriterType::Pointer nrrdWriter = WriterType::New();
  nrrdWriter->UseInputMetaDataDictionaryOn();
  nrrdWriter->SetInput( DwiImage );
  nrrdWriter->SetFileName( outputVolume );
  try
    {
    nrrdWriter->Update();
    }
  catch (itk::ExceptionObject e)
    {
    std::cout << e << std::endl;
    }

  return 0;
}

