/*=========================================================================

 Program:   GTRACT (Guided Tensor Restore Anatomical Connectivity Tractography)
 Module:    $RCSfile: $
 Language:  C++
 Date:      $Date: 2006/03/29 14:53:40 $
 Version:   $Revision: 1.9 $
 
   Copyright (c) University of Iowa Department of Radiology. All rights reserved.
   See GTRACT-Copyright.txt or http://mri.radiology.uiowa.edu/copyright/GTRACT-Copyright.txt 
   for details.
 
      This software is distributed WITHOUT ANY WARRANTY; without even 
      the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
      PURPOSE.  See the above copyright notices for more information.

=========================================================================*/

#include <iostream>
#include <fstream>

#include <itkImage.h>
#include <itkVectorImage.h>
#include <itkImageFileWriter.h>
#include <itkImageFileReader.h>
#include <itkExceptionObject.h>
#include <itkVectorImageToImageAdaptor.h>

#include "itkVectorImageRegisterVersorRigidFilter.h"
#include "itkVectorImageRegisterAffineFilter.h"

#include "gtractCoregBvaluesCLP.h"

int main (int argc, char **argv)
{  
  
  PARSE_ARGS;

  bool debug=true;
  if (debug) 
    {
    std::cout << "=====================================================" << std::endl; 
    std::cout << "Moving Image: " <<  movingVolume << std::endl; 
    std::cout << "Fixed Image: " <<  fixedVolume << std::endl; 
    std::cout << "Fixed Image Index: " <<  fixedVolumeIndex << std::endl; 
    std::cout << "Output Image: " <<  outputVolume << std::endl; 
    std::cout << "Output Transform: " <<  outputTransform << std::endl; 
    std::cout << "Eddy Current Correction: " << eddyCurrentCorrection <<std::endl;
    std::cout << "Iterations: " << numberOfIterations <<std::endl;
    std::cout << "Translation Scale: " << spatialScale <<std::endl;
    std::cout << "Maximum Step Size: " << maximumStepSize <<std::endl;
    std::cout << "Minimum Step Size: " << minimumStepSize <<std::endl;
    std::cout << "Relaxation Factor: " << relaxationFactor <<std::endl;
    std::cout << "Samples: " << numberOfSpatialSamples << std::endl;
    std::cout << "=====================================================" << std::endl; 
    }  
  
  bool violated=false;
  if (movingVolume.size() == 0) { violated = true; std::cout << "  --movingVolume Required! "  << std::endl; }
  if (fixedVolume.size() == 0) { violated = true; std::cout << "  --fixedVolume Required! "  << std::endl; }
  if (outputVolume.size() == 0) { violated = true; std::cout << "  --outputVolume Required! "  << std::endl; }
  if (outputTransform.size() == 0) { violated = true; std::cout << "  --outputTransform Required! "  << std::endl; }
  if (violated) exit(1);

  
  typedef signed short                      PixelType;
  typedef itk::VectorImage<PixelType,3>	      NrrdImageType;
  typedef itk::Image<PixelType,3>	      IndexImageType;
  
    
  typedef itk::ImageFileReader < NrrdImageType, 
                                  itk::DefaultConvertPixelTraits< PixelType > > FileReaderType;
  FileReaderType::Pointer movingImageReader = FileReaderType::New();
  movingImageReader->SetFileName( movingVolume );
  movingImageReader->Update();

  try 
    {			
    movingImageReader->Update();
    }
  catch (itk::ExceptionObject &ex)
    {
    std::cout << ex << std::endl;
    throw;
    }

  typedef itk::ImageFileReader < NrrdImageType, 
                                  itk::DefaultConvertPixelTraits< PixelType > > FileReaderType;
  FileReaderType::Pointer fixedImageReader = FileReaderType::New();
  fixedImageReader->SetFileName( fixedVolume );
  fixedImageReader->Update();

  try 
    {			
    fixedImageReader->Update();
    }
  catch (itk::ExceptionObject &ex)
    {
    std::cout << ex << std::endl;
    throw;
    }    
  
  
  /* Extract Image Index to be used for Coregistration */  
  typedef itk::VectorIndexSelectionCastImageFilter<NrrdImageType,IndexImageType> FixedImageType;
  FixedImageType::Pointer fixedImageExtractionFilter = FixedImageType::New();
  fixedImageExtractionFilter->SetIndex( fixedVolumeIndex );
  fixedImageExtractionFilter->SetInput( fixedImageReader->GetOutput() );
  fixedImageExtractionFilter->Update(  );
      
  /* Pointer Used to Hold the Resulting Coregistered Image */
  NrrdImageType::Pointer   RegisteredImage;
  
  if ( eddyCurrentCorrection == 0 )
    {
    std::cout << "Rigid Registration: " << std::endl; 
    typedef itk::VectorImageRegisterVersorRigidFilter<NrrdImageType, NrrdImageType> RegisterFilterType;
    RegisterFilterType::Pointer registerImageFilter = RegisterFilterType::New();

    registerImageFilter->SetTranslationScale( spatialScale );
    registerImageFilter->SetMaximumStepLength( maximumStepSize );
    registerImageFilter->SetMinimumStepLength( minimumStepSize );
    registerImageFilter->SetRelaxationFactor( relaxationFactor );
    registerImageFilter->SetNumberOfIterations( numberOfIterations );
    registerImageFilter->SetNumberOfSpatialSamples( numberOfSpatialSamples );
    registerImageFilter->SetInput( movingImageReader->GetOutput() );
    registerImageFilter->SetFixedImage( fixedImageExtractionFilter->GetOutput( ) ); 
    if (outputTransform.size() > 0)
      {
      registerImageFilter->SetOutputParameterFile( outputTransform );
      }


//registerImageFilter->SetDebug(true);

    try 
      {	
      registerImageFilter->Update( );
      }
    catch (itk::ExceptionObject &ex)
      {
      std::cout << ex << std::endl;
      throw;
      }
    RegisteredImage = registerImageFilter->GetOutput();
    }
  else
    {
    std::cout << "Full Affine Registration: " << std::endl; 
    typedef itk::VectorImageRegisterAffineFilter < NrrdImageType, NrrdImageType > RegisterFilterType;
    RegisterFilterType::Pointer registerImageFilter = RegisterFilterType::New();
    
    registerImageFilter->SetTranslationScale( spatialScale );
    registerImageFilter->SetMaximumStepLength( maximumStepSize );
    registerImageFilter->SetMinimumStepLength( minimumStepSize );
    registerImageFilter->SetRelaxationFactor( relaxationFactor );
    registerImageFilter->SetNumberOfIterations( numberOfIterations );
    registerImageFilter->SetNumberOfSpatialSamples( numberOfSpatialSamples );
    registerImageFilter->SetInput( movingImageReader->GetOutput() );
    registerImageFilter->SetFixedImage( fixedImageExtractionFilter->GetOutput( ) );
    registerImageFilter->DebugOn( );
    if (outputTransform.size() > 0)
      {
      registerImageFilter->SetOutputParameterFile( outputTransform );
      }
            
    try 
      {	
      registerImageFilter->Update( );
      }
    catch (itk::ExceptionObject &ex)
      {
      std::cout << ex << std::endl;
      throw;
      }
    RegisteredImage = registerImageFilter->GetOutput();
    }
  
  
  typedef itk::ImageFileWriter<NrrdImageType> WriterType;
  WriterType::Pointer nrrdWriter = WriterType::New();
  nrrdWriter->UseInputMetaDataDictionaryOn();
  nrrdWriter->SetInput( RegisteredImage );
  nrrdWriter->SetFileName( outputVolume );
  try
    {
    nrrdWriter->Update();
    }
  catch (itk::ExceptionObject e)
    {
    std::cout << e << std::endl;
    }
    
}
