/**
\file  FeatureExtractionClass.cpp

\brief Implementation of the FeatureExtractionClass

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#include "FeatureExtractionClass.h"
#include "itkVariableSizeMatrix.h"
#include "itkVariableLengthVector.h"
#include "CAPTk.h"

FeatureExtractionClass::FeatureExtractionClass()
{

}

VariableSizeMatrixType FeatureExtractionClass::ResampleTrainingData(const VariableSizeMatrixType &trainingdata, const unsigned int NumberOfNearSamples, const unsigned int NumberOfFarSamples)
{
  VariableSizeMatrixType sampledTrainingData;
  int NumberOfFeatures = trainingdata.Cols() - 1;
  unsigned int stepSize = NumberOfFarSamples / NumberOfNearSamples;
  int NumberOfSampledFarSamples = 0;
  unsigned int TotalSamples = NumberOfFarSamples + NumberOfNearSamples;

  for (unsigned int i = 0; i < NumberOfFarSamples; i = i + stepSize)
    NumberOfSampledFarSamples++;

  sampledTrainingData.SetSize(NumberOfNearSamples + NumberOfSampledFarSamples, NumberOfFeatures + 1);

  for (unsigned int i = 0; i < NumberOfNearSamples; i++)
  {
    for (unsigned int j = 0; j < trainingdata.Cols(); j++)
      sampledTrainingData(i, j) = trainingdata(i, j);
  }
  int counter = NumberOfNearSamples;

  for (unsigned int i = NumberOfNearSamples; i < TotalSamples; i = i + stepSize)
  {
    for (unsigned int j = 0; j < trainingdata.Cols(); j++)
      sampledTrainingData(counter, j) = trainingdata(i, j);
    
    counter++;
  }
  return sampledTrainingData;
}


void FeatureExtractionClass::FormulateTrainingData(const std::vector< VectorDouble > &nearintensitities, const std::vector< VectorDouble > &farintensitities)
{
  size_t NumberOfNearSamples = nearintensitities.size();
  size_t NumberOfFarSamples = farintensitities.size();
  size_t NumberOfFeatures = nearintensitities[0].size();

  mTrainingData.SetSize(NumberOfNearSamples + NumberOfFarSamples, NumberOfFeatures + 1);
  for (size_t i = 0; i < NumberOfNearSamples; i++)
  {
    for (size_t j = 0; j < NumberOfFeatures; j++)
      mTrainingData(i, j) = nearintensitities[i][j];

    mTrainingData(i, NumberOfFeatures) = TRAINING_LABEL_NEAR;
  }
  for (size_t i = 0; i < NumberOfFarSamples; i++)
  {
    for (size_t j = 0; j < NumberOfFeatures; j++)
      mTrainingData(i + NumberOfNearSamples, j) = farintensitities[i][j];

    mTrainingData(i + NumberOfNearSamples, NumberOfFeatures) = TRAINING_LABEL_FAR;
  }
}


VariableSizeMatrixType FeatureExtractionClass::FormulateTestData(const std::vector< VectorDouble > &testdata)
{
  size_t NumberOfFeatures = testdata[0].size();
  size_t NumberOfSamples = testdata.size();

  mTestData.SetSize(NumberOfSamples, NumberOfFeatures + 1);

  for (size_t i = 0; i < NumberOfSamples; i++)
  {
    for (size_t j = 0; j < NumberOfFeatures; j++)
      mTestData(i, j) = testdata[i][j];

    mTestData(i, NumberOfFeatures) = TRAINING_LABEL_FAR;
  }

  return mTestData;
}

void FeatureExtractionClass::FormulateSurvivalTrainingData(const VariableSizeMatrixType &inputFeatures, std::vector<double> inputSurvival, VariableSizeMatrixType & SixModelFeatures, VariableSizeMatrixType & EighteenModelFeatures)
{
  std::vector<int> SixModelLowerIndices;
  std::vector<int> SixModelHigherIndices;
  std::vector<int> EighteenModelLowerIndices;
  std::vector<int> EighteenModelHigherIndices;

  for (unsigned int i = 0; i < inputSurvival.size(); i++)
  {
    if (inputSurvival[i] <= 6)
      SixModelLowerIndices.push_back(i);
    else
      SixModelHigherIndices.push_back(i);

    if (inputSurvival[i] <= 18)
      EighteenModelLowerIndices.push_back(i);
    else
      EighteenModelHigherIndices.push_back(i);
  }
  SixModelFeatures.SetSize(inputFeatures.Rows(), inputFeatures.Cols() + 1);
  EighteenModelFeatures.SetSize(inputFeatures.Rows(), inputFeatures.Cols() + 1);

  for (unsigned int i = 0; i < SixModelLowerIndices.size(); i++)
  {
    unsigned int j = 0;
    for (j = 0; j < inputFeatures.Cols(); j++)
      SixModelFeatures(i, j) = inputFeatures(SixModelLowerIndices[i], j);
    SixModelFeatures(i, j) = 0;
  }
  for (unsigned int i = 0; i < SixModelHigherIndices.size(); i++)
  {
    unsigned int j = 0;
    for (j = 0; j < inputFeatures.Cols(); j++)
      SixModelFeatures(i + SixModelLowerIndices.size(), j) = inputFeatures(SixModelHigherIndices[i], j);
    SixModelFeatures(i + SixModelLowerIndices.size(), j) = 1;
  }

  for (unsigned int i = 0; i < EighteenModelLowerIndices.size(); i++)
  {
    unsigned int j = 0;
    for (j = 0; j < inputFeatures.Cols(); j++)
      EighteenModelFeatures(i, j) = inputFeatures(EighteenModelLowerIndices[i], j);
    EighteenModelFeatures(i, j) = 0;
  }
  for (unsigned int i = 0; i < EighteenModelHigherIndices.size(); i++)
  {
    unsigned int j = 0;
    for (j = 0; j < inputFeatures.Cols(); j++)
      EighteenModelFeatures(i + EighteenModelLowerIndices.size(), j) = inputFeatures(EighteenModelHigherIndices[i], j);
    EighteenModelFeatures(i + EighteenModelLowerIndices.size(), j) = 1;
  }
}
