///////////////////////////////////////////////////////////////////////////////////////
// fFeaturePanel.cpp
// Copyright (c) 2016. All rights reserved.
// Section of Biomedical Image Analysis
// Center for Biomedical Image Computing and Analytics
// Department of Radiology
// Perelman School of Medicine
// University of Pennsylvania
// Contact details: software@cbica.upenn.edu
// License Agreement: https://www.cbica.upenn.edu/sbia/software/license.html
///////////////////////////////////////////////////////////////////////////////////////
#include "fFeaturePanel.h"
#include "CAPTk.h"
#include "fMainWindow.h"
#include "TextureFeatures.h"
#include <opencv2/core/core.hpp>
using namespace cv;
static void writeFeatureList(FileStorage& fs, string name, vector<string> *featureNames, vector< double> *features)
{
  try
  {
    fs << name << "[";
    for (int i = 0; i < features->size(); i++)
    {
      fs << "{:";
      fs << featureNames->at(i) << features->at(i);
      fs << "}";
    }
    fs << "]";
  }
  catch (Exception ex)
  {
    cout << ex.msg;
  }
}
fFeaturePanel::fFeaturePanel(QWidget * parent) : QWidget(parent)
{
  m_listener = NULL;
  setupUi(this);
  connect(m_btnCompute, SIGNAL(clicked()), this, SLOT(ComputeFunctionality()));
  connect(m_btnCancel, SIGNAL(clicked()), this, SLOT(CancelFunctionality()));
  connect(m_btnBrowseSaveFile, SIGNAL(clicked()), this, SLOT(browseOutputFileName()));
  connect(m_cmbFeatureType, SIGNAL(currentIndexChanged(int)), this, SLOT(featureTypeChanged(int)));
  featureTypeChanged(TextureFeatures::Custom);

  radio1->setChecked(true);
}
void fFeaturePanel::featureTypeChanged(int type)
{
  // int type = m_cmbFeatureType->currentIndex();
  if (type == TextureFeatures::Custom)
  {
    m_Intensityfeature->setChecked(true);
    m_Statisticfeature->setChecked(true);
    m_Histogramfeature->setChecked(true);
    m_Runlenghtfeature->setChecked(true);
    m_Volumetricfeature->setChecked(true);
    m_Co_occurancefeature->setChecked(true);
    m_LBPfeature->setChecked(true);

    m_Intensityfeature->setEnabled(true);
    m_Statisticfeature->setEnabled(true);
    m_Volumetricfeature->setEnabled(true);
    m_Histogramfeature->setEnabled(true);
    m_Runlenghtfeature->setEnabled(true);
    m_Co_occurancefeature->setEnabled(true);
    m_LBPfeature->setEnabled(true);
  }
  else if (type == TextureFeatures::Torso)
  {
    m_Intensityfeature->setChecked(true);
    m_Statisticfeature->setChecked(true);
    m_Volumetricfeature->setChecked(true);
    m_Runlenghtfeature->setChecked(true);
    m_Co_occurancefeature->setChecked(true);
    m_Histogramfeature->setChecked(false);
    m_LBPfeature->setChecked(true);

    m_Intensityfeature->setEnabled(false);
    m_Statisticfeature->setEnabled(false);
    m_Volumetricfeature->setEnabled(true);
    m_Histogramfeature->setEnabled(false);
    m_Runlenghtfeature->setEnabled(false);
    m_Co_occurancefeature->setEnabled(false);
    m_LBPfeature->setEnabled(false);
  }
  else   if (type == TextureFeatures::Neuro)
  {
    m_Intensityfeature->setChecked(true);
    m_Statisticfeature->setChecked(true);
    m_Volumetricfeature->setChecked(true);
    m_Histogramfeature->setChecked(true);
    m_Runlenghtfeature->setChecked(false);
    m_Co_occurancefeature->setChecked(false);
    m_LBPfeature->setChecked(false);

    m_Intensityfeature->setEnabled(false);
    m_Statisticfeature->setEnabled(false);
    m_Volumetricfeature->setEnabled(true);
    m_Histogramfeature->setEnabled(false);
    m_Runlenghtfeature->setEnabled(false);
    m_Co_occurancefeature->setEnabled(false);
    m_LBPfeature->setEnabled(false);
  }

}
void fFeaturePanel::browseOutputFileName()
{
  QString fileName = QFileDialog::getSaveFileName(this, "Save Features", QDir::currentPath(), "Features (*.xml *.yml");
  if (!fileName.isEmpty())
  {
    m_txtSaveFileName->setText(fileName);
  }
}
void fFeaturePanel::ComputeFunctionality()
{
  int  type = m_cmbFeatureType->currentIndex();
  m_cmbFeatureType->setEnabled(false);
  computeFeature(type);
  m_cmbFeatureType->setEnabled(true);

  //emit ComputeClicked();
}
void fFeaturePanel::CancelFunctionality()
{
  cancelClicked = true;
}
void fFeaturePanel::computeFeature(int type)
{
  std::vector < ImageTypeFloat3D::Pointer> images;
  ImageTypeFloat3D::Pointer maskImg;
  std::vector < string > fileNames;
  if (m_listener != NULL)
  {
    images = ((fMainWindow*)m_listener)->getLodedImages(fileNames);
    maskImg = ((fMainWindow*)m_listener)->getMaskImage();

  }
  if (images.size() != fileNames.size() || images.size() == 0)
  {
    ((fMainWindow*)m_listener)->updateProgress(0, "Error. No valid images selected!");
    return;
  }
  else
  {
    ((fMainWindow*)m_listener)->updateProgress(5, "Computing features!");
  }
  string xmlFileName = m_txtSaveFileName->text().toStdString();
  FileStorage fs(xmlFileName, FileStorage::WRITE);
  if (fs.isOpened())
  {
    for (size_t i = 0; i < images.size(); i++)
    {
      if (cancelClicked) //Before processing m, make sure no cancel click
      {
        cancelClicked = false;
        break;
      }
      TextureFeatures txF;
      typedef itk::VectorContainer< unsigned char, OffsetType > OffsetVector;
      vector<string> featueNames;
      vector< double > features;
      typedef OffsetVector::Pointer  OffsetVectorPointer;
      OffsetType offset;
      typedef itk::Neighborhood<float, 3> NeighborhoodType;
      NeighborhoodType neighborhood;
      neighborhood.SetRadius(1);
      unsigned int centerIndex = neighborhood.GetCenterNeighborhoodIndex();
      OffsetVectorPointer offsets = OffsetVector::New();
      for (unsigned int d = 0; d < centerIndex; d++)
      {
        offset = neighborhood.GetOffset(d);
        offsets->push_back(offset);
      }

      if (type == TextureFeatures::Torso)
      {
        txF.Intensity_features<ImageTypeFloat3D>(images[i], maskImg, featueNames, features);
        if (txF.m_errormsg)
        {
          ShowErrorMessage("The mask is not provided or is invalid.");
          txF.m_errormsg = false;
          return;
        }
        else
        {
          txF.calculateTextureFeatures<ImageTypeFloat3D, OffsetVector>(txF.nonzero_pix, images[i], maskImg, features[0], features[1], offsets, featueNames, features);
          txF.calculateRunLength<ImageTypeFloat3D, OffsetVector>(txF.nonzero_pix, images[i], maskImg, features[0], features[1], offsets, featueNames, features);
          txF.ShapeFeatures(maskImg, featueNames, features);
        }
      }
      else if (type == TextureFeatures::Neuro)
      {
        txF.Intensity_features<ImageTypeFloat3D>(images[i], maskImg, featueNames, features);
        if (txF.m_errormsg)
        {
          ShowErrorMessage("The mask is not provided or is invalid.");
          txF.m_errormsg = false;
          return;
        }
        else
        {
          int tempInterval = std::max(1, static_cast<int>(std::floor((features[1] - features[0]) / 10)));
          txF.HistogramFeatures(txF.nonzero_pix, features[0], tempInterval, features[1], featueNames, features);
          txF.ShapeFeatures(maskImg, featueNames, features);
        }
      }
      else if (type == TextureFeatures::Custom)
      {
        if (m_Intensityfeature->checkState() == Qt::Checked)
          txF.Intensity_features<ImageTypeFloat3D>(images[i], maskImg, featueNames, features);
        if (txF.m_errormsg)
        {
          ShowErrorMessage("The mask is not provided or is invalid.");
          txF.m_errormsg = false;
          txF.m_errormsg = FALSE;
          return;
        }
        else
        {
          int tempInterval = std::max(1, static_cast<int>(std::floor((features[1] - features[0]) / 10)));
          if (m_Histogramfeature->checkState() == Qt::Checked)
            txF.HistogramFeatures(txF.nonzero_pix, features[0], tempInterval, features[1], featueNames, features);
          if (m_Co_occurancefeature->checkState() == Qt::Checked)
            txF.calculateTextureFeatures<ImageTypeFloat3D, OffsetVector>(txF.nonzero_pix, images[i], maskImg, features[0], features[1], offsets, featueNames, features);
          if (m_Runlenghtfeature->checkState() == Qt::Checked)
            txF.calculateRunLength<ImageTypeFloat3D, OffsetVector>(txF.nonzero_pix, images[i], maskImg, features[0], features[1], offsets, featueNames, features);
          if (m_Volumetricfeature->checkState() == Qt::Checked)
            txF.ShapeFeatures(maskImg, featueNames, features);
        }
      }

      writeFeatureList(fs, "Tex_" + to_string(i), &featueNames, &features);
      ((fMainWindow*)m_listener)->updateProgress(i + 1, "Calculating and exporting features", images.size());

    }
    fs.release();

  }
  ((fMainWindow*)m_listener)->updateProgress(0, "Feature export complete!");
  return;

}
