/**
\file  fMainWindow.cpp

\brief Implementation of fMainWindow class

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#include "stdio.h"
#include <errno.h>
#include "fMainWindow.h"
#include "fRecurrenceDialog.h"
#include "fSurvivalDialog.h"
#include "fRegistrationDialog.h"
#include "itkCSVArray2DFileReader.h"
#include "OutputInteractorStyleNavigator.h"
#include "qobject.h"
#include "qthread.h"
#include "qfuture.h"
#include "qtconcurrentrun.h"
#include "qdatetime.h"
#include "fHelpDialog.h"
#include "fHelpAppDialog.h"

//#include "NonNativeThreadManager.h"
#include "N3BiasCorrection.h"
#include "SusanDenoising.h"
//#include "Registration.h"

#include <vtkImageThreshold.h>

#include "cbicaITKImageInfo.h"
#include "cbicaITKSafeImageIO.h"

#include "Registry.h"

// this function calls an external application from CaPTk in the most generic way while waiting for output
void startExternalProcess(const QString &application, const QStringList &arguments)
{
  QProcess process;
  if (arguments.isEmpty())
  {
    process.start(application);
  }
  else
  {
    process.start(application, arguments);
  }
  process.write("exit\n\r");
  process.waitForFinished(-1);
  process.close();
  return;
}

int GetNumberOfDimensions(vtkImageData* input)
{
  int dim = 0;
#if VTK_MAJOR_VERSION <= 5
  int* extent = input->GetWholeExtent();
#else
  int* extent = input->GetExtent();
#endif
  if (extent[4] != extent[5])
  {
    dim = 3;
  }
  else if (extent[3] != extent[4])
  {
    dim = 2;
  }
  else if (extent[0] != extent[1])
  {
    dim = 1;
  }
  return dim;
}

inline std::string correctExtension(const std::string &inputFileName)
{
  std::string returnString = inputFileName, tempPath, tempBase, tempExt;
  cbica::splitFileName(returnString, tempPath, tempBase, tempExt);
  if (tempExt == "")
  {
    returnString += NII_GZ_EXT;
  }

  return returnString;
}

inline std::string correctExtension(const QString &inputFileName)
{
  std::string returnString = inputFileName.toStdString(), tempPath, tempBase, tempExt;
  cbica::splitFileName(returnString, tempPath, tempBase, tempExt);
  if (tempExt == "")
  {
    returnString += NII_GZ_EXT;
  }

  return returnString;
}


fMainWindow::fMainWindow()
{

  setupUi(this);
  featurePanel->setListner(this);//TBD bad design RK
  m_imagesTable = imagesPanel->GetImagesTable();
  m_nonVisImagesTable = imagesPanel->GetNonViewingImagesTable();
  assert(m_imagesTable != NULL);
  assert(m_nonVisImagesTable != NULL);

  mSequenceNumber = 0;

  t1cePath = "";

  QString msg = tr(EXE_NAME);
#ifdef SW_VER
  msg += " - v" + tr(SW_VER);
#endif
  this->setWindowTitle(msg);

#ifdef Q_OS_WIN32
  currentPlatform = "windows";
#endif

  mInputPathName = "";
  //mMainWidget = this;
  mCurrentSelectedImageId = "";
  mCurrentPickedImageId = "";
  mCurrentPickedImageIndex = 0;
  mCurrentNearPoints = 0;
  mCurrentFarPoints = 0;
  mCurrentInitPoints = 0;
  mCustomImageToThreshold = itk::Image< short, 3 >::New();

  tempFolderLocation = cbica::createTmpDir();
  featurePanel->setTempFolderLocation(tempFolderLocation);

  mLandmarks = new Landmarks(LANDMARK_TYPE_LANDMARKS);
  mSeedPoints = new Landmarks(LANDMARK_TYPE_SEED_POINTS);
  mTissuePoints = new Landmarks(LANDMARK_TYPE_TISSUE_POINTS);

  mMask = vtkSmartPointer<vtkImageData>::New();

  mSlicerManagers.resize(0);

  image4DSlider->setEnabled(false);
  image4DSlider->setValue(0);

  connect(imagesPanel, SIGNAL(sigOverlayCheckBoxChanged(int)), this, SLOT(overlayUseStateChanged(int)));
  connect(imagesPanel, SIGNAL(sigOverlaySliderChanged(int)), this, SLOT(overlaySliderChanged(int)));
  connect(imagesPanel, SIGNAL(sigOverlayChanged()), this, SLOT(overlayChanged()));
  connect(imagesPanel, SIGNAL(sigImageModalityChanged(int)), this, SLOT(imageModalityChanged(int)));

  connect(image4DSlider, SIGNAL(valueChanged(int)), this, SLOT(imageSliderChanged()));
  SetPresetComboBox();

  // init the sliders
  verticalSliders.push_back(AxialViewSlider);
  verticalSliders.push_back(CoronalViewSlider);
  verticalSliders.push_back(SaggitalViewSlider);
  for (int i = 0; i < 3; i++)
  {
    verticalSliders[i]->hide();
  }
  connect(AxialViewSlider, SIGNAL(valueChanged(int)), this, SLOT(AxialViewSliderChanged()));
  connect(CoronalViewSlider, SIGNAL(valueChanged(int)), this, SLOT(CoronalViewSliderChanged()));
  connect(SaggitalViewSlider, SIGNAL(valueChanged(int)), this, SLOT(SaggitalViewSliderChanged()));

  connect(actionLoad_Recurrence_Images, SIGNAL(triggered()), this, SLOT(openImages()));
  connect(actionLoad_Nifti_Images, SIGNAL(triggered()), this, SLOT(openImages()));
  
  connect(actionSave_ROI_Images, SIGNAL(triggered()), this, SLOT(SaveDrawing()));
  connect(actionSave_ROI_Dicom_Images, SIGNAL(triggered()), this, SLOT(SaveDicomDrawing()));
  connect(actionSave_Nifti_Images, SIGNAL(triggered()), this, SLOT(SaveImage()));
  connect(actionSave_Dicom_Images, SIGNAL(triggered()), this, SLOT(SaveDicomImage()));

  connect(actionLoad_Nifti_ROI, SIGNAL(triggered()), this, SLOT(LoadDrawing()));

  connect(actionExit, SIGNAL(triggered()), this, SLOT(close()));
  connect(actionAbout, SIGNAL(triggered()), this, SLOT(about()));
  connect(actionHelp_Interactions, SIGNAL(triggered()), this, SLOT(help_Interactions()));
  connect(actionHelp_Applications, SIGNAL(triggered()), this, SLOT(help_Applications()));

  // add a single function for all applications, this function will check for the specific names and then initiate that algorithm
  for (size_t i = 0; i < vectorOfApplicationActionsAndNames.size(); i++)
  {
    if (vectorOfApplicationActionsAndNames[i].name.find("EGFR") != std::string::npos)
    {
      vectorOfApplicationActionsAndNames[i].action->setText("EGFRvIII Surrogate Index");//TBD set at source
      connect(vectorOfApplicationActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ApplicationEGFR()));
    }

    else if (vectorOfApplicationActionsAndNames[i].name.find("Geodesic") != std::string::npos)
    {
      vectorOfApplicationActionsAndNames[i].action->setText("Geodesic Segmentation");//TBD set at source
      connect(vectorOfApplicationActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ApplicationGeodesic()));
    }

    else if (vectorOfApplicationActionsAndNames[i].name.find("Recurrence") != std::string::npos)
    {
      vectorOfApplicationActionsAndNames[i].action->setText("Glioblastoma Infiltration Index");//TBD set at source
      connect(vectorOfApplicationActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ApplicationRecurrence()));
    }

    else if (vectorOfApplicationActionsAndNames[i].name.find("Survival") != std::string::npos)
    {
      vectorOfApplicationActionsAndNames[i].action->setText("Survival Prediction Index");//TBD set at source
      connect(vectorOfApplicationActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ApplicationSurvival()));
    }
  }

  // add a single function for all preprocessing steps, this function will check for the specific names and then initiate that algorithm
  for (size_t i = 0; i < vectorOfPreprocessingActionsAndNames.size(); i++)
  {
    if (vectorOfPreprocessingActionsAndNames[i].name.find("Denoise") != std::string::npos)
    {
      connect(vectorOfPreprocessingActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ImageDenoising()));
    }

    else if (vectorOfPreprocessingActionsAndNames[i].name.find("BiasCorrect") != std::string::npos)
    {
      vectorOfPreprocessingActionsAndNames[i].action->setText("BiasCorrect-N4");//TBD set at source
      connect(vectorOfPreprocessingActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ImageBiasCorrection()));
    }

    else if (vectorOfPreprocessingActionsAndNames[i].name.find("Register") != std::string::npos)
    {
      connect(vectorOfPreprocessingActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ImageRegistration()));
    }

    //else if (vectorOfPreprocessingActionsAndNames[i].name.find("Custom") != std::string::npos)
    //{
    //  connect(vectorOfPreprocessingActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(CustomPreprocessing()));
    //}
  }

  // add a single function for all python GUI apps, this function will check for the specific names and then initiate that algorithm
  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("LIBRA") != std::string::npos)
    {
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyGUILIBRA()));
    }
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("ITK-SNAP") != std::string::npos)
    {
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(ApplicationITKSNAP()));
    }
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("Confetti") != std::string::npos)
    {
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyGUIConfetti()));
    }
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("WhiteStripe") != std::string::npos)
    {
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyGUIWhiteStripe()));
    }
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("SBRT_Segment") != std::string::npos)
    {
      vectorOfPythonGUIAppActionsAndNames[i].action->setText("SBRT Segment");//TBD set at source
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyCLISBRT_Segment()));
    }
    if (vectorOfPythonGUIAppActionsAndNames[i].name.find("SBRT_Analyze") != std::string::npos)
    {
      vectorOfPythonGUIAppActionsAndNames[i].action->setText("SBRT Analyze");//TBD set at source
      connect(vectorOfPythonGUIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyCLISBRT_Analyze()));
    }
  }

  // add a single function for all python CLI apps, this function will check for the specific names and then initiate that algorithm
  for (size_t i = 0; i < vectorOfPythonCLIAppActionsAndNames.size(); i++)
  {
    //if (vectorOfPythonCLIAppActionsAndNames[i].name.find("SBRT") != std::string::npos)
    //{
    //  connect(vectorOfPythonCLIAppActionsAndNames[i].action, SIGNAL(triggered()), this, SLOT(PyCLISBRT()));
    //}
  }

  connect(m_imagesTable, SIGNAL(itemSelectionChanged()), this, SLOT(DisplayChanged()));
  connect(m_imagesTable, SIGNAL(itemClicked(QTableWidgetItem*)), this, SLOT(DisplayChanged(QTableWidgetItem*)));

  connect(imagesPanel, SIGNAL(sigImageTableSelectionChanged()), this, SLOT(DisplayChanged()));

  connect(windowSpinBox, SIGNAL(editingFinished()), this, SLOT(WindowLevelEdited()));
  connect(levelSpinBox, SIGNAL(editingFinished()), this, SLOT(WindowLevelEdited()));

  connect(presetComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(UpdateWindowLevel()));
  connect(thresholdSlider, SIGNAL(valueChanged(int)), this, SLOT(thresholdSliderChanged()));
  //
  thresholdSlider->setRange(0, 255);
  thresholdSlider->setValue(20);

  connect(thresholdMaskButton, SIGNAL(toggled(bool)), this, SLOT(EnableThresholdOfMask()));
  connect(thresholdImageButton, SIGNAL(toggled(bool)), this, SLOT(EnableThresholdOfImage()));

  thresholdImageButton->isEnabled();
  mthresholdImage = true;
  mthresholdMask = false;

  if (presetComboBox->currentIndex() == PRESET_THRESHOLD)
  {
    thresholdLabel->setEnabled(true);
    thresholdSlider->setEnabled(true);
  }
  else
  {
    thresholdLabel->setEnabled(false);
    thresholdSlider->setEnabled(false);
  }

  connect(tumorPanel, SIGNAL(UpdateRenderWindows()), this, SLOT(UpdateRenderWindows()));
  connect(tumorPanel, SIGNAL(SetActiveLandmarksTypeSignal(int, int, int)), this, SLOT(SetActiveLandmarksType(int, int, int)));
  connect(tumorPanel, SIGNAL(MoveSlicerCursor(double, double, double)), this, SLOT(MoveSlicerCursor(double, double, double)));
  
  connect(drawingPanel, SIGNAL(clearMask(int)), this, SLOT(clearMask(int)));
  connect(drawingPanel, SIGNAL(CurrentBrushSizeChanged(int)), this, SLOT(ChangeBrushSize(int)));
  connect(drawingPanel, SIGNAL(UndoButtonClicked()), this, SLOT(UndoFunctionality()));
  connect(drawingPanel, SIGNAL(shapesButtonClicked(int)), this, SLOT(updateDrawMode(int)));
  connect(drawingPanel, SIGNAL(CurrentDrawingLabelChanged(int)), this, SLOT(updateDrawMode()));
  connect(drawingPanel, SIGNAL(CurrentMaskOpacityChanged(int)), this, SLOT(ChangeMaskOpacity(int)));


  connect(&recurrencePanel, SIGNAL(SubjectBasedRecurrenceEstimate(std::string, bool, bool, bool, bool, bool, bool, bool)), this, SLOT(StartRecurrenceEstimate(const std::string &, bool , bool , bool , bool , bool , bool , bool )));
  connect(&recurrencePanel, SIGNAL(ExistingModelBasedRecurrenceEstimate(std::string, std::string, std::string, bool, bool, bool, bool, bool, bool, bool)), this, SLOT(RecurrenceEstimateOnExistingModel(const std::string &, const std::string &, const std::string &, bool &, bool &, bool &, bool &, bool &, bool &, bool &)));
  connect(&recurrencePanel, SIGNAL(TrainNewModel(std::string, std::string, bool, bool, bool, bool, bool, bool, bool)), this, SLOT(TrainNewModelOnGivenData(const std::string &, const std::string &, bool &, bool &, bool &, bool &, bool &, bool &, bool &)));

  connect(&survivalPanel, SIGNAL(SurvivalPredictionOnExistingModel(const std::string, const std::string, const std::string)), this, SLOT(CallForSurvivalPredictionOnExistingModelFromMain(const std::string, const std::string, const std::string)));
  connect(&survivalPanel, SIGNAL(PrepareNewSurvivalPredictionModel(const std::string, const std::string)), this, SLOT(CallForNewSurvivalPredictionModelFromMain(const std::string, const std::string)));


  connect(this, SIGNAL(SeedPointsFocused(bool)), tumorPanel, SLOT(sTableFocused(bool)));
  connect(this, SIGNAL(TissuePointsFocused(bool)), tumorPanel, SLOT(tTableFocused(bool)));
  connect(m_tabWidget, SIGNAL(currentChanged(int)), this, SLOT(SetActiveLandmarkTypeTab(int)));
  connect(infoPanel, SIGNAL(MoveSlicerCursor(double, double, double, int)), this, SLOT(MoveSlicerCursor(double, double, double, int)));

  AxialViewWidget->hide();
  CoronalViewWidget->hide();
  SaggitalViewWidget->hide();
  infoPanel->show();

  windowLabel->setEnabled(false);
  windowSpinBox->setEnabled(false);
  levelSpinBox->setEnabled(false);
  levelLabel->setEnabled(false);
  
  thresholdLabel->setEnabled(false);
  thresholdSlider->setEnabled(false);
  presetLabel->setEnabled(false);
  presetComboBox->setEnabled(false);

  setAcceptDrops(true);

  mCustomImageToThreshold_min = 0;
  mCustomImageToThreshold_max = 0;

  m_drawShapeMode = SHAPE_MODE_NONE;

  m_messageLabel = new QLabel(":");
  statusBar()->addPermanentWidget(m_messageLabel);
  m_progressBar = new QProgressBar(this);
  QSize tempSize = this->size();
  m_progressBar->setFixedWidth(tempSize.width() * 0.75);
  statusBar()->addPermanentWidget(m_progressBar);
  m_progressBar->setValue(0);

  //connect 
  connect(m_toolTabdock, SIGNAL(topLevelChanged(bool)), this, SLOT(toolTabDockChanged(bool)));

  //Connect applications with progress/ message bar
  connect(&mRecurrenceEstimator, SIGNAL(signalProgress(int)), this->m_progressBar, SLOT(setValue(int)));
  connect(&mRecurrenceEstimator, SIGNAL(signalMessage(QString)), this->m_messageLabel, SLOT(setText(QString)));
  //connect(&mGeodesicSegmentation, SIGNAL(signalProgress(int)), this->m_progressBar, SLOT(setValue(int)));
  //connect(&mGeodesicSegmentation, SIGNAL(signalMessage(QString)), this->m_messageLabel, SLOT(setText(QString)));
  connect(&mEGFRPredictor, SIGNAL(signalProgress(int)), this->m_progressBar, SLOT(setValue(int)));
  connect(&mEGFRPredictor, SIGNAL(signalMessage(QString)), this->m_messageLabel, SLOT(setText(QString)));

}

fMainWindow::~fMainWindow()
{
  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
    if (mSlicerManagers[i] != NULL) 
    {
      delete mSlicerManagers[i];
    }
  }
  if (mLandmarks) 
  {
    delete mLandmarks;
  }
  if (mSeedPoints) 
  {
    delete mSeedPoints;
  }
  if (mTissuePoints) 
  {
    delete mTissuePoints;
  }
  cbica::deleteDir(tempFolderLocation);
}

void fMainWindow::about()
{ 
//  std::string  strVerzsionTime;
//#ifdef VERSION_TIME
//  strVerzsionTime = VERSION_TIME;
//#endif 
  QString msg;
  
  //if (strVerzsionTime.empty())
  //{
  //  msg = "Cancer Imaging Phenomics Toolkit (CaPTk) v" + tr(PROJECT_VERSION) + "\n\n";
  //}
  //else
  //{
    msg = "Cancer Imaging Phenomics Toolkit (CaPTk) v" + tr(PROJECT_VERSION) + "\n\n";
  //}

  msg += "Copyright (c) 2017 University of Pennsylvania. All rights reserved.\n";
  msg += "Contact: CBICA Group <software@cbica.upenn.edu>\n\n";
  //
  msg += "CaPTk provides a graphical interface to integrate advanced oncologic image computing and analytics tools "; // this is NOT a new line
  msg += "that have been developed by the Center for Biomedical Image Computing and Analytics (CBICA) at the University of Pennsylvania.\n\n";
  //
  msg += "The software has been designed for research purposes only and has not been ";
  msg += "reviewed or approved by the Food and Drug Administration (FDA) or by any other ";
  msg += "agency. It is not intended or recommended for clinical applications.\n\n";

  msg += "This work was supported by NIH 1U24CA189523-01A1.";
  ////
  //msg += "Software Engineers:\n";
  //msg += "\tSaima R \n";
  //msg += "\tRatheesh K \n";
  //msg += "\tPatmaa S \n";
  //msg += "\tSarthak P \n\n";
  ////
  //msg += "People Involved:\n";
  //msg += "\tMark B \n";
  //msg += "\tSpyridon B \n";
  //msg += "\tAristeidis S \n";
  //// 
  //msg += "NOTES:\n\n";
  ////
  ////msg += "1. Our application has been optimized for monitors with 16:9 aspect ratio, especially with monitors having the resolution 1920x1080.\n";
  ////msg += "1. Qt is a registered trademark of The Qt Company Ltd. and/or its subsidiaries. All other trademarks are the property of their respective owners.\n";
  //msg += "1. This package incorporates 3rd party libaries and toolkits. Trademarks for each of the following are property of their respective owners:\n\n";
  ////
  //msg += "\t(a) Qt \t\t[registered trademark of The Qt Company Ltd.]\n";
  //msg += "\t(b) ITK \t\t[developed, published and maintained by Kitware, Inc.]\n";
  //msg += "\t(c) VTK \t[developed, published and maintained by Kitware, Inc.]\n";
  //msg += "\t(d) OpenCV \t[currently maintained by Itseez, Inc.]\n";
  //msg += "\t(e) LibSVM \t[developed by the National Taiwan University]\n\n";
  ////
  //msg += "All rights to these components are subject to the terms and licenses of their respective owners. All trademarks are the property of their respective owners. \n\n";
  ////

  QMessageBox* box = new QMessageBox(this);
  box->setIcon(QMessageBox::Information);
  box->setAttribute(Qt::WA_DeleteOnClose);
  box->setStandardButtons(QMessageBox::Ok);
  box->setText(msg);
  box->setWindowTitle(tr("About"));
  box->setWindowModality(Qt::NonModal);
  box->show();
  QCoreApplication::processEvents();

  // write a single file alluding that the user has seen/read the "About" page - which opens up in the first iteration
  std::ofstream file;
  file.open(aboutScreenSeen.c_str());
  file << "User has seen the 'about' screen.\n";
  file.close();
}

void fMainWindow::help_Interactions()
{
  fHelpDialog mHelpDlg;
  mHelpDlg.exec();
}

void fMainWindow::help_Applications()
{
  //fHelpAppDialog mHelpAppDlg;
  //mHelpAppDlg.exec();
}

void fMainWindow::EnableThresholdOfMask()
{
  mthresholdImage = false;
  mthresholdMask = true;
  
  // only do calculations on current image(s)
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) 
  {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size()) 
  {
    return;
  }

  typedef itk::MinimumMaximumImageCalculator < ImageTypeFloat3D > ImageCalculatorFilterType;
  ImageCalculatorFilterType::Pointer imageCalculatorFilter = ImageCalculatorFilterType::New();
  imageCalculatorFilter->SetImage(mSlicerManagers[index]->mITKImage);
  imageCalculatorFilter->Compute();
  float actualMin = imageCalculatorFilter->GetMinimum();
  float actualMax = imageCalculatorFilter->GetMaximum();

  thresholdSlider->setRange(static_cast<int>(actualMin), static_cast<int>(actualMax));
  thresholdSlider->setValue(static_cast<int>(std::floor((actualMin + actualMax) / 2)));
}

void fMainWindow::EnableThresholdOfImage()
{
  mthresholdImage = true;
  mthresholdMask = false;

  thresholdSlider->setRange(0, 255);
  thresholdSlider->setValue(50);
}

void fMainWindow::SaveImage()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size()) {
    return;
  }
  //
  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  QFileDialog dialog/*(this, tr("Save Image"), mInputPathName, extensions)*/;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  //dialog.selectFile(QString((mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mFileName) + "_denoise.nii.gz").c_str()));
  std::string preferredName = mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mPathFileName) + "_new.nii.gz";
  QString saveFileName = dialog.getSaveFileName(this, tr("Save Image"), preferredName.c_str(), extensions, 0, QFileDialog::DontUseNativeDialog);
  if (!saveFileName.isEmpty())
  {
    file = saveFileName;
    if (mSlicerManagers[index]->GetPreset() == PRESET_THRESHOLD)
    {
      typedef ImageTypeFloat3D ImageType;
      ImageType::Pointer img = convertVtkToItk<float, 3>(mSlicerManagers[index]->mImage);
      ImageType::RegionType region = img->GetLargestPossibleRegion();
      double threshold = mSlicerManagers[index]->mThreshold;
      //
      typedef itk::ImageDuplicator< ImageType > DuplicatorType;
      DuplicatorType::Pointer duplicator = DuplicatorType::New();
      duplicator->SetInputImage(img);
      duplicator->Update();
      ImageType::Pointer seg = duplicator->GetOutput();
      //
      int vd_x, vd_y, vd_z;
      int i, j, k;
      //
      vd_x = region.GetSize()[0];
      vd_y = region.GetSize()[1];
      vd_z = region.GetSize()[2];
      //	
      for (k = 0; k < vd_z; k++)
      {
        for (j = 0; j < vd_y; j++)
        {
          for (i = 0; i < vd_x; i++)
          {
            ImageType::IndexType pixelIndex;
            pixelIndex[0] = i;
            pixelIndex[1] = j;
            pixelIndex[2] = k;

            if (img->GetPixel(pixelIndex) <= threshold)
            {
              seg->SetPixel(pixelIndex, 1);
            }
            else
            {
              seg->SetPixel(pixelIndex, 0);
            }
          }
        }
      }
      //
      typedef itk::ImageFileWriter<ImageType > WriterType;
      WriterType::Pointer writer = WriterType::New();
      writer->SetFileName(correctExtension(file));
      writer->SetInput(seg);
      writer->Update();
    }
    else {
      std::string InputPixelType = mSlicerManagers[index]->mImage->GetScalarTypeAsString();
      if (InputPixelType == "short") {
        typedef itk::Image<short, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "unsigned short") {
        typedef itk::Image<unsigned short, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<unsigned short, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "char") {
        typedef itk::Image<char, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<char, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "unsigned char") {
        typedef itk::Image<unsigned char, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<unsigned char, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "int") {
        typedef itk::Image<int, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<int, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "unsigned int") {
        typedef itk::Image<unsigned int, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<unsigned int, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "double") {
        typedef itk::Image<double, 3> ImageType;
        ImageType::Pointer img = convertVtkToItk<double, 3>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else if (InputPixelType == "float") {
        typedef ImageTypeFloat3D ImageType;
        ImageType::Pointer img = convertVtkToItk<ImageTypeFloat3D::PixelType, ImageTypeFloat3D::ImageDimension>(mSlicerManagers[index]->mImage);
        typedef itk::ImageFileWriter<ImageType > WriterType;
        WriterType::Pointer writer = WriterType::New();
        writer->SetFileName(correctExtension(file));
        writer->SetInput(img);
        writer->Update();
      }
      else 
      {
        cbica::Logging(loggerFile, "Error, input pixel type, '" + InputPixelType + "' is unknown!");
      }
    }
  }
}

void fMainWindow::InitMask(vtkImageData* image)
{
  int extent[6];
  double spacing[3];
  double origin[3];
  int i, j, k;

  image->GetExtent(extent);
  image->GetSpacing(spacing);
  image->GetOrigin(origin);

  mMask->Initialize();
  mMask->SetExtent(extent);
  mMask->SetSpacing(spacing);
  mMask->SetOrigin(origin);
#if VTK_MAJOR_VERSION <= 5
  mMask->SetScalarTypeToFloat();
  mMask->SetNumberOfScalarComponents(1);
  mMask->AllocateScalars();
#else
  mMask->AllocateScalars(VTK_FLOAT, 1);
#endif
  {
    int vd_x, vd_y, vd_z;
    vd_x = mMask->GetDimensions()[0];
    vd_y = mMask->GetDimensions()[1];
    vd_z = mMask->GetDimensions()[2];
    float* pData = (float*)mMask->GetScalarPointer();
    for (k = 0; k < vd_z; k++)
    {
      for (j = 0; j < vd_y; j++)
      {
        for (i = 0; i < vd_x; i++)
        {
          *pData = 0;
          pData++;
        }
      }
    }
  }
}
void fMainWindow::updateDrawMode(int shapeMode)
{
  if (shapeMode >= 0)
  {
    m_drawShapeMode = SHAPE_MODE(shapeMode);
  }
  if (m_drawShapeMode == SHAPE_MODE_NONE)
  {
    AxialViewWidget->unsetCursor();
    CoronalViewWidget->unsetCursor();
    SaggitalViewWidget->unsetCursor();
    return;
  }

  int color[4] = { 0, 0, 0, 0 };
  int drawLable= this->getSelectedDrawLabel();
  int drawSize = this->getSelectedDrawSize();

  switch (drawLable)
  {
  case DRAW_MODE_LABEL_1: // near
    color[0] = 255;
    color[1] = 0;
    color[2] = 0;
    color[3] = 255;
    break;
  case DRAW_MODE_LABEL_2: // far
    color[0] = 0;
    color[1] = 255;
    color[2] = 0;
    color[3] = 255;
    break;
  case DRAW_MODE_LABEL_3:
    color[0] = 255;
    color[1] = 255;
    color[2] = 0;
    color[3] = 255;
    break;
  case DRAW_MODE_LABEL_4:
    color[0] = 0;
    color[1] = 0;
    color[2] = 255;
    color[3] = 255;
    break;
  case DRAW_MODE_LABEL_5:
    color[0] = 255;
    color[1] = 0;
    color[2] = 255;
    color[3] = 255;
    break;


  default:
    break;
  }
  if (m_drawShapeMode == SHAPE_MODE_ERACER)
  {
    color[0] = 255;
    color[1] = 255;
    color[2] = 255;
    color[3] = 255;
  }



  QImage img(33, 33, QImage::Format_ARGB32);
  for (int j = -16; j <= 16; j++)
  {
    for (int i = -16; i <= 16; i++)
    {
      if (abs(i) <= drawSize && abs(j) <= drawSize)
      {
        img.setPixel(i + 16, j + 16, qRgba(color[0], color[1], color[2], color[3]));
      }
      else
      {
        img.setPixel(i + 16, j + 16, qRgba(0, 0, 0, 0));
      }
    }
  }
  AxialViewWidget->setCursor(QCursor(Qt::CrossCursor));
  CoronalViewWidget->setCursor(QCursor(Qt::CrossCursor));
  SaggitalViewWidget->setCursor(QCursor(Qt::CrossCursor));

  
}

void fMainWindow::LoadNonViewingImages(const std::string &directoryname, const int &imagetype_int, const int &imagesubtype)
{
  for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
  {
    if (mNonViewingImageManager[index]->GetPathFileName() == directoryname)
    {
      return;
    }
  }
  SimpleImageManager* nonViewingImage = new SimpleImageManager();
  nonViewingImage->SetPathFileName(directoryname);
  nonViewingImage->SetFileName(directoryname);
  if (imagetype_int == IMAGE_NIFTI)
    nonViewingImage->ReadGivenNonViewingNiftiImage(directoryname, imagesubtype);
  else
    nonViewingImage->ReadGivenNonViewingDicomImage(directoryname, imagesubtype);

  if (nonViewingImage->GetLastEncounteredError() != "")
    delete nonViewingImage;
  else
    mNonViewingImageManager.push_back(nonViewingImage);


  //Add image Information in NonViewing images table
  //----------------------------------------------
  nonViewingImage->mImageType = imagetype_int;
  nonViewingImage->mImageSubType = imagesubtype;

  std::string strImageType;
  if (nonViewingImage->mImageSubType == IMAGE_TYPE_DTI)
    strImageType = "DTI";

  int rowindex = m_nonVisImagesTable->rowCount();
  m_nonVisImagesTable->setRowCount(rowindex + 1);

  QFileInfo fileinfo(nonViewingImage->GetFileName().c_str());
  QString id = directoryname.c_str() + QString::number(mNonViewingImageManager.size() - 1);
  {
    QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
    item->setData(Qt::UserRole, directoryname.c_str());

    QTablePushButton* cButton = new QTablePushButton;
    cButton->setItem(item);
    cButton->setText(tr("X"));
    cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);


    QString in = QString::fromStdString(strImageType);
    QTableWidgetItem *item2 = new QTableWidgetItem(in);
    item2->setData(Qt::UserRole, in.toStdString().c_str());

    if (nonViewingImage->mImageSubType == IMAGE_TYPE_DTI)
      connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseNonViewingDTIImage(QTableWidgetItem*)));

    m_nonVisImagesTable->setCellWidget(rowindex, TAB_IMAGES_COLUMN_CLOSE, cButton);
    m_nonVisImagesTable->setItem(rowindex, TAB_IMAGES_COLUMN_NAME, item);
    m_nonVisImagesTable->setItem(rowindex, TAB_IMAGES_COLUMN_TYPE, item2);

    m_nonVisImagesTable->resizeRowsToContents();
  }
}



void fMainWindow::LoadSlicerImages(const string &fileName, const int &imagetype_int, const int & bSkipDup)
{
  cbica::ImageInfo imageInfo = cbica::ImageInfo(fileName);
  if (imageInfo.GetImageDimensions() == 4)
  {

    image4DSlider->setEnabled(true);
    image4DSlider->setRange(0, imageInfo.GetImageSize()[3] - 1);
  }
  else if (imageInfo.GetImageDimensions() == 3)
  {
    //imagesubtype = IMAGE_TYPE_UNDEFINED;
  }
  else
  {
    ShowErrorMessage("Unknown file: " + fileName);
    return; // 
  }

  bool bFirstLoad = false;
  if (mSlicerManagers.size() == 0)
  {
    bFirstLoad = true;
  }

  
  if (bSkipDup)//TBD check for the size integrity aswell here 
  {
    bool bDup = false;
    for (int j = 0; j < (int)mSlicerManagers.size(); j++)
    {
      if (fileName == mSlicerManagers[j]->GetPathFileName())
      {
        return;
      }
    }
  }

  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  

  SlicerManager* imageManager = new SlicerManager(3, mLandmarks, mSeedPoints, mTissuePoints);
  imageManager->mImageSubType = IMAGE_TYPE_UNDEFINED;
  if (imageInfo.GetImageDimensions() == 4)
  {
    ImageTypeFloat4D::Pointer imagePerf = cbica::ReadImage<ImageTypeFloat4D>(fileName);
    imageManager->SetPerfImage(imagePerf);
    imageManager->mImageSubType = IMAGE_TYPE_PERFUSION;
  }
  else
  {
    ImageTypeFloat3D::Pointer image = cbica::ReadImageWithOrientFix(fileName);
    imageManager->SetImage(image);
    imageManager->mImageSubType = guessImageType(fileName);


  }
  mInputPathName = cbica::getFilenamePath(fileName).c_str();
  imageManager->SetFilename(fileName);
  imageManager->SetMask(mMask);
  int rowIndex = (int)mSlicerManagers.size();

  m_imagesTable->setRowCount(rowIndex + 1);
  mSlicerManagers.push_back(imageManager);


  QFileInfo fileinfo(imageManager->GetFileName().c_str());
  QString id = fileName.c_str() + QString::number(mSlicerManagers.size() - 1);
  //
  std::string strImageType = " IMAGE ";
  {
    QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
    item->setData(Qt::UserRole, id.toStdString().c_str());
    item->setFlags(item->flags() & ~Qt::ItemIsEditable);

    QTablePushButton* cButton = new QTablePushButton;
    cButton->setItem(item);
    cButton->setText(tr("X"));
    connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));

    QLabel * label = new QLabel;
    label->setText(QString::fromStdString(strImageType));
    m_imagesTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_CLOSE, cButton);
    m_imagesTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_TYPE, label);
    m_imagesTable->setItem(rowIndex, TAB_IMAGES_COLUMN_NAME, item);
  }

  imagesPanel->NewImageLoaded(id, imageManager->GetFileName(), rowIndex, strImageType, imageManager->mImageSubType, this);



  QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
  item->setData(Qt::UserRole, id.toStdString().c_str());

  QTablePushButton* cButton = new QTablePushButton;
  cButton->setItem(item);
  cButton->setText(tr("X"));
  connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));

  mSlicerManagers.back()->SetId(id.toStdString());
  connect(mSlicerManagers.back(), SIGNAL(LeftButtonReleaseSignal(int)), this, SLOT(propogateSlicerPosition(int)));
  connect(mSlicerManagers.back(), SIGNAL(currentImageChanged(std::string)), this, SLOT(CurrentImageChanged(std::string)));
  connect(mSlicerManagers.back(), SIGNAL(currentPickedImageChanged(std::string)), this, SLOT(CurrentPickedImageChanged(std::string)));
  connect(mSlicerManagers.back(), SIGNAL(UpdatePosition(int, double, double, double, double, double, double, double)), this, SLOT(MousePositionChanged(int, double, double, double, double, double, double, double)));
  connect(mSlicerManagers.back(), SIGNAL(WindowLevelChanged()), this, SLOT(WindowLevelChanged()));
  connect(mSlicerManagers.back(), SIGNAL(UpdateSlice(int, int)), this, SLOT(UpdateSlice(int, int)));
  connect(mSlicerManagers.back(), SIGNAL(UpdateSliceRange(int, int, int)), this, SLOT(UpdateSliceRange(int, int, int)));
  connect(mSlicerManagers.back(), SIGNAL(UpdateLinkManager(std::string, int, double, double, double)), this, SLOT(UpdateLinkManager(std::string, int, double, double, double)));
  connect(mSlicerManagers.back(), SIGNAL(ChangeImageWithOrder(SlicerManager*, int)), this, SLOT(ChangeImageWithOrder(SlicerManager*, int)));
  connect(mSlicerManagers.back(), SIGNAL(UpdateBorderWidgetInMain(double, double, double, double)), this, SLOT(UpdateBorderWidget(double, double, double, double)));
  connect(mSlicerManagers.back(), SIGNAL(UpdateBorderWidgetInMain(double, double)), this, SLOT(UpdateBorderWidget(double, double)));
  bool bres = connect(mSlicerManagers.back(), SIGNAL(UpdateActionInMain(const QVariantList&)), this, SLOT(UpdateActionQ(const QVariantList&)));
  assert(bres); // Qt cannot directly connect the custome signals TBD

  connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded()), tumorPanel, SLOT(sAddPoint()));
  connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded(int, bool)), tumorPanel, SLOT(sAddPoint(int, bool)));
  connect(mSlicerManagers.back(), SIGNAL(TissuePointsAdded(int)), tumorPanel, SLOT(tAddPoint(int)));

  InitSlicers();

  if (bFirstLoad)
  {
    InitMask(mSlicerManagers.back()->mImage);
  }
  for (int j = 0; j < (int)mSlicerManagers.back()->mSlicers.size(); j++)
  {
    mSlicerManagers.back()->mSlicers[j]->SetMask(mSlicerManagers.back()->GetMask());
  }
  if (fileName.find("scan_label_map") != std::string::npos)
  {
    mSlicerManagers.back()->SetPreset(PRESET_LABEL);
  }
  if (fileName.find("gt") != std::string::npos)
  {
    mSlicerManagers.back()->SetPreset(PRESET_LABEL2);
  }

  if (mSlicerManagers.size()>0)
  {
    AxialViewWidget->show();
    CoronalViewWidget->show();
    SaggitalViewWidget->show();
    infoPanel->show();

    //overlayUse->setEnabled(true);

    windowLabel->setEnabled(true);
    windowSpinBox->setEnabled(true);
    levelLabel->setEnabled(true);
    levelSpinBox->setEnabled(true);
    presetLabel->setEnabled(true);
    presetComboBox->setEnabled(true);

    if (bFirstLoad)
    {
      for (int i = 0; i < 3; i++)
      {
        mSlicerManagers.back()->GetSlicer(i)->SetInitPosition();
      }
      QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
      item->setSelected(true);
      DisplayChanged(item);
    }
    else
    {
      QTableWidgetItem* item = NULL;
      for (int i = 0; i < (int)mSlicerManagers.size(); i++)
      {
        item = GetItemFromSlicerManager(mSlicerManagers[i]);
        if (item->isSelected())
        {
          break;
        }
      }
      DisplayChanged(item);
    }

    if (mSlicerManagers.size() > 1)
    {
      for (int i = 0; i < (int)mSlicerManagers.size(); i++)
      {
        for (int j = i + 1; j < (int)mSlicerManagers.size(); j++)
        {
          AddLink(/*QString::fromStdString*/(mSlicerManagers[i]->GetId().c_str()), /*QString::fromStdString*/(mSlicerManagers[j]->GetId().c_str()));
        }
      }
    }
    if (bFirstLoad)
    {
      QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
      item->setSelected(true);
      DisplayChanged(item);
    }
    else
    {
      QTableWidgetItem* item = NULL;
      for (int i = 0; i < (int)mSlicerManagers.size(); i++)
      {
        item = GetItemFromSlicerManager(mSlicerManagers[i]);
        if (item->isSelected())
        {
          break;
        }
      }
      DisplayChanged(item);
    }

    InitDisplay();
  }
  propogateSlicerPosition();
  updateProgress(0);
  QApplication::restoreOverrideCursor();
}

void fMainWindow::CurrentImageChanged(std::string &id)
{
  if (id == mCurrentSelectedImageId) 
  {
    return;
  }
  int selected = 0;
  for (int i = 0; i < m_imagesTable->rowCount(); i++)
  {
	if (m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString().toStdString() == id)
    {
      selected = i;
    }
    else 
    {
		m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->setSelected(false);
    }
  }
  m_imagesTable->item(selected, TAB_IMAGES_COLUMN_NAME)->setSelected(true);
  mCurrentSelectedImageId = id;
  emit SelectedImageHasChanged(mSlicerManagers[selected]);
  m_imagesTable->resizeColumnsToContents();
  m_imagesTable->resizeRowsToContents();
}

void fMainWindow::propogateSlicerPosition(int slicerId, int imageId)
{
	//TBD this not a proper fix a lot work needs to be done to make slicer, slicerManager slicerManagerCommand to be made clean OOP
	if (imageId < 0)
	{
		QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
		if (!items.empty()) 
		{
			imageId = GetSlicerIndexFromItem(items[0]);
		}

	}
	if (imageId < 0 || imageId >= (int)mSlicerManagers.size()) 
	{
		return;
	}
	const int MAX_SLICES = 3;
	if (slicerId < 0 || slicerId >= MAX_SLICES)
	{
		return;
	}
	double* pos = mSlicerManagers[imageId]->GetSlicer(slicerId)->GetCurrentPosition();
	for (size_t r = 0; r < mSlicerManagers.size(); r++)
	{
		for (size_t i = 0; i < MAX_SLICES; i++)
		{
			mSlicerManagers[r]->GetSlicer(i)->SetCurrentPosition(pos[0], pos[1], pos[2]);
		}
	}
}
void fMainWindow::CurrentPickedImageChanged(std::string id)
{
  if (id == mCurrentPickedImageId) {
    return;
  }
  int selected = 0;
  for (int i = 0; i < m_imagesTable->rowCount(); i++) {
	  if (m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString().toStdString() == id) {
      selected = i;
    }
    else {
		m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->setSelected(false);
    }
  }
  m_imagesTable->item(selected, TAB_IMAGES_COLUMN_NAME)->setSelected(true);
  mCurrentPickedImageId = id;
  mCurrentPickedImageIndex = selected;
}

void fMainWindow::ImageInfoChanged()
{
	QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size()) {
    return;
  }

  vtkSmartPointer<vtkMatrix4x4> transformation;
  QString image = m_imagesTable->selectedItems()[0]->data(Qt::DisplayRole).toString();

  vtkSmartPointer<vtkImageData> imageSelected;
  vtkSmartPointer<vtkTransform> transformSelected;
  imageSelected = mSlicerManagers[index]->GetSlicer(0)->GetImage();
  transformSelected = mSlicerManagers[index]->GetSlicer(0)->GetTransform();

  //int dimension = GetNumberOfDimensions(imageSelected);
  std::string vtktype = mSlicerManagers[index]->GetImage()->GetScalarTypeAsString();
  QString pixelType = vtktype.c_str();

  infoPanel->setFileName(image);
  infoPanel->setSizePixel(imageSelected->GetDimensions()[0], imageSelected->GetDimensions()[1], imageSelected->GetDimensions()[2]);
  infoPanel->setOrigin(imageSelected->GetOrigin()[0], imageSelected->GetOrigin()[1], imageSelected->GetOrigin()[2]);
  infoPanel->setSpacing(imageSelected->GetSpacing()[0], imageSelected->GetSpacing()[1], imageSelected->GetSpacing()[2]);
  transformation = transformSelected->GetMatrix();
  tumorPanel->SetCurrentSPoints(mSlicerManagers[index]->mSeedPoints);
  tumorPanel->SetCurrentTPoints(mSlicerManagers[index]->mTissuePoints);
  tumorPanel->SetCurrentPath(mInputPathName.toStdString());

  for (int i = 0; i < 3; i++)
  {
    mSlicerManagers[index]->UpdateInfoOnCursorPosition(i);
  }

  WindowLevelChanged();

  // reset SliceManager order
  for (int j = 0; j < (int)mSlicerManagers.size(); j++) {
    mSlicerManagers[j]->SetOrder(j);
  }
  for (int i = 0; i < m_imagesTable->rowCount(); i++) {
	  QString id_table = m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString();
    for (int j = 0; j < (int)mSlicerManagers.size(); j++) {
      QString id_sm = mSlicerManagers[j]->GetId().c_str();
      if (id_table == id_sm) {
        mSlicerManagers[j]->SetOrder(i);
        break;
      }
    }
  }
}

void fMainWindow::DisplayChanged()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
  {
    return;
  }
  DisplayChanged(items[0]);
}
void fMainWindow::DisplayChanged(QTableWidgetItem *clickedItem)
{
  int slicerManagerIndex = GetSlicerIndexFromItem(clickedItem);
  if (slicerManagerIndex < 0 || slicerManagerIndex >= (int)mSlicerManagers.size())
  {
    return;
  }

  QTableWidgetItem* clickedParentItem = m_imagesTable->item(slicerManagerIndex, TAB_IMAGES_COLUMN_NAME);
  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
	QTableWidgetItem* currentParentItem = m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME);
    if (currentParentItem != clickedParentItem)
    {
      for (int j = 0; j < 3; j++)
      {
        mSlicerManagers[i]->UpdateSlicer(j, false);
      }
    }
    else
    {
      int VisibleInWindow = 0;
      for (int j = 0; j < 3; j++)
      {
        mSlicerManagers[i]->UpdateSlicer(j, true);
        mSlicerManagers[i]->UpdateInfoOnCursorPosition(j);
        DisplaySliders(i, j);
        //
        if (mSlicerManagers[i]->GetSlicer(j)->GetActive())
        {
          VisibleInWindow = j;
        }
      }

      mSlicerManagers[i]->Picked();
      mSlicerManagers[i]->UpdateViews(VisibleInWindow);
      mSlicerManagers[i]->UpdateLinked(VisibleInWindow);
      mSlicerManagers[i]->UpdateInfoOnCursorPosition(VisibleInWindow);
    }

    for (int j = 0; j < 3; j++)
    {
      mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
    }
  }
  UpdateRenderWindows();

  ImageInfoChanged();
}



int fMainWindow::GetSlicerIndexFromItem(QTableWidgetItem* item)
{
  if (item != NULL) {
    QString id = item->data(Qt::UserRole).toString();
    std::string id_string = id.toStdString();
	for (int i = 0; i < m_imagesTable->rowCount(); i++)
    {
	  QString id_table_string = m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString();
      std::string table_string = id_table_string.toStdString();
	  if (m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString() == id)
      {
        return i;
      }
    }
  }
  return -1;
}

QTableWidgetItem* fMainWindow::GetItemFromSlicerManager(SlicerManager* sm)
{
  QString id = sm->GetId().c_str();
  for (int i = 0; i < m_imagesTable->rowCount(); i++)
  {
	if (m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString() == id)
    {
		return m_imagesTable->item(i, TAB_IMAGES_COLUMN_NAME);
    }
  }
  return NULL;
}

void fMainWindow::InitSlicers()
{
  if (mSlicerManagers.size()) {
    mSlicerManagers.back()->GenerateDefaultLookupTable();

    mSlicerManagers.back()->SetSlicerWindow(0, AxialViewWidget->GetRenderWindow());
    mSlicerManagers.back()->SetSlicerWindow(1, CoronalViewWidget->GetRenderWindow());
    mSlicerManagers.back()->SetSlicerWindow(2, SaggitalViewWidget->GetRenderWindow());
  }
}

void fMainWindow::InitDisplay()
{
  if (mSlicerManagers.size())
  {
    for (int j = 0; j < 3; j++)
    {
      InteractorStyleNavigator* style = InteractorStyleNavigator::New();
      style->SetAutoAdjustCameraClippingRange(1);
	    for (int i = 0; i < m_imagesTable->rowCount(); i++)
      {
        mSlicerManagers[i]->SetInteractorStyleNavigator(j, style);
        //
        mSlicerManagers[i]->updateToRefCam(mSlicerManagers[i]->GetSlicer(0));
        mSlicerManagers[i]->GetSlicer(j)->SetInitPosition();
      }
      style->Delete();
    }
  }
}

void fMainWindow::DisplaySliders(int slicer, int window)
{
  int range[2];
  mSlicerManagers[slicer]->GetSlicer(window)->GetSliceRange(range);
  int position = mSlicerManagers[slicer]->GetSlicer(window)->GetSlice();

  bool showVertical = false;
  if (GetNumberOfDimensions(mSlicerManagers[slicer]->GetSlicer(window)->GetImage()) >= 3) {
    showVertical = true;
  }
  if (showVertical) {
    verticalSliders[window]->show();
  }
  else {
    verticalSliders[window]->hide();
  }
  verticalSliders[window]->setRange(range[0], range[1]);
  verticalSliders[window]->setValue(position);
}

void fMainWindow::CloseImage(QTableWidgetItem* item)
{
  int index = GetSlicerIndexFromItem(item);
  if (index < 0 || index >= (int)mSlicerManagers.size())
  {
    return;
  }
  if (mSlicerManagers.size() > 1)
  {
    for (int k = 0; k < (int)mSlicerManagers.size() - 1; k++)
    {
      if (k != index)
      {
        RemoveLink(/*QString::fromStdString*/(mSlicerManagers[k]->GetId()), /*QString::fromStdString*/(mSlicerManagers[index]->GetId()));
      }
    }
  }
  std::vector<SlicerManager*>::iterator Manageriter = mSlicerManagers.begin();
  for (int i = 0; i < index; i++)
  {
    Manageriter++;
  }

  mSlicerManagers[index]->RemoveActors();
  delete mSlicerManagers[index];
  mSlicerManagers.erase(Manageriter);
  m_imagesTable->removeRow(index);

  if (mSlicerManagers.size() >= 1)
  {
    QTableWidgetItem* item_tmp = GetItemFromSlicerManager(mSlicerManagers.back());
    item_tmp->setSelected(true);
    DisplayChanged(item_tmp);
  }
  else
  {
    this->ResetNumberOfPoints();
    AxialViewWidget->hide();
    CoronalViewWidget->hide();
    SaggitalViewWidget->hide();

    for (int i = 0; i < 3; i++)
    {
      verticalSliders[i]->hide();
    }
    //
    windowLabel->setEnabled(false);
    windowSpinBox->setEnabled(false);
    levelLabel->setEnabled(false);
    levelSpinBox->setEnabled(false);
    thresholdLabel->setEnabled(false);
    thresholdSlider->setEnabled(false);
    presetLabel->setEnabled(false);
    presetComboBox->setEnabled(false);

    mLandmarks->Clear();
    mSeedPoints->Clear();
    mTissuePoints->Clear();
  }

  InitDisplay();
}

void fMainWindow::MousePositionChanged(int visibility, double x, double y, double z, double X, double Y, double Z, double value)
{
  infoPanel->setCurrentInfo(visibility, x, y, z, X, Y, Z, value);
  tumorPanel->HighlightCurrentSelctedPoints(x, y, z, X, Y, Z, value);
}

void fMainWindow::WindowLevelChanged()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size())
  {
    return;
  }
  windowSpinBox->setValue(mSlicerManagers[index]->GetColorWindow());
  levelSpinBox->setValue(mSlicerManagers[index]->GetColorLevel());
  thresholdSlider->setValue(mSlicerManagers[index]->GetThresholdIndex());
  presetComboBox->setCurrentIndex(mSlicerManagers[index]->GetPreset());
  if (presetComboBox->currentIndex() == PRESET_THRESHOLD)
  {
    thresholdLabel->setEnabled(true);
    thresholdSlider->setEnabled(true);
  }
  else
  {
    thresholdLabel->setEnabled(false);
    thresholdSlider->setEnabled(false);
  }
}

void fMainWindow::WindowLevelEdited()
{
  presetComboBox->setCurrentIndex(PRESET_USER);
  UpdateWindowLevel();
}

void fMainWindow::SetWindowLevel(double w, double l)
{
  //windowSlider->setValue(w);
  windowSpinBox->setValue(w);
  levelSpinBox->setValue(l);
  //levelSlider->setValue(l);
  presetComboBox->setCurrentIndex(PRESET_USER);
  UpdateWindowLevel();
}

void fMainWindow::UpdateWindowLevel()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size()) 
  {
    mSlicerManagers[index]->SetColorWindow(windowSpinBox->value());
    mSlicerManagers[index]->SetColorLevel(levelSpinBox->value());
    mSlicerManagers[index]->SetPreset(presetComboBox->currentIndex());
    mSlicerManagers[index]->Render();
    //
    if (presetComboBox->currentIndex() == PRESET_THRESHOLD) {
      thresholdLabel->setEnabled(true);
      thresholdSlider->setEnabled(true);
    }
    else {
      thresholdLabel->setEnabled(false);
      thresholdSlider->setEnabled(false);
    }
    //
    WindowLevelChanged();
  }
}

void fMainWindow::thresholdSliderChanged()
{
  if (mthresholdImage)
  {
    thresholdLabel->setEnabled(true);
    thresholdSlider->setEnabled(true);

	QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
    if (items.empty()) {
      return;
    }
    int index = GetSlicerIndexFromItem(items[0]);
    if (index >= 0 && index < (int)mSlicerManagers.size()) 
    {
      mSlicerManagers[index]->SetThresholdIndex(thresholdSlider->value());
      mSlicerManagers[index]->SetPreset(mSlicerManagers[index]->GetPreset());
      mSlicerManagers[index]->Render();
      WindowLevelChanged();
    }
  }
  else if (mthresholdMask)
  {
    thresholdLabel->setEnabled(true);
    thresholdSlider->setEnabled(true);

    if (mCustomImageToThreshold_max != 0)
    {
      typedef itk::BinaryThresholdImageFilter< GenericImage, GenericImage > BinaryThresholderType;
      BinaryThresholderType::Pointer binaryThresholdFilter = BinaryThresholderType::New();
      binaryThresholdFilter->SetLowerThreshold(mCustomImageToThreshold_min);
      binaryThresholdFilter->SetUpperThreshold(thresholdSlider->value());
      binaryThresholdFilter->SetInput(mCustomImageToThreshold);
      binaryThresholdFilter->SetOutsideValue(static_cast<GenericImage::PixelType>(0));
      binaryThresholdFilter->SetInsideValue(static_cast<GenericImage::PixelType>(1));
      binaryThresholdFilter->Update();

      typedef itk::ImageRegionIteratorWithIndex <GenericImage> IteratorType;
      IteratorType maskIt(binaryThresholdFilter->GetOutput(), binaryThresholdFilter->GetOutput()->GetLargestPossibleRegion());
      maskIt.GoToBegin();
      while (!maskIt.IsAtEnd())
      {
        if (maskIt.Get() > 0)
        {
          GenericImage::IndexType currentIndex = maskIt.GetIndex();
          float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)currentIndex[0], (int)currentIndex[1], (int)currentIndex[2]);
          *pData = 1;

        }
        ++maskIt;
      }

    }

    this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
    this->mSlicerManagers[0]->Render();
  }
}

void fMainWindow::UpdateLinkManager(std::string &id, int slicer, double x, double y, double z)
{
  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
    if (mSlicerManagers[i]->GetId() == id)
    {
      mSlicerManagers[i]->GetSlicer(slicer)->SetCurrentPosition(x, y, z);
      mSlicerManagers[i]->UpdateViews(slicer);
      break;
    }
  }
}
void fMainWindow::UpdateLinkedNavigation(Slicer* refSlicer)
{
  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
      mSlicerManagers[i]->updateToRefCam(refSlicer);
  }
}

void fMainWindow::CloseImage()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  CloseImage(items[0]);
}
void fMainWindow::CloseAllImages()
{
	int rows = m_imagesTable->rowCount();
	while (m_imagesTable->rowCount()>0)
	{
		m_imagesTable->selectRow(0);//TBD speedup this 
		CloseImage();
	}
}
void fMainWindow::ResetTransformationToIdentity()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size()) {
    mSlicerManagers[index]->ResetTransformationToIdentity();
    ImageInfoChanged();
  }
}

void fMainWindow::AddLink(const std::string &image1, const std::string &image2)
{
  unsigned int sm1 = 0;
  unsigned int sm2 = 0;

  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
    if (image1/*.toStdString()*/ == mSlicerManagers[i]->GetId())
    {
      mSlicerManagers[i]->AddLink(image2/*.toStdString()*/);
      sm1 = i;
    }
    if (image2/*.toStdString()*/ == mSlicerManagers[i]->GetId())
    {
      mSlicerManagers[i]->AddLink(image1/*.toStdString()*/);
      sm2 = i;
    }
  }

 /* emit UpdateLinkedNavigation(mSlicerManagers[sm1]->GetId(), mSlicerManagers[mCurrentPickedImageIndex], mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0));
  emit UpdateLinkedNavigation(mSlicerManagers[sm2]->GetId(), mSlicerManagers[mCurrentPickedImageIndex], mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0));*/
}

void fMainWindow::RemoveLink(const std::string &image1, const std::string &image2)
{
  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
  {
    if (image1/*.toStdString()*/ == mSlicerManagers[i]->GetId())
    {
      mSlicerManagers[i]->RemoveLink(image2/*.toStdString()*/);
    }
    if (image2/*.toStdString()*/ == mSlicerManagers[i]->GetId()) {
      mSlicerManagers[i]->RemoveLink(image1/*.toStdString()*/);
    }
  }
}

void fMainWindow::ChangeImageWithOrder(SlicerManager *sm, int order)
{
  if (mSlicerManagers.size() <= 1) {
    return;
  }
  if (order >= (int)mSlicerManagers.size()) {
    return;
  }
  //if (sm != mSlicerManagers[order]) {
  //	return;
  //}

  QTableWidgetItem* item;
  item = GetItemFromSlicerManager(mSlicerManagers[order]);
  item->setSelected(true);
  DisplayChanged(item);
}

void fMainWindow::AxialViewSliderChanged()
{
  static int value = -1;
  //if (value == AxialViewSlider->value()) {
  //  return;
  //}
  //else {
    value = AxialViewSlider->value();
  //}
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size()) {
    if (mSlicerManagers[index]->GetSlicer(0)->GetSlice() != value) {
      mSlicerManagers[index]->GetSlicer(0)->SetSlice(value);
      mSlicerManagers[index]->VerticalSliderHasChanged(0, value);
      mSlicerManagers[index]->UpdateSlice(0);
    }
  }
}

void fMainWindow::SaggitalViewSliderChanged()
{
  static int value = -1;
  if (value == SaggitalViewSlider->value())
  {
    return;
  }
  else {
    value = SaggitalViewSlider->value();
  }
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size()) {
    if (mSlicerManagers[index]->GetSlicer(2)->GetSlice() != value) {
      mSlicerManagers[index]->GetSlicer(2)->SetSlice(value);
      mSlicerManagers[index]->VerticalSliderHasChanged(2, value);
      mSlicerManagers[index]->UpdateSlice(2);
    }
  }
}

void fMainWindow::CoronalViewSliderChanged()
{
  static int value = -1;
  if (value == CoronalViewSlider->value())
  {
    return;
  }
  else
  {
    value = CoronalViewSlider->value();
  }
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size()) {
    if (mSlicerManagers[index]->GetSlicer(1)->GetSlice() != value) {
      mSlicerManagers[index]->GetSlicer(1)->SetSlice(value);
      mSlicerManagers[index]->VerticalSliderHasChanged(1, value);
      mSlicerManagers[index]->UpdateSlice(1);
    }
  }
}

void fMainWindow::UpdateSlice(int slicer, int slice)
{
  if (slicer == 0) 
  {
    AxialViewSlider->setValue(slice);
  }
  else if (slicer == 1) 
  {
    CoronalViewSlider->setValue(slice);
  }
  else if (slicer == 2) 
  {
    SaggitalViewSlider->setValue(slice);
  }
  propogateSlicerPosition();
}

void fMainWindow::UpdateSliceRange(int slicer, int min, int max)
{
  int position = int((min + max) / 2);
  if (slicer == 0) {
    AxialViewSlider->setValue(position);
    AxialViewSlider->setRange(min, max);
  }
  else if (slicer == 1) {
    CoronalViewSlider->setValue(position);
    CoronalViewSlider->setRange(min, max);
  }
  else if (slicer == 2) {
    SaggitalViewSlider->setValue(position);
    SaggitalViewSlider->setRange(min, max);
  }
}

void fMainWindow::UpdateRenderWindows()
{
  if (m_imagesTable->rowCount() <= 0)
  {
    return;
  }
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) {
    return;
  }
  QTableWidgetItem* item = items[0];
  if (item == NULL) {
    return;
  }
  int index = GetSlicerIndexFromItem(item);
  if (index >= 0 && index < (int)mSlicerManagers.size())
  {
    mSlicerManagers[index]->Render();
  }
  //*/
}

void fMainWindow::SetActiveLandmarksType(int type, int row, int col)
{
  mCurrentLandmarkTissueType = row;
  if (type == LANDMARK_TYPE_LANDMARKS)
  {
    emit LandmarksFocused(true);
    emit SeedPointsFocused(false);
    emit TissuePointsFocused(false);
  }
  else if (type == LANDMARK_TYPE_SEED_POINTS)
  {
    emit LandmarksFocused(false);
    emit SeedPointsFocused(true);
    emit TissuePointsFocused(false);
  }
  else if (type == LANDMARK_TYPE_TISSUE_POINTS)
  {
    emit LandmarksFocused(false);
    emit SeedPointsFocused(false);
    emit TissuePointsFocused(true);
  }
  else
  {
    emit LandmarksFocused(false);
    emit SeedPointsFocused(false);
    emit TissuePointsFocused(false);
  }

  for (int i = 0; i < (int)mSlicerManagers.size(); i++)
    mSlicerManagers[i]->SetCurrentLandmarksType(type, row, col);

  UpdateRenderWindows();
}
void fMainWindow::SetActiveLandmarkTypeTab(int current)
{
  if (current == TAB_IMAGES)
  {
    SetActiveLandmarksType(LANDMARK_TYPE_NONE, 0, 0);
  }
  else if (current == TAB_TUMOR)
  {
    SetActiveLandmarksType(LANDMARK_TYPE_NONE, 0, 0);
    tumorPanel->SetCurrentSelectedTissueType();
  }
  else if (current == TAB_LANDMARKS)
  {
    SetActiveLandmarksType(LANDMARK_TYPE_LANDMARKS, 0, 0);
  }
}

void fMainWindow::MoveSlicerCursor(double x, double y, double z, int mode)
{
  if (mCurrentPickedImageIndex < 0 || mCurrentPickedImageIndex >= (int)mSlicerManagers.size()) {
    return;
  }
  //
  if (mode == 0) 
  {
    // x, y, z are LPS
    mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->SetCurrentPosition(x, y, z);
    //
    mSlicerManagers[mCurrentPickedImageIndex]->Picked();
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateViews(0);
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateLinked(0);
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateInfoOnCursorPosition(0);
  }
  else if (mode == 1) 
  {
    // x, y, z are pixel
    x = x * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[0] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[0];
    y = y * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[1] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[1];
    z = z * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[2] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[2];
    //
    mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->SetCurrentPosition(x, y, z);
    //
    mSlicerManagers[mCurrentPickedImageIndex]->Picked();
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateViews(0);
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateLinked(0);
    mSlicerManagers[mCurrentPickedImageIndex]->UpdateInfoOnCursorPosition(0);
  }
  propogateSlicerPosition();
}

VectorVectorDouble fMainWindow::FormulateDrawingPointsForEdemaSegmentation()
{
  VectorVectorDouble Indices;
  if (mSlicerManagers.size() <= 0)
    return Indices;

  typedef itk::Image<short, 3> ImageType;
  ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (maskIt.Get() == 1 || maskIt.Get() == 2)
    {
      VectorDouble localIndex;
      localIndex.push_back(maskIt.GetIndex()[0]);
      localIndex.push_back(maskIt.GetIndex()[1]);
      localIndex.push_back(maskIt.GetIndex()[2]);

      Indices.push_back(localIndex);
    }
    ++maskIt;
  }
  return Indices;
}

VectorVectorDouble fMainWindow::GetMaskLabelIndices(const int label)
{
  VectorVectorDouble Indices;
  if (mSlicerManagers.size() <= 0)
    return Indices;

  typedef itk::Image<short, 3> ImageType;
  ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (maskIt.Get() == label)
    {
      VectorDouble localIndex;
      localIndex.push_back(maskIt.GetIndex()[0]);
      localIndex.push_back(maskIt.GetIndex()[1]);
      localIndex.push_back(maskIt.GetIndex()[2]);

      Indices.push_back(localIndex);
    }
    ++maskIt;
  }
  return Indices;
}
VectorVectorDouble fMainWindow::FormulateDrawingPointsForTumorSegmentation()
{
  VectorVectorDouble Indices;
  if (mSlicerManagers.size() <= 0)
    return Indices;

  typedef itk::Image<short, 3> ImageType;
  ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (maskIt.Get() == 1)
    {
      VectorDouble localIndex;
      localIndex.push_back(maskIt.GetIndex()[0]);
      localIndex.push_back(maskIt.GetIndex()[1]);
      localIndex.push_back(maskIt.GetIndex()[2]);

      Indices.push_back(localIndex);
    }
    ++maskIt;
  }
  return Indices;
}

void fMainWindow::SaveDicomImage()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;
  int index = GetSlicerIndexFromItem(items[0]);

  typedef short PixelType;
  const unsigned int Dimensions = 3;

  QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), mInputPathName, QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
  if (directory.isNull())
    return;

  std::string directory_wrap = directory.toStdString();
  if (directory_wrap[directory_wrap.length() - 1] != '/')
  {
    directory_wrap += "/";
  }

  // check write access
  //if (((_access(directory_wrap.c_str(), 2)) == -1) || ((_access(directory_wrap.c_str(), 6)) == -1))
  //{
  //  ShowErrorMessage("You don't have write access in selected location. Please check.");
  //  return;
  //}

  typedef itk::Image<PixelType, Dimensions> ImageType;
  typedef itk::CastImageFilter< itk::Image< float, Dimensions >, ImageType > CastFilterType;
  CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(mSlicerManagers[index]->mITKImage);
  castFilter->Update();

  const unsigned int OutputDimension = 2;
  typedef itk::Image<PixelType, OutputDimension> DicomImage2DType;
  typedef itk::GDCMImageIO ImageIOType;
  ImageIOType::Pointer dicomIO = ImageIOType::New();
  typedef itk::ImageSeriesWriter<ImageType, DicomImage2DType> SeriesWriterType;
  SeriesWriterType::Pointer serieswriter = SeriesWriterType::New();
  serieswriter->SetFileName(directory_wrap);
  serieswriter->SetImageIO(dicomIO);
  serieswriter->SetInput(castFilter->GetOutput());

}

void fMainWindow::SaveDicomDrawing()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;
  int index = GetSlicerIndexFromItem(items[0]);

  typedef short MaskPixelType;
  const unsigned int Dimensions = 3;

  typedef itk::Image<MaskPixelType, Dimensions> ImageType;
  ImageType::Pointer imageToWrite = convertVtkToItk<MaskPixelType, Dimensions>(mSlicerManagers[index]->mMask);

  typedef itk::MinimumMaximumImageCalculator< ImageType > CalculatorType;
  CalculatorType::Pointer calculator = CalculatorType::New();
  calculator->SetImage(imageToWrite);
  calculator->Compute();

  if (calculator->GetMaximum() == 0) // this means that the mask image contains no values at all
  {
    ShowErrorMessage("There should be at least one region to save.", "Notice");
    return;
  }
  
  QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), mInputPathName, QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
  if (directory.isNull())
    return;

  std::string directory_wrap = directory.toStdString();
  if (directory_wrap[directory_wrap.length() - 1] != '/')
  {
    directory_wrap += "/";
  }

  if (cbica::getFilenamePath(mSlicerManagers[index]->mSeriesReader->GetFileNames()[0]) == directory_wrap)
  {
    ShowErrorMessage("Cannot save to source directory. Please select another.", "Notice");
    return;
  }

  try
  {
    cbica::WriteDicomImage<ImageType>(mSlicerManagers[index]->mSeriesReader, imageToWrite, directory_wrap);
  }
  catch (itk::ExceptionObject & excp)
  {
    ShowErrorMessage("Couldn't write mask as DICOM: " + std::string(excp.GetDescription()));
    return;
  }

  statusBar()->showMessage("Success", 5);
}

void fMainWindow::SaveDrawing()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;
  int index = GetSlicerIndexFromItem(items[0]);

  typedef unsigned short MaskPixelType;
  const unsigned int Dimensions = 3;

  typedef itk::Image<MaskPixelType, Dimensions> ImageType;
  ImageType::Pointer imageToWrite = convertVtkToItk<MaskPixelType, Dimensions>(mSlicerManagers[index]->mMask);

  typedef itk::MinimumMaximumImageCalculator< ImageType > CalculatorType;
  CalculatorType::Pointer calculator = CalculatorType::New();
  calculator->SetImage(imageToWrite);
  calculator->Compute();

  if (calculator->GetMaximum() == 0) // this means that the mask image contains no values at all
  {
    ShowErrorMessage("There should be at least one region (near or far) for saving.");
    return;
  }

  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  QFileDialog dialog/*(this, tr("Save Image"), mInputPathName, extensions)*/;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  //dialog.selectFile(QString((mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mFileName) + "_denoise.nii.gz").c_str()));
  std::string preferredName = mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mPathFileName) + "_mask.nii.gz";
  QString saveFileName = dialog.getSaveFileName(this, tr("Save Image"), preferredName.c_str(), extensions, 0, QFileDialog::DontUseNativeDialog);
  if (!saveFileName.isEmpty())
  {
    file = saveFileName;
    std::string filename = file.toStdString();

    cbica::WriteImage< ImageType >(imageToWrite, filename);

    if (cbica::isFile(filename))
    {
      statusBar()->showMessage(("Success; File Saved: " + filename).c_str(), 5);
    }
    else
    {
      ShowErrorMessage("Couldn't write to file: " + filename);
    }
  }
}

void fMainWindow::SaveSeedDrawing()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;
  int index = GetSlicerIndexFromItem(items[0]);

  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  QFileDialog dialog/*(this, tr("Save Image"), mInputPathName, extensions)*/;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  //dialog.selectFile(QString((mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mFileName) + "_denoise.nii.gz").c_str()));
  std::string preferredName = mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mPathFileName) + "_seed.nii.gz";
  QString saveFileName = dialog.getSaveFileName(this, tr("Save Image"), preferredName.c_str(), extensions, 0, QFileDialog::DontUseNativeDialog);
  if (!saveFileName.isEmpty())
  {
    file = saveFileName;
    std::string filename = file.toStdString();



    typedef itk::Image<short, 3> ImageType;
    ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
    std::vector<ImageType::IndexType> seedIndices;
    typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
    IteratorType maskIt(img, img->GetLargestPossibleRegion());
    maskIt.GoToBegin();
    while (!maskIt.IsAtEnd())
    {
      if (maskIt.Get() == INIT_POINT_LABEL)
        seedIndices.push_back(maskIt.GetIndex());
      ++maskIt;
    }
    std::string errormsg = "";
    if (seedIndices.size() == 0)
      errormsg = "Draw seed points before saving.";
    QMessageBox box(this);
    box.setIcon(QMessageBox::Information);
    box.addButton(QMessageBox::Ok);

    if (errormsg.length() > 0)
    {
      box.setText(QString::fromStdString(errormsg));
      box.setWindowTitle(tr("Error message"));
      box.exec();
      return;
    }

    //save actual near and far region according to the current image type
    std::string InputPixelType = mSlicerManagers[0]->mImage->GetScalarTypeAsString();
	std::string subjectname = m_imagesTable->selectedItems()[0]->data(Qt::DisplayRole).toString().toStdString();
    subjectname = subjectname.substr(0, subjectname.find("_"));
    if (InputPixelType == "short")
    {
      typedef itk::Image<short, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "unsigned short")
    {
      typedef itk::Image<unsigned short, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<unsigned short, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "char")
    {
      typedef itk::Image<char, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<char, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "unsigned char")
    {
      typedef itk::Image<unsigned char, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<unsigned char, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "int")
    {
      typedef itk::Image<int, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<int, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "unsigned int")
    {
      typedef itk::Image<unsigned int, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<unsigned int, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else if (InputPixelType == "double")
    {
      typedef itk::Image<double, 3> ImageType;
      ImageType::Pointer img = convertVtkToItk<double, 3>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, subjectname, seedIndices);
    }
    else if (InputPixelType == "float")
    {
      typedef ImageTypeFloat3D ImageType;
      ImageType::Pointer img = convertVtkToItk<ImageTypeFloat3D::PixelType, ImageTypeFloat3D::ImageDimension>(mSlicerManagers[0]->mImage);
      mOutputManager.WriteSeedMasks<ImageType>(img, filename, seedIndices);
    }
    else
    {
      cbica::Logging(loggerFile, "Error, input pixel type, '" + InputPixelType + "' is unknown!");
    }

    std::string msg = "Mask saved: " + filename;
    statusBar()->showMessage("Success", 5);

  }
}


void fMainWindow::makeStroke(vector<itk::Image<short, 3>::IndexType>& indices, const int value)
{
	vector<PointVal> strokePoints;
	for (unsigned int i = 0; i < indices.size(); i++)
	{
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)indices[i][0], (int)indices[i][1], (int)indices[i][2]);

		PointVal pt;
		pt.x = indices[i][0];
		pt.y = indices[i][1];
		pt.z = indices[i][1];
		pt.value = (int)*pData;
		strokePoints.push_back(pt);
		*pData = value;
	}
	UpdateAction(strokePoints);
	return;
}
void fMainWindow::clearMask(int label)
{
  if ((mSlicerManagers.size() <= 0))
    return;

  typedef itk::Image<short, 3> ImageType;
  ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
  std::vector<ImageType::IndexType> indecesToErase;
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (label < 0)//Clear all
    {
      if (maskIt.Get()>0)
      {
        indecesToErase.push_back(maskIt.GetIndex());
      }

    }
    else
    {
      if (maskIt.Get() == label)
      {
        indecesToErase.push_back(maskIt.GetIndex());
      }
    }
    ++maskIt;
  }
  makeStroke(indecesToErase, 0);
  this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
  this->mSlicerManagers[0]->Render();
  UpdateNumberOfPointsInTable();
}




template <class ImageType>
void fMainWindow::FormulateNearFarPoints(std::vector<typename ImageType::IndexType> & nearIndices, std::vector<typename ImageType::IndexType> & farIndices)
{
  if (mSlicerManagers.size() <= 0)
    return;

  // typedef itk::Image<short, 3> ImageType;
  typename ImageType::Pointer img = convertVtkToItk<typename ImageType::PixelType, 3>(mSlicerManagers[0]->mMask);
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (maskIt.Get() == 1)
    {
      nearIndices.push_back(maskIt.GetIndex());
    }
    else if (maskIt.Get() == 2)
      farIndices.push_back(maskIt.GetIndex());
    ++maskIt;
  }
}

void fMainWindow::StartEGFREstimate()
{
}

ImageTypeFloat3D::Pointer fMainWindow::getMaskImage()
{
  ImageTypeFloat3D::Pointer img = NULL;
  if (mSlicerManagers[0]->mMask != NULL)
  {
     img = convertVtkToItk<float, 3>(mSlicerManagers[0]->mMask);
  }

  return img;
}

void fMainWindow::readMaskFile(const std::string &maskFileName)
{
  typedef itk::Image<unsigned int, 3> ImageType;
  ImageType::Pointer inputImage = cbica::ReadImageWithOrientFixMask/*< ImageType >*/(maskFileName);

  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(inputImage, inputImage->GetLargestPossibleRegion());
  maskIt.GoToBegin();

  while (!maskIt.IsAtEnd())
  {
    ImageType::IndexType currentIndex = maskIt.GetIndex();
    float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)currentIndex[0], (int)currentIndex[1], (int)currentIndex[2]);
    // this is done to take into account all possible label drawings
    switch (maskIt.Get())
    { // multiLabel: uncomment everything inside this loop and remove references to "near" and "far"
    case NEAR_POINT_SAVE_LABEL:
      *pData = DRAW_MODE_LABEL_1;
      break;
    case FAR_POINT_SAVE_LABEL:
      *pData = DRAW_MODE_LABEL_2;
      break;
    case DRAW_MODE_LABEL_1:
      *pData = DRAW_MODE_LABEL_1;
      break;
    case DRAW_MODE_LABEL_2:
      *pData = DRAW_MODE_LABEL_2;
      break;
    case DRAW_MODE_LABEL_3:
      *pData = DRAW_MODE_LABEL_3;
      break;
    case DRAW_MODE_LABEL_4:
      *pData = DRAW_MODE_LABEL_4;
      break;
    case DRAW_MODE_LABEL_5:
      *pData = DRAW_MODE_LABEL_5;
      break;
    default:
      // nothing defined for other cases 
      break;
    }
    ++maskIt;
  }

  statusBar()->showMessage("Mask loaded", 5);
}

std::vector<ImageTypeFloat3D::Pointer> fMainWindow::getLodedImages(std::vector<string>& fileNames, bool onlySelected)
{
 
  std::vector < ImageTypeFloat3D::Pointer> images;
  if (onlySelected)
  {
    QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
    if (!items.empty())
    {
      int index = GetSlicerIndexFromItem(items[0]);
      images.push_back(mSlicerManagers[index]->mITKImage);
      fileNames.push_back(mSlicerManagers[index]->mFileName);
    }
  }
  else
  {
    for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
    {
      images.push_back(mSlicerManagers[index]->mITKImage);
      fileNames.push_back(mSlicerManagers[index]->mFileName);
    }
  }
  return images;
}
void fMainWindow::StartRecurrenceEstimate(const std::string &outputdirectory, bool t1DataPresent,  bool t1ceDataPresent,  bool t2DataPresent,  bool t2flairDataPresent,  bool dtiDataPresent,  bool perfusionDataPresent,  bool distanceDataPresent)
{
  int imagetype_int = IMAGE_NIFTI;
  bool useOtherModalities = false;

  mOutputManager.SetOutputDirectoryPath(outputdirectory);

  UpdateNumberOfPointsInTable(); 
  if (!this->CheckCompletenessOfInputData(t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent))
    return;

  if (t1DataPresent || t2DataPresent || t1ceDataPresent || t2flairDataPresent || dtiDataPresent)
    useOtherModalities = true;

  typedef ImageTypeFloat3D ImageType;
  ImageType::Pointer T1CEImagePointer;
  ImageType::Pointer T2FlairImagePointer;
  ImageType::Pointer T1ImagePointer;
  ImageType::Pointer T2ImagePointer;
  std::vector<ImageType::Pointer>	PerfusionImagePointer;
  std::vector<ImageType::Pointer>	DTIImagePointer;

  ImageType::Pointer FinalT1CEImagePointer;
  ImageType::Pointer FinalT2FlairImagePointer;
  ImageType::Pointer FinalT1ImagePointer;
  ImageType::Pointer FinalT2ImagePointer;
  std::vector<ImageType::Pointer>	FinalPerfusionImagePointer;
  std::vector<ImageType::Pointer>	FinalDTIImagePointer;

  std::vector<std::string> filenames;
  std::string t1cebasefilename;
  filenames.resize(6);

  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
    {
      imagetype_int = mSlicerManagers[index]->mImageType;
      T1CEImagePointer = RescaleImageIntensity(mSlicerManagers[index]->mITKImage);
      filenames[0] = mSlicerManagers[index]->mFileName;
      t1cebasefilename = mSlicerManagers[index]->mPathFileName;
    }
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
    {
      T2FlairImagePointer = mSlicerManagers[index]->mITKImage;
      filenames[1] = mSlicerManagers[index]->mFileName;
    }
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1)
    {
      T1ImagePointer = RescaleImageIntensity(mSlicerManagers[index]->mITKImage);
      filenames[2] = mSlicerManagers[index]->mFileName;
    }
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2)
    {
      T2ImagePointer = RescaleImageIntensity(mSlicerManagers[index]->mITKImage);
      filenames[3] = mSlicerManagers[index]->mFileName;
    }
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_DTI)
    {
      DTIImagePointer.push_back(RescaleImageIntensity(mSlicerManagers[index]->mITKImage));
      filenames[5] = mSlicerManagers[index]->mFileName;
    }
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
    {
      ImageTypeFloat4D::RegionType region = mSlicerManagers[index]->mPerfusionImagePointer->GetLargestPossibleRegion();
      ImageTypeFloat4D::IndexType regionIndex;
      ImageTypeFloat4D::SizeType regionSize;

      regionSize[0] = region.GetSize()[0];
      regionSize[1] = region.GetSize()[1];
      regionSize[2] = region.GetSize()[2];
      regionSize[3] = 0; // this is 0 because we need a 3D image
      regionIndex[0] = 0;
      regionIndex[1] = 0;
      regionIndex[2] = 0;
      regionIndex[3] = 0;


      for (size_t i = 0; i < region.GetSize()[3]; i++)
      {
        regionIndex[3] = i;
        ImageTypeFloat4D::RegionType desiredRegion(regionIndex, regionSize);
        typedef itk::ExtractImageFilter< ImageTypeFloat4D, ImageTypeFloat3D > FilterType;
        FilterType::Pointer filter = FilterType::New();
        filter->SetExtractionRegion(desiredRegion);
        filter->SetInput(mSlicerManagers[index]->mPerfusionImagePointer);

        filter->SetDirectionCollapseToIdentity(); // This is required.
        filter->Update();
        PerfusionImagePointer.push_back(filter->GetOutput());
      }

      filenames[4] = mSlicerManagers[index]->mFileName;
    }
    else
    {
      cbica::Logging(loggerFile, "Unknown Image type");
    }
  }
  //------------------------------------------noise reduction----------------------------------------------
  if (imagetype_int == IMAGE_DICOM)
  {
    if (t2flairDataPresent)
      mPreprocessingObj.DenoisingMethodCall<ImageType>(T2FlairImagePointer);
    if (t1ceDataPresent)
      mPreprocessingObj.DenoisingMethodCall<ImageType>(T1CEImagePointer);
    if (t1DataPresent)
      mPreprocessingObj.DenoisingMethodCall<ImageType>(T1ImagePointer);
    if (t2DataPresent)
      mPreprocessingObj.DenoisingMethodCall<ImageType>(T2ImagePointer);
    if (perfusionDataPresent)
      for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
        mPreprocessingObj.DenoisingMethodCall<ImageType>(PerfusionImagePointer[index]);
    if (dtiDataPresent)
      for (unsigned int index = 0; index < DTIImagePointer.size(); index++)
        mPreprocessingObj.DenoisingMethodCall<ImageType>(DTIImagePointer[index]);

    mOutputManager.WriteOutput<ImageType>(T1CEImagePointer, T2FlairImagePointer, T1ImagePointer, T2ImagePointer, PerfusionImagePointer, DTIImagePointer, filenames, WRITING_TYPE_SUSAN,
      t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

    //------------------------------------------bias correction----------------------------------------------
    ImageType::Pointer T1_N3Corrected_ImagePointer;
    ImageType::Pointer T2_N3Corrected_ImagePointer;
    ImageType::Pointer T1CE_N3Corrected_ImagePointer;
    ImageType::Pointer T2Flair_N3Corrected_ImagePointer;
    std::vector<ImageType::Pointer> Perfusion_N3Corrected_ImagePointer;
    std::vector<ImageType::Pointer> DTI_N3Corrected_ImagePointer;

    if (t2flairDataPresent)
      T2Flair_N3Corrected_ImagePointer = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(T2FlairImagePointer);
    if (t1ceDataPresent)
      T1CE_N3Corrected_ImagePointer = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(T1CEImagePointer);
    if (t1DataPresent)
      T1_N3Corrected_ImagePointer = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(T1ImagePointer);
    if (t2DataPresent)
      T2_N3Corrected_ImagePointer = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(T2ImagePointer);

    if (perfusionDataPresent)
      for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
        Perfusion_N3Corrected_ImagePointer[index] = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(PerfusionImagePointer[index]);
    if (dtiDataPresent)
      for (unsigned int index = 0; index < DTIImagePointer.size(); index++)
        DTI_N3Corrected_ImagePointer[index] = mPreprocessingObj.N4BiasCorrectionMethodCall<ImageType>(DTIImagePointer[index]);

    mOutputManager.WriteOutput<ImageType>(T1CE_N3Corrected_ImagePointer, T2Flair_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer, filenames, WRITING_TYPE_BIASCORRECTION,
      t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

    //-------------------------------------------------------registration------------------------------------------
    typedef ImageTypeFloat3D InternalImageType;
    itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;
    ImageType::Pointer T2Flair_Registered = T2Flair_N3Corrected_ImagePointer;
    ImageType::Pointer T1_Registered;
    ImageType::Pointer T2_Registered;
    std::vector<ImageType::Pointer> Perfusion_Registered;
    Perfusion_Registered.resize(Perfusion_N3Corrected_ImagePointer.size());
    std::vector<ImageType::Pointer> DTI_Registered;
    DTI_Registered.resize(DTI_N3Corrected_ImagePointer.size());

    if (t2DataPresent)
    {
      Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer);
      T2_Registered = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer);
    }
    if (t1DataPresent)
    {
      Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer);
      T1_Registered = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer);
    }
    if (perfusionDataPresent)
    {
      Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer[0]);
      for (unsigned int index = 0; index < Perfusion_N3Corrected_ImagePointer.size(); index++)
        Perfusion_Registered[index] = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer[index]);
    }
    if (dtiDataPresent)
    {
      Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer[0]);
      for (unsigned int index = 0; index < DTI_N3Corrected_ImagePointer.size(); index++)
        DTI_Registered[index] = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer[index]);
    }
    mOutputManager.WriteOutput<ImageType>(T1CE_N3Corrected_ImagePointer, T2Flair_Registered, T1_Registered, T2_Registered, Perfusion_Registered, DTI_Registered, filenames, WRITING_TYPE_REGISTRATION, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

    FinalT1ImagePointer = T1_Registered;
    FinalT2ImagePointer = T2_Registered;
    FinalT1CEImagePointer = T1CE_N3Corrected_ImagePointer;
    FinalT2FlairImagePointer = T2Flair_N3Corrected_ImagePointer;
    FinalPerfusionImagePointer = Perfusion_Registered;
    FinalDTIImagePointer = DTI_Registered;
  }
  else
  {
    FinalT1ImagePointer = T1ImagePointer;
    FinalT2ImagePointer = T2ImagePointer;
    FinalT1CEImagePointer = T1CEImagePointer;
    FinalT2FlairImagePointer = T2FlairImagePointer;
    FinalPerfusionImagePointer = PerfusionImagePointer;
    FinalDTIImagePointer = DTIImagePointer;
  }

  //--------------------------------------------load edema and tumor mask--------------------------------------------------
  VectorVectorDouble allDrawingPoints = FormulateDrawingPointsForEdemaSegmentation();
  ImageType::Pointer edemaMask = mPreprocessingObj.Edema3DSegmentationInGivenImage<ImageType>(FinalT2FlairImagePointer, allDrawingPoints);
  ImageType::Pointer tumorMaskFinal;
 /* typedef itk::Image<short, 3> ImageTypeGeodesic;
  ImageTypeGeodesic::Pointer Inp = ImageTypeGeodesic::New();

  Tumor3DGeodesicSegmentation<ImageTypeGeodesic

  ImageTypeGeodesic::Pointer tumorMask = ImageTypeGeodesic::New();
  ImageType::Pointer tumorMaskFinal = ImageType::New();

  typedef itk::CastImageFilter<ImageType, ImageTypeGeodesic> CastFilterType;
  CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(FinalT1CEImagePointer);
  Inp = castFilter->GetOutput();
  Inp->Update();*/

  if (distanceDataPresent)
  {
    VectorVectorDouble tumorPoints = FormulateDrawingPointsForTumorSegmentation();
    tumorMaskFinal = mPreprocessingObj.Tumor3DSegmentationInGivenImage<ImageType>(FinalT1CEImagePointer, tumorPoints);

    //typedef itk::CastImageFilter<ImageTypeGeodesic, ImageType> CastFilterBackType;
    //CastFilterBackType::Pointer castFilter2 = CastFilterBackType::New();
    //castFilter2->SetInput(tumorMask);
    //tumorMaskFinal = castFilter2->GetOutput();
    //tumorMaskFinal->Update();
  }
  mRecurrenceEstimator.Run<ImageType>(edemaMask, tumorMaskFinal, 
    FinalT1CEImagePointer, FinalT2FlairImagePointer, FinalT1ImagePointer, FinalT2ImagePointer, FinalPerfusionImagePointer, FinalDTIImagePointer, 
    IMAGE_NIFTI, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent, 
    useOtherModalities, t1cebasefilename, GetMaskLabelIndices(1), GetMaskLabelIndices(2), mOutputManager.GetOutputDirectoryPath());
  
  LoadSlicerImages(mOutputManager.GetOutputDirectoryPath() + "/" + mRecurrenceEstimator.mRecurrenceMapFileName,IMAGE_NIFTI);
  statusbar->showMessage("Recurrence estimate for the given subject has been saved and loaded.");
}

//void fMainWindow::SetMasksTable()
//{
//  //	masksTable->horizontalHeaderItem(0)->setText(QApplication::translate("fMainWindow", "Mask type", 0, QApplication::UnicodeUTF8));
//  masksTable->horizontalHeaderItem(0)->setText(QApplication::translate("fMainWindow", " # ", 0, QApplication::UnicodeUTF8));
//
//
//  masksTable->verticalHeaderItem(MASK_RECURRENCE_ROW)->setText(QApplication::translate("fMainWindow", "Recurrence", 0, QApplication::UnicodeUTF8));
//  masksTable->verticalHeaderItem(MASK_NON_RECURRENCE_ROW)->setText(QApplication::translate("fMainWindow", "NonRecurrence e", 0, QApplication::UnicodeUTF8));
//  masksTable->verticalHeaderItem(MASK_TOTAL_ROW)->setText(QApplication::translate("fMainWindow", "Total", 0, QApplication::UnicodeUTF8));
//
//  masksTable->resizeRowsToContents();
//  //masksTable->resizeColumnsToContents();
//  masksTable->setColumnWidth(0, 130);
//}

void fMainWindow::dragEnterEvent(QDragEnterEvent *event)
{
  if (event->mimeData()->hasFormat("text/uri-list")) {
    event->acceptProposedAction();
  }
}
void fMainWindow::dropEvent(QDropEvent *event)
{
  QList<QUrl> urls = event->mimeData()->urls();
  QStringList vectorOfFiles;
  for (int i = 0; i < (int)urls.size(); i++) 
  {
    vectorOfFiles.push_back(urls[i].toLocalFile());
  }
  openImages(vectorOfFiles);
}


void fMainWindow::CloseNonViewingDTIImage(QTableWidgetItem* item)
{
  int itemIndexToBeDeleted = 0;
  m_nonVisImagesTable->removeRow(item->row());

  for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
  {
    if (mNonViewingImageManager[index]->mImageType == IMAGE_TYPE_DTI)
    {
      itemIndexToBeDeleted = index;
      delete mNonViewingImageManager[index];
      break;
    }
  }
  std::vector<SimpleImageManager*>::iterator simpleImageIterator = mNonViewingImageManager.begin();
  for (int i = 0; i < itemIndexToBeDeleted; i++)
    simpleImageIterator++;
  mNonViewingImageManager.erase(simpleImageIterator);
}

void fMainWindow::UpdateNumberOfPointsInTable()
{
  if (mSlicerManagers.size() <= 0)
    return;

  typedef itk::Image<short, 3> ImageType;
  ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
  int nearCounter = 0;
  int  farCounter = 0;
  int  initCounter = 0;
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType maskIt(img, img->GetLargestPossibleRegion());
  maskIt.GoToBegin();
  while (!maskIt.IsAtEnd())
  {
    if (maskIt.Get() == DRAW_MODE_LABEL_1)
      nearCounter++;
    else if (maskIt.Get() == DRAW_MODE_LABEL_2)
      farCounter++;
    else if (maskIt.Get() == DRAW_MODE_LABEL_3)
      initCounter++;
    ++maskIt;
  }
  mCurrentNearPoints = nearCounter;
  mCurrentFarPoints = farCounter;
  mCurrentInitPoints = initCounter;

}

void fMainWindow::SetPresetComboBox()
{
  presetComboBox->addItem("Auto Scale");
  presetComboBox->addItem("User Scale");
  presetComboBox->addItem("Label Map");
  presetComboBox->addItem("Label map 2");
  presetComboBox->addItem("Threshold");
  presetComboBox->addItem("Probability");
}
void fMainWindow::ResetNumberOfPoints()
{
  UpdateNumberOfPointsInTable();
}

bool fMainWindow::CheckCompletenessOfInputData(bool & t1DataPresent,bool & t2DataPresent,bool & t1ceDataPresent,bool & t2flairDataPresent,bool & perfusionDataPresent,bool & dtiDataPresent)
{
  bool t1P = false;
  bool t2P = false;
  bool t1ceP = false;
  bool t2flairP = false;
  bool dtiP = false;
  bool perfP = false;

  std::string msg = "";

  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
      t1ceP= true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
      t2flairP = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2)
      t2P = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1)
      t1P = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_DTI)
      dtiP = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
      perfP = true;
  }
  if (mOutputManager.GetOutputDirectoryPath() == "")
    msg = "Output directory";
  if (t1ceDataPresent == false)
    msg = msg + "\n" + "T1-Gd Data.";
  if (t2flairDataPresent == false)
  {
    msg = msg + "\n" + "T2-FLAIR Data.";
  }
  if (t1DataPresent == true && t1P == false)
    msg = msg + "\n" + "T1 Data.";
  if (t2DataPresent == true && t2P == false)
    msg = msg + "\n" + "T2 Data.";

  if (perfusionDataPresent == true && perfP == false)
    msg = msg + "\n" + "Perfusion Data.";

  if (dtiDataPresent == true && dtiP == false)
    msg = msg + "\n" + "DTI Data.";

  if (mCurrentNearPoints == 0)
    msg = msg + "\n" + "Near ROI (Label-1) mask.";
  if (mCurrentFarPoints == 0)
    msg = msg + "\n" + "Far ROI (Label-2) mask.";

  if (msg != "")
  {
    msg = "Please supply the following items:\n" + msg;
    QMessageBox box(this);
    box.setIcon(QMessageBox::Information);
    box.addButton(QMessageBox::Ok);
    box.setText(QString::fromStdString(msg));
    box.setWindowTitle(tr("Missing Data"));
    box.exec();
    return false;
  }
  else
    return true;
}
bool fMainWindow::CheckCompletenessOfInputDataForEGFR(bool & t1ceDataPresent, bool & t2flairDataPresent, bool & perfusionDataPresent)
{
  std::string msg = "";
  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
      t1ceDataPresent = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
      t2flairDataPresent = true;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
      perfusionDataPresent = true;
  }
  if (t1ceDataPresent == false)
    msg = msg + "\n" + "T1-Gd data.";
  if (t2flairDataPresent == false)
  {
    msg = msg + "\n" + "T2-FLAIR data.";
  }
  if (perfusionDataPresent == false)
    msg = msg + "\n" + "DSC-MRI data.";
  if (mCurrentNearPoints == 0)
    msg = msg + "\n" + "Near ROI (Label-1) mask.";
  if (mCurrentFarPoints == 0)
    msg = msg + "\n" + "Far ROI (Label-2) mask.";

  if (msg != "")
  {
    ShowErrorMessage("Please supply the following items:\n" + msg, "Missing Data");
    return false;
  }
  else
    return true;
}

void fMainWindow::RecurrenceEstimateOnExistingModel(const std::string &modeldirectory, const std::string &inputdirectory, const std::string &outputdirectory, bool &useT1Data, bool &useT1CEData, bool &useT2Data, bool &useT2FlairData, bool &useDTIData, bool &usePerfData, bool &useDistData)
{
  mRecurrenceEstimator.Run(modeldirectory, inputdirectory, outputdirectory, useT1Data, useT1CEData, useT2Data, useT2FlairData, useDTIData, usePerfData, useDistData);
  if (mRecurrenceEstimator.mLastEncounteredError != "")
    ShowErrorMessage(mRecurrenceEstimator.mLastEncounteredError);
  else
    statusbar->showMessage("Recurrence maps have been generated for the given subjects.");
}

void fMainWindow::CallForSurvivalPredictionOnExistingModelFromMain(const std::string modeldirectory, const std::string inputdirectory, const std::string outputdirectory)
{
  //survivalPanel.run
  mSurvivalPredictor.SurvivalPredictionOnExistingModel(modeldirectory, inputdirectory, outputdirectory);

  if (!mSurvivalPredictor.m_LastError.empty())
  {
    ShowErrorMessage(mSurvivalPredictor.m_LastError, "Error while running Survival");
    return;
  }

  std::string msg = "A Survival Prediction Index (SPI) has been calculated for the given subjects by applying the specified model. \n\n";
  msg = msg + "SPI index saved in 'results.csv' file in the output directory. \n\n";

  msg = msg + "Input Directory = " + inputdirectory + "\nOutput Directory = " + outputdirectory + "\nModel Directory = " + modeldirectory;

  ShowMessage(msg);
}

void fMainWindow::CallForNewSurvivalPredictionModelFromMain(const std::string inputdirectory, const std::string outputdirectory)
{
  mSurvivalPredictor.PrepareNewSurvivalPredictionModel(inputdirectory, outputdirectory);

  std::string msg = "A Survival Prediction Index (SPI) model has been prepared and saved. \n\nInput Directory = " + inputdirectory + "\nOutput Directory = " + outputdirectory;

  ShowMessage(msg);
}



ImageTypeFloat3D::Pointer fMainWindow::RescaleImageIntensity(ImageTypeFloat3D::Pointer image)
{
  typedef ImageTypeFloat3D ImageType;
  typedef itk::RescaleIntensityImageFilter< ImageType, ImageType > RescaleFilterType;
  RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
  rescaleFilter->SetInput(image);
  rescaleFilter->SetOutputMinimum(0);
  rescaleFilter->SetOutputMaximum(255);
  rescaleFilter->Update();

  ImageType::Pointer outputimage = rescaleFilter->GetOutput();

  return outputimage;

}

void fMainWindow::TrainNewModelOnGivenData(const std::string &directoryname, const std::string &outputdirectory, bool &useT1Data, bool &useT1CEData, bool &useT2Data, bool &useT2FlairData, bool &useDTIData, bool &usePerfData, bool &useDistData)
{
  mRecurrenceEstimator.TrainNewModelOnGivenData(directoryname, outputdirectory, useT1Data, useT1CEData, useT2Data, useT2FlairData, useDTIData, usePerfData, useDistData);
}



void fMainWindow::LoadDicomDrawing()
{
  std::string root_directory;
  QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), mInputPathName, QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
  if (directory.isNull())
    return;

  typedef itk::Image<unsigned short, 3> InputImageType;
  typedef itk::ImageSeriesReader< InputImageType > ReaderType;
  ReaderType::Pointer seriesreader = ReaderType::New();

  typedef itk::GDCMImageIO ImageIOType;
  ImageIOType::Pointer dicomIO = ImageIOType::New();
  seriesreader->SetImageIO(dicomIO);
  typedef itk::GDCMSeriesFileNames NamesGeneratorType;
  NamesGeneratorType::Pointer nameGenerator = NamesGeneratorType::New();
  nameGenerator->SetUseSeriesDetails(true);
  nameGenerator->AddSeriesRestriction("0008|0021");

  nameGenerator->SetInputDirectory(directory.toStdString());
  try
  {
    typedef std::vector< std::string > SeriesIdContainer;
    const SeriesIdContainer & seriesUID = nameGenerator->GetSeriesUIDs();

    SeriesIdContainer::const_iterator seriesItr = seriesUID.begin();
    SeriesIdContainer::const_iterator seriesEnd = seriesUID.end();
    while (seriesItr != seriesEnd)
    {
      typedef std::vector< std::string > FileNamesContainer;
      FileNamesContainer fileNames;
      fileNames = nameGenerator->GetFileNames(seriesItr->c_str());
      seriesreader->SetFileNames(fileNames);
      try
      {
        seriesreader->Update();
        typedef unsigned short ROIPixelType;
        typedef::itk::Image<ROIPixelType, 3> OutputImageType;
        OutputImageType::Pointer outputImage = seriesreader->GetOutput();
        std::vector<OutputImageType::IndexType> nearIndices;
        std::vector<OutputImageType::IndexType> farIndices;
        std::vector< std::vector<OutputImageType::IndexType> > allIndeces;
        allIndeces.resize(DRAW_MODE_LABEL_5);

        typedef itk::ImageRegionIteratorWithIndex <OutputImageType> IteratorType;
        IteratorType maskIt(outputImage, outputImage->GetLargestPossibleRegion());
        maskIt.GoToBegin();
        size_t totalIndices = 0;

        while (!maskIt.IsAtEnd())
        {
          OutputImageType::IndexType currentIndex = maskIt.GetIndex();
          float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)currentIndex[0], (int)currentIndex[1], (int)currentIndex[2]);
          // this is done to take into account all possible label drawings
          switch (maskIt.Get())
          { // multiLabel: uncomment everything inside this loop and remove references to "near" and "far"
          case NEAR_POINT_SAVE_LABEL:
            allIndeces[0].push_back(currentIndex);
            *pData = DRAW_MODE_LABEL_1;
            totalIndices++;
            break;
          case FAR_POINT_SAVE_LABEL:
            allIndeces[1].push_back(maskIt.GetIndex());
            *pData = DRAW_MODE_LABEL_2;
            totalIndices++;
            break;
          case DRAW_MODE_LABEL_1:
            allIndeces[0].push_back(currentIndex);
            *pData = DRAW_MODE_LABEL_1;
            totalIndices++;
            break;
          case DRAW_MODE_LABEL_2:
            allIndeces[1].push_back(maskIt.GetIndex());
            *pData = DRAW_MODE_LABEL_2;
            totalIndices++;
            break;
          case DRAW_MODE_LABEL_3:
            allIndeces[2].push_back(maskIt.GetIndex());
            *pData = DRAW_MODE_LABEL_3;
            totalIndices++;
            break;
          case DRAW_MODE_LABEL_4:
            allIndeces[3].push_back(maskIt.GetIndex());
            *pData = DRAW_MODE_LABEL_4;
            totalIndices++;
            break;
          case DRAW_MODE_LABEL_5:
            allIndeces[4].push_back(maskIt.GetIndex());
            *pData = DRAW_MODE_LABEL_5;
            totalIndices++;
            break;
          default:
            // nothing defined for other cases 
            break;
          }
          ++maskIt;
        }

        this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
        this->mSlicerManagers[0]->Render();
      }
      catch (itk::ExceptionObject & err)
      {
        std::stringstream error;
        error << err;
      }
      ++seriesItr;
    }
  }
  catch (itk::ExceptionObject & err)
  {
    std::stringstream error;
    error << err;
  }
}

void fMainWindow::LoadDrawing()
{
  //	SetActiveLandmarksType(NEARFAR_DRAWING_POINTS, drawingtype, 0);
  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QStringList files = QFileDialog::getOpenFileNames(this, tr("Load Masks"), mInputPathName, extensions, 0, QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
  if (files.isEmpty())
    return;

  std::string filename = files[0].toStdString();
  itk::ImageIOBase::Pointer reader = itk::ImageIOFactory::CreateImageIO(filename.c_str(), itk::ImageIOFactory::ReadMode);
  if (reader)
  {
    readMaskFile(filename);
  }

  //for (unsigned int i = 0; i < nearIndices.size(); i++)
  //{
  //  float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)nearIndices[i][0], (int)nearIndices[i][1], (int)nearIndices[i][2]);
  //  *pData = 1;
  //}
  //for (unsigned int i = 0; i < farIndices.size(); i++)
  //{
  //  float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)farIndices[i][0], (int)farIndices[i][1], (int)farIndices[i][2]);
  //  *pData = 2;
  //}
  this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
  this->mSlicerManagers[0]->Render();
  //}
}
void fMainWindow::LoadSeedDrawing()
{
  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QStringList files = QFileDialog::getOpenFileNames(this, tr("Load Masks"), mInputPathName, extensions, 0, QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
  if (files.isEmpty())
    return;

  std::string filename = files[0].toStdString();
  itk::ImageIOBase::Pointer reader = itk::ImageIOFactory::CreateImageIO(filename.c_str(), itk::ImageIOFactory::ReadMode);
  if (reader)
  {
    typedef ImageTypeFloat3D ImageType;
    ImageType::Pointer inputImage = cbica::ReadImage<ImageType>(filename);

    std::vector<ImageType::IndexType> seedIndices;
    typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
    IteratorType maskIt(inputImage, inputImage->GetLargestPossibleRegion());
    maskIt.GoToBegin();
    while (!maskIt.IsAtEnd())
    {
      if (maskIt.Get() == INIT_POINT_SAVE_LABEL)
      {
        ImageType::IndexType currentIndex = maskIt.GetIndex();
        seedIndices.push_back(currentIndex);
        float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)currentIndex[0], (int)currentIndex[1], (int)currentIndex[2]);
        *pData = INIT_POINT_LABEL;
      }
      ++maskIt;
    }
    this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
    this->mSlicerManagers[0]->Render();
  }
}

void fMainWindow::UpdateBorderWidget(double startX, double startY, double endX, double endY)
{
  mBorderStartX = std::round(startX);
  mBorderStartY = std::round(startY);
  mBorderEndX = std::round(endX);
  mBorderEndY = std::round(endY);
}
void fMainWindow::UpdateBorderWidget(double startZ, double endZ)
{
  mBorderStartZ = std::round(startZ);
  mBorderStartZ = std::round(startZ);
}

void fMainWindow::overlayUseStateChanged(int state)
{
  if (state == 0) 
  {
    for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
      for (int j = 0; j < 3; j++) {
        mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
      }
    }
    UpdateRenderWindows();
  }
}

void fMainWindow::overlaySliderChanged(int value)
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) 
  {
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index >= 0 && index < (int)mSlicerManagers.size())
  {
    for (int i = 0; i < 3; i++)
    {
      mSlicerManagers[index]->GetSlicer(i)->SetOverlayOpacity((double)value / (10 + 1e-6));
    }
  }
  UpdateRenderWindows();
}

void fMainWindow::imageModalityChanged(int value)
{
  for (size_t i = 0; i < mSlicerManagers.size(); i++)
  {
    mSlicerManagers[i]->mImageSubType = imagesPanel->getModality(i);
  }
}

//---------------------------------------------
void fMainWindow::imageSliderChanged()
{
  static int value = -1;
  if (value == image4DSlider->value()) 
    return;
  else 
    value = image4DSlider->value();

  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) 
    return;

  int index = GetSlicerIndexFromItem(items[0]);

  if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
  {
    mSlicerManagers[index]->Get3DImageAtCurrentPerfusionIndex(value);
  }
  AxialViewSliderChanged();  
  mSlicerManagers[index]->Picked();
  mSlicerManagers[index]->UpdateViews(0);
  mSlicerManagers[index]->UpdateLinked(0);
  mSlicerManagers[index]->UpdateInfoOnCursorPosition(0);
}
//---------------------------------------------
void fMainWindow::overlayChanged()
{

  overlayChanged(imagesPanel->getSelectedOverlay());
}
void fMainWindow::overlayChanged(QTableWidgetItem *clickedItem)
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty()) 
  {
    return;
  }
  int slicerManagerIndex = GetSlicerIndexFromItem(items[0]);
  if (slicerManagerIndex < 0 || slicerManagerIndex >= (int)mSlicerManagers.size())
  {
    return;
  }
  //
  int slicerManagerOverlayIndex = GetSlicerIndexFromItem(clickedItem);
  if (slicerManagerOverlayIndex < 0 || slicerManagerOverlayIndex >= (int)mSlicerManagers.size())
  {
    return;
  }
  for (unsigned int i = 0; i < mSlicerManagers.size(); i++)
  {
    if (i != static_cast<unsigned int>(slicerManagerIndex)) {
      for (int j = 0; j < 3; j++) {
        mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
      }
    }
    else
    {
      for (int j = 0; j < 3; j++) 
      {
        mSlicerManagers[slicerManagerIndex]->mSlicers[j]->SetOverlay(mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->mImage);
        //
        double window = mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetColorWindow();
        double level = mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetColorLevel();
        vtkLookupTable* LUT = static_cast<vtkLookupTable*>(mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetWindowLevel()->GetLookupTable());
        if (LUT != NULL)
        {
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetWindow(window);
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLevel(level);
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLookupTable(LUT);
        }
        else
        {
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLookupTable(NULL);
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetWindow(window);
          mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLevel(level);
        }
      }
    }
  }
  UpdateRenderWindows();
}

void fMainWindow::openImages(QStringList files)
{
  if (files.isEmpty())
  {
	  QString extensions = IMAGES_EXTENSIONS;
	  extensions += ";;All Files (*)";
    files = QFileDialog::getOpenFileNames(this, tr("Load Images"), mInputPathName, extensions, 0, QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
	  if (files.isEmpty())
		  return;
  }

  for (size_t i = 0; i < files.size(); i++)
  {
    string fileName = files[i].toStdString();
    updateProgress(i + 1, "Opening " + fileName, files.size());
    LoadSlicerImages(files[i].toStdString(), IMAGE_NIFTI);

  }
  updateProgress(0, "Loading complete", 100);
}

inline bool fMainWindow::PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, 
  std::string &scriptWithPath, std::string configFileWithPath)
{
  bool configFileMissing = false;
  if (NameToCheck.find(AppName) != std::string::npos)
  {
    scriptWithPath = NameToCheck + ".py";
    configFileWithPath = NameToCheck + ".py";
    // check for the main script
    if (cbica::isFile(scriptWithPath)) // check in the current working directory
    {
      if (!cbica::isFile(configFileWithPath))
      {
        configFileMissing = true;
      }
    }
    else if (cbica::isFile(cbica::getExecutablePath() + scriptWithPath)) // check in the directory where the executable is
    {
      scriptWithPath = cbica::getExecutablePath() + scriptWithPath;
      configFileWithPath = cbica::getExecutablePath() + configFileWithPath;
      if (!cbica::isFile(configFileWithPath))
      {
        configFileMissing = true;
      }
    }
#ifdef PROJECT_SOURCE_DIR
    else if (cbica::isFile(PROJECT_SOURCE_DIR + scriptWithPath)) // check in project source code, if defined
    {
      scriptWithPath = PROJECT_SOURCE_DIR + scriptWithPath;
      configFileWithPath = PROJECT_SOURCE_DIR + configFileWithPath;
      if (!cbica::isFile(configFileWithPath))
      {
        configFileMissing = true;
      }
    }
#endif
    else
    {
      ShowErrorMessage("The Python CLI application '" + NameToCheck +
        "' wasn't found in either the current working directory, executable directory and the source code directory.\n");
      return false;
    }
  }

  if (configFileMissing)
  {
    ShowErrorMessage("The Config File matching the Python CLI application '" + NameToCheck +
      "' wasn't found in either the current working directory, executable directory and the source code directory.\n" + 
      "It needs to be present in the same level as the Python application script.\n");
    return false;
  }

  return true;
}

inline bool fMainWindow::PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, std::string &scriptWithPath)
{
  if (NameToCheck.find(AppName) != std::string::npos)
  {
    std::string executablePath = QApplication::applicationDirPath().toStdString() + "/";
#ifdef DEVELOPER_MODE
    executablePath = QApplication::applicationDirPath().toStdString() + "/../../src/applications/binaries/windows/";
#endif
#ifdef _WIN32
    scriptWithPath = executablePath + NameToCheck + ".exe";
    if (NameToCheck == "ITK-SNAP")
    {
      scriptWithPath = executablePath + "/" + NameToCheck + "/bin/" + NameToCheck + ".exe";
    }
#else
    scriptWithPath = "./" + NameToCheck;
#endif
    // check for the main script
    if (cbica::isFile(scriptWithPath)) // check in the current working directory
    {
      return true;
    }
    else if (cbica::isFile(cbica::getExecutablePath() + scriptWithPath)) // check in the directory where the executable is
    {
      scriptWithPath = cbica::getExecutablePath() + scriptWithPath;
    }
#ifdef PROJECT_SOURCE_DIR // if not found in either of the above cases, assume that the script is located within the defined source tree
    else if (!m_allNonNativeApps.empty()) 
    {
      for (size_t i = 0; i < m_allNonNativeApps.size(); i++)
      {
        if (m_allNonNativeApps[i].name == NameToCheck)
        {
          scriptWithPath = m_allNonNativeApps[i].path;
        }
      }
    }
#endif
    else
    {
      ShowErrorMessage("The Python GUI application '" + scriptWithPath + "' wasn't found.\n");
      return false;
    }
    return true;
  }
  else
  {
    return false;
  }
}
void fMainWindow::PyGUILIBRA()
{
  std::string scriptToCall;
  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig("LIBRA", m_pyGUIApps[i], scriptToCall))
    {
      //Libra specific fix 
      scriptToCall = scriptToCall.substr(0, scriptToCall.find_last_of(".")) + ".bat";

      if (cbica::fileExists(scriptToCall))
      {
        QString exeToCall;
        QStringList args;
        args.append("/C");
        args.append(scriptToCall.c_str());
#if defined _WIN32
        exeToCall = "cmd.exe"; // this will no longer be needed once we move LIBRA to its Python variant
#endif
        startExternalProcess(exeToCall, args);
        return;
      }
      else
      {
        ShowErrorMessage("Cannot find :" + scriptToCall);
      }

    }
  }
}

void fMainWindow::PyGUIConfetti()
{
  std::string scriptToCall;
  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig("ConfettiGUI", m_pyGUIApps[i], scriptToCall))
    {
      QStringList args;
      startExternalProcess(scriptToCall.c_str(), args);
      return;
    }
  }
}
void fMainWindow::PyGUIWhiteStripe()
{
  std::string scriptToCall;
  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig("WhiteStripeGUI", m_pyGUIApps[i], scriptToCall))
    {
      QStringList args;
      startExternalProcess(scriptToCall.c_str(), args);
      return;
    }
  }
}
void fMainWindow::PyCLISBRT_Segment()
{
  if (mSlicerManagers.size() < 2)
  {
    ShowErrorMessage("Load two images with first being the CT and second being the PET image.", "Notice");
    return;
  }

  std::string ctImageFile = cbica::normalizePath(mSlicerManagers[0]->mPathFileName),
    petImageFile = cbica::normalizePath(mSlicerManagers[1]->mPathFileName), command, exePath,
    maskFile = tempFolderLocation + "tempMask.nii.gz";
  ImageTypeFloat3D::Pointer mask = getMaskImage();
  cbica::WriteImage<ImageTypeFloat3D>(mask, maskFile);

  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig("SBRT_Segmentation", m_pyGUIApps[i], exePath))
    {
      command = exePath;
      QStringList args;
      args << ctImageFile.c_str() << petImageFile.c_str() << maskFile.c_str();
      startExternalProcess(command.c_str(), args);
    }
  }

  // load written file into CapTK
  if (cbica::isFile(maskFile))
  {
    readMaskFile(maskFile);
    //LoadSlicerImages(maskFile, IMAGE_NIFTI);
    updateProgress(0, "Finished SBRT Segmentation!");
    ShowMessage("For SBRT Analyze, please Draw ROI", "Done");
  }
  else
  {
    ShowErrorMessage("Error in SBRT Segmentation");
  }

}
void fMainWindow::PyCLISBRT_Analyze()
{
  if (mSlicerManagers.size() < 2)
  {
    ShowErrorMessage("Load two images with first being the CT and second being the PET image.", "Notice");
    return;
  }
  //int index[3];
  //index[0] = -mTissuePoints[0].mLandmarks[0].coordinates[0];
  //index[1] = -mTissuePoints[0].mLandmarks[0].coordinates[1];
  //index[2] = mTissuePoints[0].mLandmarks[0].coordinates[2];


  std::string ctImageFile = cbica::normalizePath(mSlicerManagers[0]->mPathFileName),
    petImageFile = cbica::normalizePath(mSlicerManagers[1]->mPathFileName), command, exePath,
    maskFile = tempFolderLocation + "tempMask.nii.gz", outputFile = tempFolderLocation + "result.txt",
    modelfilename;
  
  modelfilename = QApplication::applicationDirPath().toStdString() + "/../data/SBRT/SVM_Model.csv";
#ifdef DEVELOPER_MODE
  modelfilename = QApplication::applicationDirPath().toStdString() + "/../../data/SBRT/SVM_Model.csv";
#endif


  ImageTypeFloat3D::Pointer mask = getMaskImage();
  if (mask.IsNull())
  {
    ShowErrorMessage("Empty mask defined");
    return;
  }
  cbica::WriteImage<ImageTypeFloat3D>(mask, maskFile);

  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig("SBRT_Analyze", m_pyGUIApps[i], exePath))
    {
      command = exePath;
      QStringList args;
      args << "-c" << ctImageFile.c_str() << "-p" << petImageFile.c_str() << "-m" << maskFile.c_str() << "-t" << modelfilename.c_str() << "-o" << outputFile.c_str();
      // full paths for exe, image1, image2, mask, tmpDir are there
      startExternalProcess(command.c_str(), args);

      float result;
      if (cbica::fileExists(outputFile))
      {
        std::ifstream infile(outputFile);
        if (infile.is_open())
        {
          infile >> result;
          if (result > 0.5)
            ShowMessage("Prediction Result: " + to_string(result) + "   No signs of nodal failure");
          else
            ShowMessage("Prediction Result: " + to_string(result) + "   Signs of nodal failure");
        }
        else
        {
          ShowErrorMessage("SBRT Analyze couldn't process the images");
        }
      }
      else
      {
        ShowErrorMessage("SBRT Analyze failed to produce a result. Please provide an ROI");
      }

      return;
    }
  }

  //cbica::deleteDir(tmpDir);
}

void fMainWindow::AllPythonGUI(const std::string &appName)
{
  std::string scriptToCall;
  for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig(appName, m_pyGUIApps[i], scriptToCall))
    {
      QProcess process;
      process.start(scriptToCall.c_str());
      process.write("exit\n\r");
      process.waitForFinished(-1);
      process.close();
      return;
    }
  }
}

inline void fMainWindow::PythonCLIToDialogAndRun(const std::string &scriptFile, const std::string &configFile)
{
  std::string commandToRun = "";
  if (cbica::isFile(configFile))
  {
    std::vector<cbica::Parameter> parametersToPass = cbica::readConfigFile(configFile);
    for (size_t i = 0; i < parametersToPass.size(); i++)
    {
      // contruct a dialog box to input the various parameters
      std::string verboseParameter = parametersToPass[i].verbose; // laconic is always empty here
      std::string description = parametersToPass[i].descriptionLine1; // put this as help text the same way for the interactive buttons
      std::string dataType = parametersToPass[i].dataType_string; // expected data type for the parameter
      std::string dataRange = parametersToPass[i].dataRange; // expected data range for the parameter
      std::string parameterValue = ""; // get the value of the parameter here from the user at this point

      bool fileError = false;
      if (parametersToPass[i].dataType_enumCode == cbica::Parameter::INTEGER)
      {
        char *end;
        for (long i = std::strtol(parameterValue.c_str(), &end, 10);
          parameterValue != end;
          i = std::strtol(parameterValue.c_str(), &end, 10))
        {
          parameterValue = end;
          if (errno == ERANGE)
          {
            fileError = true;
            errno = 0;
          }
          long i2 = i;
          i2++;
        }
      }
      else if (parametersToPass[i].dataType_enumCode == cbica::Parameter::FLOAT)
      {
        char *end;
        for (double i = std::strtod(parameterValue.c_str(), &end);
          parameterValue != end;
          i = std::strtol(parameterValue.c_str(), &end, 10))
        {
          parameterValue = end;
          if (errno == ERANGE)
          {
            fileError = true;
            errno = 0;
          }
          long i2 = i;
          i2++;
        }
      }
      else if (parametersToPass[i].dataType_enumCode == cbica::Parameter::FILE)
      {
        if (!cbica::isFile(parameterValue))
        {
          fileError = true;
        }
      }
      else if (parametersToPass[i].dataType_enumCode == cbica::Parameter::DIRECTORY)
      {
        if (!cbica::isDir(parameterValue))
        {
          fileError = true;
        }
      }
      else if (parametersToPass[i].dataType_enumCode == cbica::Parameter::BOOLEAN)
      {
        if ((parameterValue != "1") || (parameterValue != "0"))
        {
          fileError = true; // default to error state
          std::string parameterValue_toCheck = "";

          // convert parameter to lower case and then check whether there were any other flags
          std::transform(parameterValue.begin(), parameterValue.end(), parameterValue_toCheck.begin(), ::tolower);
          if ((parameterValue == "true") || (parameterValue == "false") || 
            (parameterValue == "yes") || (parameterValue == "no") || 
            (parameterValue == "")) // even if a bool parmeter is just passed as-is, assume that it means true
          {
            fileError = false;
          }
        }
      }

      if (fileError)
      {
        ShowErrorMessage("Input parameter, '" + parametersToPass[i].verbose + "' given value '" + parameterValue +
          "' but it couldn't be converted to the expected data type, '" + dataType + "'\n");
      }

      commandToRun += " --" + parametersToPass[i].verbose + parameterValue;
    }
    // construct the "command" after getting all the parameters from above dialog and pass it on to the script
    std::system((scriptFile + commandToRun).c_str());
  }
  else
  {
    ShowErrorMessage("The configuration file matching Python CLI application '" + scriptFile + "' wasn't found.\n");
  }
}

void fMainWindow::AllPythonCLI()
{
  std::string scriptToCall, configFile;
  for (size_t i = 0; i < vectorOfPythonCLIAppActionsAndNames.size(); i++)
  {
    if (PreparePythonScriptAndConfig(vectorOfPythonCLIAppActionsAndNames[i].name, m_pyCLIApps[i], scriptToCall, configFile))
    {
      PythonCLIToDialogAndRun(scriptToCall, configFile);
      //std::system((scriptToCall).c_str());
    }
  }
}

void fMainWindow::ApplicationEGFR()
{
  bool t1ceDataPresent = false;
  bool t2flairDataPresent = false;
  bool perfusionDataPresent = false;

  updateProgress(0, "Preprocessing");
  UpdateNumberOfPointsInTable();

  if (CheckCompletenessOfInputDataForEGFR(t1ceDataPresent, t2flairDataPresent, perfusionDataPresent) == false)
    return;
  updateProgress(5);


  typedef ImageTypeFloat3D ImageType;
  typedef ImageTypeFloat4D PerfusionImageType;

  ImageType::Pointer T1CEImagePointer;
  ImageType::Pointer T2FlairImagePointer;
  std::vector<ImageType::Pointer>	PerfusionImagePointer;
  typedef ImageType InternalImageType;
  itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;
  std::vector<ImageType::Pointer> Perfusion_Registered;
  PerfusionImageType::Pointer perfusionImage = NULL;

  std::vector<ImageType::IndexType> nearIndices;
  std::vector<ImageType::IndexType> farIndices;
  FormulateNearFarPoints<ImageType>(nearIndices, farIndices);
  std::string imagetype_string = "";

  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
      T1CEImagePointer = mSlicerManagers[index]->mITKImage;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
      T2FlairImagePointer = mSlicerManagers[index]->mITKImage;
    else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
    {
      if (mSlicerManagers[index]->mImageType == IMAGE_NIFTI)
      {
        perfusionImage = mSlicerManagers[index]->mPerfusionImagePointer;
        imagetype_string = "nifti";
      }
      else
      {
        for (unsigned int seriesindex = 0; seriesindex < mSlicerManagers[index]->mPerfusionImagePointerDicom.size(); seriesindex++)
          PerfusionImagePointer.push_back(mSlicerManagers[index]->mPerfusionImagePointerDicom[seriesindex]);
        imagetype_string = "dicom";
      }
    }
  }
  
  Perfusion_Registered.resize(PerfusionImagePointer.size());
  if (imagetype_string == "dicom")
  {
    Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(T1CEImagePointer, PerfusionImagePointer[0]);
    updateProgress(10);

    for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
    {
      Perfusion_Registered[index] = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, T1CEImagePointer, PerfusionImagePointer[index]);
      updateProgress((index + 1) * 2 + 10);
    }
  }
  VectorDouble EGFRStatusParams;
  if (imagetype_string == "dicom")
    EGFRStatusParams = mEGFRPredictor.PredictEGFRStatus<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, IMAGE_DICOM);
  else
    EGFRStatusParams = mEGFRPredictor.PredictEGFRStatus<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, IMAGE_NIFTI);


  QString msg;
  //if (EGFRStatusParams[0] <= 0.1377)
  msg = "PHI = " + QString::number(EGFRStatusParams[0]) + "\n\n(Near:Far) Peak Height ratio = " + 
    QString::number(EGFRStatusParams[1] / EGFRStatusParams[2]) + "\n\nNear ROI voxels used = " + 
    QString::number(EGFRStatusParams[3]) + "/" + QString::number(nearIndices.size()) + "\nFar ROI voxels used =   " + 
    QString::number(EGFRStatusParams[4]) + "/" + QString::number(farIndices.size()) +
    "\n\n\nPHI Threshold = 0.1377\n[based on 142 UPenn brain tumor scans]";
  //else
    //msg = " PHI = " + QString::number(EGFRStatusParams[0]) + "\n\nNear/Far perfusion drop ratio = " + QString::number(EGFRStatusParams[1] / EGFRStatusParams[2]) + "\n\nNear ROI voxels used = " + QString::number(EGFRStatusParams[3]) + "/" + QString::number(nearIndices.size()) + "\nFar ROI voxels used =   " + QString::number(EGFRStatusParams[4]) + "/" + QString::number(farIndices.size());
  
  updateProgress(0);
  ShowMessage(msg.toStdString());
}

void fMainWindow::ApplicationRecurrence()
{
  vector<string> fileNames;
  this->getLodedImages(fileNames,true);
  if (fileNames.size())
  {
    recurrencePanel.SetCurrentImagePath(mInputPathName.toStdString());
    recurrencePanel.exec();
  }
  else
  {
    ShowErrorMessage("No images selected!", "Notice");
  }
}

void fMainWindow::ApplicationSurvival()
{
  survivalPanel.SetCurrentImagePath(mInputPathName);
  survivalPanel.setModal(true);
  survivalPanel.exec();
}

void fMainWindow::ApplicationITKSNAP()
{
  if (mSlicerManagers.size() > 0)
  {
    ImageTypeFloat3D::Pointer mask = convertVtkToItk< ImageTypeFloat3D::PixelType, ImageTypeFloat3D::ImageDimension >(mSlicerManagers[0]->mMask);
    std::string maskFile = tempFolderLocation + "mask.nii.gz";

    cbica::WriteImage<ImageTypeFloat3D>(mask, maskFile);

    Registry r;
    r["SaveLocation"] << tempFolderLocation;
    r["Version"] << "20161029";

    for (int i = 0; i < m_imagesTable->rowCount(); i++)
    {
      auto test = m_imagesTable->item(i, 2);
      auto pathTest = mSlicerManagers[i]->mPathFileName;
      Registry &rl = r.Folder(Registry::Key("Layers.Layer[%03d]", i));
      rl["AbsolutePath"] << mSlicerManagers[i]->mPathFileName;
      if (i == 0)
      {
        rl["Role"] << "MainRole";
      }
      else
      {
        rl["Role"] << "OverlayRole";
      }

      // LayerMetaData
      Registry &rlLayerMetaData = rl.Folder(Registry::Key("LayerMetaData"));
      rlLayerMetaData["Alpha"] << "255";
      rlLayerMetaData["CustomNickName"] << "";
      rlLayerMetaData["Sticky"] << "0";
      Registry &rlMetaDataDisplayMap = rlLayerMetaData.Folder(Registry::Key("DisplayMapping"));
      Registry &rlMetaDataDisplayMapColor = rlMetaDataDisplayMap.Folder(Registry::Key("ColorMap"));
      rlMetaDataDisplayMapColor["ColorMap"] << "";
      rlMetaDataDisplayMapColor["Preset"] << "Grayscale";

      // ProjectMetaData
      Registry &rlProjectMetaData = rl.Folder(Registry::Key("ProjectMetaData"));
      rlProjectMetaData["GaussianBlurScale"] << "1";
      rlProjectMetaData["RemappingExponent"] << "3";
      rlProjectMetaData["RemappingSteepness"] << "0.04";
      Registry &rlProjectMetaData_IRIS = rlProjectMetaData.Folder(Registry::Key("IRIS"));
      rlProjectMetaData_IRIS["SliceViewLayerLayout"] << "Stacked";

    }

    // stuff for the mask
    Registry &rMask = r.Folder(Registry::Key("Layers.Layer[%03d]", m_imagesTable->rowCount()));
    rMask["AbsolutePath"] << maskFile;
    rMask["Role"] << "SegmentationRole";
    Registry &rMaskFolder = rMask.Folder(Registry::Key("LayerMetaData"));
    rMaskFolder["Alpha"] << "6.21436e-310";
    rMaskFolder["CustomNickName"] << "";
    rMaskFolder["Sticky"] << "1";

    r.WriteToXMLFile(std::string(tempFolderLocation + "testXML.itksnap").c_str());

    std::string scriptToCall;
    for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
    {
      if (PreparePythonScriptAndConfig("ITK-SNAP", m_pyGUIApps[i], scriptToCall))
      {
        break;
      }
    }

    QString itkSnapLocation = scriptToCall.c_str();
    QStringList itkSnapArgs;
    itkSnapArgs << "-w" << std::string(tempFolderLocation + "testXML.itksnap").c_str();
    
    startExternalProcess(itkSnapLocation, itkSnapArgs);

    readMaskFile(maskFile);

    statusBar()->showMessage("Mask saved from ITK-SNAP loaded", 5);
  }
  else
  {
    std::string scriptToCall;
    for (size_t i = 0; i < vectorOfPythonGUIAppActionsAndNames.size(); i++)
    {
      if (PreparePythonScriptAndConfig("ITK-SNAP", m_pyGUIApps[i], scriptToCall))
      {
        break;
      }
    }

    statusBar()->showMessage("ITK-SNAP called without loading any images", 5);

    QString itkSnapLocation = scriptToCall.c_str();
    QStringList itkSnapArgs;
    itkSnapArgs << "-w" << std::string(tempFolderLocation + "testXML.itksnap").c_str();

    startExternalProcess(itkSnapLocation, itkSnapArgs);

    std::vector< std::string > filesInTemp = cbica::filesInDirectory(tempFolderLocation);
    if (!filesInTemp.empty())
    {
      statusBar()->showMessage("Found file(s) in ITK-SNAP's working directory, loading them into memory", 10);
      for (size_t i = 0; i < filesInTemp.size(); i++)
      {
        std::string currentExtension = cbica::getFilenameExtension(filesInTemp[i]);
        if ((currentExtension == ".nii.gz") || (currentExtension == ".nii"))
        {
          LoadSlicerImages(filesInTemp[i], IMAGE_NIFTI);
        }
        else if (currentExtension == ".dcm")
        {
          LoadSlicerImages(filesInTemp[i], IMAGE_DICOM);
        }
      }
    }
  }
}

void fMainWindow::ApplicationGeodesic()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
  {
    ShowErrorMessage("Please specify an input image.", "Notice");
    return;
  }
  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size())
  {
    ShowErrorMessage("Please specify an input image.", "Notice");
    return;
  }
  updateProgress(5, "Running Geodesic Segmentation...");
  VectorVectorDouble tumorPoints = FormulateDrawingPointsForTumorSegmentation();
  if (tumorPoints.size() == 0)
  {
    ShowErrorMessage("Please draw initial seed points.", "Notice");
    return;
  }
  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  typedef ImageTypeFloat3D ImageType;
  typedef ImageTypeShort3D ImageTypeGeodesic;
  updateProgress(10, "Running Geodesic Segmentation...");
  ImageTypeGeodesic::Pointer Inp = ImageTypeGeodesic::New();
  typedef itk::CastImageFilter<ImageType, ImageTypeGeodesic> CastFilterType;
  CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(convertVtkToItk<float, 3>(mSlicerManagers[index]->mImage));
  Inp = castFilter->GetOutput();
  typedef itk::RescaleIntensityImageFilter< ImageTypeGeodesic, ImageTypeGeodesic > FilterType;
  FilterType::Pointer filter = FilterType::New();
  filter->SetInput(Inp);
  filter->SetOutputMinimum(0);
  filter->SetOutputMaximum(255);
  filter->Update();
  Inp = filter->GetOutput();
  updateProgress(15, "Running Geodesic Segmentation...");
  Inp = mGeodesicSegmentation.Run<ImageTypeGeodesic>(Inp, tumorPoints);
  updateProgress(85, "Running Geodesic Segmentation...");
  filter->SetInput(Inp);
  filter->SetOutputMinimum(0);
  filter->SetOutputMaximum(255);
  filter->Update();
  Inp = filter->GetOutput();

  itk::ImageRegionIterator<ImageTypeGeodesic> imageIterator(Inp, Inp->GetLargestPossibleRegion());
  updateProgress(90, "Displaying Geodesic Segmentation...");
  while (!imageIterator.IsAtEnd())
  {
    ImageTypeGeodesic::IndexType currentIndex = imageIterator.GetIndex();
    float val = imageIterator.Get();
    if (val < thresholdSlider->value())
    {
      float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)currentIndex[0], (int)currentIndex[1], (int)currentIndex[2]);
      *pData = 1;
    }
    ++imageIterator;
  }
  updateProgress(100);
  this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
  this->mSlicerManagers[0]->Render();
  updateProgress(0, "Geodesic Segmentation Finished!");
}

void fMainWindow::ImageDenoising()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;

  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size())
    return;

  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  QFileDialog dialog/*(this, tr("Save Image"), mInputPathName, extensions)*/;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  //dialog.selectFile(QString((mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mFileName) + "_denoise.nii.gz").c_str()));
  std::string preferredName = mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mPathFileName) + "_denoise.nii.gz";
  QString saveFileName = dialog.getSaveFileName(this, tr("Save Image"), preferredName.c_str(), extensions, 0, QFileDialog::DontUseNativeDialog);
  //int ret = dialog.exec();
  if (!saveFileName.isEmpty())
  {
    typedef	ImageTypeFloat3D ImageType;
    // typedef ImageType::Pointer ImageTypePointer;
    file = saveFileName;
    //Job TBD replace with app name 
    typedef itk::ImageDuplicator<ImageType> DuplicatorType;
    DuplicatorType::Pointer duplicator = DuplicatorType::New();
    duplicator->SetInputImage(mSlicerManagers[index]->mITKImage);
    duplicator->Update();
    ImageType::Pointer inputImage = duplicator->GetOutput();
    updateProgress(5, "Susan noise removal in process......");
    SusanDenoising denoising /*= SusanDenoising()*/;
    ImageType::Pointer outputImage = denoising.Run<ImageType>(inputImage);
    if (outputImage.IsNotNull())
    {

      updateProgress(80, "Saving file...");
      cbica::WriteImage< ImageType >(outputImage, file.toStdString());
      if (cbica::fileExists(file.toStdString()))
      {
        updateProgress(90, "Displaying output...");
        LoadSlicerImages(file.toStdString(), IMAGE_NIFTI);

      }
      updateProgress(0, "Susan noise removal finished");
    }
    else
    {
      updateProgress(0, "Error in Susan noise removal!!");
    }
  }
}
void fMainWindow::ImageBiasCorrection()
{
  QList<QTableWidgetItem*> items = m_imagesTable->selectedItems();
  if (items.empty())
    return;

  int index = GetSlicerIndexFromItem(items[0]);
  if (index < 0 || index >= (int)mSlicerManagers.size())
    return;

  QString extensions = IMAGES_EXTENSIONS;
  extensions += ";;All Files (*)";
  QString file;
  QFileDialog dialog/*(this, tr("Save Image"), mInputPathName, extensions)*/;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  std::string preferredName = mInputPathName.toStdString() + cbica::getFilenameBase(mSlicerManagers[index]->mPathFileName) + "_biasCorrect.nii.gz";
  QString saveFileName = dialog.getSaveFileName(this, tr("Save Image"), preferredName.c_str(), extensions, 0, QFileDialog::DontUseNativeDialog);
  //int ret = dialog.exec();
  if (!saveFileName.isEmpty())
  {
    typedef	ImageTypeFloat3D ImageType;
    file = saveFileName;
    typedef itk::ImageDuplicator<ImageType> DuplicatorType;
    DuplicatorType::Pointer duplicator = DuplicatorType::New();
    duplicator->SetInputImage(mSlicerManagers[index]->mITKImage);
    duplicator->Update();
    ImageType::Pointer inputImage = duplicator->GetOutput();
    updateProgress(5, "Bias correction in process...");
    N3BiasCorrection biasCorrecter /*= N3BiasCorrection()*/;
    ImageType::Pointer outputImage = biasCorrecter.Run<ImageType>(inputImage);
    if (outputImage.IsNotNull())
    {
      updateProgress(80, "Saving file...");
      cbica::WriteImage< ImageType >(outputImage, file.toStdString());
      if (cbica::fileExists(file.toStdString()))
      {
        updateProgress(90, "Displaying output...");
        LoadSlicerImages(file.toStdString(), IMAGE_NIFTI);
      }
      updateProgress(0, "Bias correction finished");
    }
    else
    {
      updateProgress(0, "Error in Bias correction!");
    }
  }
}
void fMainWindow::ImageRegistration()
{
  registrationPanel.exec();
}
void fMainWindow::CustomPreprocessing()
{

}
void fMainWindow::ChangeBrushSize(int size)
{
  updateDrawMode();
}

void fMainWindow::ChangeMaskOpacity(int newMaskOpacity) // multiLabel uncomment this function
{
  double test = newMaskOpacity * 0.1;
  for (size_t i = 0; i < this->mSlicerManagers.size(); i++)
  {
    for (size_t j = 0; j < 3; j++)
    {
      this->mSlicerManagers[i]->GetSlicer(j)->mMaskOpacity = test;
      this->mSlicerManagers[i]->GetSlicer(j)->mMaskActor->SetOpacity(test);
      this->mSlicerManagers[i]->GetSlicer(j)->mMask->Modified();
    }
  }
  this->mSlicerManagers[0]->Render();
  //UpdateRenderWindows();
}

void fMainWindow::ChangeDrawingLabel(int drawingLabel) // multiLabel uncomment this function
{
  updateDrawMode();
}
void fMainWindow::Registration(std::string &fixedFileName, std::vector<std::string> &inputFileNames, std::vector<std::string> &outputFileNames)
{
  typedef ImageTypeFloat3D ImageType;
  typedef ImageType::Pointer ImageTypePointer;
  typedef ImageType InternalImageType;
  itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;


  ImageTypePointer TargetImage = mNifiDataManager.ReadNiftiImage(fixedFileName);
  std::vector< ImageTypePointer > outputImages;
  for (unsigned int i = 0; i < inputFileNames.size(); i++)
  {
    ImageTypePointer SourceImage = mNifiDataManager.ReadNiftiImage(inputFileNames[i]);
    Registrar = mPreprocessingObj.Registration<ImageType, InternalImageType>(TargetImage, SourceImage);
    ImageTypePointer RegisteredImage = mPreprocessingObj.ResampleTransform<ImageType>(Registrar, TargetImage, SourceImage);
    // putting the registered images in a vector for visualization 
    outputImages.push_back(RegisteredImage);
    mOutputManager.WriteImageWithGivenName<ImageType>(RegisteredImage, outputFileNames[i]); // do this only if the user wants the images to be saved 
  }
}

void fMainWindow::UpdateAction(std::vector<PointVal> points)
{

  mActionPoints.push_back(points);
}

void fMainWindow::UndoFunctionality()
{
	if (mActionPoints.empty()) return;
	vector<PointVal>  OneStrkePoints = mActionPoints.back();
	//Its important to do the undo in reverse order of what happend
	for (vector<PointVal>::iterator it = OneStrkePoints.end(); it != OneStrkePoints.begin();)
	{
		--it;
		PointVal pt = *it;
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer(pt.x, pt.y, pt.z);
		*pData = pt.value;
	}
	mActionPoints.pop_back();

	this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
	this->mSlicerManagers[0]->Render();
}


void fMainWindow::SetOpacity()
{
  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    for (int i = 0; i < 3; i++)
    {
      if (this->mSlicerManagers[index]->GetSlicer(i)->GetMaskOpacity() == 0)
        this->mSlicerManagers[index]->GetSlicer(i)->SetMaskOpacity(1);
      else
        this->mSlicerManagers[index]->GetSlicer(i)->SetMaskOpacity(0);
    }
  }
  this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
  this->mSlicerManagers[0]->Render();
}

void fMainWindow::closeEvent(QCloseEvent * event)
{
  event->ignore();
  if (QMessageBox::Yes == QMessageBox::question(this, "Close Confirmation!",
    "Are you sure you want to exit?", QMessageBox::Yes | QMessageBox::No))
  {
    event->accept();
  }
};
void fMainWindow::updateProgress(int progress, string message, int max)
{
#ifdef USE_PROCESSDIALOG
	m_progressBar->setMaximum(max);
	m_progressBar->setValue(progress);
	m_messageLabel->setText(QString::fromStdString(message));
	qApp->processEvents();
#endif
}