#include "itkImage.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkTestingComparisonImageFilter.h"
#include "itkExceptionObject.h"

#include "cbicaCmdParser.h"
#include "cbicaUtilities.h"
#include "cbicaITKSafeImageIO.h"

#include "GeodesicSegmentation.h"
#include "CAPTk.h"

int main(int argc, char** argv)
{
  cbica::CmdParser parser = cbica::CmdParser(argc, argv);

  parser.setExeName("CaPTk_Tests");
  parser.addOptionalParameter("i", "inputFile", cbica::Parameter::FILE, ".nii.gz file", "Generic input file");
  parser.addOptionalParameter("geo", "geodesic", cbica::Parameter::FILE, ".nii.gz drawing", "Geodesic test");

  if (parser.isPresent("h"))
  {
    parser.echoHelp();
    return EXIT_SUCCESS;
  }

  if (parser.isPresent("u"))
  {
    parser.echoUsage();
    return EXIT_SUCCESS;
  }

  if (parser.isPresent("v"))
  {
    parser.echoVersion();
    return EXIT_SUCCESS;
  }

  const int numberOfPixelsTolerance = 10; // number of pixels that are acceptable to have intensity differences
  std::string inputFile, drawingFile;
  parser.getParameterValue("i", inputFile);
  typedef itk::Image< short, 3 > ImageType;
  ImageType::Pointer inputImage = cbica::ReadImage< ImageType >(inputFile);

  if (parser.isPresent("geo"))
  {
    std::string geosTest = cbica::getFilenamePath(inputFile) + "input_flair_geos.nii.gz";
    ImageType::Pointer testImage = cbica::ReadImage< ImageType >(geosTest);
    cbica::Logging(loggerFile, "Geodesic segmentation tests started");
    GeodesicSegmentation segmentationClass;
    parser.getParameterValue("geo", drawingFile);

    ImageType::Pointer drawing = cbica::ReadImage< ImageType >(drawingFile);
    VectorVectorDouble Indices;

    typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
    IteratorType maskIt(drawing, drawing->GetLargestPossibleRegion());
    maskIt.GoToBegin();
    while (!maskIt.IsAtEnd())
    {
      if (maskIt.Get() != 0) // anything which is not the background is taken as input 
      {
        VectorDouble localIndex;
        localIndex.push_back(maskIt.GetIndex()[0]);
        localIndex.push_back(maskIt.GetIndex()[1]);
        localIndex.push_back(maskIt.GetIndex()[2]);

        Indices.push_back(localIndex);
      }
      ++maskIt;
    }

    ImageType::Pointer geosOutput = segmentationClass.Run(inputImage, Indices);

    typedef itk::Testing::ComparisonImageFilter< ImageType, ImageType > ComparisonFilterType;
    ComparisonFilterType::Pointer comparator = ComparisonFilterType::New();
    comparator->SetValidInput(geosOutput);
    comparator->SetTestInput(testImage);
    try
    {
      comparator->UpdateLargestPossibleRegion();
    }
    catch (itk::ExceptionObject e)
    {
      cbica::Logging(loggerFile, "Exception caught while doing comparator update");
    }

    bool differenceFailed = false;
    const double averageIntensityDifference = comparator->GetTotalDifference();
    const unsigned long numberOfPixelsWithDifferences = comparator->GetNumberOfPixelsWithDifferences();
    
    // the test fails if any of the following conditions are met
    if ((averageIntensityDifference > 0.0) && // if intensity difference isn't absolute zero 
      (static_cast<int>(numberOfPixelsWithDifferences) > numberOfPixelsTolerance)) // or number of pixels with difference is above a threshold
    {
      // write some meaninfgul log if there is a failure
      cbica::Logging(loggerFile, "Geodesic test failed with parameters: averageIntensityDifference = '" + std::to_string(averageIntensityDifference) + "' and numberOfPixelsWithDifferences = '" + std::to_string(numberOfPixelsWithDifferences));
      return EXIT_FAILURE;
    }

  }


  return EXIT_SUCCESS;
}