/**
\file  SlicerManager.h

\brief Declaration of SlicerManager class

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#ifndef _SlicerManager_h_
#define _SlicerManager_h_


#include "CAPTk.h"
#include "Slicer.h"
#include "InteractorStyleNavigator.h"
#include "PreprocessingPipelineClass.h"
#include "itkImageToVTKImageFilter.h"
#include "itkExtractImageFilter.h"

#define PRESET_AUTO 0
#define PRESET_USER 1
#define PRESET_LABEL 2
#define PRESET_LABEL2 3
#define PRESET_THRESHOLD 4
#define PRESET_PROB 5

/**
\class SlicerManager

\brief This class handles the 3 different slices being visualized from an abstract level

*/
class SlicerManager : public QObject
{
  Q_OBJECT

public:
  SlicerManager(int numberOfSlicers, Landmarks* landmarks, Landmarks* seedPoints, Landmarks* tissuePoints);
  ~SlicerManager();

  std::string GetLastError()
  {
    return mLastError;
  }

  bool SetImage(const std::string &filename, ImageTypeFloat3D::Pointer t1ceimage, int imagetype, int subimagetype);

  template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
  bool SetImage(typename itk::Image<InputPixelType, VImageDimension>::Pointer image, vtkTransform* transform);

  void SetMask(vtkImageData* mask);
  vtkImageData* GetMask()
  {
    return mMask;
  }

  std::string GetPathFileName()
  { 
    return mPathFileName; 
  }
  std::string GetFileName()
  { 
    return mFileName; 
  }
  std::string GetBaseFileName()
  { 
    return mBaseFileName; 
  }

  void ToggleInterpolation();
  Slicer* GetSlicer(int i);
  void UpdateSlicer(int num, bool state);
  void SetSlicerWindow(int i, vtkRenderWindow* RW);
  void SetInteractorStyleNavigator(int i, vtkInteractorStyle* style);

  int GetNumberOfSlicers()
  { 
    return mSlicers.size(); 
  }
  vtkImageData* GetImage()
  { 
    return mImage; 
  }

  void SetId(const std::string &id)
  {
    mId = id;
  }
  std::string GetId()
  {
    return mId;
  }

  int GetDimension() 
  {
    if (mImage) {
      int dim = 0;
#if VTK_MAJOR_VERSION <= 5
      int* extent = mImage->GetWholeExtent();
#else
      int* extent = mImage->GetExtent();
#endif
      if (extent[4] != extent[5]) {
        dim = 3;
      }
      else if (extent[3] != extent[4]) {
        dim = 2;
      }
      else if (extent[0] != extent[1]) {
        dim = 1;
      }
      return dim;
    }
    else {
      return -1;
    }
  }
  void SetOrder(int order) 
  {
    mOrder = order;
  }
  int GetOrder() 
  {
    return mOrder;
  }

  void SetFilename(const std::string &f);
  void SetSliceOrientation(int slicer, int orientation);
  void SetTSlice(int slice);
  void SetNextTSlice(int originating_slicer);
  void SetPreviousTSlice(int originating_slicer);
  void SetTSliceInSlicer(int tslice, int slicer);

  void GenerateDefaultLookupTable();
  void SetColorWindow(double s);
  void SetColorLevel(double s);
  void SetOpacity(int i, double factor);
  void SetPreset(int preset);
  void SetThresholdIndex(int i) 
  {
    mThresholdIndex = i;
  }

  double GetColorWindow();
  double GetColorLevel();
  int GetPreset() 
  {
    return mPreset;
  }
  int GetThresholdIndex() 
  {
    return mThresholdIndex;
  }

  void UpdateViews(int slicer);
  void UpdateLinked(int slicer);
  void UpdateLinkedNavigation(Slicer *slicer, bool bPropagate = false);
  void ResetTransformationToIdentity();
  void Render();

  void AddLink(const std::string &newId)
  {
    mLinkedId.push_back(newId);
    mLinkedId.unique();
  }
  void RemoveLink(const std::string &oldId)
  {
    mLinkedId.remove(oldId);
  }
  bool IsLinked() 
  {
    return mLinkedId.size() > 0;
  }

  void RemoveActors();
  void Activated();
  void Picked();
  void UpdateInfoOnCursorPosition(int slicer);
  void UpdateInfoOnCurrentPosition(int slicer);
  void UpdateWindowLevel();
  void UpdateSlice(int slicer);
  void UpdateSliceRange(int slicer);

  void AddLandmark(float x, float y, float z);
  //void AddLoadedLandmark(double x, double y,double z,int landmarktype);
  void RemoveLandmark(float x, float y, float z);
  void AddLandmarkRadius(float x, float y, float z);
  void AddLandmarkShift(float x, float y, float z);

  void NextImageWithOrder(int order);
  void LeftButtonReleaseEvent(int slicer);
  void VerticalSliderHasChanged(int slicer, int slice);
  double GetScalarComponentAsDouble(vtkImageData *image, double X, double Y, double Z, int component = 0);

  void SetCurrentLandmarksType(int type, int row, int col);

  void EraseCompleteNearDrawing();
  void EraseCompleteFarDrawing();

  Landmarks* mLandmarks;
  Landmarks* mSeedPoints;
  Landmarks* mTissuePoints;
  int mCurrentLandmarksType;
  int mCurrentLandmarksRow;
  int mCurrentLandmarksCol;
  int mTissueSelectedEntry;

  template<unsigned int VImageDimension>
  void ReadImageWithDim(const std::string &filename, const std::string &InputPixelType);

  template<class OutputPixelType, unsigned int VImageDimension>
  void ReadImageWithDim(const std::string &filename, const std::string &InputPixelType, ImageTypeFloat3D::Pointer t1ceImage, int imagetype, int subimagetype);

  template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
  void ReadImageWithDimAndInputPixelType(std::string filename, ImageTypeFloat3D::Pointer t1ceImage, int imagetype, int subimagetype);

  template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
  void ReadImageWithDimAndInputPixelType(typename itk::Image<InputPixelType, VImageDimension>::Pointer inputImage);

  std::string GetFileNameInADicomDirectory(QString directoryname);
  void UpdateBorderCoordinates(double startX, double startY, double endX, double endY);
  void UpdateBorderCoordinates(double startZ, double endZ);
  void ActionAdded(VectorVectorDouble &points,int actionid);
  void Get3DImageAtCurrentPerfusionIndex(int index);
signals:
  void currentImageChanged(std::string id);
  void currentPickedImageChanged(std::string id);
  void UpdatePosition(int visibility, double x, double y, double z, double X, double Y, double Z, double value);
  void UpdateOrientation(int slicer, int orientation);
  void UpdateSlice(int slicer, int slice);
  void UpdateSliceRange(int slice, int min, int max);
  void WindowLevelChanged();
  void UpdateLinkManager(std::string, int slicer, double x, double y, double z);
  void UpdateLinkedNavigation(std::string, SlicerManager*, Slicer*);
  void LandmarkAdded();
  void UpdateNumberOfPoints(int drawn_points, VectorDouble tissueTypeCounter);
  void SeedPointsAdded();
  void SeedPointsAdded(int index, bool update);
  void TissuePointsAdded(int index);
  void HighlightTableIndexOnDeletion(double x, double y, double z, double x1, double y1, double z1, double value);
  void TissuePointsRemoved(int index);
  void ChangeImageWithOrder(SlicerManager *sm, int order);
  void LeftButtonReleaseSignal(int slicer);
  void AVerticalSliderHasChanged(int slicer, int slice);



  void UpdateNumberOfPoints(int nearpoints, int farpoints);
  void UpdateBorderWidgetInMain(double startX, double startY, double endX, double endY);
  void UpdateBorderWidgetInMain(double startZ, double endZ);
  void UpdateActionInMain(VectorVectorDouble points, int actionid);




public:
  vtkSmartPointer<vtkImageData> mImage;
  ImageTypeFloat3D::Pointer mITKImage;
  ImageTypeFloat4D::Pointer mPerfusionImagePointer;
  std::vector< ImageTypeFloat3D::Pointer > mPerfusionImagePointerDicom;
  vtkSmartPointer<vtkTransform> mTransform;
  vtkSmartPointer<vtkImageData> mMask;
  itk::ImageSeriesReader< ImageTypeFloat3D >::DictionaryArrayRawPointer mDicomDictionaryArray;
  itk::GDCMSeriesFileNames::FileNamesContainerType mFileNames;
  itk::ImageSeriesReader< ImageTypeShort3D >::Pointer mSeriesReader;
  typedef itk::ProcessObject::Pointer ConverterPointer;
  ConverterPointer mItkToVtkConverters;
  std::vector<vtkSmartPointer<Slicer> > mSlicers;

  int mPreset;
  std::string mPathFileName;
  int mImageType;
  int mImageSubType;
  std::string mFileName;
  std::string mBaseFileName;
  int mBaseFileNameNumber;
  std::string mId;
  std::string mLastError;
  std::list<std::string> mLinkedId;

  std::vector<int> mPreviousSlice;
  std::vector<int> mPreviousTSlice;

  int mOrder;
  int mThresholdIndex;
  double mThreshold;

};

template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
bool SlicerManager::SetImage(typename itk::Image<InputPixelType, VImageDimension>::Pointer image, vtkTransform* transform)
{
  ReadImageWithDimAndInputPixelType<InputPixelType, OutputPixelType, VImageDimension>(image);

  mTransform = vtkTransform::New();
  mTransform->SetMatrix(transform->GetMatrix());

  for (unsigned int i = 0; i < mSlicers.size(); i++)
  {
    mSlicers[i]->SetImage(mImage, mTransform);
    mSlicers[i]->GetImageActor()->SetInterpolate(0);
  }

  return true;
}

template<unsigned int VImageDimension>
void SlicerManager::ReadImageWithDim(const std::string &filename, const std::string &InputPixelType)
{
  if (InputPixelType == "short") {
    ReadImageWithDimAndInputPixelType<short, short, VImageDimension>(filename);
  }
  else if (InputPixelType == "unsigned_short") {
    ReadImageWithDimAndInputPixelType<unsigned short, unsigned short, VImageDimension>(filename);
  }
  else if (InputPixelType == "char") {
    ReadImageWithDimAndInputPixelType<char, char, VImageDimension>(filename);
  }
  else if (InputPixelType == "unsigned_char") {
    ReadImageWithDimAndInputPixelType<unsigned char, unsigned char, VImageDimension>(filename);
  }
  else if (InputPixelType == "int") {
    ReadImageWithDimAndInputPixelType<int, int, VImageDimension>(filename);
  }
  else if (InputPixelType == "unsigned_int") {
    ReadImageWithDimAndInputPixelType<unsigned int, unsigned int, VImageDimension>(filename);
  }
  else if (InputPixelType == "double") {
    ReadImageWithDimAndInputPixelType<double, double, VImageDimension>(filename);
  }
  else if (InputPixelType == "float") {
    ReadImageWithDimAndInputPixelType<float, float, VImageDimension>(filename);
  }
  else
  {
    std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
  }
}
template<class OutputPixelType, unsigned int VImageDimension>
void SlicerManager::ReadImageWithDim(const std::string &filename, const std::string &InputPixelType, ImageTypeFloat3D::Pointer t1ceimage, int imagetype, int subimagetype)
{
  if (InputPixelType == "short")
  {
    ReadImageWithDimAndInputPixelType<short, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "signed short")
  {
    ReadImageWithDimAndInputPixelType<short, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "unsigned_short")
  {
    ReadImageWithDimAndInputPixelType<unsigned short, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "char")
  {
    ReadImageWithDimAndInputPixelType<char, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "unsigned_char")
  {
    ReadImageWithDimAndInputPixelType<unsigned char, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype,subimagetype);
  }
  else if (InputPixelType == "int")
  {
    ReadImageWithDimAndInputPixelType<int, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "unsigned_int")
  {
    ReadImageWithDimAndInputPixelType<unsigned int, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "double")
  {
    ReadImageWithDimAndInputPixelType<double, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else if (InputPixelType == "float")
  {
    ReadImageWithDimAndInputPixelType<float, OutputPixelType, VImageDimension>(filename, t1ceimage, imagetype, subimagetype);
  }
  else
  {
    std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
  }
}

template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
void SlicerManager::ReadImageWithDimAndInputPixelType(typename itk::Image<InputPixelType, VImageDimension>::Pointer inputImage)
{
  typedef itk::Image<InputPixelType, VImageDimension> InputImageType;
  typedef itk::Image<OutputPixelType, VImageDimension> OutputImageType;
  typedef itk::CastImageFilter< InputImageType, OutputImageType > CastFilterType;

  typename CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(inputImage);

  typename OutputImageType::Pointer outputImage = castFilter->GetOutput();
  outputImage->Update();

  typedef itk::ImageToVTKImageFilter<InputImageType> ConverterType;
  typename ConverterType::Pointer converter = ConverterType::New();
  mItkToVtkConverters = dynamic_cast< itk::ProcessObject *>(converter.GetPointer());
  converter->SetInput(outputImage);
  converter->Update();

  mImage = converter->GetOutput();
}

template <typename ITK_Exporter, typename VTK_Importer>
void ConnectPipelines(ITK_Exporter &exporter, VTK_Importer* importer)
{
  importer->SetUpdateInformationCallback(exporter->GetUpdateInformationCallback());
  importer->SetPipelineModifiedCallback(exporter->GetPipelineModifiedCallback());
  importer->SetWholeExtentCallback(exporter->GetWholeExtentCallback());
  importer->SetSpacingCallback(exporter->GetSpacingCallback());
  importer->SetOriginCallback(exporter->GetOriginCallback());
  importer->SetScalarTypeCallback(exporter->GetScalarTypeCallback());
  importer->SetNumberOfComponentsCallback(exporter->GetNumberOfComponentsCallback());
  importer->SetPropagateUpdateExtentCallback(exporter->GetPropagateUpdateExtentCallback());
  importer->SetUpdateDataCallback(exporter->GetUpdateDataCallback());
  importer->SetDataExtentCallback(exporter->GetDataExtentCallback());
  importer->SetBufferPointerCallback(exporter->GetBufferPointerCallback());
  importer->SetCallbackUserData(exporter->GetCallbackUserData());
}
template <typename VTK_Exporter, typename ITK_Importer>
void ConnectPipelines(VTK_Exporter* exporter, ITK_Importer &importer)
{
  importer->SetUpdateInformationCallback(exporter->GetUpdateInformationCallback());
  importer->SetPipelineModifiedCallback(exporter->GetPipelineModifiedCallback());
  importer->SetWholeExtentCallback(exporter->GetWholeExtentCallback());
  importer->SetSpacingCallback(exporter->GetSpacingCallback());
  importer->SetOriginCallback(exporter->GetOriginCallback());
  importer->SetScalarTypeCallback(exporter->GetScalarTypeCallback());
  importer->SetNumberOfComponentsCallback(exporter->GetNumberOfComponentsCallback());
  importer->SetPropagateUpdateExtentCallback(exporter->GetPropagateUpdateExtentCallback());
  importer->SetUpdateDataCallback(exporter->GetUpdateDataCallback());
  importer->SetDataExtentCallback(exporter->GetDataExtentCallback());
  importer->SetBufferPointerCallback(exporter->GetBufferPointerCallback());
  importer->SetCallbackUserData(exporter->GetCallbackUserData());
}

template<class InputPixelType, class OutputPixelType, int VImageDimension>
typename itk::Image<OutputPixelType, VImageDimension>::Pointer
convertVtkToItk(vtkImageData* vtkInput)
{
  typedef itk::Image<InputPixelType, VImageDimension> InputImageType;
  typedef itk::Image<OutputPixelType, VImageDimension> OutputImageType;
  typedef itk::VTKImageImport<InputImageType> InputImageImportType;

  vtkSmartPointer<vtkImageExport> inputImageExporter = vtkImageExport::New();
#if VTK_MAJOR_VERSION <= 5
  inputImageExporter->SetInput(vtkInput);
#else
  inputImageExporter->SetInputData(vtkInput);
#endif
  typename InputImageImportType::Pointer inputImageImporter = InputImageImportType::New();

  ConnectPipelines(inputImageExporter.GetPointer(), inputImageImporter);

  typename InputImageType::Pointer inputImage = const_cast<InputImageType*>(inputImageImporter->GetOutput());
  inputImage->Update();
  if (inputImage.IsNull()) {
    std::cerr << "vtkImput is invalid." << std::endl;
  }

  typedef itk::CastImageFilter< InputImageType, OutputImageType > CastFilterType;
  typename CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(inputImage);
  castFilter->Update();

  typename OutputImageType::Pointer outputImage = castFilter->GetOutput();
  outputImage->Update();
  if (outputImage.IsNull()) {
    std::cerr << "inputImage is invalid." << std::endl;
    return NULL;
  }

  return outputImage;
}

template<class OutputPixelType, int VImageDimension>
typename itk::Image<OutputPixelType, VImageDimension>::Pointer convertVtkToItk(vtkImageData* vtkInput)
{
  std::string InputPixelType = vtkInput->GetScalarTypeAsString();
  if (InputPixelType == "short")
  {
    return convertVtkToItk<short, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned short")
  {
    return convertVtkToItk<unsigned short, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "char")
  {
    return convertVtkToItk<char, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned char")
  {
    return convertVtkToItk<unsigned char, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "int")
  {
    return convertVtkToItk<int, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned int")
  {
    return convertVtkToItk<unsigned int, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "double")
  {
    return convertVtkToItk<double, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "float")
  {
    return convertVtkToItk<float, OutputPixelType, VImageDimension>(vtkInput);
  }
  else {
    std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
  }
  return NULL;
}

ImageTypeFloat3D::Pointer convertVtkToItkFloat(vtkImageData* vtkInput);



template<class InputPixelType, class OutputPixelType, unsigned int VImageDimension>
void SlicerManager::ReadImageWithDimAndInputPixelType(std::string directoryName, ImageTypeFloat3D::Pointer t1ceImage, int imagetype, int subimagetype)
{
  if (imagetype == IMAGE_DICOM)
  {
    typedef itk::Image< InputPixelType, VImageDimension > InputImageType;
    typedef itk::ImageSeriesReader< InputImageType > ReaderType;
    typedef itk::Image<float, VImageDimension> OutputImageType;
    typename ReaderType::Pointer seriesreader = ReaderType::New();
    itk::ImageSeriesReader< itk::Image< short, 3 > >::Pointer seriesreader2 = itk::ImageSeriesReader< itk::Image< short, 3 > >::New();

    typedef itk::GDCMImageIO ImageIOType;
    ImageIOType::Pointer dicomIO = ImageIOType::New();
    seriesreader->SetImageIO(dicomIO);
    seriesreader2->SetImageIO(dicomIO);
    typedef itk::GDCMSeriesFileNames NamesGeneratorType;
    NamesGeneratorType::Pointer nameGenerator = NamesGeneratorType::New();
    nameGenerator->SetUseSeriesDetails(true);
    nameGenerator->AddSeriesRestriction("0008|0021");
    nameGenerator->AddSeriesRestriction("0020|0012");   //for perfusion data
	  nameGenerator->SetInputDirectory(directoryName.c_str());

    typedef std::vector< std::string > SeriesIdContainer;
    const SeriesIdContainer & seriesUID = nameGenerator->GetSeriesUIDs();

    if (subimagetype == IMAGE_TYPE_PERFUSION)
    {
      SeriesIdContainer::const_iterator seriesItr = seriesUID.begin();
      SeriesIdContainer::const_iterator seriesEnd = seriesUID.end();
      int counter = 0;
      while (seriesItr != seriesEnd)
      {
        mImageType = imagetype;
        typedef std::vector< std::string > FileNamesContainer;
        FileNamesContainer fileNames;
        fileNames = nameGenerator->GetFileNames(seriesItr->c_str());
        seriesreader->SetFileNames(fileNames);
        try
        {
          seriesreader->Update();
          typedef ImageTypeFloat3D OutputImageType;
          typedef itk::CastImageFilter< InputImageType, OutputImageType > CastFilterType;
          typename CastFilterType::Pointer castFilter = CastFilterType::New();
          castFilter->SetInput(seriesreader->GetOutput());
          typename OutputImageType::Pointer outputImage = castFilter->GetOutput();
          outputImage->Update();
          mPerfusionImagePointerDicom.push_back(outputImage);
          if (counter == 0)
          {
            mITKImage = castFilter->GetOutput();
            mITKImage->Update();
          }
          counter = counter + 1;
        }
        catch (itk::ExceptionObject & err)
        {
          std::stringstream error;
          error << err;
        }
        ++seriesItr;
      }
    }
    else
    {
      typedef std::vector< std::string > FileNamesContainer;
      FileNamesContainer fileNames;
      fileNames = nameGenerator->GetFileNames(seriesUID.begin()->c_str());
      mFileNames = nameGenerator->GetOutputFileNames();
      seriesreader->SetFileNames(fileNames);
      seriesreader2->SetFileNames(fileNames);
      try
      {
        seriesreader->Update();
        seriesreader2->Update();
      }
      catch (itk::ExceptionObject &ex)
      {
        std::cout << ex << std::endl;
      }
      mDicomDictionaryArray = seriesreader->GetMetaDataDictionaryArray();
      mFileNames = fileNames;
      mSeriesReader = seriesreader2;
      typedef itk::CastImageFilter< InputImageType, OutputImageType > CastFilterType;
      typename CastFilterType::Pointer castFilter = CastFilterType::New();
      castFilter->SetInput(seriesreader->GetOutput());
      mITKImage = castFilter->GetOutput();
      mITKImage->Update();
    }


    ////---------test index--------------------
    //typedef itk::Image<float, 3> ImageType;
    //ImageType::IndexType index;
    //index[0] = 19;
    //index[1] = 67;
    //index[2] = 0;
    //double existingValue = mITKImage.GetPointer()->GetPixel(index);
    //mITKImage.GetPointer()->SetPixel(index,200);

    ////--------------------------------------

    ////for perfusion data
    //if (mLoadingType == IMAGE_TYPE_PERFUSION)
    //{
    //	typedef itk::Image<float, 3> ImageType;
    //	typedef ImageType::Pointer ImagePointerType;

    //	std::vector<ImagePointerType> mImages;

    //	typedef std::vector< std::string > SeriesIdContainer;
    //	const SeriesIdContainer & seriesUID = nameGenerator->GetSeriesUIDs();

    //	SeriesIdContainer::const_iterator seriesItr = seriesUID.begin();
    //	SeriesIdContainer::const_iterator seriesEnd = seriesUID.end();
    //	while (seriesItr != seriesEnd)
    //	{
    //		mImageType = imagetype;
    //		typedef std::vector< std::string > FileNamesContainer;
    //		FileNamesContainer fileNames;
    //		fileNames = nameGenerator->GetFileNames(seriesItr->c_str());
    //		seriesreader->SetFileNames(fileNames);
    //		try
    //		{
    //			seriesreader->Update();
    //			typedef::itk::Image<float, 3> OutputImageType;
    //			typedef itk::CastImageFilter<InputImageType, OutputImageType> CastFilterType;
    //			typename CastFilterType::Pointer castFilter = CastFilterType::New();
    //			castFilter->SetInput(seriesreader->GetOutput());
    //			typename OutputImageType::Pointer outputImage = castFilter->GetOutput();
    //			outputImage->Update();
    //			mImages.push_back(outputImage);
    //		}
    //		catch (itk::ExceptionObject & err)
    //		{
    //		}
    //		++seriesItr;
    //	}
    //	ImageType::RegionType region = mITKImage->GetLargestPossibleRegion();
    //	int x_size = region.GetSize()[0];
    //	int y_size = region.GetSize()[1];
    //	int z_size = region.GetSize()[2];
    //	
    //	if (mImageSubType == IMAGE_TYPE_T2FLAIR)
    //	{
    //		for (int x = 0; x < x_size; x++)
    //		{
    //			for (int y = 0; y < y_size; y++)
    //			{
    //				for (int z = 0; z < z_size; z++)
    //				{
    //					double sum = 0;
    //					ImageType::IndexType index;
    //					index[0] = x;
    //					index[1] = y;
    //					index[2] = z;
    //					for (int seriesindex = 0; seriesindex < PERF_POINTS_ESTIM; seriesindex++)
    //						sum = sum + mImages[seriesindex].GetPointer()->GetPixel(index);

    //					if (x==19 && y==67 && z==0)
    //						int a = 0;
    //					double finalValue = sum / PERF_POINTS_ESTIM;
    //					mITKImage.GetPointer()->SetPixel(index, finalValue);
    //				}
    //			}
    //		}
    //	}
    ////	else if (mImageSubType == IMAGE_TYPE_T1CE)
    ////	{
    ////		for (int x = 0; x < x_size; x++)
    ////		{
    ////			for (int y = 0; y < y_size; y++)
    ////			{
    ////				for (int z = 0; z < z_size; z++)
    ////				{
    ////					double sum = 0;
    ////					ImageType::IndexType index;
    ////					index[0] = x;
    ////					index[1] = y;
    ////					index[2] = z;
    ////					double min = mImages[22].GetPointer()->GetPixel(index);
    ////					for (int seriesindex = 0; seriesindex < mImages.size(); seriesindex++)
    ////					{
    ////						if (mImages[seriesindex].GetPointer()->GetPixel(index)>0 && mImages[seriesindex].GetPointer()->GetPixel(index)< min)
    ////							min = mImages[seriesindex].GetPointer()->GetPixel(index);
    ////					}
    ////					for (int seriesindex = 0; seriesindex < mImages.size(); seriesindex++)
    ////					{
    ////						if (seriesindex < PERF_POINTS_ESTIM)
    ////							sum = sum + mImages[seriesindex].GetPointer()->GetPixel(index);
    ////					}

    ////					if (min > 0)
    ////						int a = 0;
    ////					
    ////					double finalValue = (sum / PERF_POINTS_ESTIM) - min;
    ////					mITKImage.GetPointer()->SetPixel(index, finalValue);
    ////				}
    ////			}
    ////		}
    ////	}
    ////	mITKImage->Update();

    //}
    ////---------test index--------------------
    //ImageType::IndexType newindex;
    //newindex[0] = 19;
    //newindex[1] = 67;
    //newindex[2] = 0;
    //double newexistingValue = mITKImage.GetPointer()->GetPixel(newindex);


    typename OutputImageType::Pointer registeredimage;
    typename OutputImageType::Pointer scaledimage;
    //if (mImageSubType == IMAGE_TYPE_T2FLAIR)
    //{
    //    std::cout << "Registering dicom image if it is Flair." << std::endl;
    //	typedef itk::Image<float, 3> InternalImageType;
    //	typename itk::MultiResolutionImageRegistrationMethod<OutputImageType, OutputImageType>::Pointer Registrar;
    //	PreprocessingPipelineClass * obj = new PreprocessingPipelineClass();
    //	Registrar = obj->Registration<OutputImageType, InternalImageType>(t1ceImage, mITKImage);
    //	registeredimage = obj->ResampleTransform<OutputImageType>(Registrar, t1ceImage, mITKImage);
    //	mITKImage = registeredimage;
    //	delete obj;
    //}
    //else if (this->mImageType == IMAGE_TYPE_RECURRENCE_OUTPUT)
    //{
    //    std::cout << "Manipulating Dicom image if it is recurrence output." << std::endl;
    //	typename OutputImageType::RegionType region = mITKImage->GetLargestPossibleRegion();
    //	int vd_x = region.GetSize()[0];
    //	int vd_y = region.GetSize()[1];
    //	int vd_z = region.GetSize()[2];

    //	for (int i = 0; i < vd_x; i++)
    //		for (int j = 0; j < vd_y; j++)
    //			for (int k = 0; k < vd_z; k++)
    //			{
    //				typename OutputImageType::IndexType index;
    //				index[0] = i;
    //				index[1] = j;
    //				index[2] = k;
    //				float val = mITKImage->GetPixel(index);
    //				val = val * 0.001;
    //				mITKImage->SetPixel(index, val);
    //			}
    //}
    //-----------------------------------------------------
    //typedef itk::RescaleIntensityImageFilter<OutputImageType, OutputImageType> RescaleFilterType;
    //RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
    //rescaleFilter->SetInput(mITKImage);
    //rescaleFilter->SetOutputMinimum(0);
    //rescaleFilter->SetOutputMaximum(255);
    //rescaleFilter->Update();
    //mITKImage = rescaleFilter->GetOutput();
    //--------------------

    // Convert from ITK object to VTK object
    typedef itk::ImageToVTKImageFilter<OutputImageType> ConverterType;
    typename ConverterType::Pointer converter = ConverterType::New();
    mItkToVtkConverters = dynamic_cast<itk::ProcessObject *>(converter.GetPointer());

    if (this->mImageType == IMAGE_TYPE_T2FLAIR)
      converter->SetInput(registeredimage);
    else if (this->mImageType == IMAGE_TYPE_RECURRENCE_OUTPUT)
      converter->SetInput(mITKImage);
    else
      converter->SetInput(mITKImage);

    converter->Update();
    mImage = converter->GetOutput();

    vtkSmartPointer<vtkMatrix4x4> matrix = vtkSmartPointer<vtkMatrix4x4>::New();
    matrix->Identity();
    mTransform->SetMatrix(matrix);
  }
  else
  {
    try
    {        
      typedef itk::Image<InputPixelType, 3> InputImageType3D;
      typedef itk::Image<OutputPixelType, 3> OutputImageType3D;

      typename OutputImageType3D::Pointer outputImage;
      typename InputImageType3D::Pointer  inputImage;
      if (subimagetype == IMAGE_TYPE_PERFUSION)
      {
        typedef itk::Image<InputPixelType, 4> InputImageType4D;
        typedef itk::Image<OutputPixelType, 4> OutputImageType4D;
        typedef itk::ImageFileReader<InputImageType4D> ReaderType;
        typename ReaderType::Pointer reader = ReaderType::New();
        reader->SetFileName(directoryName);
        reader->ReleaseDataFlagOn();

        typename InputImageType4D::Pointer inputImage = reader->GetOutput();
        inputImage->Update();
        typedef itk::CastImageFilter< InputImageType4D, OutputImageType4D > CastFilterType;
        typename CastFilterType::Pointer castFilter = CastFilterType::New();
        castFilter->SetInput(inputImage);

        typename OutputImageType4D::Pointer outputImage4D = castFilter->GetOutput();
        outputImage4D->Update();

        //assign to original 4D nifti image
        mPerfusionImagePointer = outputImage4D;

        //crop one 3D image from the 4D series
        typename OutputImageType4D::RegionType region = outputImage4D->GetLargestPossibleRegion();
        typename OutputImageType4D::IndexType regionIndex;
        typename OutputImageType4D::SizeType regionSize;
        regionSize[0]  = region.GetSize()[0];
        regionSize[1]  = region.GetSize()[1];
        regionSize[2]  = region.GetSize()[2];
        regionSize[3]  = 0;
        regionIndex[0] = 0;
        regionIndex[1] = 0;
        regionIndex[2] = 0;
        regionIndex[3] = 0;
        typename OutputImageType4D::RegionType desiredRegion(regionIndex, regionSize);


        typedef itk::ExtractImageFilter< OutputImageType4D, OutputImageType3D > FilterType;
        typename FilterType::Pointer filter = FilterType::New();
        filter->SetExtractionRegion(desiredRegion);
        filter->SetInput(outputImage4D);
        #if ITK_VERSION_MAJOR >= 4
        filter->SetDirectionCollapseToIdentity(); // This is required.
        #endif
        filter->Update();
        outputImage = filter->GetOutput();
      }
      else
      {

        typedef itk::ImageFileReader<InputImageType3D> ReaderType;
        typename ReaderType::Pointer reader = ReaderType::New();
        reader->SetFileName(directoryName);
        reader->ReleaseDataFlagOn();

        inputImage = reader->GetOutput();
        inputImage->Update();
        typedef itk::CastImageFilter< InputImageType3D, OutputImageType3D > CastFilterType;
        typename CastFilterType::Pointer castFilter = CastFilterType::New();
        castFilter->SetInput(inputImage);

        outputImage = castFilter->GetOutput();
        outputImage->Update();
      }

      // Convert from ITK object to VTK object
      typedef itk::ImageToVTKImageFilter<OutputImageType3D> ConverterType;
      typename ConverterType::Pointer converter = ConverterType::New();
      mItkToVtkConverters = dynamic_cast< itk::ProcessObject *>(converter.GetPointer());
      converter->SetInput(outputImage);
      converter->Update();

      mImage = converter->GetOutput();
      mITKImage = outputImage;

      vtkSmartPointer<vtkMatrix4x4> matrix = vtkSmartPointer<vtkMatrix4x4>::New();
      matrix->Identity();

      if (subimagetype == IMAGE_TYPE_PERFUSION)
      { 
        for (unsigned int i = 0; i < outputImage->GetImageDimension(); i++)
        {
          for (unsigned int j = 0; j < outputImage->GetImageDimension(); j++)
          {
            (*matrix)[i][j] = outputImage->GetDirection()[i][j];
            (*matrix)[i][3] -= (*matrix)[i][j] * outputImage->GetOrigin()[j];
          }
          (*matrix)[i][3] += outputImage->GetOrigin()[i];
        }
      }
      else
      {
        for (unsigned int i = 0; i < inputImage->GetImageDimension(); i++)
        {
          for (unsigned int j = 0; j < inputImage->GetImageDimension(); j++)
          {
            (*matrix)[i][j] = inputImage->GetDirection()[i][j];
            (*matrix)[i][3] -= (*matrix)[i][j] * inputImage->GetOrigin()[j];
          }
          (*matrix)[i][3] += inputImage->GetOrigin()[i];
        }
      }

      matrix->Invert();

      mTransform->SetMatrix(matrix);

#if (ITK_VERSION_MAJOR < 4) || defined(ITKV3_COMPATIBILITY)
      itk::AnalyzeImageIO *analyzeImageIO = dynamic_cast<itk::AnalyzeImageIO*>(reader->GetImageIO());

      if (analyzeImageIO) {
        const double m[16] = {
          1.0, 0.0, 0.0, 0.0,
          0.0, 0.0, 1.0, 0.0,
          0.0, -1.0, 0.0, 0.0,
          0.0, 0.0, 0.0, 1.0
        };
        int i;
        for (i = 0; i < 16 && m[i] == mTransform->GetMatrix()->GetElement(i % 4, i / 4); i++);
        if (i == 16) {
          std::cout << "Analyze image file format detected with unknown orientation. Forcing identity orientation, use other file format if not ok." << std::endl;
          mTransform->Identity();
        }
      }
#endif
    }
    catch (itk::ExceptionObject & err)
    {
      std::cerr << "Error while reading " << mFileName.c_str() << " " << err << std::endl;
      std::stringstream error;
      error << err;
      mLastError = error.str();
      return;
    }
  }

}


#endif
