///////////////////////////////////////////////////////////////////////////////////////
// fTumorPanel.cxx
//
// Copyright (c) 2016. All rights reserved.
// Section of Biomedical Image Analysis
// Center for Biomedical Image Computing and Analytics
// Department of Radiology
// Perelman School of Medicine
// University of Pennsylvania
//
// Contact details: software@cbica.upenn.edu
//
// License Agreement: https://www.cbica.upenn.edu/sbia/software/license.html
///////////////////////////////////////////////////////////////////////////////////////

#include "fTumorPanel.h"
#include "CAPTk.h"
#include "fMainWindow.h"
#include "SlicerManager.h"
#include "Landmarks.h"

class tEventFilter : public QObject
{
public:
  fTumorPanel * mTObj;
  tEventFilter(fTumorPanel * obj) :QObject()
  {
    mTObj = obj;
  };
  ~tEventFilter(){};

  bool eventFilter(QObject* object, QEvent* event)
  {
    if (event->type() == QEvent::KeyPress)
    {
      QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

      switch (keyEvent->key())
      {
      case Qt::Key_Return:
      {
        mTObj->HandleKeyPressingEventTTable();
        return true;
      }
      case Qt::Key_Down:
      {
        mTObj->HandleDownKeyEventTTable();
        return true;
      }
      case Qt::Key_Up:
      {
        mTObj->HandleUpKeyEventTTable();
        return true;
      }
      case Qt::Key_Delete:
      {
        mTObj->HandleDeleteKeyEventTTable();
        return true;
      }
      default:
      {
        cbica::Logging(loggerFile, "Undefined Key press.");
        //exit(EXIT_FAILURE); // probably not required
        return false;
      }
      }
    }
    else
    {
      return QObject::eventFilter(object, event);
    }
  }
};
class sEventFilter : public QObject
{
public:
  fTumorPanel * mTObj;
  sEventFilter(fTumorPanel * obj) :QObject()
  {
    mTObj = obj;
  };
  ~sEventFilter(){};

  bool eventFilter(QObject* object, QEvent* event)
  {
    if (event->type() == QEvent::KeyPress)
    {
      QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

      switch (keyEvent->key())
      {
      case Qt::Key_Return:
      {
        mTObj->HandleKeyPressingEventSTable();
        return true;
      }
      case Qt::Key_Down:
      {
        mTObj->HandleDownKeyEventSTable();
        return true;
      }
      case Qt::Key_Up:
      {
        mTObj->HandleUpKeyEventTTable();
        return true;
      }
      case Qt::Key_Delete:
      {
        mTObj->HandleDeleteKeyEventTTable();
        return true;
      }
      default:
      {
        cbica::Logging(loggerFile, "Undefined Key press.");
        //exit(EXIT_FAILURE); // probably not required
        return false;
      }
      }
    }
    else
    {
      return QObject::eventFilter(object, event);
    }
  }
};
//static fMainWindow* getMainWindow()
//{
//	QWidgetList widgets = qApp->topLevelWidgets();
//	for (QWidgetList::iterator i = widgets.begin(); i != widgets.end(); ++i)
//	{
//		if ((*i)->objectName() == "MainWindow")
//		{
//			return qobject_cast<fMainWindow*>(*i);
//		}
//	}
//	return qobject_cast<fMainWindow*>(qApp->activeWindow());
//}
fTumorPanel::fTumorPanel(QWidget * parent) : QWidget(parent)
{
  initializationFileName = "initiazedPoints_all.txt";
  saveType_GLISTR = false;
  saveType_PORTRPre = false;
  saveType_PORTRPost = false;
  setupUi(this);
  mTissueType = 0;
  mCurrentSPoints = NULL;
  mCurrentTPoints = NULL;
  sTableWidget->setSelectionBehavior(QAbstractItemView::SelectRows);
  sTableWidget->setSelectionMode(QAbstractItemView::SingleSelection);

  sLoadButton->setEnabled(false);
  sSaveButton->setEnabled(false);
  sRemoveButton->setEnabled(false);
  sRemoveAllButton->setEnabled(false);
  tLoadButton->setEnabled(false);
  tSaveButton->setEnabled(false);
  tRemoveButton->setEnabled(false);
  tRemoveAllButton->setEnabled(false);

  connect(sLoadButton, SIGNAL(clicked()), this, SLOT(sLoad()));
  connect(sSaveButton, SIGNAL(clicked()), this, SLOT(sSave()));
  connect(sRemoveButton, SIGNAL(clicked()), this, SLOT(sRemoveSelectedPoints()));
  connect(sRemoveAllButton, SIGNAL(clicked()), this, SLOT(sRemoveAllPoints()));
  connect(tLoadButton, SIGNAL(clicked()), this, SLOT(tLoad()));
  connect(tSaveButton, SIGNAL(clicked()), this, SLOT(tSave()));
  connect(tRemoveButton, SIGNAL(clicked()), this, SLOT(tRemoveSelectedPoints()));
  connect(tRemoveAllButton, SIGNAL(clicked()), this, SLOT(tRemoveAllPoints()));
  //connect(sTableWidget, SIGNAL(cellClicked(int, int)), this, SLOT(sTableClicked(int, int)));
  //connect(tTableWidget, SIGNAL(cellClicked(int, int)), this, SLOT(tTableClicked(int, int)));
  connect(sTableWidget, SIGNAL(cellDoubleClicked(int, int)), this, SLOT(sTableDoubleClicked(int, int)));
  connect(tTableWidget, SIGNAL(cellDoubleClicked(int, int)), this, SLOT(tTableDoubleClicked(int, int)));

  connect(RadioButton_CSF, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_CSF()));
  connect(RadioButton_GM, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_GM()));
  connect(RadioButton_WM, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_WM()));
  connect(RadioButton_VS, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_VS()));
  connect(RadioButton_ED, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_ED()));
  connect(RadioButton_NCR, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_NCR()));
  connect(RadioButton_TU, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_TU()));
  connect(RadioButton_NE, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_NE()));
  connect(RadioButton_CB, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_CB()));
  connect(RadioButton_VT, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_VT()));
  connect(RadioButton_CAN, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_CAN()));
  connect(RadioButton_CAE, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_CAE()));
  connect(RadioButton_RTN, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_RTN()));
  connect(RadioButton_RTE, SIGNAL(toggled(bool)), this, SLOT(SetTissueType_RTE()));

  connect(m_typeRadBtnTumor, SIGNAL(toggled(bool)), this, SLOT(SetSeedType()));
  connect(m_typeRadBtnAllTissues, SIGNAL(toggled(bool)), this, SLOT(SetTissueType()));
  connect(m_typeRadBtnGlister, SIGNAL(toggled(bool)), this, SLOT(SetGLISTRTissueType()));
  connect(m_typeRadBtnPorterPre, SIGNAL(toggled(bool)), this, SLOT(SetPORTRPRETissueType()));
  connect(m_typeRadBtnPorterPost, SIGNAL(toggled(bool)), this, SLOT(SetPORTRPOSTTissueType()));




  tTableWidget->installEventFilter(new tEventFilter(this));
  sTableWidget->installEventFilter(new sEventFilter(this));



  sTableWidget->resizeColumnsToContents();
  sTableWidget->resizeRowsToContents();
  tTableWidget->resizeColumnsToContents();
  tTableWidget->resizeRowsToContents();

  this->SetTissueType_CSF();
  emit SetTissueCounter(mType);
  RadioButton_CSF->setChecked(true);
  m_typeRadBtnTumor->setChecked(false);
  m_typeRadBtnAllTissues->setChecked(true);


}

void fTumorPanel::Clear()
{
  sTableWidget->clearContents();
  tTableWidget->clearContents();
  sLoadButton->setEnabled(false);
  sSaveButton->setEnabled(false);
  sRemoveButton->setEnabled(false);
  sRemoveAllButton->setEnabled(false);
  tLoadButton->setEnabled(false);
  tSaveButton->setEnabled(false);
  tRemoveButton->setEnabled(false);
  tRemoveAllButton->setEnabled(false);
}

void fTumorPanel::sLoad()
{
  QString file;
  QFileDialog dialog(this, tr("Load Seed Points"), mCurrentPath.c_str(), tr("Seed Points (*.txt)"));
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptOpen);

  dialog.selectFile("init_seed.txt");

  int ret = dialog.exec();
  if (ret == QDialog::Accepted)
  {
    file = dialog.selectedFiles()[0];
    double scan_seeds_info[MaxNumberOFTumorSeeds][4];
    int scan_seeds_num;

    QFile inputFile(file);
    if (inputFile.open(QIODevice::ReadOnly))
    {
      QTextStream in(&inputFile);
      scan_seeds_num = (int)atoi(in.readLine().toStdString().c_str());
      if (scan_seeds_num > MaxNumberOFTumorSeeds)
      {
        return;
      }
      mCurrentSPoints->Clear();
      for (int i = 0; i < scan_seeds_num; i++)
      {
        std::vector<std::string > values = cbica::stringSplit(in.readLine().toStdString(), " ");
        scan_seeds_info[i][0] = static_cast<double>(std::atof(values[0].c_str()));
        scan_seeds_info[i][1] = static_cast<double>(std::atof(values[1].c_str()));
        scan_seeds_info[i][2] = static_cast<double>(std::atof(values[2].c_str()));
        scan_seeds_info[i][3] = static_cast<double>(std::atof(values[3].c_str()));

#ifdef USE_LANDMARKS_LPS_COORD
        mCurrentSPoints->AddLandmark(scan_seeds_info[i][0], scan_seeds_info[i][1], scan_seeds_info[i][2], scan_seeds_info[i][3] / 2.0, 0, i);
#endif

#ifdef USE_LANDMARKS_RAS_COORD
        mCurrentSPoints->AddLandmark(-scan_seeds_info[i][0], -scan_seeds_info[i][1], scan_seeds_info[i][2], scan_seeds_info[i][3] / 2.0, 0, i);
#endif
      }
    }
  }
  SetCurrentSPoints(mCurrentSPoints);
  emit UpdateRenderWindows();
}
void fTumorPanel::sSave()
{
  QString file;
  QFileDialog dialog(this, tr("Save Seed Points"), mCurrentPath.c_str(), tr("Seed Points (*.txt)"));
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  dialog.selectFile("init_seed.txt");

  int ret = dialog.exec();
  if (ret == QDialog::Accepted)
  {
    file = dialog.selectedFiles()[0];
    //*/
    FILE* fp;

    //QFile fileToWrite(file);
    //if (!fileToWrite.open(QIODevice::WriteOnly | QIODevice::Text))
    //{
    //  printf("Cannot open %s\n", file.toStdString().c_str());
    //  return;
    //}
    //QTextStream out(&fileToWrite);
    //out << mCurrentSPoints->GetNumberOfPoints() << "\n";

#ifdef _WIN32
    fopen_s(&fp, file.toStdString().c_str(), "w");
#else
    fp = fopen(file.toStdString().c_str(), "w");
#endif

    if (fp == NULL) {
      printf("Cannot open %s\n", file.toStdString().c_str());
      return;
    }
    fprintf(fp, "%d\n", mCurrentSPoints->GetNumberOfPoints());
    for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
    {
#ifdef USE_LANDMARKS_LPS_COORD
      fprintf(fp, "%f %f %f %f\n", mCurrentSPoints->mLandmarks[i].coordinates[0], mCurrentSPoints->mLandmarks[i].coordinates[1], mCurrentSPoints->mLandmarks[i].coordinates[2], mCurrentSPoints->mLandmarks[i].radius * 2.0);
      //out << mCurrentSPoints->mLandmarks[i].coordinates[0] << " " << mCurrentSPoints->mLandmarks[i].coordinates[1] << " " << mCurrentSPoints->mLandmarks[i].coordinates[2] << " " << mCurrentSPoints->mLandmarks[i].radius * 2.0 << "\n";
#endif
#ifdef USE_LANDMARKS_RAS_COORD
      fprintf(fp, "%f %f %f %f\n", -mCurrentSPoints->mLandmarks[i].coordinates[0], -mCurrentSPoints->mLandmarks[i].coordinates[1], mCurrentSPoints->mLandmarks[i].coordinates[2], mCurrentSPoints->mLandmarks[i].radius * 2.0);
      //out << -mCurrentSPoints->mLandmarks[i].coordinates[0] << " " << -mCurrentSPoints->mLandmarks[i].coordinates[1] << " " << mCurrentSPoints->mLandmarks[i].coordinates[2] << " " << mCurrentSPoints->mLandmarks[i].radius * 2.0 << "\n";
#endif
    }
    //fileToWrite.close();
    fclose(fp);
  }
}
//
void fTumorPanel::tLoad()
{
  QString file;
  QFileDialog dialog(this, tr("Load Tissue Points"), mCurrentPath.c_str(), tr("Tissue Points (*.txt)"));
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptOpen);
  dialog.selectFile("init_point.txt");

  int ret = dialog.exec();
  if (ret == QDialog::Accepted)
  {
    file = dialog.selectedFiles()[0];
    unsigned int i;
    int j;
    QFile inputFile(file);
    mCurrentTPoints->Clear();


    if (inputFile.open(QIODevice::ReadOnly))
    {
      QTextStream in(&inputFile);
      while (!in.atEnd())
      {
        double fx, fy, fz;
        std::string id = in.readLine().toStdString();
        std::vector< std::string > values = cbica::stringSplit(in.readLine().toStdString(), " ");
      
        fx = static_cast<double>(std::atof(values[0].c_str()));
        fy = static_cast<double>(std::atof(values[1].c_str()));
        fz = static_cast<double>(std::atof(values[2].c_str()));

        for (j = 0; j < NumberOfPriorChannels; j++)
        {
          if (strcmp(id.c_str(), labels[j]) == 0)
          {
            if (j == 0)
            {
              // skip BG
            }
            else
            {
#ifdef USE_LANDMARKS_LPS_COORD
              mCurrentTPoints->AddLandmark(fx, fy, fz, 0, 0, j);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
              mCurrentTPoints->AddLandmark(-fx, -fy, fz, 0, 0, j);
#endif
            }
            break;
          }
        }
        if (j == NumberOfPriorChannels)
        {
          printf("class id is wrong\n");
          mCurrentTPoints->Clear();
          return;
        }
      }
      inputFile.close();
    }
    for (i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
    {
      if (mCurrentTPoints->mLandmarks[i].bValid)
      {
        printf("point %d: %s %f %f %f\n", i, labels[mCurrentTPoints->mLandmarks[i].id], mCurrentTPoints->mLandmarks[i].coordinates[0], mCurrentTPoints->mLandmarks[i].coordinates[1], mCurrentTPoints->mLandmarks[i].coordinates[2]);
      }
    }
  }
  SetCurrentTPoints(mCurrentTPoints);
  emit UpdateRenderWindows();
}
void fTumorPanel::tSave()
{
  QString file;
  QFileDialog dialog(this, tr("Save Tissue Points"), mCurrentPath.c_str(), tr("Tissue Points (*.txt)"));
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  dialog.selectFile(initializationFileName.c_str());

  // check for saveType_* for three and compare different tissue types
  std::vector< unsigned int > tissuesToCheck;
  /// these "required" tissue types have been decided after discussion with Spyros
  if (saveType_GLISTR)
  {
    tissuesToCheck.push_back(CSF);
    tissuesToCheck.push_back(GM);
    tissuesToCheck.push_back(WM);
    tissuesToCheck.push_back(VS);
    //tissuesToCheck.push_back(ED);
    //tissuesToCheck.push_back(NCR);
    tissuesToCheck.push_back(TU);
    tissuesToCheck.push_back(NE); // one of either this or above should be present
    tissuesToCheck.push_back(CB);
  }
  else if (saveType_PORTRPre)
  {
    tissuesToCheck.push_back(CSF);
    tissuesToCheck.push_back(VT); // one of either this or above should be present
    tissuesToCheck.push_back(GM);
    tissuesToCheck.push_back(WM);
    tissuesToCheck.push_back(VS);
    //tissuesToCheck.push_back(ED);
    tissuesToCheck.push_back(NCR);
    tissuesToCheck.push_back(TU); // one of either this or above should be present
  }
  else if (saveType_PORTRPost)
  {
    tissuesToCheck.push_back(CSF);
    tissuesToCheck.push_back(VT); // one of either this or above should be present
    tissuesToCheck.push_back(GM);
    tissuesToCheck.push_back(WM);
    tissuesToCheck.push_back(VS);
    //tissuesToCheck.push_back(ED);
    tissuesToCheck.push_back(CAN);
    tissuesToCheck.push_back(CAE); // one of either this or above should be present
    //tissuesToCheck.push_back(RTN);
    //tissuesToCheck.push_back(RTE);
  }
  //else
  //{
  //  // do nothing
  //}

  if (!tissuesToCheck.empty())
  {
    for (size_t i = 0; i < mCurrentTPoints->mLandmarks.size(); i++)
    {
      if (!tissuesToCheck.empty() && mCurrentTPoints->mLandmarks[i].bValid)
      {
        std::vector< unsigned int >::iterator iterator = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), mCurrentTPoints->mLandmarks[i].id);
        if (iterator != tissuesToCheck.end())
        {
          tissuesToCheck.erase(iterator);
        }
      }
      else
      {
        break;
      }
    }
  }

  bool glistrOptionalsFound = false, portrPreOptionalsFound = false, portrPostOptionalsFound = false;

  if (tissuesToCheck.size() > 0)
  {
    // check for optional tissue types for the specific applications
    std::string tissues_later = "";
    std::vector< unsigned int >::iterator it_1, it_2;
    if (saveType_GLISTR)
    {
      it_1 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), TU);
      it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), NE);
      if ((it_1 != tissuesToCheck.end()) && (it_2 != tissuesToCheck.end()))
      {
        tissues_later += "  Either TU (Enhacing Tumor) or NE (Non-enhacing Tumor) is required.";
        // ensure the checked tissue is removed
        if (!mCurrentTPoints->mLandmarks.empty())
        {
          if (it_1 != tissuesToCheck.end())
          {
            tissuesToCheck.erase(it_1);
            it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), NE);
          }
          
          if (it_2 != tissuesToCheck.end())
            tissuesToCheck.erase(it_2);
        }
      }
      else
      {
        if (!mCurrentTPoints->mLandmarks.empty())
          glistrOptionalsFound = true;
      }
    }
    else if (saveType_PORTRPre || saveType_PORTRPost)
    {
      it_1 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), CSF);
      it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), VT);
      if ((it_1 != tissuesToCheck.end()) && (it_2 != tissuesToCheck.end()))
      {
        tissues_later += "  Either CSF (Cerebrospinal Fluid) or VT (Ventricular CSF) is required.\n";
        // ensure the checked tissue is removed
        if (!mCurrentTPoints->mLandmarks.empty())
        {
          if (it_1 != tissuesToCheck.end())
          {
            tissuesToCheck.erase(it_1);
            it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), VT);
          }

          if (it_2 != tissuesToCheck.end())
            tissuesToCheck.erase(it_2);
        }
      }
      else
      {
        if (!mCurrentTPoints->mLandmarks.empty())
          portrPostOptionalsFound = true;
      }

      it_1 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), CAN);
      it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), CAE);
      if ((it_1 != tissuesToCheck.end()) && (it_2 != tissuesToCheck.end()))
      {
        tissues_later += "  Either CAN (Cavity Non Enhancing) or CAE (Cavity Enhancing) is required.\n";
        // ensure the checked tissue is removed
        if (!mCurrentTPoints->mLandmarks.empty())
        {
          if (it_1 != tissuesToCheck.end())
          {
            tissuesToCheck.erase(it_1);
            it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), CAE);
          }

          if (it_2 != tissuesToCheck.end())
            tissuesToCheck.erase(it_2);
        }
      }
      else
      {
        if (!mCurrentTPoints->mLandmarks.empty())
          portrPostOptionalsFound = true;
      }

      if (saveType_PORTRPre)
      {
        it_1 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), NCR);
        it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), TU);
        if ((it_1 != tissuesToCheck.end()) && (it_2 != tissuesToCheck.end()))
        {
          tissues_later += "  Either NCR (Necrosis) or TU (Enhacing Tumor) is required.";
          // ensure the checked tissue is removed
          if (!mCurrentTPoints->mLandmarks.empty())
          {
            if (it_1 != tissuesToCheck.end())
            {
              tissuesToCheck.erase(it_1);
              it_2 = std::find(tissuesToCheck.begin(), tissuesToCheck.end(), TU);
            }

            if (it_2 != tissuesToCheck.end())
              tissuesToCheck.erase(it_2);
          }
        }
        else
        {
          if (!mCurrentTPoints->mLandmarks.empty())
            portrPreOptionalsFound = true;
        }
      }

      if (saveType_PORTRPost)
      {
        portrPreOptionalsFound = true;
      }
    }

    if (!tissuesToCheck.empty() || !tissues_later.empty())
    {
      std::string tissues = "";
      for (size_t i = 0; i < tissuesToCheck.size(); i++)
      {
        switch (tissuesToCheck[i])
        {
        case CSF:
          if (!portrPostOptionalsFound || !portrPreOptionalsFound)
            tissues += "  CSF (Cerebrospinal Fluid)\n";
          break;
        case GM:
          tissues += "  GM  (Gray Matter)\n";
          break;
        case WM:
          tissues += "  WM  (White Matter)\n";
          break;
        case VS:
          tissues += "  VS  (Vessels)\n";
          break;
        case ED:
          tissues += "  ED  (Edema)\n";
          break;
        case NCR:
          if (!portrPreOptionalsFound)
            tissues += "  NCR (Necrosis)\n";
          break;
        case TU:
          if (!glistrOptionalsFound && !portrPreOptionalsFound)
            tissues += "  TU  (Enhancing Tumor)\n";
          break;
        case NE:
          if (!glistrOptionalsFound)
            tissues += "  NE  (Non-enhancing Tumor)\n";
          break;
        case CB:
          tissues += "  CB  (Cerebellum)\n";
          break;
        case VT:
          if (!portrPostOptionalsFound && !portrPreOptionalsFound)
            tissues += "  VT  (Ventricular CSF)\n";
          break;
        case CAN:
          if (!portrPostOptionalsFound)
            tissues += "  CAN (Non-enhancing Cavity)\n";
          break;
        case CAE:
          if (!portrPostOptionalsFound)
            tissues += "  CAE (Enhacing Cavity)\n";
          break;
        case RTN:
          tissues += "  RTN (Tumor Recurrence)\n";
          break;
        case RTE:
          tissues += "  RTE (Enhanced Tumor Recurrence)\n";
          break;
        default:
          break;
        }
      }
      
      if (!(tissues.empty() && tissues_later.empty()))
      {
        // this needs to show the text in a better aligned way
        QMessageBox msgBox;
        msgBox.setText("The following required tissue type(s) are either not initialized or aren't valid:");
        if (!tissues.empty())
        {
          tissues = tissues.substr(0, tissues.size() - 1); // remove trailing '\n'
          msgBox.setInformativeText((tissues + "\n\n" + tissues_later).c_str());
        }
        else
        {
          msgBox.setInformativeText(tissues_later.c_str());
        }
        msgBox.setStandardButtons(QMessageBox::Ok);
        msgBox.setDefaultButton(QMessageBox::Ok);
        msgBox.exec();

        return;
      }
    }
  }

  int ret = dialog.exec();
  if (ret == QDialog::Accepted)
  {
    file = dialog.selectedFiles()[0];
    //*/
    FILE* fp;
#ifdef _WIN32
    fopen_s(&fp, file.toStdString().c_str(), "w");
#else
    fp = fopen(file.toStdString().c_str(), "w");
#endif

    if (fp == NULL)
    {
      printf("Cannot open %s\n", file.toStdString().c_str());
      return;
    }
    for (int i = 0; i < NumberOfPriorChannels; i++)
    {
      for (unsigned int j = 0; j < mCurrentTPoints->GetNumberOfPoints(); j++)
      {
        if (mCurrentTPoints->mLandmarks[j].id == i && mCurrentTPoints->mLandmarks[j].bValid)
        {
#ifdef USE_LANDMARKS_LPS_COORD
          fprintf(fp, "%s\n%f %f %f\n", labels[i], mCurrentTPoints->mLandmarks[j].coordinates[0], mCurrentTPoints->mLandmarks[j].coordinates[1], mCurrentTPoints->mLandmarks[j].coordinates[2]);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
          fprintf(fp, "%s\n%f %f %f\n", labels[i], -mCurrentTPoints->mLandmarks[j].coordinates[0], -mCurrentTPoints->mLandmarks[j].coordinates[1], mCurrentTPoints->mLandmarks[j].coordinates[2]);
#endif
        }
      }
    }
    fclose(fp);
  }
}

void fTumorPanel::sRemoveSelectedPoints()
{
  QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
  if (items.empty())
    return;

  int rowIndex = items[0]->row();
  std::vector<int> rowIndices;
  rowIndices.push_back(rowIndex);

  for (int i = 0; i < items.size(); i++)
  {
    if ((std::find(rowIndices.begin(), rowIndices.end(), items[i]->row()) != rowIndices.end()) == false)
      rowIndices.push_back(items[i]->row());
  }
  for (unsigned int j = 0; j < rowIndices.size(); j++)
    sRemoveCurrentIndexPoint(rowIndex);
}

void fTumorPanel::tRemoveCurrentIndexPoint(unsigned int rowIndex)
{
  mCurrentTPoints->RemoveLandmark(rowIndex);
  tTableWidget->removeRow(rowIndex);
  emit UpdateRenderWindows();

  if (rowIndex > mCurrentTPoints->GetNumberOfPoints())
    return;

  tTableWidget->selectRow(rowIndex);
  QTableWidgetItem * item2 = tTableWidget->item(rowIndex, 0);
  tTableWidget->setItemSelected(item2, true);
  UpdateCurrentPoints();
}

void fTumorPanel::sRemoveCurrentIndexPoint(unsigned int rowIndex)
{
  mCurrentSPoints->RemoveLandmark(rowIndex);
  sTableWidget->removeRow(rowIndex);

  emit UpdateRenderWindows();

  if (rowIndex>mCurrentSPoints->GetNumberOfPoints())
    return;

  sTableWidget->selectRow(rowIndex);
  QTableWidgetItem * item2 = sTableWidget->item(rowIndex, 0);
  sTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::tRemoveSelectedPoints()
{
  QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
  if (items.empty())
    return;

  int rowIndex = items[0]->row();
  std::vector<int> rowIndices;
  rowIndices.push_back(rowIndex);

  for (int i = 0; i < items.size(); i++)
  {
    if ((std::find(rowIndices.begin(), rowIndices.end(), items[i]->row()) != rowIndices.end()) == false)
      rowIndices.push_back(items[i]->row());
  }
  for (unsigned int j = 0; j < rowIndices.size(); j++)
    tRemoveCurrentIndexPoint(rowIndex);

  UpdateCurrentPoints();
}

void fTumorPanel::sAddPoint()
{
  sAddPoint(mCurrentSPoints->GetNumberOfPoints() - 1, false);
}

void fTumorPanel::sAddPoint(int landmarksIndex, bool update)
{
  int rowIndex = landmarksIndex;// mCurrentSPoints->GetNumberOfPoints() - 1; //sTableWidget->rowCount() //landmarksIndex;
  float x, y, z;
  // double val;
  bool bValid;
  float r;

  //sTableWidget->setRowCount(rowIndex+1);
  //sTableWidget->setRowHeight(rowIndex, 20);

  if (update == false)
    sTableWidget->setRowCount(rowIndex + 1);


#ifdef USE_LANDMARKS_LPS_COORD
  x = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[0];
  y = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[1];
  z = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
  x = -mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[0];
  y = -mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[1];
  z = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[2];
#endif
  // val = mCurrentSPoints->mLandmarks[landmarksIndex].pixel_value;
  bValid = mCurrentSPoints->mLandmarks[landmarksIndex].bValid;
  r = mCurrentSPoints->mLandmarks[landmarksIndex].radius;

  if (bValid) {
    QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
    //xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    sTableWidget->setItem(rowIndex, 0, xItem);

    QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
    //yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    sTableWidget->setItem(rowIndex, 1, yItem);

    QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
    //zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    sTableWidget->setItem(rowIndex, 2, zItem);

    QTableWidgetItem* rItem = new QTableWidgetItem(QString::number(r, 'f', 3));
    //rItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    rItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    sTableWidget->setItem(rowIndex, 3, rItem);
  }
}

void fTumorPanel::tAddPoint(int rowIndex)
{
  float x, y, z;
  // double val;
  bool bValid;

  int idx = mCurrentTPoints->GetNumberOfPoints() - 1;
  tTableWidget->setRowCount(mCurrentTPoints->GetNumberOfPoints());

#ifdef USE_LANDMARKS_LPS_COORD
  x = mCurrentTPoints->mLandmarks[idx].coordinates[0];
  y = mCurrentTPoints->mLandmarks[idx].coordinates[1];
  z = mCurrentTPoints->mLandmarks[idx].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
  x = -mCurrentTPoints->mLandmarks[idx].coordinates[0];
  y = -mCurrentTPoints->mLandmarks[idx].coordinates[1];
  z = mCurrentTPoints->mLandmarks[idx].coordinates[2];
#endif
  // val = mCurrentTPoints->mLandmarks[idx].pixel_value;
  bValid = mCurrentTPoints->mLandmarks[idx].bValid;

  std::string tissue_label = labels[mCurrentTPoints->mLandmarks[idx].id];

  if (bValid)
  {

    QTableWidgetItem* labelItem = new QTableWidgetItem(QString::fromStdString(tissue_label));
    //xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    labelItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    tTableWidget->setItem(idx, 0, labelItem);

    QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
    //xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    tTableWidget->setItem(idx, 1, xItem);

    QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
    //yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    tTableWidget->setItem(idx, 2, yItem);

    QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
    //zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
    tTableWidget->setItem(idx, 3, zItem);

    emit UpdateRenderWindows();
    tTableWidget->scrollToItem(labelItem, QAbstractItemView::PositionAtTop);
  }
  UpdateCurrentPoints();
}

void fTumorPanel::SetCurrentSPoints(Landmarks* lm)
{
  sLoadButton->setEnabled(true);
  sSaveButton->setEnabled(true);
  sRemoveButton->setEnabled(true);
  sRemoveAllButton->setEnabled(true);

  mCurrentSPoints = lm;
  sTableWidget->clearContents();
  for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
  {
    sAddPoint(i, false);
  }
  sTableWidget->resizeColumnsToContents();
}
void fTumorPanel::SetCurrentTPoints(Landmarks* lm)
{
  tLoadButton->setEnabled(true);
  tSaveButton->setEnabled(true);
  tRemoveButton->setEnabled(true);
  tRemoveAllButton->setEnabled(true);
  mCurrentTPoints = lm;

  tTableWidget->clearContents();


  float x, y, z;
  // double val;
  bool bValid;
  tTableWidget->setRowCount(mCurrentTPoints->GetNumberOfPoints());
  for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
  {
#ifdef USE_LANDMARKS_LPS_COORD
    x = mCurrentTPoints->mLandmarks[i].coordinates[0];
    y = mCurrentTPoints->mLandmarks[i].coordinates[1];
    z = mCurrentTPoints->mLandmarks[i].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
    x = -mCurrentTPoints->mLandmarks[i].coordinates[0];
    y = -mCurrentTPoints->mLandmarks[i].coordinates[1];
    z = mCurrentTPoints->mLandmarks[i].coordinates[2];
#endif
    // val = mCurrentTPoints->mLandmarks[i].pixel_value;
    bValid = mCurrentTPoints->mLandmarks[i].bValid;

    std::string tissue_label = labels[mCurrentTPoints->mLandmarks[i].id];


    if (bValid)
    {

      QTableWidgetItem* labelItem = new QTableWidgetItem(QString::fromStdString(tissue_label));
      //xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      labelItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
      tTableWidget->setItem(i, 0, labelItem);

      QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
      //xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
      tTableWidget->setItem(i, 1, xItem);

      QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
      //yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
      tTableWidget->setItem(i, 2, yItem);

      QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
      //zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
      zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
      tTableWidget->setItem(i, 3, zItem);
    }
  }
  emit UpdateRenderWindows();
  //	
  //	for (unsigned int i = 0; i < tTableWidget->rowCount(); i++) 
  //{
  //	tAddPoint(i);
  //}
  tTableWidget->resizeColumnsToContents();
  UpdateCurrentPoints();
}

void fTumorPanel::sTableDoubleClicked(int row, int col)
{
  if (mCurrentSPoints == NULL)
    return;
  if (mCurrentSPoints->GetNumberOfPoints() == 0)
    return;

  emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, row, col);

  if (row >= 0 && row < (int)mCurrentSPoints->GetNumberOfPoints())
  {
    if (mCurrentSPoints->mLandmarks[row].bValid)
    {
      double x, y, z;
      x = mCurrentSPoints->mLandmarks[row].coordinates[0];
      y = mCurrentSPoints->mLandmarks[row].coordinates[1];
      z = mCurrentSPoints->mLandmarks[row].coordinates[2];
      emit MoveSlicerCursor(x, y, z);
    }
  }
}
void fTumorPanel::tTableDoubleClicked(int row, int col)
{
  if (mCurrentTPoints == NULL)
    return;
  if (mCurrentTPoints->GetNumberOfPoints() == 0)
    return;

  mTissueType = mCurrentTPoints->mLandmarks[row].id - 1;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, col);
  emit SetTissueTableIndex(row);

  switch (mTissueType)
  {
  case 0:
    RadioButton_CSF->setChecked(true);
    break;
  case 1:
    RadioButton_GM->setChecked(true);
    break;
  case 2:
    RadioButton_WM->setChecked(true);
    break;
  case 3:
    RadioButton_VS->setChecked(true);
    break;
  case 4:
    RadioButton_ED->setChecked(true);
    break;
  case 5:
    RadioButton_NCR->setChecked(true);
    break;
  case 6:
    RadioButton_TU->setChecked(true);
    break;
  case 7:
    RadioButton_NE->setChecked(true);
    break;
  case 8:
    RadioButton_CB->setChecked(true);
    break;
  case 9:
    RadioButton_VT->setChecked(true);
    break;
  case 10:
    RadioButton_CAN->setChecked(true);
    break;
  case 11:
    RadioButton_CAE->setChecked(true);
    break;
  case 12:
    RadioButton_RTN->setChecked(true);
    break;
  case 13:
    RadioButton_RTE->setChecked(true);
    break;
  default:
    break;
  }

  double x, y, z;
  x = mCurrentTPoints->mLandmarks[row].coordinates[0];
  y = mCurrentTPoints->mLandmarks[row].coordinates[1];
  z = mCurrentTPoints->mLandmarks[row].coordinates[2];
  emit MoveSlicerCursor(x, y, z);
}

void fTumorPanel::sTableFocused(bool bFocused)
{

}

void fTumorPanel::tTableFocused(bool bFocused)
{

}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void fTumorPanel::SetTissueType_CSF()
{
  mTissueType = 0; // can be made to 'CSF - 1' and "-1" is needed because background is '0'
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_GM()
{
  mTissueType = 1;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_WM()
{
  mTissueType = 2;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_VS()
{
  mTissueType = 3;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_ED()
{
  mTissueType = 4;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_NCR()
{
  mTissueType = 5;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_TU()
{
  mTissueType = 6;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_NE()
{
  mTissueType = 7;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_CB()
{
  mTissueType = 8;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_VT()
{
  mTissueType = 9;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_CAN()
{
  mTissueType = 10;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_CAE()
{
  mTissueType = 11;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_RTN()
{
  mTissueType = 12;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::SetTissueType_RTE()
{
  mTissueType = 13;
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}


void fTumorPanel::SetCurrentSelectedTissueType()
{
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::tRemoveAllPoints()
{
  int rowCount = mCurrentTPoints->GetNumberOfPoints();
  for (int rowIndex = rowCount - 1; rowIndex >= 0; rowIndex--)
  {
    mCurrentTPoints->RemoveLandmark(rowIndex);
    tTableWidget->removeRow(rowIndex);
  }
  emit UpdateRenderWindows();
  UpdateCurrentPoints();
}
void fTumorPanel::sRemoveAllPoints()
{
  int rowCount = mCurrentSPoints->GetNumberOfPoints();
  for (int rowIndex = rowCount - 1; rowIndex >= 0; rowIndex--)
  {
    mCurrentSPoints->RemoveLandmark(rowIndex);
    sTableWidget->removeRow(rowIndex);
  }
  emit UpdateRenderWindows();
}
void fTumorPanel::HighlightCurrentSelctedPoints(double x, double y, double z, double X, double Y, double Z, double value)
{
  if (mCurrentTPoints == NULL)
    return;

  int rowCount = mCurrentTPoints->GetNumberOfPoints();
  for (int rowIndex = 0; rowIndex<rowCount; rowIndex++)
  {
    float roundX = floorf(x * 1000) / 1000;
    float roundY = floorf(y * 1000) / 1000;
    float roundZ = floorf(z * 1000) / 1000;

    float roundCurrentX = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[0] * 1000) / 1000;
    float roundCurrentY = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[1] * 1000) / 1000;
    float roundCurrentZ = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[2] * 1000) / 1000;

    if (roundX == roundCurrentX && roundY == roundCurrentY && roundZ == roundCurrentZ)
    {
      tTableWidget->selectRow(rowIndex);
      QTableWidgetItem * item = tTableWidget->item(rowIndex, 0);
      tTableWidget->scrollToItem(item, QAbstractItemView::PositionAtTop);
      tTableWidget->setItemSelected(item, true);
      break;
    }
  }
}
void fTumorPanel::HandleKeyPressingEventTTable()
{
  if (mCurrentTPoints == NULL)
    return;

  if (mCurrentTPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
  if (items.empty())
    return;

  int row = items[0]->row();

  double x, y, z;
  x = mCurrentTPoints->mLandmarks[row].coordinates[0];
  y = mCurrentTPoints->mLandmarks[row].coordinates[1];
  z = mCurrentTPoints->mLandmarks[row].coordinates[2];
  emit MoveSlicerCursor(x, y, z);
}
void fTumorPanel::HandleDownKeyEventTTable()
{
  if (mCurrentTPoints == NULL)
    return;

  if (mCurrentTPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
  if (items.empty())
    return;

  unsigned int rowindex = items[0]->row() + 1;
  if (rowindex > mCurrentTPoints->GetNumberOfPoints())
    return;

  for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
  {
    QTableWidgetItem * item3 = tTableWidget->item(i, 0);
    tTableWidget->setItemSelected(item3, false);
  }
  tTableWidget->selectRow(rowindex);
  QTableWidgetItem * item2 = tTableWidget->item(rowindex, 0);
  tTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
  tTableWidget->setItemSelected(item2, true);

}
void fTumorPanel::HandleUpKeyEventTTable()
{
  if (mCurrentTPoints == NULL)
    return;

  if (mCurrentTPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
  if (items.empty())
    return;

  int rowindex = items[0]->row() - 1;
  if (rowindex < 0)
    return;
  for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
  {
    QTableWidgetItem * item3 = tTableWidget->item(i, 0);
    tTableWidget->setItemSelected(item3, false);
  }
  tTableWidget->selectRow(rowindex);
  QTableWidgetItem * item2 = tTableWidget->item(rowindex, 0);
  tTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
  tTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::HandleDeleteKeyEventTTable()
{
  tRemoveSelectedPoints();
}


void fTumorPanel::HandleKeyPressingEventSTable()
{
  if (mCurrentSPoints == NULL)
    return;

  if (mCurrentSPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
  if (items.empty())
    return;

  int row = items[0]->row();

  double x, y, z;
  x = mCurrentSPoints->mLandmarks[row].coordinates[0];
  y = mCurrentSPoints->mLandmarks[row].coordinates[1];
  z = mCurrentSPoints->mLandmarks[row].coordinates[2];
  emit MoveSlicerCursor(x, y, z);
}
void fTumorPanel::HandleDownKeyEventSTable()
{
  if (mCurrentSPoints == NULL)
    return;

  if (mCurrentSPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
  if (items.empty())
    return;

  unsigned int rowindex = items[0]->row() + 1;
  if (rowindex > mCurrentSPoints->GetNumberOfPoints())
    return;



  for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
  {
    QTableWidgetItem * item3 = sTableWidget->item(i, 0);
    sTableWidget->setItemSelected(item3, false);
  }
  sTableWidget->selectRow(rowindex);
  QTableWidgetItem * item2 = sTableWidget->item(rowindex, 0);
  sTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
  sTableWidget->setItemSelected(item2, true);

}
void fTumorPanel::HandleUpKeyEventSTable()
{
  if (mCurrentSPoints == NULL)
    return;

  if (mCurrentSPoints->GetNumberOfPoints() == 0)
    return;

  QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
  if (items.empty())
    return;

  int rowindex = items[0]->row() - 1;
  if (rowindex < 0)
    return;

  for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
  {
    QTableWidgetItem * item3 = sTableWidget->item(i, 0);
    sTableWidget->setItemSelected(item3, false);
  }
  sTableWidget->selectRow(rowindex);
  QTableWidgetItem * item2 = sTableWidget->item(rowindex, 0);
  sTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
  sTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::HandleDeleteKeyEventSTable()
{
  sRemoveSelectedPoints();
}

void fTumorPanel::SetSeedType()
{
  if (mCurrentSPoints == NULL)
  {
    emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, 0, 0);
  }
  else
  {
    emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, mCurrentSPoints->GetNumberOfPoints(), 0);
  }

  RadioButton_CSF->setEnabled(false);
  RadioButton_GM->setEnabled(false);
  RadioButton_WM->setEnabled(false);
  RadioButton_VS->setEnabled(false);
  RadioButton_ED->setEnabled(false);
  RadioButton_NCR->setEnabled(false);
  RadioButton_TU->setEnabled(false);
  RadioButton_NE->setEnabled(false);
  RadioButton_CB->setEnabled(false);
  RadioButton_VT->setEnabled(false);
  RadioButton_CAN->setEnabled(false);
  RadioButton_CAE->setEnabled(false);
  RadioButton_RTN->setEnabled(false);
  RadioButton_RTE->setEnabled(false);

  

}
void fTumorPanel::SetTissueType()
{
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);

  RadioButton_CSF->setEnabled(true);
  RadioButton_GM->setEnabled(true);
  RadioButton_WM->setEnabled(true);
  RadioButton_VS->setEnabled(true);
  RadioButton_ED->setEnabled(true);
  RadioButton_NCR->setEnabled(true);
  RadioButton_TU->setEnabled(true);
  RadioButton_NE->setEnabled(true);
  RadioButton_CB->setEnabled(true);
  RadioButton_VT->setEnabled(true);
  RadioButton_CAN->setEnabled(true);
  RadioButton_CAE->setEnabled(true);
  RadioButton_RTN->setEnabled(true);
  RadioButton_RTE->setEnabled(true);
  initializationFileName = "initiazedPoints_all.txt";

  saveType_GLISTR = false;
  saveType_PORTRPost = false;
  saveType_PORTRPre = false;
}
void fTumorPanel::SetGLISTRTissueType()
{
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
  
  RadioButton_CSF->setEnabled(true);
  RadioButton_GM->setEnabled(true);
  RadioButton_WM->setEnabled(true);
  RadioButton_VS->setEnabled(true);
  RadioButton_ED->setEnabled(true);
  RadioButton_NCR->setEnabled(true);
  RadioButton_TU->setEnabled(true);
  RadioButton_NE->setEnabled(true);
  RadioButton_CB->setEnabled(true);
  RadioButton_VT->setEnabled(false);
  RadioButton_CAN->setEnabled(false);
  RadioButton_CAE->setEnabled(false);
  RadioButton_RTN->setEnabled(false);
  RadioButton_RTE->setEnabled(false);

  initializationFileName = "initiazedPoints_glistr.txt";
  saveType_GLISTR = true;
  saveType_PORTRPre = false;
  saveType_PORTRPost = false;
}
void fTumorPanel::SetPORTRPRETissueType()
{
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);

  RadioButton_CSF->setEnabled(true);
  RadioButton_GM->setEnabled(true);
  RadioButton_WM->setEnabled(true);
  RadioButton_VS->setEnabled(true);
  RadioButton_ED->setEnabled(true);
  RadioButton_NCR->setEnabled(true);
  RadioButton_TU->setEnabled(true);
  RadioButton_NE->setEnabled(false);
  RadioButton_CB->setEnabled(false);
  RadioButton_VT->setEnabled(true);
  RadioButton_CAN->setEnabled(false);
  RadioButton_CAE->setEnabled(false);
  RadioButton_RTN->setEnabled(false);
  RadioButton_RTE->setEnabled(false);


  initializationFileName = "initiazedPoints_portrPre.txt";
  saveType_GLISTR = false;
  saveType_PORTRPre = true;
  saveType_PORTRPost = false;
}
void fTumorPanel::SetPORTRPOSTTissueType()
{
  emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);

  RadioButton_CSF->setEnabled(true);
  RadioButton_GM->setEnabled(true);
  RadioButton_WM->setEnabled(true);
  RadioButton_VS->setEnabled(true);
  RadioButton_ED->setEnabled(true);
  RadioButton_NCR->setEnabled(false);
  RadioButton_TU->setEnabled(false);
  RadioButton_NE->setEnabled(false);
  RadioButton_CB->setEnabled(false);
  RadioButton_VT->setEnabled(true);
  RadioButton_CAN->setEnabled(true);
  RadioButton_CAE->setEnabled(true);
  RadioButton_RTN->setEnabled(true);
  RadioButton_RTE->setEnabled(true);

  initializationFileName = "initiazedPoints_portrPost.txt";
  saveType_GLISTR = false;
  saveType_PORTRPre = false;
  saveType_PORTRPost = true;
}

void fTumorPanel::UpdateCurrentPoints()
{
  if (mCurrentTPoints == NULL)
    return;

  int csfCounter = 0;
  int gmCounter = 0;
  int wmCounter = 0;
  int vsCounter = 0;
  int edCounter = 0;
  int ncrCounter = 0;
  int tuCounter = 0;
  int neCounter = 0;
  int cbCounter = 0;
  int vtCounter = 0;
  int canCounter = 0;
  int caeCounter = 0;
  int rtnCounter = 0;
  int rteCounter = 0;

  for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
  {
    switch (mCurrentTPoints->mLandmarks[i].id)
    {
    case CSF:
      csfCounter++;
      break;
    case GM:
      gmCounter++;
      break;
    case WM:
      wmCounter++;
      break;
    case VS:
      vsCounter++;
      break;
    case ED:
      edCounter++;
      break;
    case NCR:
      ncrCounter++;
      break;
    case TU:
      tuCounter++;
      break;
    case NE:
      neCounter++;
      break;
    case CB:
      cbCounter++;
      break;
    case VT:
      vtCounter++;
      break;
    case CAN:
      canCounter++;
      break;
    case CAE:
      caeCounter++;
      break;
    case RTN:
      rtnCounter++;
      break;
    case RTE:
      rteCounter++;
      break;
    default:
      cbica::Logging(loggerFile, "Undefined mCurrentTPoints->mLandmarks");
      exit(EXIT_FAILURE);
    }
  }

  RadioButton_CSF->setText("CSF:" + QString::number(csfCounter));
  RadioButton_GM->setText("GM:" + QString::number(gmCounter));
  RadioButton_WM->setText("WM:" + QString::number(wmCounter));
  RadioButton_VS->setText("VS:" + QString::number(vsCounter));
  RadioButton_ED->setText("ED:" + QString::number(edCounter));
  RadioButton_NCR->setText("NCR:" + QString::number(ncrCounter));
  RadioButton_TU->setText("TU:" + QString::number(tuCounter));
  RadioButton_NE->setText("NE:" + QString::number(neCounter));
  RadioButton_CB->setText("CB:" + QString::number(cbCounter));
  RadioButton_VT->setText("VT:" + QString::number(vtCounter));
  RadioButton_CAN->setText("CAN:" + QString::number(canCounter));
  RadioButton_CAE->setText("CAE:" + QString::number(caeCounter));
  RadioButton_RTN->setText("RTN:" + QString::number(rtnCounter));
  RadioButton_RTE->setText("RTE:" + QString::number(rteCounter));

}
