/**
\file RecurrenceEstimator.h

This file holds the declaration of the class RecurrenceEstimator.

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html
*/

#ifndef _RecurrenceEstimator_h_
#define _RecurrenceEstimator_h_

#include "CAPTk.h"
#include "NiftiDataManager.h"
#include "OutputWritingManager.h"
#include "FeatureReductionClass.h"
#include "FeatureScalingClass.h"
#include "FeatureExtractionClass.h"
#include "SVMClassificationClass.h"
#include "fProgressDialog.h"

#include "ApplicationBase.h"

#if (_WIN32)
static std::string cSeparator = "\\";
//  static const char* cSeparators = "\\/";
#else
static std::string cSeparator = "/";
//  static const char* cSeparators = "/";
#endif

/**
\class RecurrenceEstimator

\brief A small description of RecurrenceEstimator

Give more details about the class (like its usage, default parameters, etc.) along with the correct reference paper.

References:

@inproceedings{,
title={Imaging Surrogates of Infiltration Obtained Via Multiparametric Imaging Pattern Analysis Predict Subsequent Location of Recurrence of Glioblastoma},
author={Akbari, Hamed MD, PhD; Macyszyn, Luke MD, MA; Da, Xiao MSc; Bilello, Michel MD, PhD; Wolf, Ronald L. MD, PhD; Martinez-Lage, Maria MD; Biros, George PhD; Alonso-Basanta, Michelle MD, PhD; O'Rourke, Donald M. MD; Davatzikos, Christos PhD},
pages={572–580},
year={2016},
organization={Neurosurgery}
}

@inproceedings{,
title={Pattern Analysis of Dynamic Susceptibility Contrast-enhanced MR Imaging Demonstrates Peritumoral Tissue Heterogeneity},
author={Hamed Akbari, MD, PhD Luke Macyszyn, MD, MA Xiao Da, MS Ronald L. Wolf, MD, PhD Michel Bilello, MD, PhD Ragini Verma, PhD Donald M. O’Rourke, MD Christos Davatzikos, PhD},
pages={502-510},
year={2014},
organization={RSNA Radiology}
}
*/
class RecurrenceEstimator : public ApplicationBase
{
public:
  /**
  \brief Default Constructor
  */
  RecurrenceEstimator()
  {
	  //mNiftiLocalPtr = new NiftiDataManager();
	  //mOutputLocalPtr = new OutputWritingManager();
	  //mFeatureReductionLocalPtr = new FeatureReductionClass();
	  //mFeatureScalingLocalPtr = new FeatureScalingClass();
	  //mFeatureExtractionLocalPtr = new FeatureExtractionClass();
    mTrainedModelName = "FinalModelFile.xml";
  };

  //! Default Destructor
  ~RecurrenceEstimator();

  //! Gets last error
  std::string mLastEncounteredError;

  std::string mTrainedModelName;

  /**
  \brief Get the size of the Feature Vector

  */
  int GetFeatureVectorSize(bool &useT1Data, bool &useT1CEData, bool &useT2FlairData, bool &useT2Data, bool &useDTIData, bool &usePerfData, bool &useDistData);

  /**
  \brief Rescale Image Intensity using the itk::RescaleIntensityImageFilter to the 0-255 range
  */
  ImageTypeFloat3D::Pointer RescaleImageIntensity(ImageTypeFloat3D::Pointer image);

  /**
  \brief Train a new model based on the inputs
  */
  void TrainNewModelOnGivenData(const std::string &directoryname, const std::string &outputdirectory,  bool &useT1Data,  bool &useT1CEData,  bool &useT2Data,  bool &useT2FlairData,  bool &useDTIData,  bool &usePerfData,  bool &useDistData);

  /**
  \brief Recurrence Estimation using existing model
  */
  void RecurrenceEstimateOnExistingModel(const std::string &modeldirectory, const std::string &inputdirectory, const std::string &outputdirectory,  bool &useT1Data,  bool &useT1CEData,  bool &useT2Data,  bool &useT2FlairData,  bool &useDTIData,  bool &usePerfData,  bool &useDistData);


  /**
  \brief Load qualified subjects
  */
  void LoadQualifiedSubjectsFromGivenDirectory(std::string type, std::string directoryname, bool useT1Data, bool useT1CEData, bool useT2Data, bool useT2FlairData, bool useDTIData, bool usePerfData, bool useDistData,
    std::vector<std::string> & qualifiedSubjectNames,
    std::vector<std::string> & t1FileNames,
    std::vector<std::string> & t1ceFileNames,
    std::vector<std::string> & t2FileNames,
    std::vector<std::string> & t2FlairFileNames,
    std::vector<std::string> & axFileNames,
    std::vector<std::string> & faFileNames,
    std::vector<std::string> & radFileNames,
    std::vector<std::string> & trFileNames,
    std::vector<std::string> & perfFileNames,
    std::vector<std::string> & labelNames,
    std::vector<std::string> & nearFileNames,
    std::vector<std::string> & farFileNames,
    std::vector<std::string> & rejectedSubjectNames);
  
  /**
  \brief Recurrence Estimation using existing model on given subject
  */
  template<class ImageType>
  void RecurrenceEstimateOnGivenSubject(typename ImageType::Pointer edemaMask,
    typename ImageType::Pointer tumorMask,
    typename ImageType::Pointer FinalT1CEImagePointer,
    typename ImageType::Pointer FinalT2FlairImagePointer,
    typename ImageType::Pointer FinalT1ImagePointer,
    typename ImageType::Pointer FinalT2ImagePointer,
    std::vector<typename ImageType::Pointer> FinalPerfusionImagePointer,
    std::vector<typename ImageType::Pointer> FinalDTIImagePointer,
    int imagetype, bool t1DataPresent, bool t2DataPresent, bool t1ceDataPresent, bool t2flairDataPresent, bool dtiDataPresent, bool perfusionDataPresent, bool distanceDataPresent,
    bool useOtherModalities, std::string t1cebasefilename,
    const VectorVectorDouble &nearRegionIndices,
    const VectorVectorDouble &farRegionIndices);

  NiftiDataManager mNiftiLocalPtr;
  OutputWritingManager mOutputLocalPtr;
  FeatureReductionClass mFeatureReductionLocalPtr;
  FeatureScalingClass mFeatureScalingLocalPtr;
  FeatureExtractionClass mFeatureExtractionLocalPtr;
  SVMClassificationClass mClassificationLocalPtr;

  void Run(const std::string &modeldirectory, const std::string &inputdirectory, const std::string &outputdirectory, bool useT1Data, bool useT1CEData, bool useT2Data, bool useT2FlairData, bool useDTIData, bool usePerfData, bool useDistData)
  {
    RecurrenceEstimateOnExistingModel(modeldirectory, inputdirectory, outputdirectory, useT1Data, useT1CEData, useT2Data, useT2FlairData, useDTIData, usePerfData, useDistData);
  }

  template<class ImageType>
  void Run(typename ImageType::Pointer edemaMask,
    typename ImageType::Pointer tumorMask,
    typename ImageType::Pointer FinalT1CEImagePointer,
    typename ImageType::Pointer FinalT2FlairImagePointer,
    typename ImageType::Pointer FinalT1ImagePointer,
    typename ImageType::Pointer FinalT2ImagePointer,
    std::vector<typename ImageType::Pointer> FinalPerfusionImagePointer,
    std::vector<typename ImageType::Pointer> FinalDTIImagePointer,
    int imagetype, bool t1DataPresent, bool t2DataPresent, bool t1ceDataPresent, bool t2flairDataPresent, bool dtiDataPresent, bool perfusionDataPresent, bool distanceDataPresent,
    bool useOtherModalities, const std::string &t1cebasefilename,
    const VectorVectorDouble &nearRegionIndices,
    const VectorVectorDouble &farRegionIndices, const std::string &outputDirName)
  {
    mCurrentOutputDir = outputDirName;
    RecurrenceEstimateOnGivenSubject<ImageType>(edemaMask, tumorMask, FinalT1CEImagePointer, FinalT2FlairImagePointer, FinalT1ImagePointer, FinalT2ImagePointer,
      FinalPerfusionImagePointer, FinalDTIImagePointer,
      imagetype, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent,
      useOtherModalities, t1cebasefilename, nearRegionIndices, farRegionIndices);
  }

  std::string mRecurrenceMapFileName;
private:
  std::string mCurrentOutputDir;
};

template<class ImageType>
void RecurrenceEstimator::RecurrenceEstimateOnGivenSubject(typename ImageType::Pointer edemaMask,
  typename ImageType::Pointer tumorMask,
  typename ImageType::Pointer FinalT1CEImagePointer,
  typename ImageType::Pointer FinalT2FlairImagePointer,
  typename ImageType::Pointer FinalT1ImagePointer,
  typename ImageType::Pointer FinalT2ImagePointer,
  std::vector<typename ImageType::Pointer> FinalPerfusionImagePointer,
  std::vector<typename ImageType::Pointer> FinalDTIImagePointer,
  int imagetype, bool t1DataPresent, bool t2DataPresent, bool t1ceDataPresent, bool t2flairDataPresent, bool dtiDataPresent, bool perfusionDataPresent, bool distanceDataPresent,
  bool useOtherModalities, std::string t1cebasefilename,
  const VectorVectorDouble &nearRegionIndices,
  const VectorVectorDouble &farRegionIndices)
{
  //------------------------------------------training data formulation------------------------------------------
  mOutputLocalPtr.SetOutputDirectoryPath(mCurrentOutputDir);
  VectorVectorDouble mNearIntensities;
  VectorVectorDouble mFarIntensities;
  VectorVectorDouble pNearIntensities;
  VectorVectorDouble pFarIntensities;
  VectorDouble tNearIntensities;
  VectorDouble tFarIntensities;

  VectorVectorDouble perfusionIntensities;
  VectorVectorDouble otherIntensities;
  VectorDouble distanceIntensities;
  VectorVectorDouble fNearIntensities;
  VectorVectorDouble fFarIntensities;
  messageUpdate("Recurrence Estimation");
  progressUpdate(0);
  mNiftiLocalPtr.LoadTrainingData(tumorMask, nearRegionIndices, farRegionIndices, FinalT1CEImagePointer, FinalT2FlairImagePointer, FinalT1ImagePointer, FinalT2ImagePointer, FinalPerfusionImagePointer, FinalDTIImagePointer, pNearIntensities, pFarIntensities, mNearIntensities, mFarIntensities, tNearIntensities, tFarIntensities, IMAGE_NIFTI, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent);

  for (unsigned int i = 0; i < mNearIntensities.size(); i++)
  {
    if (perfusionDataPresent)
      perfusionIntensities.push_back(pNearIntensities[i]);
    if (useOtherModalities)
      otherIntensities.push_back(mNearIntensities[i]);
    if (distanceDataPresent)
      distanceIntensities.push_back(tNearIntensities[i]);
  }
  for (unsigned int i = 0; i < mFarIntensities.size(); i++)
  {
    if (perfusionDataPresent)
      perfusionIntensities.push_back(pFarIntensities[i]);
    if (useOtherModalities)
      otherIntensities.push_back(mFarIntensities[i]);
    if (distanceDataPresent)
      distanceIntensities.push_back(tFarIntensities[i]);
  }
  progressUpdate(10);

  VariableLengthVectorType perfMeanVector;
  vtkSmartPointer<vtkTable> reducedPerfusionFeatures;
  //------------------------------------------reduce perfusion intensities------------------------------------------
  if (perfusionDataPresent)
  {
    reducedPerfusionFeatures = mFeatureReductionLocalPtr.GetDiscerningPerfusionTimePoints(perfusionIntensities);
  }
  //-----------------------------------develope final near and far vectors------------------------------------------
  for (unsigned int i = 0; i < mNearIntensities.size(); i++)
  {
    VectorDouble cIntensityVectorPerSub;
    if (perfusionDataPresent)
      for (int j = 0; j < NO_OF_PCS; j++)
        cIntensityVectorPerSub.push_back(reducedPerfusionFeatures->GetValue(i, j).ToDouble());

    if (useOtherModalities)
      for (unsigned int j = 0; j < otherIntensities[i].size(); j++)
        cIntensityVectorPerSub.push_back(otherIntensities[i][j]);

    if (distanceDataPresent)
      cIntensityVectorPerSub.push_back(distanceIntensities[i]);

    fNearIntensities.push_back(cIntensityVectorPerSub);
  }

  for (unsigned int i = mNearIntensities.size(); i < mNearIntensities.size() + mFarIntensities.size(); i++)
  {
    VectorDouble cIntensityVectorPerSub;
    if (perfusionDataPresent)
      for (int j = 0; j < NO_OF_PCS; j++)
        cIntensityVectorPerSub.push_back(reducedPerfusionFeatures->GetValue(i, j).ToDouble());

    if (useOtherModalities)
      for (unsigned int j = 0; j < otherIntensities[i].size(); j++)
        cIntensityVectorPerSub.push_back(otherIntensities[i][j]);

    if (distanceDataPresent)
      cIntensityVectorPerSub.push_back(distanceIntensities[i]);
    fFarIntensities.push_back(cIntensityVectorPerSub);
  }
  mFeatureExtractionLocalPtr.FormulateTrainingData(fNearIntensities, fFarIntensities);
  VariableSizeMatrixType TrainingData = mFeatureExtractionLocalPtr.GetTrainingData();

  progressUpdate(20);

  //typedef vnl_matrix<double> MatrixType;
  //MatrixType data;
  //data.set_size(120, 45);
  //for (int i = 0; i < perfusionIntensities.size(); i++)
  //	for (int j = 0; j < 45; j++)
  //		data(i, j) = perfusionIntensities[i][j];
  //typedef itk::CSVNumericObjectFileWriter<double, 120, 45> WriterType;
  //WriterType::Pointer writer = WriterType::New();
  //writer->SetFileName("perfusionData.csv");
  //writer->SetInput(&data);
  //writer->Write();


  //data.set_size(120,15);
  //for (unsigned int i = 0; i < TrainingData.Rows(); i++)
  //  for (unsigned int j = 0; j < TrainingData.Cols(); j++)
  //		data(i, j) = TrainingData[i][j];
  //writer->SetFileName("tData.csv");
  //writer->SetInput(&data);
  //writer->Write();


  VariableSizeMatrixType ScaledTrainingData = mFeatureScalingLocalPtr.ScaleGivenTrainingFeatures(TrainingData);

  //data.set_size(120, 15);
  //for (unsigned int i = 0; i < ScaledTrainingData.Rows(); i++)
  //  for (unsigned int j = 0; j < ScaledTrainingData.Cols(); j++)
  //		data(i, j) = ScaledTrainingData[i][j];
  //writer->SetFileName("sData.csv");
  //writer->SetInput(&data);
  //writer->Write();

  ////------------------------------------------training process---------------------------------------------------
  //FILE *t;
  //t = fopen("TrainingData.txt", "w");

  //for (int i = 0; i < ScaledTrainingData.Rows(); i++)
  //{
  //	fprintf(t, "%f ", ScaledTrainingData[i][ScaledTrainingData.Cols() - 1]);
  //	for (int j = 0; j < ScaledTrainingData.Cols() - 1; j++)
  //		fprintf(t, "%d:%lf ", j + 1, ScaledTrainingData[i][j]);
  //	fprintf(t, "\n");
  //}
  //fclose(t);

  int size = GetFeatureVectorSize(t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent);
  mOutputLocalPtr.SaveModelResults(ScaledTrainingData, mFeatureScalingLocalPtr.GetMeanVector(), mFeatureScalingLocalPtr.GetStdVector(), mFeatureReductionLocalPtr.GetPerfusionMeanVector(), mFeatureReductionLocalPtr.GetPCATransformationMatrix(),
    t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent, size);
  progressUpdate(30);

   VectorDouble trainDistances = trainOpenCVSVM(ScaledTrainingData, mOutputLocalPtr.mOutputDirectoryPath + "/" + mTrainedModelName, true, true);
//  mClassificationLocalPtr.Training(ScaledTrainingData, mOutputLocalPtr.mOutputDirectoryPath);
  //if (mClassificationLocalPtr->GetLastEncounteredError() != "")
  //  mLastEncounteredError = mClassificationLocalPtr->GetLastEncounteredError();
  progressUpdate(40);
  ////------------------------------------------testing data formulation---------------------------------------------------

  const typename ImageType::RegionType region = FinalT1CEImagePointer->GetLargestPossibleRegion();
  typename ImageType::Pointer RecurrenceProbabilityMap = ImageType::New();
  RecurrenceProbabilityMap->SetRegions(region);
  RecurrenceProbabilityMap->Allocate();
  RecurrenceProbabilityMap->SetSpacing(FinalT1CEImagePointer->GetSpacing());
  RecurrenceProbabilityMap->SetOrigin(FinalT1CEImagePointer->GetOrigin());
  RecurrenceProbabilityMap->SetDirection(FinalT1CEImagePointer->GetDirection());

  perfusionIntensities.clear();
  otherIntensities.clear();
  distanceIntensities.clear();
  std::vector<typename ImageType::IndexType> testindices;
  testindices = mNiftiLocalPtr.LoadTestData(FinalT1CEImagePointer, FinalT2FlairImagePointer, FinalT1ImagePointer, FinalT2ImagePointer, FinalPerfusionImagePointer, FinalDTIImagePointer, tumorMask, edemaMask, perfusionIntensities, otherIntensities, distanceIntensities, IMAGE_DICOM, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent);



  VectorVectorDouble reducedTestPerfusionFeatures;
  if (perfusionDataPresent)
    reducedTestPerfusionFeatures = mFeatureReductionLocalPtr.ApplyPCAOnTestData(perfusionIntensities);

  int NumberOfPCs = 5;
  VectorVectorDouble globaltestintensities;

  for (unsigned int k = 0; k < testindices.size(); k++)
  {
    VectorDouble inten;

    if (perfusionDataPresent)
      for (int j = 0; j < NumberOfPCs; j++)
        inten.push_back(reducedTestPerfusionFeatures[k][j]);

    if (useOtherModalities)
      for (unsigned int j = 0; j < otherIntensities[0].size(); j++)
        inten.push_back(otherIntensities[k][j]);

    if (distanceDataPresent)
      inten.push_back(distanceIntensities[k]);

    if (inten.size()>0)
      globaltestintensities.push_back(inten);
  }
  VariableSizeMatrixType TestingData = mFeatureExtractionLocalPtr.FormulateTestData(globaltestintensities);
  VariableSizeMatrixType ScaledTestingData = mFeatureScalingLocalPtr.ScaleGivenTestingFeatures(TestingData);

  progressUpdate(60);

  //qSleep(5000);
  //typedef vnl_matrix<double> MatrixTypeT;
  //MatrixTypeT dataT;
  //dataT.set_size(1517, 45);
  //for (int ii = 0; ii < perfusionIntensities.size(); ii++)
  //	for (int jj = 0; jj < 45; jj++)
  //		dataT(ii, jj) = reducedTestPerfusionFeatures[ii][jj];
  //typedef itk::CSVNumericObjectFileWriter<double, 1517, 45> WriterTypeT;
  //WriterTypeT::Pointer writerT = WriterTypeT::New();
  //writerT->SetFileName("perfusionDataTest.csv");
  //writerT->SetInput(&dataT);
  //writerT->Write();


  //dataT.set_size(1517, 15);
  //for (int ii = 0; ii < TestingData.Rows(); ii++)
  //	for (int jj = 0; jj < TestingData.Cols(); jj++)
  //		dataT(ii, jj) = TestingData[ii][jj];
  //writerT->SetFileName("tDataTest.csv");
  //writerT->SetInput(&dataT);
  //writerT->Write();

  //dataT.set_size(1517, 15);
  //for (int ii = 0; ii < ScaledTestingData.Rows(); ii++)
  //	for (int jj = 0; jj < ScaledTestingData.Cols(); jj++)
  //		dataT(ii, jj) = ScaledTestingData[ii][jj];
  //writerT->SetFileName("sDataTest.csv");
  //writerT->SetInput(&dataT);
  //writerT->Write();

  try
  {
    VectorDouble result = testOpenCVSVM(ScaledTestingData, mOutputLocalPtr.mOutputDirectoryPath + "/" + mTrainedModelName);
    //--------------------------------------------------------
    VectorDouble positiveDistances;
    VectorDouble negativeDistances;
    for (int x = 0; x < trainDistances.size(); x++)
    {
      if (trainDistances[x] > 0)
        positiveDistances.push_back(trainDistances[x]);
      else if (trainDistances[x] < 0)
        negativeDistances.push_back(trainDistances[x]);
    }
    std::sort(positiveDistances.begin(), positiveDistances.end());
    std::sort(negativeDistances.rbegin(), negativeDistances.rend());

    double positivePercentileCutOff = 0;
    double negativePercentileCutOff = 0;

    for (int x = 0; x < positiveDistances.size(); x++)
    {
      double percentile = (100 * (x + 0.5)) / positiveDistances.size();
      if (percentile > 99)
      {
        positivePercentileCutOff = result[x];
        break;
      }
    }
    for (int x = 0; x < negativeDistances.size(); x++)
    {
      double percentile = (100 * (x + 0.5)) / negativeDistances.size();
      if (percentile > 99)
      {
        negativePercentileCutOff = result[x];
        break;
      }
    }


    for (int x = 0; x < result.size(); x++)
    {
      if (result[x] > positivePercentileCutOff)
        result[x] = positivePercentileCutOff;
      if (result[x] < negativePercentileCutOff)
        result[x] = negativePercentileCutOff;
    }
    double min = 0;
    double max = positivePercentileCutOff;
    for (int x = 0; x < result.size(); x++)
    {
      if (result[x] > 0)
        result[x] = (result[x] - min)/ (max - min);
    }
    min = negativePercentileCutOff;
    max = 0;
    for (int x = 0; x < result.size(); x++)
    {
      if (result[x] < 0)
        result[x] = (result[x] - min)/ (max - min);
    }
    for (int x = 0; x < result[x]; x++)
      result[x] = (result[x] + 1) / 2;


    //--------------------------------------------------------
    //mClassificationLocalPtr.SetModelFileName(mOutputLocalPtr.mOutputDirectoryPath + "/FinalModelFile.model");
    //VectorVectorDouble result = mClassificationLocalPtr.Testing(ScaledTestingData, false, mOutputLocalPtr.mOutputDirectoryPath +  "/FinalModelFile.model");
	progressUpdate(70);

    for (unsigned int x = 0; x < result.size(); x++)
      RecurrenceProbabilityMap->SetPixel(testindices[x], result[x]);

	progressUpdate(80);
    typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
    IteratorType RecIt(RecurrenceProbabilityMap, RecurrenceProbabilityMap->GetLargestPossibleRegion());
    IteratorType EdeIt(edemaMask, edemaMask->GetLargestPossibleRegion());
    RecIt.GoToBegin();
    EdeIt.GoToBegin();
    while (!RecIt.IsAtEnd())
    {
      typename ImageType::IndexType index = RecIt.GetIndex();
      if (EdeIt.Get() != 255)
        RecIt.Set(0);
      ++RecIt;
      ++EdeIt;
    }
    //if (x_index >= mBorderStartX && x_index <= mBorderEndX && y_index >= mBorderStartY && y_index <= mBorderEndY && z_index >= 80 && z_index <= 120)
  }
  catch (itk::ExceptionObject & excp)
  {
    std::string str(excp.GetDescription());
  }
  progressUpdate(90);

  //------------------------------------------Writing final output--------------------------------------------------
  mRecurrenceMapFileName = "RecurrenceMap.nii.gz";
  mOutputLocalPtr.WriteRecurrenceOutput<ImageType>(RecurrenceProbabilityMap, t1cebasefilename, mRecurrenceMapFileName);
  progressUpdate(100);
}

#endif
