/**
\file  FeatureReductionClass.h

\brief Declaration of the FeatureReductionClass

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#pragma once

#include "iostream"
#include "CAPTk.h"

using namespace std;

class  FeatureReductionClass
{
public:
  //!Constructor
  FeatureReductionClass();

  //!Destructor
  ~FeatureReductionClass();
  
  /**
  \brief Finds the first few discerning principal components in perfusion data
  \param intensities Input perfusion data
  */
  vtkSmartPointer<vtkTable> GetDiscerningPerfusionTimePoints(VectorVectorDouble &intensities);

  /**
  \brief Applies the exsiting PCA model (developed on training data) on test data
  \param intensities Test data
  */
  VectorVectorDouble ApplyPCAOnTestData(VectorVectorDouble &intensities);

  /**
  \brief Calculates the transpose of an input matrix
  \param inputmatrix Input matrix
  */
  VariableSizeMatrixType MatrixTranspose(VariableSizeMatrixType &inputmatrix);

  /**
  \brief Calculates average of each feature present in input data 
  \param inputdata Input matrix (in vtkTable format)
  */
  VariableLengthVectorType ComputeMeanOfGivenFeatureVectors(vtkSmartPointer< vtkTable > &inputdata);

  /**
  \brief Calculates average of each feature present in input data
  \param inputdata Input matrix (in vector format)
  */
  VariableLengthVectorType ComputeMeanOfGivenFeatureVectors(VectorVectorDouble &inputdata);

  //!Returns the PCA transformation matrix
  VariableSizeMatrixType GetPCATransformationMatrix()
  {
    return PCATransformationMatrix;
  }

  //!Returns the avergae perfusion signal
  VariableLengthVectorType GetPerfusionMeanVector()
  {
    return mPMeanvector;
  }

  /**
  \brief Sets the values of PCA transformation matrix and average perfusion signal
  \param pcaMatrix PCA transformation matrix
  \param pcaMean Avergae perfusion signal
  */
  void SetParameters(VariableSizeMatrixType &pcaMatrix, VariableLengthVectorType &pcaMean)
  {
    PCATransformationMatrix = pcaMatrix;
    mPMeanvector = pcaMean;
  }

  /**
  \brief Resets the parameters 
  */
  void ResetParameters()
  {
    PCATransformationMatrix.SetSize(0, 0);
    mPMeanvector.SetSize(0);
  }

private:
  VariableSizeMatrixType PCATransformationMatrix;
  VariableLengthVectorType mPMeanvector;

};
