#pragma once
#include <iostream>
#include <iterator>
#include <stdlib.h>
#include <string>
#include <algorithm>
#include <exception>
#include <typeinfo>
#include <stdexcept>
#include <stdio.h>
#include <numeric>
#include <functional>
#include <vector> 

#include "itkImageFileReader.h"
#include "itkScalarImageToRunLengthFeaturesFilter.h"
#include "itkHistogramToTextureFeaturesFilter.h"
#include <itkLabelImageToShapeLabelMapFilter.h>
#include <itkVectorImage.h>
#include "itkScalarImageToCooccurrenceMatrixFilter.h"
#include "itkRegionOfInterestImageFilter.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkVectorContainer.h"
#include "itkStatisticsImageFilter.h"
#include "itkImageToHistogramFilter.h"
#include <itkShapeLabelObject.h>
#include <itkConnectedComponentImageFilter.h>

#include "CAPTk.h"
using namespace std;

typedef ImageTypeFloat3D::OffsetType OffsetType;
typedef itk::VectorContainer< unsigned char, OffsetType > OffsetVector;
typedef itk::Statistics::ScalarImageToCooccurrenceMatrixFilter<ImageTypeFloat3D>Image2CoOccuranceType;
typedef Image2CoOccuranceType::HistogramType HistogramType;
typedef itk::Statistics::HistogramToTextureFeaturesFilter<HistogramType> Hist2FeaturesType1;

class TextureFeatures
{
public:
  typedef enum
  {
    Custom,
    Torso,
    Neuro
  };

  //! Default Constructor
  TextureFeatures();

  //! Default Destructor
  ~TextureFeatures();


  /**/
  template <class TImageType = ImageTypeFloat3D>
  void Intensity_features(typename TImageType::Pointer image, typename TImageType::Pointer mask, std::vector<std::string> &intensity_featurename, std::vector<double> &intensity_feature);

  /**
  \brief Calculate Texture Features

  This function computes features showcases the texture of a given image. this function uses ITK ScalarImageToTextureFeaturesFilter.
  The texture features have proven to be useful in image classification for biological and medical imaging.
  This function computes the texture features of a specified ROI(region of interest) got through the mask provided as an input para.
  The features that are calculated are contrast, correlation, energy, entropy, haralick correlation, cluster shde etc.
  The user can set the number of bins if required. The deafult value used is 16.

  \param image The input Image on which to calculate the features
  \param mask The image specificying the roi
  \param min Minimum intensity in given roi
  \param max Maximum intensity in given roi
  \param offsetvector vector with 13 offset direction
  \param featurename Vector holing name of the features
  \param feature A vector holding features of each offset direction

  */
  template <class TImageType = ImageTypeFloat3D, typename Offsets = OffsetVector>
  void calculateTextureFeatures(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max, Offsets *offset, std::vector< std::string > &glcm_featurename, std::vector<double> &glcm_feature)
  {
    //glcm_featureNames.push_back("dummyVal");
    //glcm_features.push_back(vector<double>(1, 1));
    //return;
    //https://itk.org/Wiki/ITK/Examples/Statistics/TextureFeatures RK
    //glcm_featureNames
    typename TImageType::Pointer inertia = TImageType::New();
    typename TImageType::Pointer correlation = TImageType::New();
    typename TImageType::Pointer energy = TImageType::New();
    typename TImageType::Pointer entropy = TImageType::New();
    inertia->SetRegions(image->GetLargestPossibleRegion());
    correlation->SetRegions(image->GetLargestPossibleRegion());
    energy->SetRegions(image->GetLargestPossibleRegion());
    entropy->SetRegions(image->GetLargestPossibleRegion());
    Offsets::Pointer offsets = Offsets::New();
    offsets = offset;

    typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
    std::vector< TImageType::IndexType> index_vec;
    //   std::vector<double> nonzero_pix;
    IteratorType inputIt(mask, mask->GetLargestPossibleRegion());

    for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
    {
      if (mask->GetPixel(inputIt.GetIndex()) != 0)
      {
        index_vec.push_back(inputIt.GetIndex());
        nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));

      }
    }

    for (int i = 0; i < offsets->size(); i++)
    {
      Image2CoOccuranceType::Pointer glcmGenerator = Image2CoOccuranceType::New();
      glcmGenerator->SetOffset(offsets->at(i));
      glcmGenerator->SetNumberOfBinsPerAxis(16); //reasonable number of bins
      glcmGenerator->SetPixelValueMinMax(min, max); //for input UCHAR pixel type
      glcmGenerator->SetMaskImage(mask);
      glcmGenerator->SetInput(image);
      Hist2FeaturesType1::Pointer featureCalc = Hist2FeaturesType1::New();
      glcmGenerator->Update();
      featureCalc->SetInput(glcmGenerator->GetOutput());
      featureCalc->Update();
      // std::cout << "correlation" << offsets->at(i) << featureCalc->GetFeature(Hist2FeaturesType1::Correlation);
      // std::cout << "energy" << featureCalc->GetFeature(Hist2FeaturesType1::Energy);
      // std::cout << "contrast" << featureCalc->GetFeature(Hist2FeaturesType1::Inertia);
      // std::cout << "entropy" << featureCalc->GetFeature(Hist2FeaturesType1::Entropy);

      //  std::cout << offsets->GetElement(0).operator[](0);

      glcm_featurename.push_back("correlation" + to_string(i));
      glcm_featurename.push_back("energy" + to_string(i));
      glcm_featurename.push_back("contrast" + to_string(i));
      glcm_featurename.push_back("entropy" + to_string(i));
      glcm_featurename.push_back("inversediffmom" + to_string(i));
      glcm_featurename.push_back("Clustershade" + to_string(i));
      glcm_featurename.push_back("ClusterProminence" + to_string(i));
      glcm_featurename.push_back("haralickCorrelation" + to_string(i));

      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::Correlation));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::Energy));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::Inertia));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::Entropy));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::InverseDifferenceMoment));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::ClusterShade));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::ClusterProminence));
      glcm_feature.push_back(featureCalc->GetFeature(Hist2FeaturesType1::HaralickCorrelation));

    }

 }

  template <class TImageType = ImageTypeFloat3D, typename Offsets = OffsetVector >
  void calculateRunLength(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max,
    Offsets *offset, std::vector<std::string> &runLength_featurename, std::vector<double> &runLength_feature);

  template <class TImageType = ImageTypeFloat3D >
  void ShapeFeatures(typename TImageType::Pointer mask, std::vector<std::string> &runLength_featurename, std::vector<double> &runLength_feature);

  void HistogramFeatures(std::vector<double> &intensities, int start, int interval, int end, std::vector<std::string> &histogram_featurename, std::vector<double> &histogram_feature);
  bool m_errormsg =FALSE;
  std::vector<double> nonzero_pix;
};

template <class TImageType>
void TextureFeatures::Intensity_features(typename TImageType::Pointer image, typename TImageType::Pointer maskin, std::vector<std::string> &intensity_featurename, std::vector<double> &intensity_feature)
{
  typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
  std::vector< TImageType::IndexType> index_vec;

  IteratorType inputIt(maskin, maskin->GetLargestPossibleRegion());

  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  {
    if (maskin->GetPixel(inputIt.GetIndex()) != 0)
    {
      index_vec.push_back(inputIt.GetIndex());
      nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));
    }
  }
  if (nonzero_pix.empty())
  {
    m_errormsg = TRUE;
    return;
  }

  //if (nonzero_pix.size() == 0)
  //{
  //  IteratorType inputIt(image, image->GetLargestPossibleRegion());
  //  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  //  {
  //    if (image->GetPixel(inputIt.GetIndex()) != 0)
  //    {
  //      index_vec.push_back(inputIt.GetIndex());
  //      nonzero_pix.push_back(image->GetPixel(inputIt.GetIndex()));
  //    }
  //  }
  //}

  double min = *min_element(nonzero_pix.begin(), nonzero_pix.end());
  //std::cout << "Min value: " << min;
  double max = *max_element(nonzero_pix.begin(), nonzero_pix.end());
  //std::cout << "Max value: " << max;

  double sum = std::accumulate(nonzero_pix.begin(), nonzero_pix.end(), 0.0);
  double mean = sum / nonzero_pix.size();

  std::vector<double> diff(nonzero_pix.size());
  std::transform(nonzero_pix.begin(), nonzero_pix.end(), diff.begin(),
    std::bind2nd(std::minus<double>(), mean));
  double sq_sum = std::inner_product(diff.begin(), diff.end(), diff.begin(), 0.0);
  double stdev = std::sqrt(sq_sum / nonzero_pix.size());

  double voxvol = image->GetSpacing().GetElement(0)*image->GetSpacing().GetElement(1)*image->GetSpacing().GetElement(2);
  double volume = nonzero_pix.size() * voxvol;

  intensity_featurename.push_back("min_int"); intensity_feature.push_back(min);
  intensity_featurename.push_back("max_int"); intensity_feature.push_back(max);
  intensity_featurename.push_back("mean_int"); intensity_feature.push_back(mean);
  intensity_featurename.push_back("std"); intensity_feature.push_back(stdev);
  intensity_featurename.push_back("volume"); intensity_feature.push_back(volume);

  return;
}


template <class TImageType, typename Offsets>
void TextureFeatures::calculateRunLength(std::vector<double> &intensity, typename TImageType::Pointer image, typename TImageType::Pointer mask, int min, int max,
  Offsets *offset, std::vector<std::string> &runLength_featurename, std::vector<double> &runLength_feature)
{
  //typedef itk::VectorContainer< unsigned char, OffsetType > OffsetVector;
  Offsets::Pointer offsets = Offsets::New();
  offsets = offset;

  typedef itk::ImageRegionIteratorWithIndex< TImageType > IteratorType;
  std::vector< TImageType::IndexType> index_vec;
  //std::vector<double> nonzero_pix;
  IteratorType inputIt(mask, mask->GetLargestPossibleRegion());

  for (inputIt.GoToBegin(); !inputIt.IsAtEnd(); ++inputIt)
  {
    if (mask->GetPixel(inputIt.GetIndex()) != 0)
    {
      index_vec.push_back(inputIt.GetIndex());
      nonzero_pix.push_back(inputIt.Get());
    }
  }
  typedef itk::Statistics::DenseFrequencyContainer2 HistogramFrequencyContainerType;
  typedef itk::Statistics::ScalarImageToRunLengthFeaturesFilter
    <ImageTypeFloat3D, HistogramFrequencyContainerType> RunLengthFilterType;
  typedef  RunLengthFilterType::RunLengthMatrixFilterType RunLengthMatrixGenerator;
  typedef  RunLengthFilterType::RunLengthFeaturesFilterType RunLengthFeatures;
  RunLengthMatrixGenerator::Pointer matrix_generator = RunLengthMatrixGenerator::New();
  typedef RunLengthFilterType::RunLengthFeaturesFilterType    RunLengthFeatureName;
  typedef RunLengthFeatureName::RunLengthFeatureName InternalRunLengthFeatureName;
  RunLengthFilterType::FeatureNameVectorPointer requestedFeatures = RunLengthFilterType::FeatureNameVector::New();

  requestedFeatures->push_back(RunLengthFeatureName::ShortRunEmphasis);
  requestedFeatures->push_back(RunLengthFeatureName::LongRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::GreyLevelNonuniformity);
  requestedFeatures->push_back(
    RunLengthFeatureName::RunLengthNonuniformity);
  requestedFeatures->push_back(
    RunLengthFeatureName::LowGreyLevelRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::HighGreyLevelRunEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::ShortRunLowGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::ShortRunHighGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::LongRunLowGreyLevelEmphasis);
  requestedFeatures->push_back(
    RunLengthFeatureName::LongRunHighGreyLevelEmphasis);
  std::vector <std::string> featurename;
  featurename.push_back("SRE"); featurename.push_back("LRE");
  featurename.push_back("GLN"); featurename.push_back("RLN");
  featurename.push_back("LGRE"); featurename.push_back("HGRE");
  featurename.push_back("SRLGE");  featurename.push_back("SRHGE");
  featurename.push_back("LRLGE");  featurename.push_back("LRHGE");

  // texFilter->SetRequestedFeatures(requestedFeatures);
  // texFilter->Update();
  // FeatureValueVectorPointer means, stds;
  //  means = texFilter->GetFeatureMeans();
  // stds = texFilter->GetFeatureStandardDeviations();

  OffsetVector::ConstIterator offsetIt;
  size_t offsetNum, featureNum;
  size_t numOffsets = offsets->size();
  size_t numFeatures = requestedFeatures->size();
  double **features;
  features = new double *[numOffsets];

  for (size_t i = 0; i < numOffsets; i++)
  {
    features[i] = new double[numFeatures];
  }

  for (offsetIt = offsets->Begin(), offsetNum = 0;
    offsetIt != offsets->End(); offsetIt++, offsetNum++){

    matrix_generator->SetOffset(offsetIt.Value());
    matrix_generator->SetMaskImage(mask);
    matrix_generator->SetInput(image);
    
    matrix_generator->SetInsidePixelValue(1);
    matrix_generator->SetPixelValueMinMax(min, max);
    matrix_generator->SetDistanceValueMinMax(0, 4); // TOCHECK - why is this only between 0-4? SP
    matrix_generator->SetNumberOfBinsPerAxis(10); // TOCHECK - needs to be statistically significant
    matrix_generator->Update();

    std::stringstream ss;

    RunLengthFeatures::Pointer runLengthMatrixCalculator = RunLengthFeatures::New();
    runLengthMatrixCalculator->SetInput(matrix_generator->GetOutput());
    runLengthMatrixCalculator->Update();
    RunLengthFilterType::FeatureNameVector::ConstIterator fnameIt;
    for (fnameIt = requestedFeatures->Begin(), featureNum = 0;
      fnameIt != requestedFeatures->End(); fnameIt++, featureNum++)
    {
      ss << offsetIt.Value();
      std::string current_offset = ss.str();
      runLength_featurename.push_back(featurename.at(fnameIt.Value()) + to_string(offsetNum));
      runLength_feature.push_back(runLengthMatrixCalculator->GetFeature((InternalRunLengthFeatureName)fnameIt.Value()));
      features[offsetNum][featureNum] = runLengthMatrixCalculator->GetFeature((InternalRunLengthFeatureName)fnameIt.Value());
    }
  }
}


template <class TImageType>
void TextureFeatures::ShapeFeatures(typename TImageType::Pointer mask, std::vector<std::string> &shape_featurename, std::vector<double> &shape_feature)
{
  typedef  short                                LabelType;
  typedef itk::Image< LabelType, 3 >            OutputImageType;
  typedef itk::ShapeLabelObject< LabelType, 3 > ShapeLabelObjectType;
  typedef itk::LabelMap< ShapeLabelObjectType >         LabelMapType;

  typedef itk::ConnectedComponentImageFilter < TImageType, OutputImageType >
    ConnectedComponentImageFilterType;
  typedef itk::LabelImageToShapeLabelMapFilter < OutputImageType, LabelMapType >
    I2LType;
  ConnectedComponentImageFilterType::Pointer connected = ConnectedComponentImageFilterType::New();
  connected->SetInput(mask);
  connected->Update();


  I2LType::Pointer i2l = I2LType::New();
  i2l->SetInput(connected->GetOutput());
  i2l->SetComputePerimeter(true);
  i2l->Update();
  LabelMapType *labelMap = i2l->GetOutput();
  //std::cout << " has " << labelMap->GetNumberOfLabelObjects() << " labels." << std::endl;
  ShapeLabelObjectType *labelObject = labelMap->GetNthLabelObject(0);

  // std::cout << labelObject->GetPrincipalMoments() << labelObject->GetElongation() <<
  //    labelObject->GetPerimeter() << labelObject->GetRoundness() << labelObject->GetFlatness();

  shape_featurename.push_back("no_of_pix");
  shape_featurename.push_back("principleMomemt_1");
  shape_featurename.push_back("principleMomemt_2");
  shape_featurename.push_back("principleMomemt_3");
  shape_featurename.push_back("Elongation");
  shape_featurename.push_back("Perimeter");
  shape_featurename.push_back("Roundness");
  shape_featurename.push_back("Flatness");

  shape_feature.push_back(labelObject->GetNumberOfPixels());
  shape_feature.push_back(labelObject->GetPrincipalMoments().operator[](0));
  shape_feature.push_back(labelObject->GetPrincipalMoments().operator[](1));
  shape_feature.push_back(labelObject->GetPrincipalMoments().operator[](2));
  shape_feature.push_back(labelObject->GetElongation());
  shape_feature.push_back(labelObject->GetPerimeter());
  shape_feature.push_back(labelObject->GetRoundness());
  shape_feature.push_back(labelObject->GetFlatness());

  //FileToWorkWith.open(filename, std::fstream::in | std::fstream::out | std::fstream::app);


}

