/**
\file  fMainWindow.h

\brief Declaration of fMainWindow class

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#pragma once


#ifndef _fMainWindow_h_
#define _fMainWindow_h_


#include "CAPTk.h"
#include "Slicer.h"
#include "SlicerManager.h"
#include "NiftiDataManager.h"
#include "EGFRvIIISurrogateIndex.h"
#include "RecurrenceEstimator.h"
#include "SurvivalPredictor.h"
#include "ui_fMainWindow.h"
#include "InteractorStyleNavigator.h"
#include "QTablePushButton.h"
#include "SimpleImageManager.h"
#include "OutputWritingManager.h"
#include <PreprocessingPipelineClass.h>
#include <FeatureExtractionClass.h>
#include <FeatureReductionClass.h>
#include <FeatureScalingClass.h>
#include <SVMClassificationClass.h>
#include <GeodesicSegmentation.h>

#define USE_PROCESSDIALOG

enum TAB_TYPE
{
  TAB_IMAGES, TAB_TUMOR, TAB_LANDMARKS
};

typedef itk::Image< short, 3 > GenericImage;

// multiLabel
enum DRAW_MODE
{
  DRAW_MODE_LABEL_1=1, DRAW_MODE_LABEL_2, DRAW_MODE_LABEL_3, DRAW_MODE_LABEL_4, DRAW_MODE_LABEL_5
};

enum SHAPE_MODE
{
  SHAPE_MODE_NONE = 0, SHAPE_MODE_ERACER, SHAPE_MODE_FREE_HAND, SHAPE_MODE_LINE, SHAPE_MODE_RECTANGLE, SHAPE_MODE_CIRCLE
};

//TBD move all this to util class 
template <typename ITK_Exporter, typename VTK_Importer>
void ConnectPipelines(ITK_Exporter &exporter, VTK_Importer* importer)
{
  importer->SetUpdateInformationCallback(exporter->GetUpdateInformationCallback());
  importer->SetPipelineModifiedCallback(exporter->GetPipelineModifiedCallback());
  importer->SetWholeExtentCallback(exporter->GetWholeExtentCallback());
  importer->SetSpacingCallback(exporter->GetSpacingCallback());
  importer->SetOriginCallback(exporter->GetOriginCallback());
  importer->SetScalarTypeCallback(exporter->GetScalarTypeCallback());
  importer->SetNumberOfComponentsCallback(exporter->GetNumberOfComponentsCallback());
  importer->SetPropagateUpdateExtentCallback(exporter->GetPropagateUpdateExtentCallback());
  importer->SetUpdateDataCallback(exporter->GetUpdateDataCallback());
  importer->SetDataExtentCallback(exporter->GetDataExtentCallback());
  importer->SetBufferPointerCallback(exporter->GetBufferPointerCallback());
  importer->SetCallbackUserData(exporter->GetCallbackUserData());
}
template <typename VTK_Exporter, typename ITK_Importer>
void ConnectPipelines(VTK_Exporter* exporter, ITK_Importer &importer)
{
  importer->SetUpdateInformationCallback(exporter->GetUpdateInformationCallback());
  importer->SetPipelineModifiedCallback(exporter->GetPipelineModifiedCallback());
  importer->SetWholeExtentCallback(exporter->GetWholeExtentCallback());
  importer->SetSpacingCallback(exporter->GetSpacingCallback());
  importer->SetOriginCallback(exporter->GetOriginCallback());
  importer->SetScalarTypeCallback(exporter->GetScalarTypeCallback());
  importer->SetNumberOfComponentsCallback(exporter->GetNumberOfComponentsCallback());
  importer->SetPropagateUpdateExtentCallback(exporter->GetPropagateUpdateExtentCallback());
  importer->SetUpdateDataCallback(exporter->GetUpdateDataCallback());
  importer->SetDataExtentCallback(exporter->GetDataExtentCallback());
  importer->SetBufferPointerCallback(exporter->GetBufferPointerCallback());
  importer->SetCallbackUserData(exporter->GetCallbackUserData());
}


template<class InputPixelType, class OutputPixelType, int VImageDimension>
typename itk::Image<OutputPixelType, VImageDimension>::Pointer convertVtkToItk(vtkImageData* vtkInput)
{
  typedef itk::Image<InputPixelType, VImageDimension> InputImageType;
  typedef itk::Image<OutputPixelType, VImageDimension> OutputImageType;
  typedef itk::VTKImageImport<InputImageType> InputImageImportType;

  vtkSmartPointer<vtkImageExport> inputImageExporter = vtkImageExport::New();
#if VTK_MAJOR_VERSION <= 5
  inputImageExporter->SetInput(vtkInput);
#else
  inputImageExporter->SetInputData(vtkInput);
#endif
  typename InputImageImportType::Pointer inputImageImporter = InputImageImportType::New();

  ConnectPipelines(inputImageExporter.GetPointer(), inputImageImporter);

  typename InputImageType::Pointer inputImage = const_cast<InputImageType*>(inputImageImporter->GetOutput());
  inputImage->Update();
  if (inputImage.IsNull()) {
    std::cerr << "vtkImput is invalid." << std::endl;
  }

  typedef itk::CastImageFilter< InputImageType, OutputImageType > CastFilterType;
  typename CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(inputImage);
  castFilter->Update();

  typename OutputImageType::Pointer outputImage = castFilter->GetOutput();
  outputImage->Update();
  if (outputImage.IsNull()) {
    std::cerr << "inputImage is invalid." << std::endl;
    return NULL;
  }

  return outputImage;
}

template<class OutputPixelType, int VImageDimension>
typename itk::Image<OutputPixelType, VImageDimension>::Pointer convertVtkToItk(vtkImageData* vtkInput) //TBD move to cbica util 
{
  std::string InputPixelType = vtkInput->GetScalarTypeAsString();
  if (InputPixelType == "short")
  {
    return convertVtkToItk<short, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned short")
  {
    return convertVtkToItk<unsigned short, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "char")
  {
    return convertVtkToItk<char, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned char")
  {
    return convertVtkToItk<unsigned char, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "int")
  {
    return convertVtkToItk<int, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "unsigned int")
  {
    return convertVtkToItk<unsigned int, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "double")
  {
    return convertVtkToItk<double, OutputPixelType, VImageDimension>(vtkInput);
  }
  else if (InputPixelType == "float")
  {
    return convertVtkToItk<float, OutputPixelType, VImageDimension>(vtkInput);
  }
  else {
    std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
  }
  return NULL;
}

/**
\class fMainWindow

\brief This is the main UI class for CaPTk
*/
class fMainWindow : public QMainWindow, private Ui::fMainWindow
{
  Q_OBJECT

public:
  //! Default constructor
  fMainWindow();

  //! Default destructor
  ~fMainWindow();
  
  QTableWidget * m_imagesTable;
  QTableWidget * m_nonVisImagesTable;

  /**
  \brief Guess Image Type

  \param str String to guess  guess
  \return deduced type
  */
  int guessImageType(const std::string &fileName)
  {
    int ImageSubType = IMAGE_TYPE_UNDEFINED;
    std::string fileName_wrap = fileName;
    std::transform(fileName_wrap.begin(), fileName_wrap.end(), fileName_wrap.begin(), ::tolower);
    if ((fileName_wrap.find("_t1ce") != std::string::npos) || (fileName_wrap.find("_t1-gad") != std::string::npos) ||
      (fileName_wrap.find("_t1-ce") != std::string::npos) || (fileName_wrap.find("_t1-gd") != std::string::npos) ||
      (fileName_wrap.find("_t1gd") != std::string::npos) || (fileName_wrap.find("t1gd_") != std::string::npos) ||
      (fileName_wrap.find("t1ce") != std::string::npos) || (fileName_wrap.find("t1-gad") != std::string::npos) ||
      (fileName_wrap.find("t1-ce") != std::string::npos) || (fileName_wrap.find("t1-gd") != std::string::npos))
    {
      ImageSubType = IMAGE_TYPE_T1CE;
    }
    else if ((fileName_wrap.find("_t1") != std::string::npos) || (fileName_wrap.find("t1_") != std::string::npos))
    {
      ImageSubType = IMAGE_TYPE_T1;
    }
    else if ((fileName_wrap.find("_t2") != std::string::npos) || (fileName_wrap.find("t2_") != std::string::npos))
    {
      if ((fileName_wrap.find("flair") != std::string::npos))
      {
        ImageSubType = IMAGE_TYPE_T2FLAIR;
      }
      else
      {
        ImageSubType = IMAGE_TYPE_T2;
      }
    }
    else if ((fileName_wrap.find("_flair") != std::string::npos) || (fileName_wrap.find("flair_") != std::string::npos))
    {
      ImageSubType = IMAGE_TYPE_T2FLAIR;
    }
    else if ((fileName_wrap.find("_dti") != std::string::npos) || (fileName_wrap.find("dti_") != std::string::npos))
    {
      ImageSubType = IMAGE_TYPE_DTI;
    }
    return ImageSubType;
  }

  /**
  \brief Load images into memory
  
  \param filenames Vector of image filenames which need to be loaded (comes from "Load" dialog)
  \param imagetype_int Either NIfTI or DICOM
  \param bSkipDup Skip duplicates, defaults to true
  */
  void LoadSlicerImages(const string &fileName, const int &imagetype_int, const int &bSkipDup = true);


  /**
  \brief Load non-viewing images into memory
  
  \param directoryname Directory name in which the non-viewing images are present
  \param imagetype_int Either NIfTI or DICOM
  \param imagesubtype Modality of the image (T1, T2, ...)
  */
  void LoadNonViewingImages(const std::string &directoryname, const int &imagetype_int, const int &imagesubtype);

  /**
  \brief Initialize drawing mask on the image
  */
  void InitMask(vtkImageData* image);



  /**
  \brief Initializer for the entire rendering pipeline
  */
  void InitDisplay();

  /**
  \brief Initializes difference slicer views of a single image
  */
  void InitSlicers();

  /**
  \brief Render the sliders to adjust the view of the image slices.

  Initializes it at the middle of the whole image.

  \param slicer If many images are present, this keeps a count of which image is being rendered
  \param window The window number (goes from 1 to 3)
  */
  void DisplaySliders(int slicer, int window);

  /**
  \brief Get image index from the visualization table
  */
  int GetSlicerIndexFromItem(QTableWidgetItem* item);

  /**
  \brief Get the corresponding table item from the image
  */
  QTableWidgetItem* GetItemFromSlicerManager(SlicerManager* sm);
  
  /**
  \brief Construct near and far indeces from the initialized mask
  */
  template<class ImageType>
  void FormulateNearFarPoints(std::vector<typename ImageType::IndexType> &nearIndices, std::vector<typename ImageType::IndexType> &farIndices);

  /**
  \brief Rescales image intensity in the range of 0-255
  */
  ImageTypeFloat3D::Pointer RescaleImageIntensity(ImageTypeFloat3D::Pointer image);

  /*
  \brief Drag event initialization
  */
  void dragEnterEvent(QDragEnterEvent *event);

  /*
  \brief Drop Event parsing
  */
  void dropEvent(QDropEvent *event);

  /*
  \brief For fast developer testing .TBD:  remove or disable.
  */
  void runDevTestMode( QString file = "" )
  {
    QStringList lst(file);
	  this->openImages(lst);
	  
  }

signals:
  void SelectedImageHasChanged(SlicerManager *);
  void LandmarksFocused(bool bFocused);
  void SeedPointsFocused(bool bFocused);
  void TissuePointsFocused(bool bFocused);

  public slots:
  /**
  \brief Updates draw mode when drawing panel changes
  */
  void updateDrawMode(int mode = -1);

  /**
  \brief Updates number of near and far points in the table
  */
  void UpdateNumberOfPointsInTable();

  /**
  \brief Initilializes preset combobox with the default options
  */
  void SetPresetComboBox();

  /**
  \brief Main function that performs EGFRvIII estimation on the displayed image
  */
  void StartEGFREstimate();

  /**
  \brief get the mask image 
  */
 ImageTypeFloat3D::Pointer getMaskImage();

  /**
  \brief get images loaded and theair file names
  */
  std::vector<ImageTypeFloat3D::Pointer> getLodedImages(std::vector<string>& fileNames, bool onlySelected=false);

  /**
  \brief Main function that estimates recurrence on the displayed test subject
  param outputdirectory The outputdirectory where recurrence map will be written
  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void StartRecurrenceEstimate(const std::string &outputdirectory,  bool cbT1Data,  bool cbT1ceData,  bool cbT2Data,  bool cbT2FlairData,  bool cbDTIData,  bool cbPerfData,  bool cbDistData);

  /**
  \brief Main function that estimates recurrence on test subjects by using an existing model
  param modeldirectory The directory where model related files are stored
  param inputdirectory The directory where test data is stored
  param outputdirectory The directory where output data will be stored

  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void RecurrenceEstimateOnExistingModel(const std::string &modeldirectory, const std::string &inputdirectory, const std::string &outputdirectory, bool &cbT1Data,  bool &cbT1ceData,  bool &cbT2Data,  bool &cbT2FlairData,  bool &cbDTIData,  bool &cbPerfData,  bool &cbDistData);

  /**
  \brief Main function that trains a model on many training subjects
  param directory The directory that contains training data
  param outputdirectory The directory where trainined model related data will be stored
  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void TrainNewModelOnGivenData(const std::string &directory, const std::string &outputdirectory,  bool &cbT1Data,  bool &cbT1ceData,  bool &cbT2Data,  bool &cbT2FlairData,  bool &cbDTIData,  bool &cbPerfData,  bool &cbDistData);

  void CallForSurvivalPredictionOnExistingModelFromMain(const std::string modeldirectory, const std::string inputdirectory, const std::string outputdirectory);
  void CallForNewSurvivalPredictionModelFromMain(const std::string inputdirectory, const std::string outputdirectory);

  /**
  \brief Function that updates the co-ordicates (X and Y) of border
  param startX starting X co-ordinate
  param startY starting Y co-ordinate
  param endX ending X co-ordinate
  param endY ending Y co-ordinate
  */
  void UpdateBorderWidget(double startX, double startY, double endX, double endY);

  /**
  \brief Function that updates the co-ordicates (Z) of border
  param startX starting Z co-ordinate
  param startY starting Z co-ordinate
  */
  void UpdateBorderWidget(double startZ, double endZ);

  /**
  \brief Adds actions in the action log
  param points The voxels corresponding to action
  param actionid The specific id of the action
  */
  void UpdateActionQ(const QVariantList& list)
  {
	  vector<PointVal> points;
	  for (size_t i = 0; i < list.size(); i++)
	  {
		  points.push_back(qvariant_cast<PointVal>(list[i]));
	  }
	  UpdateAction(points);
  }
  void UpdateAction(vector<PointVal> points = vector<PointVal>());

  /**
  \brief Brief description about the software
  */
  void about();

  /**
  \brief Help for interactions
  */
  void help_Interactions();

  /**
  \brief Help for interactions
  */
  void help_Applications();



  /**
  \brief Open Nifti image functionality. Shows dialog to select nifti files
  */
  void openImages(QStringList files = QStringList());

  /**
  \brief Function called when the sliders of axial view is changed
  */
  void AxialViewSliderChanged();
  
  /**
  \brief Function called when the sliders of coronal view is changed
  */
  void CoronalViewSliderChanged();
  
  /**
  \brief Function called when the sliders of saggital view is changed
  */
  void SaggitalViewSliderChanged();
  
  /**
  \brief Closing viewing image by pressing X in front of the image
  param item The current selected item of the table
  */
  void CloseImage(QTableWidgetItem* item);

  /**
  \brief Closing non-viewing image by pressing X in front of the image
  param item The current selected item of the table
  */
  void CloseNonViewingDTIImage(QTableWidgetItem* item);


  void clearMask(int label=-1);

  void makeStroke(vector<itk::Image<short, 3>::IndexType>& indices, const int value);
  
  int getSelectedDrawLabel()
  {
    return drawingPanel->getSelectedDrawLabel();
  }
  int getSelectedDrawSize()
  {
    return drawingPanel->getSelectedDrawSize();
  }
  
  /**
  \brief Save near/far drawing in Nifti format
  */
  void SaveDrawing();
  
  /**
  \brief Save near/far drawing in DICOM format
  */
  void SaveDicomDrawing();
  
  /**
  \brief Save initial seed drawing in Nifti format
  */
  void SaveSeedDrawing();

  /**
  \brief Load seed drawing from a Nifti file
  */
  void LoadSeedDrawing();

  /**
  \brief Load near/far drawing from a Nifti file
  */
  void LoadDrawing();

  /**
  \brief Load near/far drawing from a DICOM file
  */
  void LoadDicomDrawing();

  /**
  \brief Combines near and far drawn points in one long vector to be used for edema segmentation based on region growing
  */
  VectorVectorDouble FormulateDrawingPointsForEdemaSegmentation();

  /**
  \brief Puts initial seed points in one vector to be used for tumor segmentation 
  */
  VectorVectorDouble FormulateDrawingPointsForTumorSegmentation();



  /**
  \brief Save the current selected Nifti image
  */
  void SaveImage();
  
  /**
  \brief Save the current selected DICOM image
  */
  void SaveDicomImage();

  /**
  \brief Get indices of particular label
  */
  VectorVectorDouble GetMaskLabelIndices(const int label);
  
  /**
  \brief Read ROI from a file into the memory
  */
  void readMaskFile(const std::string &maskFileName);

  /**
  \brief Change the size of drawing/erasing brush
  */
  void ChangeBrushSize(int size);

  /**
  \brief Changes the label in the drawing
  */
  void ChangeDrawingLabel(int drawingLabel); // multiLabel uncomment this function

  /**
  \brief Changes the opacity in the drawing
  */
  void ChangeMaskOpacity(int newMaskOpacity); // multiLabel uncomment this function

  /**
  \brief Checks whether required images are present for the recurrence estimation application
  */
  bool CheckCompletenessOfInputData(bool &t1DataPresent, bool &t2DataPresent, bool &t1ceDataPresent, bool &t2flairDataPresent, bool &perfusionDataPresent, bool &dtiDataPresent);
  /**
  \brief Checks whether required images are present for the EGFRvIII estimation application
  */
  bool CheckCompletenessOfInputDataForEGFR(bool &t1ceDataPresent, bool &t2flairDataPresent, bool &perfusionDataPresent);

  /**
  \brief Sets current tissue type to be drawn
  */
  void SetActiveLandmarksType(int type, int row, int col);

  /**
  \brief Sets current landmark type based on the current selected tab
  */
  void SetActiveLandmarkTypeTab(int current);

  void propogateSlicerPosition(int slicerId =0,int imageId=-1);


  /**
  \brief Sets the current selected image
  \param id The id of the current selected image
  */
  void CurrentImageChanged(std::string &id);

  /**
  \brief Updates the information in the bottom panel based on the current selected image
  */
  void ImageInfoChanged();

  /**
  \brief Close the current selected image
  */
  void CloseImage();

  /**
  \brief Close all loaded images
  */
  void CloseAllImages();


  
  /**
  \brief Reset the number of points in the table when all the images are closed
  */
  void ResetNumberOfPoints();

  /**
  \brief This function deals with undo  
  */
  void UndoFunctionality();



  /**
  \brief Sets the opacity of the mask for the displayed images
  */
  void SetOpacity();

  /**
  \brief This function deals with the functionality associated with the change in the overlay slider
  */
  void overlaySliderChanged(int value);

  /**
  \brief This function deals with the functionality associated with the change in the overlay slider
  */
  void imageModalityChanged(int value);

  /**
  \brief This function deals with the functionality associated with the change in the image slider for the perfusion image
  */
  void imageSliderChanged();

  /**
  \brief Resets the transfrmation of each slicer to identity, resets the camera and renders the slicer again
  */
  void ResetTransformationToIdentity();

  /**
  \brief Renders all the images again
  */
  void UpdateRenderWindows();

  /**
  \brief Enable or disable overlay
  \param State of the overlay to set (true/false)
  */
  void overlayUseStateChanged(int state);

  /**
  \brief Passes the item, whose overlay needs to be update, to overlayChanged(QTableWidgetItem *item) function
  */
  void overlayChanged();

  /**
  \brief Sets the item selected in overlay table as overlay on item selected in images table
  */
  void overlayChanged(QTableWidgetItem *item);

  /**
  \brief Maintains record of currentlty displayed image
  */
  void CurrentPickedImageChanged(std::string id);

  /**
  \brief Displays the image currently selected in images table. Internally calls DisplayChanged(QTableWidgetItem *item) function
  */
  void DisplayChanged();

  /**
  \brief Displays the image (item) currently selected in images table
  */
  void DisplayChanged(QTableWidgetItem *item);

  /**
  \brief Updates co-ordinates based on each change in the mouse position
  \param visibity This param shows whether coordinates need to be displayed or not
  \param x LPS/RAS co-ordinate X
  \param y LPS/RAS co-ordinate Y
  \param z LPS/RAS co-ordinate Z
  \param X pixel co-ordiante X
  \param Y pixel co-ordiante Y
  \param Z pixel co-ordiante Z
  \param value Intensity value at current voxel
  */
  void MousePositionChanged(int visibility, double x, double y, double z, double X, double Y, double Z, double value);


  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void WindowLevelChanged();

  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void WindowLevelEdited();
  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void SetWindowLevel(double w, double l);
  /**
  \brief Applies the window and level values selected from spin boxes on currently displayed images
  */
  void UpdateWindowLevel();

  /**
  \brief Applies the value of threshold slider on the displayed images
  */
  void thresholdSliderChanged();

  /**
  \brief Enable image thresholding using the radio button
  */
  void EnableThresholdOfImage();

  /**
  \brief Enable mask thresholding using the radio button
  */
  void EnableThresholdOfMask();

  /**
  \brief Creates a link between two currently displayed images
  \param image1 ID of the first image
  \param image2 ID of the second image
  */
  void AddLink(const std::string &image1, const std::string &image2);

  /**
  \brief Removes link between two images
  \param image1 ID of the first image
  \param image2 ID of the second image
  */
  void RemoveLink(const std::string &image1, const std::string &image2);
  /**
  \brief Sets the value of the axial, coronal or sagital sliders
  \param slicer Number of the slicer
  \param slice Value to bet set on the slider
  */
  void UpdateSlice(int slicer, int slice);
  /**
  \brief Sets the minimum and maximum values of axial, coronal or sagital sliders
  \param slicer Number of the slicer
  \param min Minimum value of the slider
  \param max Maximum value of the slider
  */
  void UpdateSliceRange(int slicer, int min, int max);

  /**
  \brief Moves the cursor on the given co-ordinates
  */
  void MoveSlicerCursor(double x, double y, double z, int mode = 0);


  /**
  \brief Displays the next image in order based on keyboard input (1,2,3,...)
  */
  void ChangeImageWithOrder(SlicerManager *sm, int order);

  /**
  \brief Called internally within DisplayChange function to update link between images
  */
  void UpdateLinkManager(std::string &id, int slicer, double x, double y, double z);

  /**
  \brief Called internally within DisplayChange function to update link between images
  */
  void UpdateLinkedNavigation( Slicer* refSlicer);

  void toolTabDockChanged(bool bUnDocked)//TBD - move to cpp file 
  {
	  if (bUnDocked)
	  {
		  
		  m_tabWidget->setMaximumHeight(m_tabWidget->minimumHeight() * 10);
		  m_toolTabdock->show();
	  }
	  else
	  {
		  m_tabWidget->setMaximumHeight(m_tabWidget->minimumHeight());
	  }
  }

  void AllPythonGUI(const std::string &appName); //! call for all python GUI applications
  void PyGUILIBRA();
  void PyGUIConfetti();
  void PyGUIWhiteStripe();
  void PyCLISBRT_Segment();
  void PyCLISBRT_Analyze();
  void AllPythonCLI(); //! call for all python CLI applications

  /**
  \brief Construct a dialog for the python CLI application using the script and config file
  */
  inline void PythonCLIToDialogAndRun(const std::string &scriptFile, const std::string &configFile);
  
  /**
  \brief Search for the Python script in many locations WRT to the running graphical layer executable
  */
  inline bool PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, std::string &scriptWithPath);

  /**
  \brief Search for the Python script and its matchign the config file in many locations WRT to the running graphical layer executable
  */
  inline bool PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, std::string &scriptWithPath, std::string configFileWithPath);

  void ApplicationEGFR();
  void ApplicationRecurrence();
  void ApplicationSurvival();
  void ApplicationGeodesic();
  void ApplicationITKSNAP();
  void ImageDenoising();
  void ImageBiasCorrection();
  void ImageRegistration();
  void CustomPreprocessing();

  void Registration(std::string &fixedfilename, std::vector<std::string> &inputFileNames, std::vector<std::string> &outputFileNames);

  //confirm before exit
  void closeEvent(QCloseEvent * event);

  // Progress Update
  void updateProgress(int progress, string message = "",int max=100 );

public:

  std::string currentPlatform;

  std::string tempFolderLocation;

  std::vector<SlicerManager*> mSlicerManagers;
  std::vector<SimpleImageManager*> mNonViewingImageManager;

  //QWidget* mMainWidget;
  QString mInputPathName;
  std::vector<QSlider*> verticalSliders;
  //
  std::string mCurrentSelectedImageId;
  std::string mCurrentPickedImageId;
  int mCurrentPickedImageIndex;

  //progress Bar
  QProgressBar *m_progressBar;
  QLabel* m_messageLabel;

  //
  Landmarks* mLandmarks;
  Landmarks* mSeedPoints;
  Landmarks* mTissuePoints;
  int mCurrentLandmarkTissueType;
  vtkSmartPointer<vtkImageData> mMask;

  SHAPE_MODE m_drawShapeMode;
  int mCurrentNearPoints;
  int mCurrentFarPoints;
  int mCurrentInitPoints;


  int mBorderStartX;
  int mBorderStartY;
  int mBorderStartZ;

  int mBorderEndX;
  int mBorderEndY;
  int mBorderEndZ;


  PreprocessingPipelineClass mPreprocessingObj;
  OutputWritingManager		mOutputManager;
  NiftiDataManager			 mNifiDataManager;

  //Applications
  EGFRStatusPredictor mEGFRPredictor;
  RecurrenceEstimator mRecurrenceEstimator;
  SurvivalPredictor mSurvivalPredictor;
  GeodesicSegmentation mGeodesicSegmentation;

  std::string t1cePath;
  std::string m_imagetype_string;
  Ui::fMainWindow	m_mainWindowUI;

  std::vector<int> mActionIds;
  std::vector<int> mActionSequenceIds;
  
  vector< vector<PointVal> > mActionPoints;
  
  GenericImage::Pointer mCustomImageToThreshold;

  int mSequenceNumber, mCustomImageToThreshold_min, mCustomImageToThreshold_max;

  bool mthresholdImage, mthresholdMask;

  private:

    struct DicomDictTagAndVal
    {
      std::string tag;
      std::string value;

      DicomDictTagAndVal(const std::string &input_tag, const std::string &input_value) :
        tag(input_tag), value(input_value)
      { }

      DicomDictTagAndVal(const std::string &input_tag) :
      tag(input_tag)
      { }

      void SetValue(const std::string &input_value)
      {
        value = input_value;
      }
    };

};
//-------------------------------------------------------------------------------------

#endif
