/**
\file  fMainWindow.h

\brief Declaration of fMainWindow class

https://www.cbica.upenn.edu/sbia/software/ <br>
software@cbica.upenn.edu

Copyright (c) 2016 University of Pennsylvania. All rights reserved. <br>
See COPYING file or https://www.cbica.upenn.edu/sbia/software/license.html

*/

#pragma once


#ifndef _fMainWindow_h_
#define _fMainWindow_h_


#include "CAPTk.h"
#include "Slicer.h"
#include "SlicerManager.h"
#include "NiftiDataManager.h"
#include "EGFRStatusPredictor.h"
#include "RecurrenceEstimator.h"
#include "ui_fMainWindow.h"
#include "InteractorStyleNavigator.h"
#include "QTablePushButton.h"
#include "SimpleImageManager.h"
#include "OutputWritingManager.h"
#include <PreprocessingPipelineClass.h>
#include <FeatureExtractionClass.h>
#include <FeatureReductionClass.h>
#include <FeatureScalingClass.h>
#include <SVMClassificationClass.h>
#include <GeodesicSegmentation.h>

#define USE_PROCESSDIALOG

enum TAB_TYPE
{
  TAB_IMAGES, TAB_TUMOR, TAB_LANDMARKS
};

typedef itk::Image< short, 3 > GenericImage;



enum DRAW_MODE
{
  DRAW_MODE_NORMAL, DRAW_MODE_NEAR, DRAW_MODE_FAR, DRAW_MODE_SEED, DRAW_MODE_ERASE
};

/**
\class fMainWindow

\brief This is the main UI class for CapTk
*/
class fMainWindow : public QMainWindow, private Ui::fMainWindow
{
  Q_OBJECT

public:
  //! Default constructor
  fMainWindow();

  //! Default destructor
  ~fMainWindow();
  
  QTableWidget * m_imagesTable;
  QTableWidget * m_nonVisImagesTable;


  /**
  \brief Load images into memory
  
  \param filenames Vector of image filenames which need to be loaded (comes from "Load" dialog)
  \param imagetype_int Either NIfTI or DICOM
  \param imagesubtype Modality of the image (T1, T2, ...)
  \param bSkipDup Skip duplicates, defaults to true
  */
  void LoadSlicerImages(const std::vector<std::string> &directories, const int &imagetype_int, int imagesubtype, const int &bSkipDup = true);

  void LoadSlicerImages(const std::string &singleFile, const int &imagetype_int, int imagesubtype, const int &bSkipDup = true);

  /**
  \brief Load non-viewing images into memory
  
  \param directoryname Directory name in which the non-viewing images are present
  \param imagetype_int Either NIfTI or DICOM
  \param imagesubtype Modality of the image (T1, T2, ...)
  */
  void LoadNonViewingImages(const std::string &directoryname, const int &imagetype_int, const int &imagesubtype);

  /**
  \brief Initialize drawing mask on the image
  */
  void InitMask(vtkImageData* image);

  /**
  \brief Sets the drawing mode to either drawing or viewing

  \param mode Drawing mode (modes are DRAW_MODE enum)
  \param size Size of the pointer in drawing mode
  */
  void SetDrawMode(const int &mode, const int &size);

  /**
  \brief Initializer for the entire rendering pipeline
  */
  void InitDisplay();

  /**
  \brief Initializes difference slicer views of a single image
  */
  void InitSlicers();

  /**
  \brief Render the sliders to adjust the view of the image slices.

  Initializes it at the middle of the whole image.

  \param slicer If many images are present, this keeps a count of which image is being rendered
  \param window The window number (goes from 1 to 3)
  */
  void DisplaySliders(int slicer, int window);

  /**
  \brief Get image index from the visualization table
  */
  int GetSlicerIndexFromItem(QTableWidgetItem* item);

  /**
  \brief Get the corresponding table item from the image
  */
  QTableWidgetItem* GetItemFromSlicerManager(SlicerManager* sm);
  
  /**
  \brief Construct near and far indeces from the initialized mask
  */
  template<class ImageType>
  void FormulateNearFarPoints(std::vector<typename ImageType::IndexType> &nearIndices, std::vector<typename ImageType::IndexType> &farIndices);

  /**
  \brief Rescales image intensity in the range of 0-255
  */
  ImageTypeFloat3D::Pointer RescaleImageIntensity(ImageTypeFloat3D::Pointer image);

  /*
  \brief Drag event initialization
  */
  void dragEnterEvent(QDragEnterEvent *event);

  /*
  \brief Drop Event parsing
  */
  void dropEvent(QDropEvent *event);

  /*
  \brief For fast developer testing .TBD:  remove or disable.
  */
  void runDevTestMode( QString file = "" )
  {
    QStringList lst(file);
	  this->OpenNiftiImages(lst);
	  
  }

signals:
  void SelectedImageHasChanged(SlicerManager *);
  void LandmarksFocused(bool bFocused);
  void SeedPointsFocused(bool bFocused);
  void TissuePointsFocused(bool bFocused);

  public slots:
  /**
  \brief Updates number of near and far points in the table
  */
  void UpdateNumberOfPointsInTable();

  /**
  \brief Initilializes preset combobox with the default options
  */
  void SetPresetComboBox();

  /**
  \brief Main function that performs EGFRvIII estimation on the displayed image
  */
  void StartEGFREstimate();

  /**
  \brief Main function that estimates recurrence on the displayed test subject
  param outputdirectory The outputdirectory where recurrence map will be written
  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void StartRecurrenceEstimate(const std::string &outputdirectory,  bool cbT1Data,  bool cbT1ceData,  bool cbT2Data,  bool cbT2FlairData,  bool cbDTIData,  bool cbPerfData,  bool cbDistData);

  /**
  \brief Main function that estimates recurrence on test subjects by using an existing model
  param modeldirectory The directory where model related files are stored
  param inputdirectory The directory where test data is stored
  param outputdirectory The directory where output data will be stored

  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void RecurrenceEstimateOnExistingModel(const std::string &modeldirectory, const std::string &inputdirectory, const std::string &outputdirectory, bool &cbT1Data,  bool &cbT1ceData,  bool &cbT2Data,  bool &cbT2FlairData,  bool &cbDTIData,  bool &cbPerfData,  bool &cbDistData);

  /**
  \brief Main function that trains a model on many training subjects
  param directory The directory that contains training data
  param outputdirectory The directory where trainined model related data will be stored
  param cbT1Data Whether T1 data is present or not
  param cbT1ceData Whether T1CE data is present or not
  param cbT2Data Whether T2 data is present or not
  param cbT2FlairData Whether T2-Flair data is present or not
  param cbDTIData Whether DTI data is present or not
  param cbPerfData Whether Perfusion data is present or not
  param cbDistData Whether Distance feature need to be used or not
  */
  void TrainNewModelOnGivenData(const std::string &directory, const std::string &outputdirectory,  bool &cbT1Data,  bool &cbT1ceData,  bool &cbT2Data,  bool &cbT2FlairData,  bool &cbDTIData,  bool &cbPerfData,  bool &cbDistData);

  /**
  \brief Function that updates the co-ordicates (X and Y) of border
  param startX starting X co-ordinate
  param startY starting Y co-ordinate
  param endX ending X co-ordinate
  param endY ending Y co-ordinate
  */
  void UpdateBorderWidget(double startX, double startY, double endX, double endY);

  /**
  \brief Function that updates the co-ordicates (Z) of border
  param startX starting Z co-ordinate
  param startY starting Z co-ordinate
  */
  void UpdateBorderWidget(double startZ, double endZ);

  /**
  \brief Adds actions in the action log
  param points The voxels corresponding to action
  param actionid The specific id of the action
  */
  void UpdateAction(VectorVectorDouble &points, int actionid);

  /**
  \brief Brief description about the software
  */
  void about();

  /**
  \brief Help section
  */
  void help();

  /**
  \brief Open image functionality
  param type Image type i.e. Recurrence output
  param directoryname Name of the directory where dicom data is stored
  */
  void OpenImages(int type, std::vector<std::string> &directoryname);

  /**
  \brief Open recurrence image functionality
  */
  void OpenRecurrenceImages();

  /**
  \brief Open Dicom image functionality. Shows dialog to select directories
  */
  void OpenDicomImages();

  /**
  \brief Open Nifti image functionality. Shows dialog to select nifti files
  */
  void OpenNiftiImages(QStringList files = QStringList());

  /**
  \brief Function called when the sliders of axial view is changed
  */
  void AxialViewSliderChanged();
  
  /**
  \brief Function called when the sliders of coronal view is changed
  */
  void CoronalViewSliderChanged();
  
  /**
  \brief Function called when the sliders of saggital view is changed
  */
  void SaggitalViewSliderChanged();
  
  /**
  \brief Closing viewing image by pressing X in front of the image
  param item The current selected item of the table
  */
  void CloseImage(QTableWidgetItem* item);

  /**
  \brief Closing non-viewing image by pressing X in front of the image
  param item The current selected item of the table
  */
  void CloseNonViewingDTIImage(QTableWidgetItem* item);

  /**
  \brief Erasing seed mask drawing
  */
  void EraseInitDrawing();
  
  /**
  \brief Erasing near region drawing
  */
  void EraseNearDrawing();
  
  /**
  \brief Erasing far region drawing
  */
  void EraseFarDrawing();
    
  /**
  \brief Change the mode to near region drawing mode
  */
  void EnableNearRegionDrawing();
  
  /**
  \brief Change the mode to far region drawing mode
  */
  void EnableFarRegionDrawing();

  /**
  \brief Change the mode to initial seed drawing mode
  */
  void EnableInitDrawing();
  
  /**
  \brief Save near/far drawing in Nifti format
  */
  void SaveDrawing();
  
  /**
  \brief Save near/far drawing in DICOM format
  */
  void SaveDicomDrawing();
  
  /**
  \brief Save initial seed drawing in Nifti format
  */
  void SaveSeedDrawing();

  /**
  \brief Load seed drawing from a Nifti file
  */
  void LoadSeedDrawing();

  /**
  \brief Load near/far drawing from a Nifti file
  */
  void LoadDrawing();

  /**
  \brief Load near/far drawing from a DICOM file
  */
  void LoadDicomDrawing();

  /**
  \brief Combines near and far drawn points in one long vector to be used for edema segmentation based on region growing
  */
  VectorVectorDouble FormulateDrawingPointsForEdemaSegmentation();

  /**
  \brief Puts initial seed points in one vector to be used for tumor segmentation 
  */
  VectorVectorDouble FormulateDrawingPointsForTumorSegmentation();



  /**
  \brief Save the current selected Nifti image
  */
  void SaveImage();
  
  /**
  \brief Save the current selected DICOM image
  */
  void SaveDicomImage();

  /**
  \brief Get indices of the far region in one vector
  */
  VectorVectorDouble GetFarRegionIndices();
  
  /**
  \brief Get indices of the near region in one vector
  */
  VectorVectorDouble GetNearRegionIndices();

  /**
  \brief Convert the system from drawing mode to eraser mode
  */
  void SetDrawingModeToEraserMode();

  /**
  \brief Change the size of drawing/erasing brush
  */
  void ChangeBrushSize(int size);

  /**
  \brief Checks whether required images are present for the recurrence estimation application
  */
  bool CheckCompletenessOfInputData(bool &t1DataPresent, bool &t2DataPresent, bool &t1ceDataPresent, bool &t2flairDataPresent, bool &perfusionDataPresent, bool &dtiDataPresent);
  /**
  \brief Checks whether required images are present for the EGFRvIII estimation application
  */
  bool CheckCompletenessOfInputDataForEGFR(bool &t1ceDataPresent, bool &t2flairDataPresent, bool &perfusionDataPresent);

  /**
  \brief Sets current tissue type to be drawn
  */
  void SetActiveLandmarksType(int type, int row, int col);

  /**
  \brief Sets current landmark type based on the current selected tab
  */
  void SetActiveLandmarkTypeTab(int current);

  void propogateSlicerPosition(int slicerId =0,int imageId=-1);


  /**
  \brief Sets the current selected image
  \param id The id of the current selected image
  */
  void CurrentImageChanged(std::string &id);

  /**
  \brief Updates the information in the bottom panel based on the current selected image
  */
  void ImageInfoChanged();

  /**
  \brief Close the current selected image
  */
  void CloseImage();

  /**
  \brief Close all loaded images
  */
  void CloseAllImages();

  /**
  \brief Converts the interface to normal mode from the drawing mode
  */
  void EscapeMode();
  
  /**
  \brief Reset the number of points in the table when all the images are closed
  */
  void ResetNumberOfPoints();

  /**
  \brief This function deals with undo  
  */
  void UndoFunctionality();

  /**
  \brief Sets the opacity of the mask for the displayed images
  */
  void SetOpacity();

  /**
  \brief This function deals with the functionality associated with the change in the overlay slider
  */
  void overlaySliderChanged(int value);

  /**
  \brief This function deals with the functionality associated with the change in the image slider for the perfusion image
  */
  void imageSliderChanged();

  /**
  \brief Resets the transfrmation of each slicer to identity, resets the camera and renders the slicer again
  */
  void ResetTransformationToIdentity();

  /**
  \brief Renders all the images again
  */
  void UpdateRenderWindows();

  /**
  \brief Enable or disable overlay
  \param State of the overlay to set (true/false)
  */
  void overlayUseStateChanged(int state);

  /**
  \brief Passes the item, whose overlay needs to be update, to overlayChanged(QTableWidgetItem *item) function
  */
  void overlayChanged();

  /**
  \brief Sets the item selected in overlay table as overlay on item selected in images table
  */
  void overlayChanged(QTableWidgetItem *item);

  /**
  \brief Maintains record of currentlty displayed image
  */
  void CurrentPickedImageChanged(std::string id);

  /**
  \brief Displays the image currently selected in images table. Internally calls DisplayChanged(QTableWidgetItem *item) function
  */
  void DisplayChanged();

  /**
  \brief Displays the image (item) currently selected in images table
  */
  void DisplayChanged(QTableWidgetItem *item);

  /**
  \brief Updates co-ordinates based on each change in the mouse position
  \param visibity This param shows whether coordinates need to be displayed or not
  \param x LPS/RAS co-ordinate X
  \param y LPS/RAS co-ordinate Y
  \param z LPS/RAS co-ordinate Z
  \param X pixel co-ordiante X
  \param Y pixel co-ordiante Y
  \param Z pixel co-ordiante Z
  \param value Intensity value at current voxel
  */
  void MousePositionChanged(int visibility, double x, double y, double z, double X, double Y, double Z, double value);


  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void WindowLevelChanged();

  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void WindowLevelEdited();
  /**
  \brief Sets the window and level values. Calls UpdateWindowLevel function internally
  */
  void SetWindowLevel(double w, double l);
  /**
  \brief Applies the window and level values selected from spin boxes on currently displayed images
  */
  void UpdateWindowLevel();

  /**
  \brief Applies the value of threshold slider on the displayed images
  */
  void thresholdSliderChanged();

  /**
  \brief Enable image thresholding using the radio button
  */
  void EnableThresholdOfImage();

  /**
  \brief Enable mask thresholding using the radio button
  */
  void EnableThresholdOfMask();

  /**
  \brief Creates a link between two currently displayed images
  \param image1 ID of the first image
  \param image2 ID of the second image
  */
  void AddLink(const std::string &image1, const std::string &image2);

  /**
  \brief Removes link between two images
  \param image1 ID of the first image
  \param image2 ID of the second image
  */
  void RemoveLink(const std::string &image1, const std::string &image2);
  /**
  \brief Sets the value of the axial, coronal or sagital sliders
  \param slicer Number of the slicer
  \param slice Value to bet set on the slider
  */
  void UpdateSlice(int slicer, int slice);
  /**
  \brief Sets the minimum and maximum values of axial, coronal or sagital sliders
  \param slicer Number of the slicer
  \param min Minimum value of the slider
  \param max Maximum value of the slider
  */
  void UpdateSliceRange(int slicer, int min, int max);

  /**
  \brief Moves the cursor on the given co-ordinates
  */
  void MoveSlicerCursor(double x, double y, double z, int mode = 0);


  /**
  \brief Displays the next image in order based on keyboard input (1,2,3,...)
  */
  void ChangeImageWithOrder(SlicerManager *sm, int order);

  /**
  \brief Called internally within DisplayChange function to update link between images
  */
  void UpdateLinkManager(std::string &id, int slicer, double x, double y, double z);

  /**
  \brief Called internally within DisplayChange function to update link between images
  */
  void UpdateLinkedNavigation(std::string id, SlicerManager *sm, Slicer* refSlicer);

  void toolTabDockChanged(bool bUnDocked)//TBD - move to cpp file 
  {
	  if (bUnDocked)
	  {
		  
		  m_tabWidget->setMaximumHeight(m_tabWidget->minimumHeight() * 10);
		  m_toolTabdock->show();
	  }
	  else
	  {
		  m_tabWidget->setMaximumHeight(m_tabWidget->minimumHeight());
	  }
  }

  void AllPythonGUI(); //! call for all python GUI applications
  void AllPythonCLI(); //! call for all python CLI applications

  /**
  \brief Construct a dialog for the python CLI application using the script and config file
  */
  inline void PythonCLIToDialogAndRun(const std::string &scriptFile, const std::string &configFile);
  
  /**
  \brief Search for the Python script in many locations WRT to the running graphical layer executable
  */
  inline bool PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, std::string &scriptWithPath);

  /**
  \brief Search for the Python script and its matchign the config file in many locations WRT to the running graphical layer executable
  */
  inline bool PreparePythonScriptAndConfig(const std::string &NameToCheck, const std::string &AppName, std::string &scriptWithPath, std::string configFileWithPath);

  void ApplicationEGFR();
  void ApplicationRecurrence();
  void ApplicationGeodesic();
  void ImageDenoising();
  void ImageBiasCorrection();
  void ImageRegistration();
  void CustomPreprocessing();

  void Registration(std::string &fixedfilename, std::vector<std::string> &inputFileNames, std::vector<std::string> &outputFileNames);

  //confirm before exit
  void closeEvent(QCloseEvent * event);

  // Progress Update
  void updateProgress(int progress, string message = "",int max=100 );

public:

  std::vector<SlicerManager*> mSlicerManagers;
  std::vector<SimpleImageManager*> mNonViewingImageManager;

  //QWidget* mMainWidget;
  QString mInputPathName;
  std::vector<QSlider*> verticalSliders;
  //
  std::string mCurrentSelectedImageId;
  std::string mCurrentPickedImageId;
  int mCurrentPickedImageIndex;

  //progress Bar
  QProgressBar *m_progressBar;
  QLabel* m_messageLabel;

  //
  Landmarks* mLandmarks;
  Landmarks* mSeedPoints;
  Landmarks* mTissuePoints;
  vtkSmartPointer<vtkImageData> mMask;
  //
  int mDrawMode;
  int mDrawSize;
  int mCurrentNearPoints;
  int mCurrentFarPoints;
  int mCurrentInitPoints;


  int mBorderStartX;
  int mBorderStartY;
  int mBorderStartZ;

  int mBorderEndX;
  int mBorderEndY;
  int mBorderEndZ;


  PreprocessingPipelineClass	* mPreprocessingObj;
  OutputWritingManager		* mOutputManager;
  NiftiDataManager			* mNifiDataManager;

  //Applications
  EGFRStatusPredictor* mEGFRPredictor;
  RecurrenceEstimator* mRecurrenceEstimator;
  GeodesicSegmentation* mGeodesicSegmentation;

  std::string t1cePath;
  std::string m_imagetype_string;
  Ui::fMainWindow	m_mainWindowUI;

  std::vector<int> mActionIds;
  std::vector<int> mActionSequenceIds;
  std::vector<VectorVectorDouble> mActionPoints;

  GenericImage::Pointer mCustomImageToThreshold;

  int mSequenceNumber, mCustomImageToThreshold_min, mCustomImageToThreshold_max;

  bool mthresholdImage, mthresholdMask;

  private:

    struct DicomDictTagAndVal
    {
      std::string tag;
      std::string value;

      DicomDictTagAndVal(const std::string &input_tag, const std::string &input_value) :
        tag(input_tag), value(input_value)
      { }

      DicomDictTagAndVal(const std::string &input_tag) :
      tag(input_tag)
      { }

      void SetValue(const std::string &input_value)
      {
        value = input_value;
      }
    };

};
//-------------------------------------------------------------------------------------

#endif
