#ifndef _SurvivalPredictor_h_
#define _SurvivalPredictor_h_

#include "CAPTk.h"
#include "NiftiDataManager.h"
#include "OutputWritingManager.h"
#include "FeatureReductionClass.h"
#include "FeatureScalingClass.h"
#include "FeatureExtractionClass.h"
#include "SVMClassificationClass.h"
#include "fProgressDialog.h"
#include "ApplicationBase.h"
#include "itkCSVArray2DFileReader.h"
#include "itkCSVNumericObjectFileWriter.h"
#include "itkConnectedComponentAlgorithm.h"
#include "itkConnectedComponentImageFilter.h"
#include "itkSignedMaurerDistanceMapImageFilter.h"

class SurvivalPredictor : public ApplicationBase
{
public:
  //! Default constructor
  SurvivalPredictor()
  {
    //mFeatureExtractionLocalPtr = new FeatureExtractionClass();
    //mFeatureScalingLocalPtr = new FeatureScalingClass();
    mSixTrainedFile= "SixModelFile.xml";
    mEighteenTrainedFile = "EighteenModelFile.xml";
  }

  //! Default destructor
  ~SurvivalPredictor();

  std::string mLastEncounteredError;

  std::string mEighteenTrainedFile, mSixTrainedFile;

  FeatureExtractionClass mFeatureExtractionLocalPtr;
  FeatureScalingClass mFeatureScalingLocalPtr;

  
  void LoadQualifiedSubjectsFromGivenDirectory(std::string directoryname,
  std::vector<std::string> & qualifiedSubjectNames,
  std::vector<std::string> & t1FileNames,
  std::vector<std::string> & t1ceFileNames,
  std::vector<std::string> & t2FileNames,
  std::vector<std::string> & t2FlairFileNames,
  std::vector<std::string> & axFileNames,
  std::vector<std::string> & faFileNames,
  std::vector<std::string> & radFileNames,
  std::vector<std::string> & trFileNames,
  std::vector<std::string> & rcbvFileNames,
  std::vector<std::string> & psrFileNames,
  std::vector<std::string> & phFileNames,
  std::vector<std::string> & labelNames,
  std::vector<std::string> & rejectedSubjectNames);
  
  template<class ImageType>
  std::vector<double>  LoadTestData(typename ImageType::Pointer t1ceImagePointer,
    typename ImageType::Pointer t2flairImagePointer,
    typename ImageType::Pointer t1ImagePointer,
    typename ImageType::Pointer t2ImagePointer,
    typename ImageType::Pointer rcbvImagePointer,
    typename ImageType::Pointer psrImagePointer,
    typename ImageType::Pointer phImagePointer,
    typename ImageType::Pointer axImagePointer,
    typename ImageType::Pointer faImagePointer,
    typename ImageType::Pointer radImagePointer,
    typename ImageType::Pointer trImagePointer,
    typename ImageType::Pointer labelImagePointer);


  std::vector<double> GetStatisticalFeatures(std::vector<double> itensities);
  std::vector<double> GetHistogramFeatures(std::vector<double> intensities, int start, int interval, int end);
  std::vector<double> GetVolumetricFeatures(double edemaSize, double tuSize, double neSize, double totalsize);



  template <class ImageType>
  typename ImageType::Pointer ReadNiftiImage(std::string filename);


  template<class ImageType>
  std::vector<typename ImageType::Pointer> RevisedTumorArea(typename ImageType::Pointer labelImagePointer);

  template<class ImageType>
  typename ImageType::Pointer RescaleImageIntensity(typename ImageType::Pointer image);

  template<class ImageType>
  std::vector<double> GetDistanceFeatures(typename ImageType::Pointer,
    std::vector<typename ImageType::IndexType> edemaIndices,
    std::vector<typename ImageType::IndexType> tumorIndices,
    std::vector<typename ImageType::IndexType> ventIndices);

  template<class ImageType>
  std::vector<double> GetDistanceFeatures2(typename ImageType::Pointer labelImagePointer, std::vector<typename ImageType::IndexType> edemaIndices,
    std::vector<typename ImageType::IndexType> tumorIndices,
    std::vector<typename ImageType::IndexType> ventIndices);

  template<class ImageType>
  std::vector<double> GetDistanceFeatures3(typename ImageType::Pointer edemaImage, typename ImageType::Pointer tumorImage, typename ImageType::Pointer ventImage);


  template<class ImageType>
  typename ImageType::Pointer GetDistanceMap(typename ImageType::Pointer labelImagePointer,int label);


  void PrepareNewSurvivalPredictionModel(const std::string inputdirectory, const std::string outputdirectory);
  void SurvivalPredictionOnExistingModel(const std::string modeldirectory, const std::string inputdirectory, const std::string outputdirectory);


  void Run()
  {

  }
private:

};


template<class ImageType>
std::vector<double> SurvivalPredictor::GetDistanceFeatures(typename ImageType::Pointer labelImagePointer, std::vector<typename ImageType::IndexType> edemaIndices,
  std::vector<typename ImageType::IndexType> tumorIndices,
  std::vector<typename ImageType::IndexType> ventIndices)
{
  std::vector<double> DistanceFeatures;
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType imIt(labelImagePointer, labelImagePointer->GetLargestPossibleRegion());
  imIt.GoToBegin();
  std::vector<double> TumorVent;
  std::vector<double> EdemaVent;

  while (!imIt.IsAtEnd())
  {
    ImageType::IndexType index = imIt.GetIndex();

    std::vector<double> tumorDistances;
    std::vector<double> edemaDistances;
    std::vector<double> ventDistances;
    for (unsigned int j = 0; j < tumorIndices.size(); j++)
    {
      ImageType::IndexType tumorIndex = tumorIndices[j];
      double distance = std::sqrt(std::pow(index[0] - tumorIndex[0], 2) + std::pow(index[1] - tumorIndex[1], 2) + std::pow(index[2] - tumorIndex[2], 2));
      tumorDistances.push_back(distance);
    }
    for (unsigned int j = 0; j < edemaIndices.size(); j++)
    {
      ImageType::IndexType edemaIndex = edemaIndices[j];
      double distance = std::sqrt(std::pow(index[0] - edemaIndex[0], 2) + std::pow(index[1] - edemaIndex[1], 2) + std::pow(index[2] - edemaIndex[2], 2));
      edemaDistances.push_back(distance);
    }
    for (unsigned int j = 0; j < ventIndices.size(); j++)
    {
      ImageType::IndexType ventIndex = ventIndices[j];
      double distance = std::sqrt(std::pow(index[0] - ventIndex[0], 2) + std::pow(index[1] - ventIndex[1], 2) + std::pow(index[2] - ventIndex[2], 2));
      ventDistances.push_back(distance);
    }

    double min_distance_tumor = *std::min_element(std::begin(tumorDistances), std::end(tumorDistances));
    double min_distance_edema = *std::min_element(std::begin(edemaDistances), std::end(edemaDistances));
    double min_distance_vent = *std::min_element(std::begin(ventDistances), std::end(ventDistances));

    double TumorVentDistance = min_distance_tumor + min_distance_vent;
    if (TumorVentDistance > 0)
      TumorVent.push_back(TumorVentDistance);
    double EdemaVentDistance = min_distance_edema + min_distance_vent;
    if (EdemaVentDistance > 0)
      EdemaVent.push_back(EdemaVentDistance);

    ++imIt;
  }
  std::sort(TumorVent.begin(), TumorVent.end());
  std::sort(EdemaVent.begin(), EdemaVent.end());

  TumorVent.resize(100);
  EdemaVent.resize(100);

  double min_distance_tumor = *std::min_element(std::begin(TumorVent), std::end(TumorVent));
  double min_distance_edema = *std::min_element(std::begin(EdemaVent), std::end(EdemaVent));
  DistanceFeatures.push_back(min_distance_tumor);
  DistanceFeatures.push_back(min_distance_edema);
  /*



  std::vector<double> edemaDistances;

  for (int i = 0; i < ventIndices.size(); i++)
  {
  ImageType::IndexType ventIndex = ventIndices[i];
  for (unsigned int j = 0; j < tumorIndices.size(); j++)
  {
  ImageType::IndexType tumorIndex = tumorIndices[j];
  double distance = std::sqrt(std::pow(ventIndex[0] - tumorIndex[0], 2) + std::pow(ventIndex[1] - tumorIndex[1], 2) + std::pow(ventIndex[2] - tumorIndex[2], 2));
  tumorDistances.push_back(distance);
  }
  for (unsigned int j = 0; j < edemaIndices.size(); j++)
  {
  ImageType::IndexType edemaIndex = edemaIndices[j];
  double distance = std::sqrt(std::pow(ventIndex[0] - edemaIndex[0], 2) + std::pow(ventIndex[1] - edemaIndex[1], 2) + std::pow(ventIndex[2] - edemaIndex[2], 2));
  edemaDistances.push_back(distance);
  }
  }



  *///DistanceFeatures.push_back(tumorDistances[std::distance(std::begin(tumorDistances), result)]);

  //std::vector<int>::iterator result = std::min_element(std::begin(edemaDistances), std::end(edemaDistances));
  //DistanceFeatures.push_back(edemaDistances[std::distance(std::begin(edemaDistances), result)]);
  return DistanceFeatures;
}
template<class ImageType>
typename ImageType::Pointer SurvivalPredictor::RescaleImageIntensity(typename ImageType::Pointer image)
{
  typedef ImageType ImageType;
  typedef itk::RescaleIntensityImageFilter< ImageType, ImageType > RescaleFilterType;
  RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
  rescaleFilter->SetInput(image);
  rescaleFilter->SetOutputMinimum(0);
  rescaleFilter->SetOutputMaximum(255);
  rescaleFilter->Update();
  ImageType::Pointer outputimage = rescaleFilter->GetOutput();
  return outputimage;
}
template <class ImageType>
typename ImageType::Pointer SurvivalPredictor::ReadNiftiImage(std::string filename)
{
  typedef itk::ImageFileReader<ImageType> ImageReaderType;
  typename ImageReaderType::Pointer reader = ImageReaderType::New();
  reader->SetFileName(filename);

  try
  {
    reader->Update();
  }
  catch (itk::ExceptionObject& e)
  {
    std::cerr << "Error caught: " << e.what() << "\n";
    //cbica::Logging(loggerFile, "Error caught during testing: " + std::string(e.GetDescription()));
    exit(EXIT_FAILURE);
  }


  return reader->GetOutput();
}

template<class ImageType>
std::vector<double> SurvivalPredictor::GetDistanceFeatures2(typename ImageType::Pointer labelImagePointer, std::vector<typename ImageType::IndexType> edemaIndices,
  std::vector<typename ImageType::IndexType> tumorIndices,
  std::vector<typename ImageType::IndexType> ventIndices)
{
  std::vector<double> VentEdemaDistances;
  std::vector<double> VentTumorDistances;

  for (int i = 0; i < ventIndices.size(); i++)
  {
    ImageType::IndexType index = ventIndices[i];
    std::vector<double> tumorDistances;
    std::vector<double> edemaDistances;
    for (unsigned int j = 0; j < tumorIndices.size(); j++)
    {
      ImageType::IndexType tumorIndex = tumorIndices[j];
      double distance = std::sqrt(std::pow(index[0] - tumorIndex[0], 2) + std::pow(index[1] - tumorIndex[1], 2) + std::pow(index[2] - tumorIndex[2], 2));
      tumorDistances.push_back(distance);
    }
    for (unsigned int j = 0; j < edemaIndices.size(); j++)
    {
      ImageType::IndexType edemaIndex = edemaIndices[j];
      double distance = std::sqrt(std::pow(index[0] - edemaIndex[0], 2) + std::pow(index[1] - edemaIndex[1], 2) + std::pow(index[2] - edemaIndex[2], 2));
      edemaDistances.push_back(distance);
    }
    double min_distance_tumor = *std::min_element(std::begin(tumorDistances), std::end(tumorDistances));
    double min_distance_edema = *std::min_element(std::begin(edemaDistances), std::end(edemaDistances));
    VentEdemaDistances.push_back(min_distance_edema);
    VentTumorDistances.push_back(min_distance_tumor);
  }

  std::vector<double> DistanceFeatures;
  std::sort(VentEdemaDistances.begin(), VentEdemaDistances.end());
  std::sort(VentTumorDistances.begin(), VentTumorDistances.end());

  VentTumorDistances.resize(100);
  VentEdemaDistances.resize(100);

  double min_distance_tumor = *std::min_element(std::begin(VentTumorDistances), std::end(VentTumorDistances));
  double min_distance_edema = *std::min_element(std::begin(VentEdemaDistances), std::end(VentEdemaDistances));
  DistanceFeatures.push_back(min_distance_tumor);
  DistanceFeatures.push_back(min_distance_edema);
  /*



  std::vector<double> edemaDistances;

  for (int i = 0; i < ventIndices.size(); i++)
  {
  ImageType::IndexType ventIndex = ventIndices[i];
  for (unsigned int j = 0; j < tumorIndices.size(); j++)
  {
  ImageType::IndexType tumorIndex = tumorIndices[j];
  double distance = std::sqrt(std::pow(ventIndex[0] - tumorIndex[0], 2) + std::pow(ventIndex[1] - tumorIndex[1], 2) + std::pow(ventIndex[2] - tumorIndex[2], 2));
  tumorDistances.push_back(distance);
  }
  for (unsigned int j = 0; j < edemaIndices.size(); j++)
  {
  ImageType::IndexType edemaIndex = edemaIndices[j];
  double distance = std::sqrt(std::pow(ventIndex[0] - edemaIndex[0], 2) + std::pow(ventIndex[1] - edemaIndex[1], 2) + std::pow(ventIndex[2] - edemaIndex[2], 2));
  edemaDistances.push_back(distance);
  }
  }



  *///DistanceFeatures.push_back(tumorDistances[std::distance(std::begin(tumorDistances), result)]);

  //std::vector<int>::iterator result = std::min_element(std::begin(edemaDistances), std::end(edemaDistances));
  //DistanceFeatures.push_back(edemaDistances[std::distance(std::begin(edemaDistances), result)]);
  return DistanceFeatures;
}
//--------------------------------------------------------------------------------------

template<class ImageType>
std::vector<double> SurvivalPredictor::GetDistanceFeatures3(typename ImageType::Pointer edemaImage, typename ImageType::Pointer tumorImage, typename ImageType::Pointer ventImage)
{
  //-----------------------create distance image----------------------------
  std::vector<double> VentEdemaSum;
  std::vector<double> VentTumorSum;
  //-------------------------get sum of images----------------------------------
  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType tumorIt(tumorImage, tumorImage->GetLargestPossibleRegion());
  IteratorType ventIt(ventImage, ventImage->GetLargestPossibleRegion());
  IteratorType edemaIt(edemaImage, edemaImage->GetLargestPossibleRegion());

  tumorIt.GoToBegin();
  ventIt.GoToBegin();
  edemaIt.GoToBegin();

  while (!tumorIt.IsAtEnd())
  {
    VentEdemaSum.push_back(edemaIt.Get() + ventIt.Get());
    VentTumorSum.push_back(tumorIt.Get() + ventIt.Get());
    ++tumorIt;
    ++ventIt;
    ++edemaIt;
  }
  std::sort(VentEdemaSum.begin(),VentEdemaSum.end());
  std::sort(VentTumorSum.begin(), VentTumorSum.end());

  double sumVentEdema = 0;
  double sumVentTumor = 0;
  for (int i = 0; i < 100; i++)
  {
    sumVentEdema = sumVentEdema + VentEdemaSum[i];
    sumVentTumor = sumVentTumor + VentTumorSum[i];
  }
  std::vector<double> DistanceFeatures;
  DistanceFeatures.push_back(sumVentEdema / 100);
  DistanceFeatures.push_back(sumVentTumor / 100);

  return DistanceFeatures;
}
//--------------------------------------------------------------------------------------

template<class ImageType>
typename ImageType::Pointer SurvivalPredictor::GetDistanceMap(typename ImageType::Pointer labelImagePointer, int label)
{
  ImageType::Pointer interImage = ImageType::New();
  interImage->CopyInformation(labelImagePointer);
  interImage->SetRequestedRegion(labelImagePointer->GetLargestPossibleRegion());
  interImage->SetBufferedRegion(labelImagePointer->GetBufferedRegion());
  interImage->Allocate();
  interImage->FillBuffer(0);


  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType imIt(labelImagePointer, labelImagePointer->GetLargestPossibleRegion());
  IteratorType interIt(interImage, interImage->GetLargestPossibleRegion());
  imIt.GoToBegin();
  interIt.GoToBegin();
  while (!interIt.IsAtEnd())
  {
    ImageType::IndexType index = imIt.GetIndex();
    if (label == 200)
    {
      if (imIt.Get() == label || imIt.Get() == 175)
        interIt.Set(1);
      else
        interIt.Set(0);
    }
    else
    {
      if (imIt.Get() == label)
        interIt.Set(1);
      else
        interIt.Set(0);
    }
    ++interIt;
    ++imIt;
  }
  typedef itk::Image<unsigned char, 3>  UnsignedCharImageType;
  typedef itk::CastImageFilter< ImageType, UnsignedCharImageType > CastFilterType;
  CastFilterType::Pointer castFilter = CastFilterType::New();
  castFilter->SetInput(interImage);
  typedef  itk::SignedMaurerDistanceMapImageFilter< UnsignedCharImageType, ImageType  > SignedMaurerDistanceMapImageFilterType;
  SignedMaurerDistanceMapImageFilterType::Pointer distanceMapImageFilter = SignedMaurerDistanceMapImageFilterType::New();
  distanceMapImageFilter->SetInput(castFilter->GetOutput());
  distanceMapImageFilter->Update();
  ImageType::Pointer distanceimage = distanceMapImageFilter->GetOutput();
  return distanceimage;
}
//---------------------------------------------------------------------------------------

template<class ImageType>
std::vector<double>  SurvivalPredictor::LoadTestData(typename ImageType::Pointer t1ceImagePointer,
  typename ImageType::Pointer t2flairImagePointer,
  typename ImageType::Pointer t1ImagePointer,
  typename ImageType::Pointer t2ImagePointer,
  typename ImageType::Pointer rcbvImagePointer,
  typename ImageType::Pointer psrImagePointer,
  typename ImageType::Pointer phImagePointer,
  typename ImageType::Pointer axImagePointer,
  typename ImageType::Pointer faImagePointer,
  typename ImageType::Pointer radImagePointer,
  typename ImageType::Pointer trImagePointer,
  typename ImageType::Pointer labelImagePointer)
{
  std::vector<ImageType::IndexType> edemaIndices;
  std::vector<ImageType::IndexType> etumorIndices;
  std::vector<ImageType::IndexType> tumorIndices;
  std::vector<ImageType::IndexType> necoreIndices;
  std::vector<ImageType::IndexType> brainIndices;
  std::vector<ImageType::IndexType> ventIndices;


  std::vector<double> tumorIntensitiesT1;
  std::vector<double> tumorIntensitiesT2;
  std::vector<double> tumorIntensitiesT1CE;
  std::vector<double> tumorIntensitiesT2Flair;
  std::vector<double> tumorIntensitiesAX;
  std::vector<double> tumorIntensitiesFA;
  std::vector<double> tumorIntensitiesRAD;
  std::vector<double> tumorIntensitiesTR;
  std::vector<double> tumorIntensitiesRCBV;
  std::vector<double> tumorIntensitiesPSR;
  std::vector<double> tumorIntensitiesPH;

  std::vector<double> edemaIntensitiesT1;
  std::vector<double> edemaIntensitiesT2;
  std::vector<double> edemaIntensitiesT1CE;
  std::vector<double> edemaIntensitiesT2Flair;
  std::vector<double> edemaIntensitiesAX;
  std::vector<double> edemaIntensitiesFA;
  std::vector<double> edemaIntensitiesRAD;
  std::vector<double> edemaIntensitiesTR;
  std::vector<double> edemaIntensitiesRCBV;
  std::vector<double> edemaIntensitiesPSR;
  std::vector<double> edemaIntensitiesPH;

  std::vector<double> necoreIntensitiesT1;
  std::vector<double> necoreIntensitiesT2;
  std::vector<double> necoreIntensitiesT1CE;
  std::vector<double> necoreIntensitiesT2Flair;
  std::vector<double> necoreIntensitiesAX;
  std::vector<double> necoreIntensitiesFA;
  std::vector<double> necoreIntensitiesRAD;
  std::vector<double> necoreIntensitiesTR;
  std::vector<double> necoreIntensitiesRCBV;
  std::vector<double> necoreIntensitiesPSR;
  std::vector<double> necoreIntensitiesPH;

  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType imIt(labelImagePointer, labelImagePointer->GetLargestPossibleRegion());
  imIt.GoToBegin();
  while (!imIt.IsAtEnd())
  {
    if (imIt.Get() == 200) // these values are predetermined by GLISTR output
    {
      etumorIndices.push_back(imIt.GetIndex());
      //tumorIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //tumorIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(imIt.GetIndex()));
    }
    else if (imIt.Get() == 175)
    {
      necoreIndices.push_back(imIt.GetIndex());
      //necoreIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //necoreIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(imIt.GetIndex()));
    }
    else if (imIt.Get() == 100)
    {
      edemaIndices.push_back(imIt.GetIndex());
      //edemaIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(imIt.GetIndex())));
      //edemaIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(imIt.GetIndex()));
    }
    else if (imIt.Get() == 10)
      ventIndices.push_back(imIt.GetIndex());
    else
    {
    }
    if (imIt.Get() > 0)
      brainIndices.push_back(imIt.GetIndex());
    if (imIt.Get() == 200 || imIt.Get() == 175)
      tumorIndices.push_back(imIt.GetIndex());
    ++imIt;
  }
  for (unsigned int i = 0; i < etumorIndices.size(); i++)
  {
    tumorIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(etumorIndices[i])));
    tumorIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(etumorIndices[i]));
  }
  
  for (unsigned int i = 0; i < edemaIndices.size(); i++)
  {
    edemaIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(edemaIndices[i])));
    edemaIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(edemaIndices[i]));
  }
  
  for (unsigned int i = 0; i < necoreIndices.size(); i++)
  {
    necoreIntensitiesT1.push_back(std::round(t1ImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesT2.push_back(std::round(t2ImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesT1CE.push_back(std::round(t1ceImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesT2Flair.push_back(std::round(t2flairImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesAX.push_back(std::round(axImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesFA.push_back(std::round(faImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesRAD.push_back(std::round(radImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesTR.push_back(std::round(trImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesRCBV.push_back(std::round(rcbvImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesPSR.push_back(std::round(psrImagePointer.GetPointer()->GetPixel(necoreIndices[i])));
    necoreIntensitiesPH.push_back(phImagePointer.GetPointer()->GetPixel(necoreIndices[i]));
  }

  //-------------------------------Find connected components and get volumetric features -----------------------------
  std::vector<ImageType::Pointer> RevisedImages = RevisedTumorArea<ImageType>(labelImagePointer);
  ImageType::Pointer etumorImage = RevisedImages[1];
  ImageType::Pointer ncrImage = RevisedImages[2];

  IteratorType etIt(etumorImage, etumorImage->GetLargestPossibleRegion());
  IteratorType ncrIt(ncrImage, ncrImage->GetLargestPossibleRegion());
  imIt.GoToBegin();
  ncrIt.GoToBegin();
  etIt.GoToBegin();

  while (!imIt.IsAtEnd())
  {
    if (imIt.Get() == 200 && etIt.Get() == 0)
      imIt.Set(0);
    if (imIt.Get() == 175 && ncrIt.Get() == 0)
      imIt.Set(0);
    ++imIt;
    ++ncrIt;
    ++etIt;
  }

  //for (size_t i = 0; i < etumorIndices.size(); i++)
  //{
  //  if ((labelImagePointer.GetPointer()->GetPixel(etumorIndices[i]) == 200) &&
  //    (etumorImage.GetPointer()->GetPixel(etumorIndices[i]) == 0))
  //  {
  //    labelImagePointer.GetPointer()->SetPixel(etumorIndices[i], 0);
  //  }
  //}
  //for (size_t i = 0; i < necoreIndices.size(); i++)
  //{
  //  if ((labelImagePointer.GetPointer()->GetPixel(necoreIndices[i]) == 175) &&
  //    (ncrImage.GetPointer()->GetPixel(necoreIndices[i]) == 0))
  //  {
  //    labelImagePointer.GetPointer()->SetPixel(etumorIndices[i], 0);
  //  }
  //}

  imIt.GoToBegin();
  while (!imIt.IsAtEnd())
  {
    if (imIt.Get() == 200)
      etumorIndices.push_back(imIt.GetIndex());
    else if (imIt.Get() == 175)
      necoreIndices.push_back(imIt.GetIndex());
    else if (imIt.Get() == 100)
      edemaIndices.push_back(imIt.GetIndex());
    else if (imIt.Get() == 10)
      ventIndices.push_back(imIt.GetIndex());
    else
    {
    }
    if (imIt.Get() > 0)
      brainIndices.push_back(imIt.GetIndex());
    if (imIt.Get() == 200 || imIt.Get() == 175)
      tumorIndices.push_back(imIt.GetIndex());
    ++imIt;
  }
  std::vector<double> VolumetricFeatures = GetVolumetricFeatures(edemaIndices.size(), etumorIndices.size(), necoreIndices.size(), brainIndices.size());
  //--------------------------------Find distance features----------------------------------------------
 // std::vector<double> DistanceFeatures = GetDistanceFeatures2<ImageType>(labelImagePointer, edemaIndices, tumorIndices, ventIndices);
 
  //--------------------------------Alternate function for distance features----------------------------------------------  
  int edemalabel = 100;
  int etumorlabel = 200;
  int ventlabel = 10;

  ImageType::Pointer  edemaDistanceMap  = GetDistanceMap<ImageType>(labelImagePointer, edemalabel);
  ImageType::Pointer tumorDistanceMap   = GetDistanceMap<ImageType>(labelImagePointer, etumorlabel);
  ImageType::Pointer ventDistanceMap    = GetDistanceMap<ImageType>(labelImagePointer, ventlabel);

  std::vector<double> DistanceFeatures = GetDistanceFeatures3<ImageType>(edemaDistanceMap, tumorDistanceMap, ventDistanceMap);
  //BIN_ET_PSR = 0:84 : 255;
  //BIN_ET_PH = 0:100 : 255;
  //BIN_ET_RCBV = 0:98 : 255;
  //BIN_ET_FA = 0:70 : 137;
  //BIN_ET_AX = 10:100 : 209;
  //BIN_ET_RAD = 14:100 : 193;
  //BIN_ET_ADC = 13:100 : 200;
  //BIN_ET_T1CE = 27:92 : 206;
  //BIN_ET_T1 = 51:98 : 198;
  //BIN_ET_T2 = 44:98 : 227;
  //BIN_ET_Flair = 11:70 : 218;

  std::vector<double> TumorBinsT1     = GetHistogramFeatures(tumorIntensitiesT1, 51, 98, 198);
  std::vector<double> TumorBinsT2       = GetHistogramFeatures(tumorIntensitiesT2, 44, 98, 227);
  std::vector<double> TumorBinsT1CE     = GetHistogramFeatures(tumorIntensitiesT1CE, 27, 92, 206);
  std::vector<double> TumorBinsT2Flair = GetHistogramFeatures(tumorIntensitiesT2Flair, 11, 70, 218);
  std::vector<double> TumorBinsAX = GetHistogramFeatures(tumorIntensitiesAX, 10, 100, 209);
  std::vector<double> TumorBinsFA = GetHistogramFeatures(tumorIntensitiesFA, 0, 70, 137);
  std::vector<double> TumorBinsRAD = GetHistogramFeatures(tumorIntensitiesRAD, 14, 100, 193);
  std::vector<double> TumorBinsTR = GetHistogramFeatures(tumorIntensitiesTR, 13, 100, 200);
  std::vector<double> TumorBinsRCBV = GetHistogramFeatures(tumorIntensitiesRCBV, 0, 98, 255);
  std::vector<double> TumorBinsPSR = GetHistogramFeatures(tumorIntensitiesPSR, 0, 84, 255);
  std::vector<double> TumorBinsPH = GetHistogramFeatures(tumorIntensitiesPH, 0, 100, 255);

  //BIN_Edema_PSR = 10:25 : 255;
  //BIN_Edema_PH = 20:32 : 255;
  //BIN_Edema_RCBV = 0:100 : 220;
  //BIN_Edema_FA = 0:90 : 175;
  //BIN_Edema_AX = 10:100 : 195;
  //BIN_Edema_RAD = 14:100 : 192;
  //BIN_Edema_ADC = 13:100 : 200;
  //BIN_Edema_T1CE = 29:36 : 164;
  //BIN_Edema_T1 = 51:14 : 163;
  //BIN_Edema_T2 = 30:24 : 222;
  //BIN_Edema_Flair = 13:20 : 174;

  std::vector<double> edemaBinsT1 = GetHistogramFeatures(edemaIntensitiesT1, 51, 14, 163);
  std::vector<double> edemaBinsT2 = GetHistogramFeatures(edemaIntensitiesT2, 30, 24, 222);
  std::vector<double> edemaBinsT1CE = GetHistogramFeatures(edemaIntensitiesT1CE, 29, 36, 164);
  std::vector<double> edemaBinsT2Flair = GetHistogramFeatures(edemaIntensitiesT2Flair, 13, 20, 174);
  std::vector<double> edemaBinsAX = GetHistogramFeatures(edemaIntensitiesAX, 10, 100, 195);
  std::vector<double> edemaBinsFA = GetHistogramFeatures(edemaIntensitiesFA, 0, 90, 175);
  std::vector<double> edemaBinsRAD = GetHistogramFeatures(edemaIntensitiesRAD, 14, 100, 192);
  std::vector<double> edemaBinsTR = GetHistogramFeatures(edemaIntensitiesTR, 13, 100, 200);
  std::vector<double> edemaBinsRCBV = GetHistogramFeatures(edemaIntensitiesRCBV, 0, 100, 220);
  std::vector<double> edemaBinsPSR = GetHistogramFeatures(edemaIntensitiesPSR, 10, 25, 255);
  std::vector<double> edemaBinsPH = GetHistogramFeatures(edemaIntensitiesPH, 20, 32, 255);


  //BIN_NCR_PSR = 10:100 : 255;
  //BIN_NCR_PH = 10:88 : 255;
  //BIN_NCR_RCBV = 0:100 : 255;
  //BIN_NCR_FA = 0:98 : 130;
  //BIN_NCR_AX = 30:72 : 221;
  //BIN_NCR_RAD = 27:60 : 205;
  //BIN_NCR_ADC = 28:72 : 208;
  //BIN_NCR_T1CE = 22:84 : 186;
  //BIN_NCR_T1 = 45:94 : 156;
  //BIN_NCR_T2 = 49:94 : 236;
  //BIN_NCR_Flair = 17:32 : 235;

  std::vector<double> necoreBinsT1 = GetHistogramFeatures(necoreIntensitiesT1, 45, 94, 156);
  std::vector<double> necoreBinsT2 = GetHistogramFeatures(necoreIntensitiesT2, 49, 94, 236);
  std::vector<double> necoreBinsT1CE = GetHistogramFeatures(necoreIntensitiesT1CE, 22, 84, 186);
  std::vector<double> necoreBinsT2Flair = GetHistogramFeatures(necoreIntensitiesT2Flair, 17, 32, 235);
  std::vector<double> necoreBinsAX = GetHistogramFeatures(necoreIntensitiesAX, 30, 72, 221);
  std::vector<double> necoreBinsFA = GetHistogramFeatures(necoreIntensitiesFA, 0, 98, 130);
  std::vector<double> necoreBinsRAD = GetHistogramFeatures(necoreIntensitiesRAD, 27, 60, 205);
  std::vector<double> necoreBinsTR = GetHistogramFeatures(necoreIntensitiesTR, 28, 72, 208);
  std::vector<double> necoreBinsRCBV = GetHistogramFeatures(necoreIntensitiesRCBV, 0, 100, 255);
  std::vector<double> necoreBinsPSR = GetHistogramFeatures(necoreIntensitiesPSR, 10, 100, 255);
  std::vector<double> necoreBinsPH = GetHistogramFeatures(necoreIntensitiesPH, 10, 88, 255);

  std::vector<double> TumorStatisticsT1 = GetStatisticalFeatures(tumorIntensitiesT1);
  std::vector<double> TumorStatisticsT2 = GetStatisticalFeatures(tumorIntensitiesT2);
  std::vector<double> TumorStatisticsT1CE = GetStatisticalFeatures(tumorIntensitiesT1CE);
  std::vector<double> TumorStatisticsT2Flair = GetStatisticalFeatures(tumorIntensitiesT2Flair);
  std::vector<double> TumorStatisticsAX = GetStatisticalFeatures(tumorIntensitiesAX);
  std::vector<double> TumorStatisticsFA = GetStatisticalFeatures(tumorIntensitiesFA);
  std::vector<double> TumorStatisticsRAD = GetStatisticalFeatures(tumorIntensitiesRAD);
  std::vector<double> TumorStatisticsTR = GetStatisticalFeatures(tumorIntensitiesTR);
  std::vector<double> TumorStatisticsRCBV = GetStatisticalFeatures(tumorIntensitiesRCBV);
  std::vector<double> TumorStatisticsPSR = GetStatisticalFeatures(tumorIntensitiesPSR);
  std::vector<double> TumorStatisticsPH = GetStatisticalFeatures(tumorIntensitiesPH);

  std::vector<double> edemaStatisticsT1 = GetStatisticalFeatures(edemaIntensitiesT1);
  std::vector<double> edemaStatisticsT2 = GetStatisticalFeatures(edemaIntensitiesT2);
  std::vector<double> edemaStatisticsT1CE = GetStatisticalFeatures(edemaIntensitiesT1CE);
  std::vector<double> edemaStatisticsT2Flair = GetStatisticalFeatures(edemaIntensitiesT2Flair);
  std::vector<double> edemaStatisticsAX = GetStatisticalFeatures(edemaIntensitiesAX);
  std::vector<double> edemaStatisticsFA = GetStatisticalFeatures(edemaIntensitiesFA);
  std::vector<double> edemaStatisticsRAD = GetStatisticalFeatures(edemaIntensitiesRAD);
  std::vector<double> edemaStatisticsTR = GetStatisticalFeatures(edemaIntensitiesTR);
  std::vector<double> edemaStatisticsRCBV = GetStatisticalFeatures(edemaIntensitiesRCBV);
  std::vector<double> edemaStatisticsPSR = GetStatisticalFeatures(edemaIntensitiesPSR);
  std::vector<double> edemaStatisticsPH = GetStatisticalFeatures(edemaIntensitiesPH);

  std::vector<double> necoreStatisticsT1 = GetStatisticalFeatures(necoreIntensitiesT1);
  std::vector<double> necoreStatisticsT2 = GetStatisticalFeatures(necoreIntensitiesT2);
  std::vector<double> necoreStatisticsT1CE = GetStatisticalFeatures(necoreIntensitiesT1CE);
  std::vector<double> necoreStatisticsT2Flair = GetStatisticalFeatures(necoreIntensitiesT2Flair);
  std::vector<double> necoreStatisticsAX = GetStatisticalFeatures(necoreIntensitiesAX);
  std::vector<double> necoreStatisticsFA = GetStatisticalFeatures(necoreIntensitiesFA);
  std::vector<double> necoreStatisticsRAD = GetStatisticalFeatures(necoreIntensitiesRAD);
  std::vector<double> necoreStatisticsTR = GetStatisticalFeatures(necoreIntensitiesTR);
  std::vector<double> necoreStatisticsRCBV = GetStatisticalFeatures(necoreIntensitiesRCBV);
  std::vector<double> necoreStatisticsPSR = GetStatisticalFeatures(necoreIntensitiesPSR);
  std::vector<double> necoreStatisticsPH = GetStatisticalFeatures(necoreIntensitiesPH);

  std::vector<double> TestFeatures;

  TestFeatures.insert(TestFeatures.end(), TumorStatisticsT1CE.begin(), TumorStatisticsT1CE.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsT1.begin(), TumorStatisticsT1.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsT2.begin(), TumorStatisticsT2.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsT2Flair.begin(), TumorStatisticsT2Flair.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsPH.begin(), TumorStatisticsPH.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsPSR.begin(), TumorStatisticsPSR.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsRCBV.begin(), TumorStatisticsRCBV.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsFA.begin(), TumorStatisticsFA.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsAX.begin(), TumorStatisticsAX.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsRAD.begin(), TumorStatisticsRAD.end());
  TestFeatures.insert(TestFeatures.end(), TumorStatisticsTR.begin(), TumorStatisticsTR.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsT1CE.begin(), TumorBinsT1CE.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsT1CE.begin(), edemaBinsT1CE.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsT1CE.begin(), necoreBinsT1CE.end());

  TestFeatures.insert(TestFeatures.end(), TumorBinsT1.begin(), TumorBinsT1.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsT1.begin(), edemaBinsT1.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsT1.begin(), necoreBinsT1.end());

  TestFeatures.insert(TestFeatures.end(), TumorBinsT2.begin(), TumorBinsT2.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsT2.begin(), edemaBinsT2.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsT2.begin(), necoreBinsT2.end());

  TestFeatures.insert(TestFeatures.end(), TumorBinsT2Flair.begin(), TumorBinsT2Flair.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsT2Flair.begin(), edemaBinsT2Flair.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsT2Flair.begin(), necoreBinsT2Flair.end());

  TestFeatures.insert(TestFeatures.end(), TumorBinsPH.begin(), TumorBinsPH.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsPH.begin(), edemaBinsPH.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsPH.begin(), necoreBinsPH.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsPSR.begin(), TumorBinsPSR.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsPSR.begin(), edemaBinsPSR.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsPSR.begin(), necoreBinsPSR.end());

  TestFeatures.insert(TestFeatures.end(), TumorBinsPSR.begin(), TumorBinsPSR.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsPSR.begin(), edemaBinsPSR.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsPSR.begin(), necoreBinsPSR.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsRCBV.begin(), TumorBinsRCBV.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsRCBV.begin(), edemaBinsRCBV.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsRCBV.begin(), necoreBinsRCBV.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsFA.begin(), TumorBinsFA.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsFA.begin(), edemaBinsFA.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsFA.begin(), necoreBinsFA.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsAX.begin(), TumorBinsAX.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsAX.begin(), edemaBinsAX.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsAX.begin(), necoreBinsAX.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsRAD.begin(), TumorBinsRAD.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsRAD.begin(), edemaBinsRAD.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsRAD.begin(), necoreBinsRAD.end());


  TestFeatures.insert(TestFeatures.end(), TumorBinsTR.begin(), TumorBinsTR.end());
  TestFeatures.insert(TestFeatures.end(), edemaBinsTR.begin(), edemaBinsTR.end());
  TestFeatures.insert(TestFeatures.end(), necoreBinsTR.begin(), necoreBinsTR.end());




  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsT1.begin(), edemaStatisticsT1.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsT1CE.begin(), edemaStatisticsT1CE.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsT2.begin(), edemaStatisticsT2.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsT2Flair.begin(), edemaStatisticsT2Flair.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsAX.begin(), edemaStatisticsAX.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsFA.begin(), edemaStatisticsFA.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsRAD.begin(), edemaStatisticsRAD.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsTR.begin(), edemaStatisticsTR.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsPSR.begin(), edemaStatisticsPSR.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsPH.begin(), edemaStatisticsPH.end());
  //TestFeatures.insert(TestFeatures.end(), edemaStatisticsRCBV.begin(), edemaStatisticsRCBV.end());

  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsT1.begin(), necoreStatisticsT1.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsT1CE.begin(), necoreStatisticsT1CE.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsT2.begin(), necoreStatisticsT2.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsT2Flair.begin(), necoreStatisticsT2Flair.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsAX.begin(), necoreStatisticsAX.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsFA.begin(), necoreStatisticsFA.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsRAD.begin(), necoreStatisticsRAD.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsTR.begin(), necoreStatisticsTR.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsPSR.begin(), necoreStatisticsPSR.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsPH.begin(), necoreStatisticsPH.end());
  //TestFeatures.insert(TestFeatures.end(), necoreStatisticsRCBV.begin(), necoreStatisticsRCBV.end());

  TestFeatures.insert(TestFeatures.end(), VolumetricFeatures.begin(), VolumetricFeatures.end());
  TestFeatures.insert(TestFeatures.end(), DistanceFeatures.begin(), DistanceFeatures.end());
  return TestFeatures;
}


template<class ImageType>
std::vector<typename ImageType::Pointer> SurvivalPredictor::RevisedTumorArea(typename ImageType::Pointer labelImagePointer)
{
  std::vector<ImageType::Pointer> RevisedImages;

  ImageType::Pointer tumorImage = ImageType::New();
  tumorImage->CopyInformation(labelImagePointer);
  tumorImage->SetRequestedRegion(labelImagePointer->GetLargestPossibleRegion());
  tumorImage->SetBufferedRegion(labelImagePointer->GetBufferedRegion());
  tumorImage->Allocate();
  tumorImage->FillBuffer(0);

  ImageType::Pointer etumorImage = ImageType::New();
  etumorImage->CopyInformation(labelImagePointer);
  etumorImage->SetRequestedRegion(labelImagePointer->GetLargestPossibleRegion());
  etumorImage->SetBufferedRegion(labelImagePointer->GetBufferedRegion());
  etumorImage->Allocate();
  etumorImage->FillBuffer(0);

  ImageType::Pointer ncrImage = ImageType::New();
  ncrImage->CopyInformation(labelImagePointer);
  ncrImage->SetRequestedRegion(labelImagePointer->GetLargestPossibleRegion());
  ncrImage->SetBufferedRegion(labelImagePointer->GetBufferedRegion());
  ncrImage->Allocate();
  ncrImage->FillBuffer(0);


  typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
  IteratorType imIt(labelImagePointer, labelImagePointer->GetLargestPossibleRegion());
  IteratorType ncrIt(ncrImage, ncrImage->GetLargestPossibleRegion());
  IteratorType etIt(etumorImage, etumorImage->GetLargestPossibleRegion());
  IteratorType tumorIt(tumorImage, tumorImage->GetLargestPossibleRegion());
  imIt.GoToBegin();
  tumorIt.GoToBegin();
  etIt.GoToBegin();
  ncrIt.GoToBegin();

  while (!tumorIt.IsAtEnd())
  {
    if (imIt.Get() == 200 || imIt.Get() == 175)
      tumorIt.Set(1);
    else
      tumorIt.Set(0);

    if (imIt.Get() == 200)
      etIt.Set(1);
    else
      etIt.Set(0);

    if (imIt.Get() == 175)
      ncrIt.Set(1);
    else
      ncrIt.Set(0);

    ++tumorIt;
    ++etIt;
    ++imIt;
    ++ncrIt;
  }

  typedef itk::Image< unsigned short, 3 > OutputImageType;

  typedef itk::ConnectedComponentImageFilter <ImageType, OutputImageType> ConnectedComponentImageFilterType;
  ConnectedComponentImageFilterType::Pointer connected = ConnectedComponentImageFilterType::New();
  connected->FullyConnectedOn();
  connected->SetInput(tumorImage);
  connected->Update();
  OutputImageType::Pointer labeledImage = connected->GetOutput();

  connected->GetObjectCount();
  std::vector<int> sizes;
  typedef itk::ImageRegionIteratorWithIndex <OutputImageType> OutputIteratorType;
  OutputIteratorType lbimIt(labeledImage, labeledImage->GetLargestPossibleRegion());

  for (int i = 0; i < connected->GetObjectCount(); i++)
  {
    int counter = 0;
    lbimIt.GoToBegin();
    while (!lbimIt.IsAtEnd())
    {
      if (lbimIt.Get() == i + 1)
        counter++;
      ++lbimIt;
    }
    sizes.push_back(counter);
  }
  for (int i = 0; i < connected->GetObjectCount(); i++)
  {
    if (sizes[i] < 100)
    {
      lbimIt.GoToBegin();
      etIt.GoToBegin();
      ncrIt.GoToBegin();
      tumorIt.GoToBegin();
      while (!lbimIt.IsAtEnd())
      {
        if (lbimIt.Get() == i + 1)
        {
          tumorIt.Set(0);
          etIt.Set(0);
          ncrIt.Set(0);
        }
        ++lbimIt;
        ++tumorIt;
        ++ncrIt;
        ++etIt;
      }
    }
  }
  RevisedImages.push_back(tumorImage);
  RevisedImages.push_back(etumorImage);
  RevisedImages.push_back(ncrImage);

  return RevisedImages;
}


#endif





